/** @file scim_panel_gtk.cpp
 */

/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2002-2005 James Su <suzhe@tsinghua.org.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * Evan JIANG	<firstfan@gmail.com>	2009/12/07
 */

#include <sys/wait.h>
#include <sys/types.h>
#include <sys/time.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <glib.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <X11/X.h>
#include <X11/Xatom.h>
#include <X11/Xlib.h>
#include <gtk/gtk.h>
#include <stdlib.h>
#include <list>
#include <hildon/hildon.h>


#define Uses_C_STDIO
#define Uses_C_STDLIB
#define Uses_SCIM_LOOKUP_TABLE
#define Uses_SCIM_SOCKET
#define Uses_SCIM_TRANSACTION
#define Uses_SCIM_TRANS_COMMANDS
#define Uses_SCIM_CONFIG
#define Uses_SCIM_CONFIG_MODULE
#define Uses_SCIM_DEBUG
#define Uses_SCIM_HELPER
#define Uses_SCIM_HELPER_MODULE
#define Uses_SCIM_PANEL_AGENT

#include "scim_private.h"
#include "scim.h"
#include "scim_stl_map.h"

#include "scimstringview.h"

#include "scimtrayicon.h"

using namespace scim;

/*
#include "icons/up.xpm"
#include "icons/down.xpm"
#include "icons/left.xpm"
#include "icons/right.xpm"
#include "icons/setup.xpm"
#include "icons/help.xpm"
#include "icons/trademark.xpm"
#include "icons/pin-up.xpm"
#include "icons/pin-down.xpm"
#include "icons/menu.xpm"
*/
#define SCIM_CONFIG_PANEL_GTK_TOOLBAR_POS_X             "/Panel/Gtk/ToolBar/POS_X"
#define SCIM_CONFIG_PANEL_GTK_TOOLBAR_POS_Y             "/Panel/Gtk/ToolBar/POS_Y"
#define SCIM_CONFIG_PANEL_GTK_DEFAULT_STICKED           "/Panel/Gtk/DefaultSticked"
#define SCIM_CONFIG_PANEL_GTK_SHOW_TRAY_ICON            "/Panel/Gtk/ShowTrayIcon"

#define SCIM_KEYBOARD_ICON_FILE     (SCIM_ICONDIR "/keyboard.png")
#define SCIM_TRADEMARK_ICON_FILE    (SCIM_ICONDIR "/trademark.png")
#define SCIM_SETUP_ICON_FILE        (SCIM_ICONDIR "/setup.png")
#define SCIM_HELP_ICON_FILE         (SCIM_ICONDIR "/help.png")
#define SCIM_MENU_ICON_FILE         (SCIM_ICONDIR "/menu.png")
#define SCIM_UP_ICON_FILE           (SCIM_ICONDIR "/up.png")
#define SCIM_DOWN_ICON_FILE         (SCIM_ICONDIR "/down.png")
#define SCIM_LEFT_ICON_FILE         (SCIM_ICONDIR "/left.png")
#define SCIM_RIGHT_ICON_FILE        (SCIM_ICONDIR "/right.png")
#define SCIM_PIN_UP_ICON_FILE       (SCIM_ICONDIR "/pin-up.png")
#define SCIM_PIN_DOWN_ICON_FILE     (SCIM_ICONDIR "/pin-down.png")

#define LOOKUP_ICON_SIZE                      36
#define TRAY_ICON_SIZE                        18
#define FACTORY_ICON_SIZE                     30


struct PropertyInfo {
    Property   property;
    GtkWidget *widget;

    PropertyInfo () : widget (0) { }
};

typedef std::vector <PropertyInfo>               PropertyRepository;


static void       ui_initialize                        (void);
static void       ui_config_reload_callback            (const ConfigPointer &config);

static bool       initialize_panel_agent               (const String &config, const String &display, bool resident);
static bool       run_panel_agent                      (void);
static gpointer   panel_agent_thread_func              (gpointer data);


static void       ui_settle_input_window               (bool            relative = false,
                                                        bool            force    = false);
static void       ui_lookup_table_up_button_click_cb   (GtkButton      *button,
                                                        gpointer        user_data);
static void       ui_lookup_table_down_button_click_cb (GtkButton      *button,
                                                        gpointer        user_data);
static gboolean   ui_factory_button_click_cb           (GtkWidget      *button,
                                                        GdkEvent       *event,
                                                        gpointer        user_data);
static void       ui_factory_menu_activate_cb          (GtkMenuItem    *item,
                                                        gpointer        user_data);
static void       ui_factory_menu_deactivate_cb        (GtkMenuItem    *item,
                                                        gpointer        user_data);
static void       ui_menu_button_click_cb              (GtkButton      *button,
							gpointer 	user_data);
static GtkWidget* ui_create_factory_menu_entry         (const PanelFactoryInfo &info,
                                                        int                    id,
                                                        GtkMenuShell           *menu);
static GtkWidget * ui_create_icon 			(const String  &iconfile);

static void       slot_transaction_start               (void);
static void       slot_transaction_end                 (void);
static void       slot_reload_config                   (void);
static void       slot_turn_on                         (void);
static void       slot_turn_off                        (void);
static void       slot_update_screen                   (int screen);
static void       slot_update_spot_location            (int x, int y);
static void       slot_update_factory_info             (const PanelFactoryInfo &info);
static void       slot_show_help                       (const String &help);
static void       slot_show_factory_menu               (const std::vector <PanelFactoryInfo> &menu);
                  
static void       slot_show_preedit_string             (void);
static void       slot_show_aux_string                 (void);
static void       slot_show_lookup_table               (void);
static void       slot_hide_preedit_string             (void);
static void       slot_hide_aux_string                 (void);
static void       slot_hide_lookup_table               (void);
static void       slot_update_preedit_string           (const String &str, const AttributeList &attrs);
static void       slot_update_preedit_caret            (int caret);
static void       slot_update_aux_string               (const String &str, const AttributeList &attrs);
static void       slot_update_lookup_table             (const LookupTable &table);
static void       slot_register_properties             (const PropertyList &props);
static void       slot_update_property                 (const Property &prop);
                  
static void       slot_register_helper_properties      (int id, const PropertyList &props);
static void       slot_update_helper_property          (int id, const Property &prop);
static void       slot_register_helper                 (int id, const HelperInfo &helper);
static void       slot_remove_helper                   (int id);
static void       slot_lock                            (void);
static void       slot_unlock                          (void);


static gboolean       ui_lookup_table_click_cb 	       (GtkWidget      *item,
                          				GdkEventButton *event,
                          				gpointer        user_data);


static void       create_properties                    (GtkWidget            *container,
                                                        PropertyRepository &repository,
                                                        const PropertyList   &properties,
                                                        int                   client,
                                                        int                   level);

static GtkWidget* create_properties_node               (PropertyRepository         &repository,
                                                        PropertyList::const_iterator  begin,
                                                        PropertyList::const_iterator  end,
                                                        int                           client,
                                                        int                           level);

static void       register_frontend_properties         (const PropertyList &properties);
static void       update_frontend_property             (const Property     &property);

static void       update_property                      (PropertyRepository &repository,
                                                        const Property       &property);

static void       restore_properties                   (void);


// Client Property Callback
static void       ui_property_activate_cb              (GtkWidget      *widget,
                                                        gpointer        user_data);

static void       ui_property_menu_deactivate_cb       (GtkWidget      *item,
                                                        gpointer        user_data);

//static gboolean   ui_create_tray_icon_when_idle        (gpointer        data);
//static void       ui_tray_icon_destroy_cb              (GtkObject      *object,
//                                                        gpointer        user_data);
#if GDK_MULTIHEAD_SAFE
static void       ui_switch_screen                     (GdkScreen      *screen);
#endif
static int        ui_screen_width                      (void);
static int        ui_screen_height                     (void);

#if GDK_MULTIHEAD_SAFE
static GdkScreen         *_current_screen              = 0;
#endif
static GtkWidget         *_input_window                = 0;
static GtkWidget	 *_input_bin		       = 0;
static GtkWidget	 *_table_box		       = 0;
static GtkWidget         *_text_area                   = 0;
static GtkWidget         *_lookup_table_items [SCIM_LOOKUP_TABLE_MAX_PAGESIZE];
static GtkWidget         *_lookup_table_up_button      = 0;
static GtkWidget         *_lookup_table_down_button    = 0;
static GtkWidget         *_lookup_table_up_arrow       = 0;
static GtkWidget         *_lookup_table_down_arrow     = 0;
static GtkWidget         *_factory_button              = 0;
static GtkWidget	 *_menu_button		       = 0;
static GtkWidget	 *_factory_icon		       = 0;


static guint              _check_exit_timeout          = 0;
static bool               _panel_is_on                 = false;

static ConfigModule      *_config_module               = 0;
static ConfigPointer      _config;

static int                _spot_location_x             = -1;
static int                _spot_location_y             = -1;
static gint               _input_window_x              = 0;
static gint               _input_window_y              = 0;

//static bool               _show_tray_icon              = true;
static bool               _should_exit                 = false;
static GThread           *_panel_agent_thread          = 0;

static PanelAgent        *_panel_agent                 = 0;

static bool              _lookup_table_index [SCIM_LOOKUP_TABLE_MAX_PAGESIZE];
static GString* 	 _preedit_string	       = NULL;
static GString*		 _aux_string                   = NULL;
static GString*		 _input_window_str             = NULL;

static bool               _factory_menu_activated      = false;
static GtkWidget         *_factory_menu                = 0;
static bool               _property_menu_activated     = false;


static GtkWidget         *_client_properties_area      = 0;
static GtkWidget         *_frontend_properties_area    = 0;

static std::vector<String> _factory_menu_uuids;

static struct timeval     _last_menu_deactivate_time = {0, 0};

static PropertyRepository            _frontend_property_repository;
//static gboolean 		 _follow_cursor = false;
static gboolean			 _hide_table	     = false;
G_LOCK_DEFINE_STATIC     (_global_resource_lock);
G_LOCK_DEFINE_STATIC     (_panel_agent_lock);


static void
ui_config_reload_callback (const ConfigPointer &config)
{
    _config = config;
    ui_initialize ();
    //restore_properties ();
}

static void
ui_build_input_window (void)
{
   	_input_window = 0;
   	_input_window = gtk_window_new (GTK_WINDOW_POPUP);
	//_input_window = hildon_window_new ();
	
        //g_object_get_property((GObject*)_input_window,"type",(GValue*)&window_type);
	//g_object_set((GObject*)_input_window,"type",GTK_WINDOW_POPUP,NULL);
        gtk_window_set_resizable (GTK_WINDOW (_input_window), FALSE);	
	//gtk_window_set_opacity(GTK_WINDOW (_input_window),0.5);
	gtk_window_set_accept_focus(GTK_WINDOW (_input_window), FALSE);
	gtk_window_set_decorated(GTK_WINDOW (_input_window), FALSE);
	//gtk_window_set_modal (GTK_WINDOW (_input_window), TRUE);
	/*
	g_object_set_data(G_OBJECT(GTK_WIDGET(_input_window)->window),
			     "_NEW_WM_STATE", (gpointer) PropModeAppend);
	Atom atoms[2];
	Window win=GDK_WINDOW_XID(GTK_WIDGET(_input_window)->window);
	Display *dpy=GDK_DISPLAY();
	atoms[0] = XInternAtom(dpy, "_NET_WM_WINDOW_TYPE", False);
	atoms[1] = XInternAtom(dpy, "_NET_WM_WINDOW_TYPE_INPUT", False);
	XChangeProperty(dpy, win, atoms[0], XA_ATOM, 32, PropModeReplace,
			(unsigned char *) &atoms[1], 1);
	*/

	//gtk_window_set_type_hint (GTK_WINDOW (_input_window),GDK_WINDOW_TYPE_HINT_POPUP_MENU);
	gtk_window_set_type_hint (GTK_WINDOW (_input_window),GDK_WINDOW_TYPE_HINT_DIALOG);
	gtk_container_set_border_width (GTK_CONTAINER (_input_window),HILDON_MARGIN_DEFAULT);
	//gtk_widget_set_name (GTK_WIDGET(_input_window),"hildon-framed-window");
	gtk_widget_set_name (_input_window, "hildon-input-method-ui");
        gtk_widget_set_size_request (GTK_WIDGET(_input_window), ui_screen_width(),-1);
	
	//gtk_window_set_has_frame(GTK_WINDOW (_input_window), TRUE);
	//gtk_window_set_skip_taskbar_hint(GTK_WINDOW (_input_window), TRUE);
	//gtk_window_set_skip_pager_hint(GTK_WINDOW (_input_window), TRUE);
	
	//gtk_widget_modify_bg(_input_window,GTK_STATE_NORMAL,&bg_color);
	//frame = gtk_frame_new (0);
        //gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_OUT);
        //gtk_container_add (GTK_CONTAINER (_input_window), frame);
	//gtk_widget_modify_bg(frame,GTK_STATE_NORMAL,&bg_color);
        //gtk_container_set_border_width (GTK_CONTAINER (_input_window), 2);
}

static void
ui_initialize (void)
{
        GtkWidget *hbox;
        GtkWidget *frame;	
	GtkWidget *image;
	//GdkColor bg_color;

	//_follow_cursor = _config->read (String (SCIM_CONFIG_PANEL_GTK_DEFAULT_STICKED),false);
	//gdk_color_parse("#efcb00",&bg_color);
   	//_show_tray_icon = _config->read (String (SCIM_CONFIG_PANEL_GTK_SHOW_TRAY_ICON),_show_tray_icon);

	ui_build_input_window();
	
	_input_bin = gtk_vbox_new (FALSE, 0); 
	g_object_ref(G_OBJECT(_input_bin));
	gtk_container_add (GTK_CONTAINER (_input_window), _input_bin);
	hbox = gtk_hbox_new (FALSE, 0);


	    _factory_button = gtk_button_new(); //hildon_button_new (HILDON_SIZE_AUTO,HILDON_BUTTON_ARRANGEMENT_VERTICAL);
            gtk_button_set_relief (GTK_BUTTON (_factory_button), GTK_RELIEF_NONE);
	    if (_factory_icon != 0)
		gtk_button_set_image(GTK_BUTTON(_factory_button), _factory_icon);
            //hildon_button_set_alignment((HildonButton*)_factory_button,0,0,1,1);
	    //gtk_widget_set_size_request (_factory_button, FACTORY_ICON_SIZE , FACTORY_ICON_SIZE );
            gtk_box_pack_start (GTK_BOX (hbox), _factory_button, FALSE, FALSE, 0);
            g_signal_connect (G_OBJECT (_factory_button), "button-release-event",
                              G_CALLBACK (ui_factory_button_click_cb),
                              NULL);

	_client_properties_area = gtk_hbox_new (TRUE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), _client_properties_area, FALSE, FALSE, 0);

	_text_area = gtk_label_new("");
	//gtk_widget_set_size_request (_text_area, 600, 40);
	gtk_box_pack_start (GTK_BOX (hbox), _text_area, TRUE, TRUE, 0);

	_menu_button = gtk_button_new();
	gtk_button_set_relief (GTK_BUTTON (_menu_button), GTK_RELIEF_NONE);
	GtkWidget* _menu_icon = ui_create_icon(SCIM_MENU_ICON_FILE);	
	gtk_button_set_image((GtkButton*)_menu_button, _menu_icon);
	/*
	gtk_box_pack_start (GTK_BOX (hbox), _menu_button, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (_factory_button), "button-release-event",
			G_CALLBACK (ui_menu_button_click_cb),
			NULL);
	*/
	    //New left button
            //image = ui_create_left_icon ();
            _lookup_table_up_button = gtk_button_new();
	    _lookup_table_up_arrow =gtk_arrow_new(GTK_ARROW_LEFT,GTK_SHADOW_NONE);
            gtk_button_set_relief (GTK_BUTTON (_lookup_table_up_button), GTK_RELIEF_NONE);
            //hildon_button_set_alignment((HildonButton*)_lookup_table_up_button,0,0,1,1);
            //gtk_container_set_border_width(GTK_CONTAINER (_lookup_table_up_button),0);
            //gtk_widget_set_size_request (_lookup_table_up_button, LOOKUP_ICON_SIZE , LOOKUP_ICON_SIZE );
            gtk_container_add (GTK_CONTAINER (_lookup_table_up_button), _lookup_table_up_arrow);
	    //GtkWidget *event_box;
	    //event_box = gtk_event_box_new ();
	    //_lookup_table_up_button =gtk_arrow_new(GTK_ARROW_LEFT,GTK_SHADOW_NONE);
 	    //gtk_widget_set_size_request (_lookup_table_up_button, LOOKUP_ICON_SIZE, LOOKUP_ICON_SIZE);
            //gtk_container_add (GTK_CONTAINER (event_box), _lookup_table_up_button);
            gtk_box_pack_start (GTK_BOX (hbox), _lookup_table_up_button, FALSE, FALSE, 0);
 	    g_signal_connect (G_OBJECT (_lookup_table_up_button), "button-press-event",
                                G_CALLBACK (ui_lookup_table_up_button_click_cb),
                                NULL);


            //New right button
            //image = ui_create_right_icon ();
            _lookup_table_down_button = gtk_button_new();
            _lookup_table_down_arrow =gtk_arrow_new(GTK_ARROW_RIGHT,GTK_SHADOW_NONE);
            gtk_button_set_relief (GTK_BUTTON (_lookup_table_down_button), GTK_RELIEF_NONE);
            //hildon_button_set_alignment((HildonButton*)_lookup_table_up_button,0,0,1,1);
            //gtk_widget_set_size_request (_lookup_table_down_button, LOOKUP_ICON_SIZE , LOOKUP_ICON_SIZE );
            gtk_container_add (GTK_CONTAINER (_lookup_table_down_button), _lookup_table_down_arrow);
	    //event_box = gtk_event_box_new ();
	    //_lookup_table_down_button =gtk_arrow_new(GTK_ARROW_RIGHT,GTK_SHADOW_NONE);
            //gtk_widget_set_size_request (_lookup_table_down_button, LOOKUP_ICON_SIZE, LOOKUP_ICON_SIZE);
	    //gtk_container_add (GTK_CONTAINER (_lookup_table_down_button), _lookup_table_down_button);
            gtk_box_pack_start (GTK_BOX (hbox), _lookup_table_down_button, FALSE, FALSE, 0);
	    g_signal_connect (G_OBJECT (_lookup_table_down_button), "clicked",
                                G_CALLBACK (ui_lookup_table_down_button_click_cb),
                                NULL);

	gtk_box_pack_start (GTK_BOX (_input_bin), hbox, TRUE, TRUE, 0);
        
	_table_box = gtk_hbox_new (TRUE, 0);
	//gtk_widget_modify_bg(_table_box,GTK_STATE_NORMAL,&bg_color);
	GtkRadioButton *radio_group = NULL;
	for (int i=0; i<SCIM_LOOKUP_TABLE_MAX_PAGESIZE; ++i) {
             _lookup_table_items [i] = hildon_gtk_radio_button_new_from_widget
(
				//	HILDON_SIZE_AUTO_WIDTH | 
					HILDON_SIZE_FINGER_HEIGHT,
				//	HILDON_BUTTON_ARRANGEMENT_VERTICAL);
					radio_group);
	     radio_group = (GtkRadioButton *) _lookup_table_items[i];
	     gtk_toggle_button_set_mode ((GtkToggleButton *)radio_group, FALSE);
	     hildon_helper_set_logical_font(_lookup_table_items [i], "SmallSystemFont");
	     //gtk_widget_set_size_request (_lookup_table_items [i], 120, 80);
	     //hildon_button_set_style ( (HildonButton *)_lookup_table_items [i],HILDON_BUTTON_STYLE_PICKER);
	     g_signal_connect (G_OBJECT (_lookup_table_items [i]), "button-press-event",
                                  G_CALLBACK (ui_lookup_table_click_cb),
                                  GINT_TO_POINTER (i));
	}
	gtk_box_pack_start (GTK_BOX (_table_box), _lookup_table_items [3], TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (_table_box), _lookup_table_items [1], TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (_table_box), _lookup_table_items [0], TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (_table_box), _lookup_table_items [2], TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (_table_box), _lookup_table_items [4], TRUE, TRUE, 0);

        gtk_box_pack_start (GTK_BOX (_input_bin), _table_box, TRUE, TRUE, 0);
	gtk_widget_show_all (_input_window);
        gtk_widget_hide (_input_window);
}

static void
ui_update_input_string ()
{
    g_string_assign(_input_window_str,_aux_string->str);
    g_string_append(_input_window_str,_preedit_string->str);

    gtk_label_set_text((GtkLabel*)_text_area, _input_window_str->str); 
}

static void
ui_settle_input_window (bool relative, bool force)
{
    SCIM_DEBUG_MAIN (2) << " Settle input window...\n";

    GtkRequisition ws;
    gint spot_x, spot_y;

    gtk_widget_size_request (_input_window, &ws);


    spot_x = 0; //(ui_screen_width() - ws.width) / 2; 
    spot_y = _spot_location_y; 
     

//    if (spot_x < 0) spot_x = 0;
//    if (spot_y < 0) spot_y = 0;


    if (spot_y + ws.height > ui_screen_height ())
	    spot_y -= ws.height + 40;
    else
    	spot_y = ui_screen_height () - ws.height - 1;

    ui_update_input_string();        

    if (spot_x != _input_window_x || spot_y != _input_window_y || force) {
        gtk_window_move (GTK_WINDOW (_input_window), spot_x, spot_y);
        _input_window_x = spot_x;
        _input_window_y = spot_y;
    }
    if (force)
    {
	//gtk_window_deiconify(GTK_WINDOW(_input_window));
	gtk_widget_show (_input_window);
	//gtk_window_present(GTK_WINDOW (_input_window));
	gtk_window_set_keep_above (GTK_WINDOW (_input_window), TRUE);
	//gtk_window_stick (GTK_WINDOW (_input_window));
    }
/*
{
    FILE *fileObj = fopen ("/tmp/scim.log", "a");

    fprintf (fileObj, "%s %d %d %d %d\n", _input_window_str->str,
    	 	    spot_x,spot_y,ws.width,ws.height);
//    fprintf(fileObj,"%s %d %d %d\n",_input_window_str->str,
//		    GTK_WIDGET_DRAWABLE(_input_window),
//		    GTK_WIDGET_REALIZED(_input_window),
//		    GTK_IS_WIDGET(_input_window));
    fclose (fileObj);
}
*/
}

static void
ui_lookup_table_up_button_click_cb (GtkButton *button,
                                    gpointer user_data)
{
    SCIM_DEBUG_MAIN (3) << "  ui_lookup_table_up_button_click_cb...\n";

    _panel_agent->lookup_table_page_up ();
}

static void
ui_lookup_table_down_button_click_cb (GtkButton *button,
                                      gpointer user_data)
{
    SCIM_DEBUG_MAIN (3) << "  ui_lookup_table_down_button_click_cb...\n";

    _panel_agent->lookup_table_page_down ();
}

static gboolean
ui_lookup_table_click_cb (GtkWidget      *item,
                          GdkEventButton *event,
                          gpointer        user_data)
{

    uint32 i=(uint32)GPOINTER_TO_INT (user_data);
    if (_lookup_table_index[i])
    	_panel_agent->select_candidate (i);
}


static int
ui_screen_width (void)
{
#if GDK_MULTIHEAD_SAFE
    if (_current_screen)
        return gdk_screen_get_width (_current_screen);
#endif
    return gdk_screen_width ();
}

static int
ui_screen_height (void)
{
#if GDK_MULTIHEAD_SAFE
    if (_current_screen)
        return gdk_screen_get_height (_current_screen);
#endif
    return gdk_screen_height ();
}

#if GDK_MULTIHEAD_SAFE
static void
ui_switch_screen (GdkScreen *screen)
{
    if (screen) {
        if (_input_window) {
            gtk_window_set_screen (GTK_WINDOW (_input_window), screen);

//            _input_window_x = ui_screen_width ();
//            _input_window_y = ui_screen_height ();

//            gtk_window_move (GTK_WINDOW (_input_window), _input_window_x, _input_window_y);
        }


        //if (_tray_icon) {
        //    gtk_window_set_screen (GTK_WINDOW (_tray_icon), screen);
        //}

        ui_settle_input_window ();
    }
}
#endif

static GtkWidget *
ui_create_icon (const String  &iconfile)
{
   String path = iconfile;

    if (path.length ()) {
        if (path [0] != SCIM_PATH_DELIM)
            path = String (SCIM_ICONDIR) + String (SCIM_PATH_DELIM_STRING) + path;
    }

    GtkWidget* icon = gtk_image_new_from_file (path.c_str());
    return icon;
}

static GtkWidget*
ui_create_factory_menu_entry (const PanelFactoryInfo &info,
                              gint                   id,
                              GtkMenuShell           *menu)
{
    GtkWidget *menu_item;
    GtkWidget *icon_image;

    menu_item = gtk_image_menu_item_new_with_label (info.name.c_str ());
    //gtk_icon_size_lookup (GTK_ICON_SIZE_MENU, &width, &height);
    icon_image = ui_create_icon (info.icon);
    //if (icon_image)
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (menu_item), icon_image);

    g_signal_connect (G_OBJECT (menu_item), "activate",
                      G_CALLBACK (ui_factory_menu_activate_cb),
                      GINT_TO_POINTER ((gint)id));
    gtk_widget_show (menu_item);

    gtk_menu_shell_append (GTK_MENU_SHELL (menu), menu_item);

    return menu_item;
}

static gboolean
ui_factory_button_click_cb (GtkWidget *button,
                            GdkEvent  *event,
                            gpointer   user_data)
{

    struct timeval cur_time;
    gettimeofday (&cur_time, 0);

    if (cur_time.tv_sec < _last_menu_deactivate_time.tv_sec ||
        (cur_time.tv_sec == _last_menu_deactivate_time.tv_sec &&
         cur_time.tv_usec < _last_menu_deactivate_time.tv_usec + 200000))
        return FALSE;
	
    _panel_agent->request_factory_menu ();
    return TRUE;
}


static void
ui_factory_menu_activate_cb (GtkMenuItem *item,
                             gpointer     user_data)
{
    int id = GPOINTER_TO_INT (user_data);

    if (id >= 0 && id < (int) _factory_menu_uuids.size ())
        _panel_agent->change_factory (_factory_menu_uuids [id]);
    else
        _panel_agent->change_factory ("");
}

static void
ui_factory_menu_deactivate_cb (GtkMenuItem *item,
                               gpointer     user_data)
{
    _factory_menu_activated = false;
    gettimeofday (&_last_menu_deactivate_time, 0);
}

static void
ui_menu_button_click_cb (GtkButton *button,
		gpointer   user_data)
{
    _panel_agent->reload_config ();

    if (!_config.null ()) _config->reload ();

    struct timeval cur_time;
    gettimeofday (&cur_time, 0);

    if (cur_time.tv_sec < _last_menu_deactivate_time.tv_sec ||
	(cur_time.tv_sec == _last_menu_deactivate_time.tv_sec &&
	 cur_time.tv_usec < _last_menu_deactivate_time.tv_usec + 200000))
	 return;


}


static void
slot_transaction_start (void)
{
    gdk_threads_enter ();
}

static void
slot_transaction_end (void)
{
    gdk_threads_leave ();
}

static void
slot_reload_config (void)
{
    if (!_config.null ()) _config->reload ();
}

static void
slot_turn_on (void)
{
    _panel_is_on = true;
    gtk_widget_hide (_input_window);
}

static void
slot_turn_off (void)
{
    _panel_is_on = false;
    //gtk_widget_hide (_input_window);
    GtkWidget * tmp = _input_window;
    ui_build_input_window(); 
    gtk_container_remove(GTK_CONTAINER(tmp),_input_bin);
    gtk_container_add (GTK_CONTAINER (_input_window), _input_bin);
    gtk_widget_set_size_request (GTK_WIDGET(_input_window), ui_screen_width(),-1);
    gtk_widget_destroy(tmp);
    //gtk_widget_show(_input_window);
    ui_settle_input_window(true,true);
    gtk_widget_hide(_input_window);
}

static void
slot_update_screen (int num)
{
#if GDK_MULTIHEAD_SAFE
    if (gdk_display_get_n_screens (gdk_display_get_default ()) > num) {

        GdkScreen *screen = gdk_display_get_screen (gdk_display_get_default (), num);

        if (screen) {
            _current_screen = screen;
            ui_switch_screen (screen);
        }
    }
#endif
}


static bool
ui_can_hide_input_window (void)
{
    if (!_panel_is_on) return true;

      if (_preedit_string->len || _aux_string->len || _lookup_table_index[0])
        return false;
      return true;
}

static void
slot_update_factory_info (const PanelFactoryInfo &info)
{
    //Change UI for hangul
    if (info.uuid == "d75857a5-4148-4745-89e2-1da7ddaf70a9"
     || info.uuid == "61ea7bfd-d62a-4dd6-842a-c7f91db5d6ff"
     ||	info.uuid == "d6468711-00e1-47a9-bc4d-a3efe64ccd9c"
	)
	    
    {
       if (!_hide_table)
       {
	   gtk_widget_hide(_client_properties_area);
	   gtk_widget_hide(_lookup_table_up_button);
	   gtk_widget_hide(_lookup_table_down_button);
	   gtk_widget_hide(_table_box);
	   _hide_table = true;
       }
    } else {
       if (_hide_table)
       {
           gtk_widget_show_all(_client_properties_area);
           gtk_widget_show_all(_lookup_table_up_button);
           gtk_widget_show_all(_lookup_table_down_button);
           gtk_widget_show_all(_table_box);
	   _hide_table = false;
       }
    }
    _factory_icon = ui_create_icon(info.icon);
//    gtk_widget_set_size_request (icon, HILDON_ICON_SIZE_FINGER , HILDON_ICON_SIZE_FINGER );
/*    GtkWidget * old = gtk_bin_get_child (GTK_BIN (_factory_button));
            if (old)
                gtk_container_remove (GTK_CONTAINER (_factory_button), old);
            gtk_container_add (GTK_CONTAINER (_factory_button), icon);
    //ui_create_icon (info.icon, NULL, HILDON_ICON_SIZE_FINGER, HILDON_ICON_SIZE_FINGER, true);
*/    
    //gtk_widget_set_size_request (icon, FACTORY_ICON_SIZE , FACTORY_ICON_SIZE );
    //gtk_container_add (GTK_CONTAINER (_factory_button), icon);
    gtk_button_set_image((GtkButton*)_factory_button, _factory_icon);
    gtk_widget_show (_factory_button);
}

static void
slot_show_help (const String &help)
{

}

static void
slot_show_factory_menu (const std::vector <PanelFactoryInfo> &factories)
{
    if (!_factory_menu_activated && factories.size ()) {
        size_t i;
	GtkWidget *menu_item;
        PanelFactoryInfo info;
	guint32 activate_time = gtk_get_current_event_time ();
	_factory_menu_uuids.clear ();
        _factory_menu_activated = true;

	if (_factory_menu) {
            gtk_widget_destroy (_factory_menu);
            _factory_menu = 0;
        }

        _factory_menu = hildon_gtk_menu_new ();

	for (i = 0; i < factories.size (); ++i) {
            _factory_menu_uuids.push_back (factories [i].uuid);
	    info = factories [i];

            ui_create_factory_menu_entry (info, i, GTK_MENU_SHELL (_factory_menu));
	}

	info = PanelFactoryInfo (String (""), String (_("English/Keyboard")), String ("C"), String (SCIM_KEYBOARD_ICON_FILE));
        ui_create_factory_menu_entry (info, -1, GTK_MENU_SHELL (_factory_menu));

        g_signal_connect (G_OBJECT (_factory_menu), "deactivate",
                          G_CALLBACK (ui_factory_menu_deactivate_cb),
                          NULL);
        gtk_menu_popup (GTK_MENU (_factory_menu), 0, 0, 0, 0, 1, activate_time);
    }
}

static void
slot_update_spot_location (int x, int y)
{
    if (x > 0 && x < ui_screen_width () && y > 0 && y < ui_screen_height ()) {
        _spot_location_x = x;
	    if (_spot_location_y != y)
	    {
            _spot_location_y = y;
            ui_settle_input_window ();
	    }
    }
}

static void
slot_show_preedit_string (void)
{
    if (_panel_is_on && !GTK_WIDGET_VISIBLE (_input_window))
        gtk_widget_show (_input_window);

    //ui_settle_input_window (true, true);
}

static void
slot_show_aux_string (void)
{
    if (_panel_is_on && !GTK_WIDGET_VISIBLE (_input_window))
        gtk_widget_show (_input_window);

    //ui_settle_input_window (true, true);
}

static void
slot_show_lookup_table (void)
{
    if (_panel_is_on && !GTK_WIDGET_VISIBLE (_input_window)) {
        gtk_widget_show (_input_window);
        ui_settle_input_window (true, true);
    }
}

static void
slot_hide_preedit_string (void)
{
//    scim_string_view_set_text (SCIM_STRING_VIEW (_preedit_area), "");
    g_string_assign(_preedit_string,"");
    if (ui_can_hide_input_window ())
        gtk_widget_hide (_input_window);

}

static void
slot_hide_aux_string (void)
{

//    scim_string_view_set_text (SCIM_STRING_VIEW (_aux_area), "");
    g_string_assign(_aux_string,"");
    if (ui_can_hide_input_window ())
        gtk_widget_hide (_input_window);

}

static void
slot_hide_lookup_table (void)
{

    for (int i = 0; i < SCIM_LOOKUP_TABLE_MAX_PAGESIZE; ++ i) {
	_lookup_table_index[i]=false;
	gtk_button_set_label((GtkButton *)_lookup_table_items [i]," ");
    }
    if (ui_can_hide_input_window ())
        gtk_widget_hide (_input_window);
}

static void
slot_update_preedit_string (const String &str, const AttributeList &attrs)
{
    //PangoAttrList  *attrlist = create_pango_attrlist (str, attrs);

    //scim_string_view_set_attributes (SCIM_STRING_VIEW (_preedit_area), attrlist);
    //scim_string_view_set_text (SCIM_STRING_VIEW (_preedit_area), str.c_str ());
    g_string_assign(_preedit_string, str.c_str());
    //ui_update_input_string();
    //pango_attr_list_unref (attrlist);
    if (_hide_table)
       ui_settle_input_window (true,true);
    else
       ui_update_input_string();
    //ui_settle_lookup_table_window ();
}

static void
slot_update_preedit_caret (int caret)
{
    //scim_string_view_set_position (SCIM_STRING_VIEW (_preedit_area), caret);
}

static void
slot_update_aux_string (const String &str, const AttributeList &attrs)
{
    //PangoAttrList  *attrlist = create_pango_attrlist (str, attrs);

    //scim_string_view_set_attributes (SCIM_STRING_VIEW (_aux_area), attrlist);
    //scim_string_view_set_text (SCIM_STRING_VIEW (_aux_area), str.c_str ());
    g_string_assign(_aux_string, str.c_str());
    ui_update_input_string();
    //pango_attr_list_unref (attrlist);
    //ui_settle_input_window (true);
    /*
    if (_hide_table)
       ui_settle_input_window (true,true);
    else
       ui_update_input_string();
    */
}

static void
slot_update_lookup_table (const LookupTable &table)
{
    size_t i;
    size_t item_num = table.get_current_page_size ();

    String         mbs;
    WideString     wcs;
    WideString     label;
    GtkRequisition size;
    AttributeList  attrs;
    PangoAttrList  *attrlist;

        for (i = 0; i < SCIM_LOOKUP_TABLE_MAX_PAGESIZE; ++ i) {
            if (i < item_num) {
                mbs = String ();
		_lookup_table_index[i]=true;
                wcs = table.get_candidate_in_current_page (i);
        WideString new_line=utf8_mbstowcs("\n");
		if (wcs.length() > 8) {
		   wcs.insert(5,new_line);
		} else if (wcs.length() > 5) {
           wcs.insert(4,new_line);
        }
                //label = table.get_candidate_label (i);

                //if (label.length ()) {
                //    label += utf8_mbstowcs (". ");
                //} else {
                //    label = utf8_mbstowcs (" ");
                //}

                //mbs = utf8_wcstombs (label+wcs);
		mbs = utf8_wcstombs (wcs);
                //scim_string_view_set_text (SCIM_STRING_VIEW (_lookup_table_items [i]),
                //                           mbs.c_str ());
		//hildon_button_set_text ((HildonButton*)_lookup_table_items [i],mbs.c_str(), NULL);
		gtk_button_set_label((GtkButton *)_lookup_table_items [i],mbs.c_str());
            } else {
		_lookup_table_index[i]=false;
		gtk_button_set_label((GtkButton *)_lookup_table_items [i]," ");
		//hildon_button_set_text ((HildonButton*)_lookup_table_items [i]," ", NULL);
	    }
            gtk_widget_show (_lookup_table_items [i]);
        }
/*
    for (i=0; i< SCIM_LOOKUP_TABLE_MAX_PAGESIZE; i++)
    {
	if (table.is_cursor_visible() && i==table.get_cursor_pos_in_current_page())
		gtk_toggle_button_set_active ((GtkToggleButton *)_lookup_table_items[i], TRUE);
	else
		gtk_toggle_button_set_active ((GtkToggleButton *)_lookup_table_items[i], FALSE);
    }
*/
/*
    {
      FILE *fileObj = fopen ("/tmp/scim.log", "a");
      fprintf (fileObj, "%s %d %d %d\n", __FUNCTION__, __LINE__,
	table.is_cursor_visible (),
	table.get_cursor_pos_in_current_page ());
      fclose (fileObj);
    }
*/
    //if (table.is_cursor_visible ())
    {
	i = table.get_cursor_pos_in_current_page ();
	if (i >= 0 && i< SCIM_LOOKUP_TABLE_MAX_PAGESIZE)
	{
	gtk_toggle_button_set_active((GtkToggleButton *)_lookup_table_items[i], TRUE);
	//gtk_widget_grab_focus (_lookup_table_items[i]);
	}
    }
    if (table.get_current_page_start ())
        gtk_widget_set_sensitive (_lookup_table_up_arrow, TRUE);
    else
        gtk_widget_set_sensitive (_lookup_table_up_arrow, FALSE);

    if (table.get_current_page_start () + item_num < table.number_of_candidates ())
        gtk_widget_set_sensitive (_lookup_table_down_arrow, TRUE);
    else
        gtk_widget_set_sensitive (_lookup_table_down_arrow, FALSE);

    if (item_num < table.get_current_page_size ())
        _panel_agent->update_lookup_table_page_size (item_num);


    ui_settle_input_window (true, true);
}

static void
slot_register_properties (const PropertyList &props)
{
	register_frontend_properties (props);
}

static void
slot_update_property (const Property &prop)
{
	update_frontend_property (prop);
}

static void
slot_register_helper_properties (int id, const PropertyList &props)
{

}

static void
slot_update_helper_property (int id, const Property &prop)
{

}

static void
slot_register_helper (int id, const HelperInfo &helper)
{
}

static void
slot_remove_helper (int id)
{

}

static void
slot_lock (void)
{
    G_LOCK (_panel_agent_lock);
}

static void
slot_unlock (void)
{
    G_UNLOCK (_panel_agent_lock);
}



static void
ui_property_activate_cb (GtkWidget      *widget,
                         gpointer        user_data)
{
    GtkWidget *submenu = (GtkWidget *) g_object_get_data (G_OBJECT (widget), "property_submenu");

    if (submenu) {
        guint32 activate_time = gtk_get_current_event_time ();
        _property_menu_activated = true;
        gtk_menu_popup (GTK_MENU (submenu), 0, 0, 0, 0, 1, activate_time);
        return;
    }

    gchar * key = (gchar *) g_object_get_data (G_OBJECT (widget), "property_key");

    if (key) {
        int client = GPOINTER_TO_INT (user_data);

        if (client < 0)
            _panel_agent->trigger_property (key);
        else
            _panel_agent->trigger_helper_property (client, key);
    }
}

static void
ui_property_menu_deactivate_cb (GtkWidget   *item,
                                gpointer     user_data)
{
    _property_menu_activated = false;
}

static GtkWidget *
create_properties_node (PropertyRepository           &repository,
                        PropertyList::const_iterator  begin,
                        PropertyList::const_iterator  end,
                        int                           client,
                        int                           level)
{
    PropertyList::const_iterator it;
    PropertyList::const_iterator next;

    GtkWidget * node;
    PropertyInfo info;
    bool leaf = true;

    if (begin >= end) return 0;

    // If the level is zero, then create the this node as button, otherwise create as a menu item.
    if (!level) {
        GtkWidget * label = ui_create_icon (begin->get_icon ());

        node = gtk_button_new ();
        gtk_button_set_image(GTK_BUTTON (node),label);
        gtk_button_set_relief (GTK_BUTTON (node), GTK_RELIEF_NONE);
    } else {
        GtkWidget * icon = ui_create_icon (begin->get_icon ());
        node = gtk_image_menu_item_new_with_label (begin->get_label ().c_str ());
        if (icon)
            gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (node), icon);
    }

    if (begin->visible ())
        gtk_widget_show (node);
    else
        gtk_widget_hide (node);

    gtk_widget_set_sensitive (node, begin->active ());

    g_object_set_data_full (G_OBJECT (node), "property_key", g_strdup (begin->get_key ().c_str ()), g_free);

    info.property = *begin;
    info.widget = node;

    repository.push_back (info);

    it = begin + 1;

    if (it != end) {
        GtkWidget * submenu = hildon_gtk_menu_new ();
        GtkWidget * child;
        int menu_item_idx = 0;

        // Create all leafs of the first child.
        while (it != end) {
            // Find all leafs of the first child.
            for (next = it + 1; next != end; ++ next)
                if (!next->is_a_leaf_of (*it)) break;

            child = create_properties_node (repository, it, next, client, level + 1);
            if (child) {
                gtk_menu_shell_append (GTK_MENU_SHELL (submenu), child);
                g_object_set_data (G_OBJECT (child), "menu_item_idx", GINT_TO_POINTER (menu_item_idx));
                ++ menu_item_idx;
            }

            it = next;
        }

        // The node is a button, so attach the submenu as its data.
        if (!level) {
            g_object_set_data_full (G_OBJECT (node), "property_submenu", submenu, (void (*)(void*)) gtk_widget_destroy);

            g_signal_connect (G_OBJECT (submenu), "deactivate",
                              G_CALLBACK (ui_property_menu_deactivate_cb),
                              NULL);
        } else // The node is a menu item, so attach the submenu directly.
            gtk_menu_item_set_submenu (GTK_MENU_ITEM (node), submenu);

        leaf = false;
    }

    if (leaf || level == 0) {
        g_signal_connect (G_OBJECT (node),
                          ((level > 0) ? "activate" : "clicked"),
                          G_CALLBACK (ui_property_activate_cb),
                          GINT_TO_POINTER (client));
    }

    return node;
}

static void
create_properties (GtkWidget *container,
                   PropertyRepository &repository,
                   const PropertyList &properties,
                   int client,
                   int level)
{

    PropertyList::const_iterator it;
    PropertyList::const_iterator next;
    PropertyList::const_iterator begin = properties.begin ();
    PropertyList::const_iterator end = properties.end ();

    GtkWidget *root;

    int menu_item_idx = 0;

    if (begin == end) return;

    root = gtk_hbox_new (FALSE, 0);

    it = begin;
    next = begin + 1;

    while (it != end) {
        if (next == end || !next->is_a_leaf_of (*it)) {
            GtkWidget * node = create_properties_node (repository, it, next, client, level);

            if (node) {
                // The container is a hbox.
                if (!level)
                    gtk_box_pack_start (GTK_BOX (container), node, TRUE, TRUE, 0);
                // The container is a menu.
                else {
                    gtk_menu_shell_append (GTK_MENU_SHELL (container), node);
                    g_object_set_data (G_OBJECT (node), "menu_item_idx", GINT_TO_POINTER (menu_item_idx));
                    ++ menu_item_idx;
                }
            }
            it = next;
        }
        ++ next;
    }
}

static void
register_frontend_properties (const PropertyList &properties)
{
    bool same = true;

    PropertyList::const_iterator pit = properties.begin ();

    if (properties.size () == 0) {
        same = false;
    } else if (properties.size () == _frontend_property_repository.size ()) {
        // Check if the properties are same as old ones.
        PropertyRepository::iterator it = _frontend_property_repository.begin ();

        for (; it != _frontend_property_repository.end (); ++it, ++pit) {
            if (it->property != *pit) {
                same = false;
                break;
            }
        }
    } else {
        same = false;
    }

    // Only update the properties.
    if (same) {
        for (pit = properties.begin (); pit != properties.end (); ++pit)
            update_frontend_property (*pit);

        gtk_widget_show (_frontend_properties_area);
    } else { // Construct all properties.
        if (_frontend_properties_area)
            gtk_widget_destroy (_frontend_properties_area);

        _frontend_properties_area = 0;

        _frontend_property_repository.clear ();

        if (properties.size ()) {
            _frontend_properties_area = gtk_hbox_new (FALSE, 0);

            create_properties (_frontend_properties_area,
                               _frontend_property_repository,
                               properties,
                               -1,
                               0);

            gtk_widget_show (_frontend_properties_area);

            gtk_box_pack_start (GTK_BOX (_client_properties_area), _frontend_properties_area, TRUE,TRUE, 0);
        }
    }
}

static void
update_frontend_property (const Property &property)
{
    update_property (_frontend_property_repository, property);
}

static void
update_property (PropertyRepository &repository,
                 const Property       &property)
{
    PropertyRepository::iterator it = repository.begin ();

    for (; it != repository.end (); ++ it) {
        if (it->property == property) {

            if (!it->widget) break;

            if (it->property.get_label () != property.get_label () ||
                it->property.get_icon () != property.get_icon ()) {
                if (GTK_IS_BUTTON (it->widget)) {
                    GtkWidget *label = ui_create_icon (property.get_icon ());
                    gtk_button_set_image (GTK_BUTTON (it->widget), label);
                } else if (GTK_IS_MENU_ITEM (it->widget)) {
                    GtkWidget * menu = gtk_widget_get_parent (it->widget);
                    int menu_item_idx = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (it->widget), "menu_item_idx"));

                    GtkWidget * icon = ui_create_icon (property.get_icon ());
                    GtkWidget * new_item = gtk_image_menu_item_new_with_label (property.get_label ().c_str ());

                    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (new_item), icon);

                    GtkWidget * submenu = gtk_menu_item_get_submenu (GTK_MENU_ITEM (it->widget));

                    gtk_menu_item_set_submenu (GTK_MENU_ITEM (new_item), submenu);

                    g_object_set_data_full (G_OBJECT (new_item), "property_key", g_strdup (property.get_key ().c_str ()), g_free);
                    g_object_set_data (G_OBJECT (new_item), "menu_item_idx", GINT_TO_POINTER (menu_item_idx));

                    gtk_widget_destroy (it->widget);

                    it->widget = new_item;
                    gtk_menu_shell_insert (GTK_MENU_SHELL (menu), new_item, menu_item_idx);
                }
            }

            if (property.visible ())
                gtk_widget_show (it->widget);
            else
                gtk_widget_hide (it->widget);

            gtk_widget_set_sensitive (it->widget, property.active ());

            it->property = property;
            break;
        }
    }
}

static void
restore_properties (void)
{
    PropertyList properties;

    _frontend_properties_area = 0;

    PropertyRepository::iterator it = _frontend_property_repository.begin ();


    for (; it != _frontend_property_repository.end (); ++it)
        properties.push_back (it->property);

    if (properties.size ()) {
        _frontend_property_repository.clear ();
        register_frontend_properties (properties);
    }
}


static bool
initialize_panel_agent (const String &config, const String &display, bool resident)
{
    _panel_agent = new PanelAgent ();

    if (!_panel_agent->initialize (config, display, resident))
        return false;

    _panel_agent->signal_connect_transaction_start          (slot (slot_transaction_start));
    _panel_agent->signal_connect_transaction_end            (slot (slot_transaction_end));
    _panel_agent->signal_connect_reload_config              (slot (slot_reload_config));
    _panel_agent->signal_connect_turn_on                    (slot (slot_turn_on));
    _panel_agent->signal_connect_turn_off                   (slot (slot_turn_off));
    _panel_agent->signal_connect_update_screen              (slot (slot_update_screen));
    _panel_agent->signal_connect_update_spot_location       (slot (slot_update_spot_location));
    _panel_agent->signal_connect_update_factory_info        (slot (slot_update_factory_info));
    _panel_agent->signal_connect_show_help                  (slot (slot_show_help));
    _panel_agent->signal_connect_show_factory_menu          (slot (slot_show_factory_menu));
    _panel_agent->signal_connect_show_preedit_string        (slot (slot_show_preedit_string));
    _panel_agent->signal_connect_show_aux_string            (slot (slot_show_aux_string));
    _panel_agent->signal_connect_show_lookup_table          (slot (slot_show_lookup_table));
    _panel_agent->signal_connect_hide_preedit_string        (slot (slot_hide_preedit_string));
    _panel_agent->signal_connect_hide_aux_string            (slot (slot_hide_aux_string));
    _panel_agent->signal_connect_hide_lookup_table          (slot (slot_hide_lookup_table));
    _panel_agent->signal_connect_update_preedit_string      (slot (slot_update_preedit_string));
    _panel_agent->signal_connect_update_preedit_caret       (slot (slot_update_preedit_caret));
    _panel_agent->signal_connect_update_aux_string          (slot (slot_update_aux_string));
    _panel_agent->signal_connect_update_lookup_table        (slot (slot_update_lookup_table));
    _panel_agent->signal_connect_register_properties        (slot (slot_register_properties));
    _panel_agent->signal_connect_update_property            (slot (slot_update_property));
    _panel_agent->signal_connect_register_helper_properties (slot (slot_register_helper_properties));
    _panel_agent->signal_connect_update_helper_property     (slot (slot_update_helper_property));
    _panel_agent->signal_connect_register_helper            (slot (slot_register_helper));
    _panel_agent->signal_connect_remove_helper              (slot (slot_remove_helper));
    _panel_agent->signal_connect_lock                       (slot (slot_lock));
    _panel_agent->signal_connect_unlock                     (slot (slot_unlock));

 //   _panel_agent->get_helper_list (_helper_list);

    return true;
}


static bool
run_panel_agent (void)
{
    SCIM_DEBUG_MAIN(1) << "run_panel_agent ()\n";

    _panel_agent_thread = NULL;

    if (_panel_agent && _panel_agent->valid ())
        _panel_agent_thread = g_thread_create (panel_agent_thread_func, NULL, TRUE, NULL);

    return (_panel_agent_thread != NULL);
}

static gpointer
panel_agent_thread_func (gpointer data)
{
    SCIM_DEBUG_MAIN(1) << "panel_agent_thread_func ()\n";

    if (!_panel_agent->run ())
        std::cerr << "Failed to run Panel.\n";

    G_LOCK (_global_resource_lock);
    _should_exit = true;
    G_UNLOCK (_global_resource_lock);

    g_thread_exit (NULL);
    return ((gpointer) NULL);
}

static void
start_auto_start_helpers (void)
{
    SCIM_DEBUG_MAIN(1) << "start_auto_start_helpers ()\n";
/*
    // Add Helper object items.
    for (size_t i = 0; i < _helper_list.size (); ++i) {
        if ((_helper_list [i].option & SCIM_HELPER_AUTO_START) != 0) {
            _panel_agent->start_helper (_helper_list [i].uuid);
        }
    }
*/
}

static gboolean
check_exit_timeout_cb (gpointer data)
{
    G_LOCK (_global_resource_lock);
    if (_should_exit) {
        gdk_threads_enter ();
        gtk_main_quit ();
        gdk_threads_leave ();
    }
    G_UNLOCK (_global_resource_lock);

    return TRUE;
}

static void
signalhandler(int sig)
{
    SCIM_DEBUG_MAIN (1) << "In signal handler...\n";
    if (_panel_agent != NULL) {
        _panel_agent->stop ();
    }
}

int main (int argc, char *argv [])
{
    std::vector<String>  config_list;

    int i;

    bool daemon = false;

    int    new_argc = 0;
    char **new_argv = new char * [40];

    String config_name ("simple");
    String display_name;
    bool should_resident = true;

    setlocale ( LC_ALL, "" );

    //Display version info
    std::cerr << "GTK Panel of SCIM " << SCIM_VERSION << "\n\n";

    //get modules list
    scim_get_config_module_list (config_list);

    //Add a dummy config module, it's not really a module!
    config_list.push_back ("dummy");

    //Use socket Config module as default if available.
    if (config_list.size ()) {
        if (std::find (config_list.begin (),
                       config_list.end (),
                       config_name) == config_list.end ())
            config_name = config_list [0];
    }

    DebugOutput::disable_debug (SCIM_DEBUG_AllMask);
    DebugOutput::enable_debug (SCIM_DEBUG_MainMask);

    //parse command options
    i = 0;
    while (i<argc) {
        if (++i >= argc) break;

        if (String ("-l") == argv [i] ||
            String ("--list") == argv [i]) {
            std::vector<String>::iterator it;

            std::cout << "\n";
            std::cout << "Available Config module:\n";
            for (it = config_list.begin (); it != config_list.end (); it++)
                std::cout << "    " << *it << "\n";

            return 0;
        }

        if (String ("-c") == argv [i] ||
            String ("--config") == argv [i]) {
            if (++i >= argc) {
                std::cerr << "no argument for option " << argv [i-1] << "\n";
                return -1;
            }
            config_name = argv [i];
            continue;
        }

        if (String ("-h") == argv [i] ||
            String ("--help") == argv [i]) {
            std::cout << "Usage: " << argv [0] << " [option]...\n\n"
                 << "The options are: \n"
                 << "  --display DISPLAY    Run on display DISPLAY.\n"
                 << "  -l, --list           List all of available config modules.\n"
                 << "  -c, --config NAME    Uses specified Config module.\n"
                 << "  -d, --daemon         Run " << argv [0] << " as a daemon.\n"
                 << "  -ns, --no-stay       Quit if no connected client.\n"
#if ENABLE_DEBUG
                 << "  -v, --verbose LEVEL  Enable debug info, to specific LEVEL.\n"
                 << "  -o, --output FILE    Output debug information into FILE.\n"
#endif
                 << "  -h, --help           Show this help message.\n";
            return 0;
        }

        if (String ("-d") == argv [i] ||
            String ("--daemon") == argv [i]) {
            daemon = true;
            continue;
        }

        if (String ("-ns") == argv [i] ||
            String ("--no-stay") == argv [i]) {
            should_resident = false;
            continue;
        }

        if (String ("-v") == argv [i] ||
            String ("--verbose") == argv [i]) {
            if (++i >= argc) {
                std::cerr << "no argument for option " << argv [i-1] << "\n";
                return -1;
            }
            DebugOutput::set_verbose_level (atoi (argv [i]));
            continue;
        }

        if (String ("-o") == argv [i] ||
            String ("--output") == argv [i]) {
            if (++i >= argc) {
                std::cerr << "No argument for option " << argv [i-1] << "\n";
                return -1;
            }
            DebugOutput::set_output (argv [i]);
            continue;
        }

        if (String ("--display") == argv [i]) {
            if (++i >= argc) {
                std::cerr << "No argument for option " << argv [i-1] << "\n";
                return -1;
            }
            display_name = argv [i]; 
            continue;
        }

        if (String ("--") == argv [i])
            break;

        std::cerr << "Invalid command line option: " << argv [i] << "\n";
        return -1;
    } //End of command line parsing.

    new_argv [new_argc ++] = argv [0];

    // Store the rest argvs into new_argv.
    for (++i; i < argc && new_argc < 40; ++i) {
        new_argv [new_argc ++] = argv [i];
    }

    // Make up DISPLAY env.
    if (display_name.length ()) {
        new_argv [new_argc ++] = const_cast <char*> ("--display");
        new_argv [new_argc ++] = const_cast <char*> (display_name.c_str ());

        setenv ("DISPLAY", display_name.c_str (), 1);
    }

    new_argv [new_argc] = 0;

    if (!config_name.length ()) {
        std::cerr << "No Config module is available!\n";
        return -1;
    }

    if (config_name != "dummy") {
        //load config module
        _config_module = new ConfigModule (config_name);

        if (!_config_module || !_config_module->valid ()) {
            std::cerr << "Can not load " << config_name << " Config module.\n";
            return -1;
        }

        //create config instance
        _config = _config_module->create_config ();
    } else {
        _config = new DummyConfig ();
    }

    if (_config.null ()) {
        std::cerr << "Failed to create Config instance from "
             << config_name << " Config module.\n";
        return -1;
    }

    /* init threads */
    g_thread_init (NULL);
    gdk_threads_init ();

    signal(SIGQUIT, signalhandler);
    signal(SIGTERM, signalhandler);
    signal(SIGINT,  signalhandler);
    signal(SIGHUP,  signalhandler);

    //hildon_gtk_init(&new_argc, &new_argv);
    gtk_init(&new_argc, &new_argv);

    if (_input_window_str != NULL)
    {
	g_string_free(_input_window_str, TRUE);
    }
    _input_window_str = g_string_new(NULL);
    if (_preedit_string != NULL)
    {
	    g_string_free(_preedit_string, TRUE);
    }
    _preedit_string = g_string_new(NULL);
    if (_aux_string != NULL)
    {
	    g_string_free(_aux_string, TRUE);
    }
    _aux_string = g_string_new(NULL);

    ui_initialize ();

    // get current display.
    {
#if GDK_MULTIHEAD_SAFE
        const char *p = gdk_display_get_name (gdk_display_get_default ());
#else
        const char *p = getenv ("DISPLAY");
#endif
        if (p) display_name = String (p);
    }

    if (!initialize_panel_agent (config_name, display_name, should_resident)) {
        std::cerr << "Failed to initialize Panel Agent!\n";
        return -1;
    }
 
    if (daemon)
        scim_daemon ();

    // connect the configuration reload signal.
    _config->signal_connect_reload (slot (ui_config_reload_callback));

    if (!run_panel_agent()) {
        std::cerr << "Failed to run Socket Server!\n";
        return -1;
    }

    start_auto_start_helpers ();

    _check_exit_timeout = gtk_timeout_add (500, check_exit_timeout_cb, NULL);

    gdk_threads_enter ();
    gtk_main ();
    gdk_threads_leave ();

    // Exiting...
    g_thread_join (_panel_agent_thread);
    _config.reset ();

    std::cerr << "Successfully exited.\n";

    return 0;
}
