#include "qmessagelistwidget.h"
#include "ui_qmessagelistwidget.h"

#include <QToolBar>
#include <QAbstractItemModel>

#include "qmessagedelegate.h"
#include "filter/servicefilter.h"
#include "widgets/filter/filterpanel.h"
#include "config.h"
#include "filter/filtermodel.h"

QMessageListWidget::QMessageListWidget(QFont uf, ServiceMgr *sm, QWidget *parent) :
    QWidget(parent),
    ui(new Ui::QMessageListWidget), sm(sm), isMultiPanel(true)
{
    // set up search buttons
    ui->setupUi(this);
    ui->layoutMessagesBottom->addWidget(editOwnr = new HintEdit(this));
    ui->layoutMessagesBottom->addWidget(buttonEditOwnr = new QPushButton(this));

    buttonEditOwnr->setIconSize(QSize(ICON_SIZE, ICON_SIZE));
    buttonEditOwnr->setIcon(QIcon::fromTheme("general_backspace",
                                              QIcon(":/res/general_backspace.png")));
    buttonEditOwnr->setFlat(TRUE);
    editOwnr->setHintText(tr("Type a title..."));

    // set up list view
    ui->treeView->setFont(uf);
    ui->treeView->setItemDelegate(new QMessageDelegate(this->sm, ui->treeView));
    ui->treeView->setSelectionMode(QTreeView::SingleSelection);
    ui->treeView->setRootIsDecorated(false);
    ui->treeView->setSortingEnabled(true);

#if defined(Q_WS_MAEMO_5) || defined(MEEGO_EDITION_HARMATTAN)
    ui->treeView->setHeaderHidden(TRUE);
    ui->treeView->setStyleSheet("QTreeView {background-color: black;}");
#endif

    // prepare model
    filterModel = new FilterModel(sm, this);
    filterModel->setSortRole(QMessageListModel::SortRole);
    model = new QMessageListModel(this);
    connect(this->sm, SIGNAL(updateMessageList(MessageList,bool)), model, SLOT(updateList(MessageList)));
    connect(this->sm, SIGNAL(messageReaded(QString,QString)), model, SLOT(readMessage(QString, QString)));
    filterModel->setSourceModel(model);
    ui->treeView->setModel(filterModel);
    ui->treeView->header()->setResizeMode(QHeaderView::Stretch);
    ui->treeView->sortByColumn(1, Qt::DescendingOrder);
    this->setFocusProxy(ui->treeView);

#if defined(Q_WS_MAEMO_5) || defined(MEEGO_EDITION_HARMATTAN)
    connect(ui->treeView,
            SIGNAL(clicked(QModelIndex)),
            this, SLOT(itemSelected(QModelIndex)));
#else
    // it allows navigation by arrow keys
    connect(ui->treeView->selectionModel(),
            SIGNAL(currentChanged(QModelIndex,QModelIndex)),
            this, SLOT(itemSelected(QModelIndex)));
#endif

    connect(filterModel, SIGNAL(layoutChanged()), this, SLOT(modelLayoutChanged()));
    connect(model, SIGNAL(messageCountChanged(int)), this, SIGNAL(messageCountChanged(int)));

    //service filters
    filter = sm->serviceFilter();
    filterPanel = new FilterPanel;
    filter->registerFilterInterface(filterPanel);
    ui->filtersLayout->addWidget(filterPanel);

    // This is toolbar to datatype filters
    filtersToolBar_ = new QToolBar;
    filtersToolBar_->setOrientation(Qt::Vertical);
    filtersToolBar_->setIconSize(QSize(TOOLBAR_ICON_SIZE, TOOLBAR_ICON_SIZE));
    ui->filtersLayout->addWidget(filtersToolBar_);
#if !defined(Q_WS_MAEMO_5) && !defined(MEEGO_EDITION_HARMATTAN)
    ui->filtersLayout->addStretch();
#endif

    connect(this->sm, SIGNAL(updateMessageList(MessageList,bool)), this, SLOT(updateMessages(MessageList, bool)));

#if defined(Q_WS_MAEMO_5) || defined(MEEGO_EDITION_HARMATTAN)
    // device rotation
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
    orientationChanged();
#endif

    // Connect filter field and button
    connect(editOwnr, SIGNAL(textChanged(QString)),
            filterModel, SLOT(setFilterString(QString)));
    connect(buttonEditOwnr, SIGNAL(clicked()),
            editOwnr, SLOT(clear()));
}

QMessageListWidget::~QMessageListWidget()
{
    delete ui;
}

void QMessageListWidget::itemSelected(QModelIndex in)
{
    QModelIndex srcIn = filterModel->mapToSource(in);

    if (!srcIn.isValid()) {
        emit ownerSelected(Message());
        return;
    }

    Message selectedMessage = srcIn.data(Qt::UserRole).value<Message>();

    this->selectedItem = srcIn;
    this->selectedMsgAccount = selectedMessage.accountId();
    this->selectedMsgId = selectedMessage.messageId();

    // FIXME: if time is empty this message is draft!
    if (!selectedMessage.isReaded() && !selectedMessage.time().isEmpty())
        this->sm->readMessage(selectedMessage.accountId(), selectedMessage.messageId());

    emit ownerSelected(selectedMessage);
}

Message QMessageListWidget::getSelectedMessage() const
{
    // if message was selected
    if (selectedItem.isValid())
        return selectedItem.data(Qt::UserRole).value<Message>();

    // or empty message
    return Message();
}

void QMessageListWidget::selectedMessage(const Message &msg)
{
    QModelIndex idx = this->model->getIndex(msg);
    if (idx.isValid()) {
        this->ui->treeView->setCurrentIndex(filterModel->mapFromSource(idx));
        this->selectedItem = idx;
        this->selectedMsgAccount = msg.accountId();
        this->selectedMsgId = msg.messageId();
        emit ownerSelected(msg);
    }
}

void QMessageListWidget::updateMessages(MessageList, bool)
{
    QModelIndex idx = this->model->getIndex(this->selectedMsgAccount, this->selectedMsgId);
    if (idx.isValid()) {
        if (isMultiPanel)
            this->ui->treeView->setCurrentIndex(filterModel->mapFromSource(idx));
        this->selectedItem = idx;
    } else {
        if (isMultiPanel)
            this->ui->treeView->setCurrentIndex(model->index(0,0));
        this->itemSelected(filterModel->index(0, 0));
    }
}

void QMessageListWidget::showFriendMessages(Friend fr)
{
    MessageList ret;
    for (int i = 0; i < this->model->msgList.length(); i++) {
        bool isFound = false;
        for (int j = 0; j < this->model->msgList.at(i).recepientListConst().count(); j++) {
            if (this->model->msgList.at(i).recepientListConst().at(j).id == fr.ownerId())
                isFound = true;
        }

        if ((this->model->msgList.at(i).senderId() == fr.ownerId()) ||
                isFound)
            ret.append(this->model->msgList.at(i));
    }

    if (ret.length() > 0) {
        QModelIndex idx = this->model->getIndex(ret.first());
        if (isMultiPanel)
            this->ui->treeView->setCurrentIndex(filterModel->mapFromSource(idx));
        this->selectedMsgAccount = ret.first().accountId();
        this->selectedMsgId = ret.first().messageId();
    }

    this->model->setMessagesList(ret);
}

QToolBar* QMessageListWidget::filtersToolBar() const
{
    return filtersToolBar_;
}

void QMessageListWidget::modelLayoutChanged()
{
    QModelIndex idx = this->model->getIndex(this->selectedMsgAccount, this->selectedMsgId);
    QModelIndex filterIdx = filterModel->mapFromSource(idx);

    if (filterIdx.isValid()) {
        if (isMultiPanel)
            this->ui->treeView->setCurrentIndex(filterIdx);
        this->selectedItem = idx;
    } else {

        if (isMultiPanel)
            this->ui->treeView->setCurrentIndex(filterModel->index(0,0));

        itemSelected(filterModel->index(0, 0));
    }
}

void QMessageListWidget::setMultipanelEnabled(const bool isMultiPanel)
{
    this->isMultiPanel = isMultiPanel;
}

void QMessageListWidget::orientationChanged()
{
    bool isPortraitMode = true;
    QRect geom = QApplication::desktop()->screenGeometry();
    if (geom.width() > geom.height())
        isPortraitMode = false;

    if (!isPortraitMode) {
        this->filterPanel->setVisible(false);
    } else {
        this->filterPanel->setVisible(true);
    }
}
