#include "albumlistwidget.h"

#include <QtCore/QDebug>
#include <QtCore/QMetaType>
#include <QtCore/QMimeData>
#include <QtGui/QDragEnterEvent>
#include <QtGui/QDragMoveEvent>
#include <QtGui/QMessageBox>

#include "servicemgr.h"

AlbumListWidget::AlbumListWidget(ServiceMgr* mgr, QWidget *parent) :
    QListWidget(parent), mServiceMgr(mgr)
{
    setIconSize(QSize(90, 130));
    viewport()->setAcceptDrops(true);

    // Disable itemview standard drag'n'drop
    setDragDropMode(QListWidget::NoDragDrop);
    setDragEnabled(false);
}

void AlbumListWidget::dragEnterEvent(QDragEnterEvent* event)
{
    if (event->proposedAction() == Qt::CopyAction && event->mimeData()->formats().contains("application/msa-photo"))
        event->acceptProposedAction();
    else
        QListWidget::dragEnterEvent(event);
}

void AlbumListWidget::dragMoveEvent(QDragMoveEvent* event)
{
    bool accepted = false;
    if (event->proposedAction() == Qt::CopyAction && event->mimeData()->formats().contains("application/msa-photo")) {
        QModelIndex index = indexAt(event->pos());

        if (index.isValid()) {
            const Album& album = index.data(Qt::UserRole).value<Album>();
            if (mServiceMgr->canUploadPhoto(album.accountId()) && mServiceMgr->isLocalFriend(album.ownerId())) {
                accepted = true;
            }
        }
    }

    if (accepted)
        event->acceptProposedAction();
    else
        QListWidget::dragMoveEvent(event);
}

void AlbumListWidget::dropEvent(QDropEvent* event)
{
    // If event contains msa-photo
    if (event->proposedAction() == Qt::CopyAction && event->mimeData()->formats().contains("application/msa-photo")) {
        QModelIndex index = indexAt(event->pos());

        // Restore photo from byte array
        QByteArray data = event->mimeData()->data("application/msa-photo");
        QDataStream in(&data, QIODevice::ReadOnly);
        Photo photo;
        in >> photo;

        // Get album from index
        const Album& album = index.data(Qt::UserRole).value<Album>();

        // Upload photo
        mServiceMgr->uploadPhoto(album, photo.photo(), photo.description());
        event->acceptProposedAction();
    } else {
        QListWidget::dropEvent(event);
    }
}
