/********************
 * Callback functions
 ********************/
/*
 * Include declarations
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <glob.h>

#include <dbus/dbus.h>
#include <mce/mode-names.h>
#include <mce/dbus-names.h>
#include <hildon/hildon.h>
#include <hildon/hildon-fm.h>
#include <hildon/hildon-defines.h>
#include <gtk/gtk.h>
#include <gdk/gdkpixbuf.h>
#include <glib/gprintf.h>
#include <glib/gfileutils.h>
#include <pango/pango.h>

#include "appdata.h"
#include "interface.h"
#include "callbacks.h"
#include "helpers.h"
#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

/*
 * Image file selector
 */
void open_image_file(GtkWidget *callerobj,app_data_t *myapp) {
	GtkWidget *dialog;
	GtkFileFilter *filter;
	gint retval;
	gchar *filename = NULL;
	gchar *filterext;
	guint i;
	GList *li = NULL;

    dialog = hildon_file_chooser_dialog_new(GTK_WINDOW (myapp->mainwin),GTK_FILE_CHOOSER_ACTION_OPEN);
	if (myapp->currdir != NULL) gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER (dialog),myapp->currdir);
	/* Show only supported RAW files */
    filter = gtk_file_filter_new();
	for(i = 0;myapp->rawext[i] != NULL;i++) {
		filterext = g_strconcat("*.",myapp->rawext[i],NULL);
		gtk_file_filter_add_pattern(filter,filterext);
	    g_free(filterext);
	}
	gtk_file_chooser_set_filter(GTK_FILE_CHOOSER (dialog),filter);
    retval = gtk_dialog_run(GTK_DIALOG (dialog));
	if (retval == GTK_RESPONSE_ACCEPT || retval == GTK_RESPONSE_OK)
		filename = g_strndup(gtk_file_chooser_get_filename(GTK_FILE_CHOOSER (dialog)),(MAXDIRLEN + MAXFNLEN + 1));
	gtk_widget_destroy(dialog);
	if (filename != NULL) {
		g_free(myapp->currdir);
		myapp->currdir = g_strndup(g_path_get_dirname(filename),MAXDIRLEN);
		store_filenames_in_dir(myapp);
		li = g_list_find_custom(myapp->allimgindir,(gconstpointer) g_path_get_basename(filename),(GCompareFunc) g_strcmp0);
		myapp->currfindex = g_list_position(myapp->allimgindir,li);
		/* DEBUG */
		/* g_printerr("Current directory: %s - Current file: %s - Current index in list: %d\n",myapp->currdir,g_path_get_basename(filename),myapp->currfindex); */
		load_image(myapp,filename);
	}
	g_free(filename);
}

/*
 * Open next image in the same directory (if any)
 */
void load_next_image(GtkWidget *callerobj,app_data_t *myapp) {
	gchar *nextfn;
	GtkWidget *dialog;

	if (myapp->allimgindir != NULL) {
		if (g_list_length(myapp->allimgindir) > 1) {
			nextfn = (gchar*) g_list_nth_data(myapp->allimgindir,++myapp->currfindex);
			if (nextfn == NULL) {
				myapp->currfindex = 0;
				nextfn = (gchar*) g_list_nth_data(myapp->allimgindir,myapp->currfindex);
			}
			nextfn = g_strconcat(myapp->currdir,"/",nextfn,NULL);
			if (!g_file_test(nextfn,G_FILE_TEST_EXISTS)) {
				g_free(nextfn);
				g_free(g_list_nth_data(myapp->allimgindir,myapp->currfindex));
				myapp->allimgindir = g_list_delete_link(myapp->allimgindir,g_list_nth(myapp->allimgindir,myapp->currfindex));
				myapp->currfindex = ((myapp->currfindex > 0) ? (myapp->currfindex - 1) : (g_list_length(myapp->allimgindir) - 1));
				load_next_image(callerobj,myapp);
			} else {
				load_image(myapp,nextfn);
				g_free(nextfn);
			}
		} else {
			toggle_prev_next_menu_button("disable",myapp);
			dialog = gtk_message_dialog_new(GTK_WINDOW (myapp->mainwin),
											GTK_DIALOG_DESTROY_WITH_PARENT,
											GTK_MESSAGE_WARNING,
											GTK_BUTTONS_CLOSE,
											"No more images in directory '%s'",
											myapp->currdir);
			 gtk_dialog_run(GTK_DIALOG (dialog));
			 gtk_widget_destroy(dialog);
		}
	}
}

/*
 * Open previous image in the same directory (if any)
 */
void load_prev_image(GtkWidget *callerobj,app_data_t *myapp) {
	gchar *prevfn;
	GtkWidget *dialog;

	if (myapp->allimgindir != NULL) {
		if (g_list_length(myapp->allimgindir) > 1) {
			prevfn = (gchar*) g_list_nth_data(myapp->allimgindir,--myapp->currfindex);
			if (prevfn == NULL) {
				myapp->currfindex = g_list_length(myapp->allimgindir) - 1;
				prevfn = (gchar*) g_list_nth_data(myapp->allimgindir,myapp->currfindex);
			}
			prevfn = g_strconcat(myapp->currdir,"/",prevfn,NULL);
			if (!g_file_test(prevfn,G_FILE_TEST_EXISTS)) {
				g_free(prevfn);
				g_free(g_list_nth_data(myapp->allimgindir,myapp->currfindex));
				myapp->allimgindir = g_list_delete_link(myapp->allimgindir,g_list_nth(myapp->allimgindir,myapp->currfindex));
				myapp->currfindex = ((myapp->currfindex != (g_list_length(myapp->allimgindir) - 1)) ? (myapp->currfindex + 1) : 0);
				load_next_image(callerobj,myapp);
			} else {
				load_image(myapp,prevfn);
				g_free(prevfn);
			}
		} else {
			toggle_prev_next_menu_button("disable",myapp);
			dialog = gtk_message_dialog_new(GTK_WINDOW (myapp->mainwin),
											GTK_DIALOG_DESTROY_WITH_PARENT,
											GTK_MESSAGE_WARNING,
											GTK_BUTTONS_CLOSE,
											"No more images in directory '%s'",
											myapp->currdir);
			 gtk_dialog_run(GTK_DIALOG (dialog));
			 gtk_widget_destroy(dialog);
		}
	}
}

/*
 * Draw image details
 */
gboolean draw_image_details(GtkWidget *self,GdkEventExpose *event,app_data_t *myapp) {
	PangoLayout *textbuff;
	cairo_t *cr;
	gint x0, y0, r, txtwidth, txtheight;

	/* Get the Cairo context */
	cr = gdk_cairo_create(GDK_DRAWABLE (self->window));
	/* Clear the surface first */
	cairo_set_operator(cr,CAIRO_OPERATOR_CLEAR);
	cairo_paint(cr);
	/* Create the text layout */
	textbuff = pango_cairo_create_layout(cr);
	pango_layout_set_markup(textbuff,myapp->imgparam,-1);
	pango_layout_get_pixel_size(textbuff,&txtwidth,&txtheight);
	/* Draw a rounded rectangle */
	x0 = y0 = r = 10;
	cairo_set_operator(cr,CAIRO_OPERATOR_OVER);
	cairo_set_source_rgba(cr,0,0,0,0.4);
	cairo_move_to(cr,x0,y0 - r);
	cairo_line_to(cr,x0 + txtwidth,y0 -r);
	cairo_curve_to(cr,x0 + txtwidth + r,y0 -r,x0 + txtwidth + r,y0 -r,x0 + txtwidth + r,y0);
	cairo_line_to(cr,x0 + txtwidth + r,y0 + txtheight);
	cairo_curve_to(cr,x0 + txtwidth + r,y0 + txtheight + r,x0 + txtwidth + r,y0 + txtheight + r,x0 + txtwidth,y0 + txtheight + r);
	cairo_line_to(cr,x0,y0 + txtheight + r);
	cairo_curve_to(cr,x0 -r,y0 + txtheight + r,x0 -r,y0 + txtheight + r,x0 -r,y0 + txtheight);
	cairo_line_to(cr,x0 - r,y0);
	cairo_curve_to(cr,x0 - r,y0 -r,x0 -r,y0 -r,x0,y0 - r);
	cairo_close_path(cr);
	cairo_fill(cr);
	/* Layout the text on it */
	cairo_new_path(cr);
	cairo_move_to(cr,x0,y0);
	cairo_set_line_width(cr,1);
	cairo_set_source_rgb(cr,0.8,0.8,0.8);
	pango_cairo_update_layout(cr,textbuff);
	pango_cairo_layout_path(cr,textbuff);
	cairo_fill_preserve(cr);
	/*cairo_set_line_width(cr,0.5);
	cairo_set_source_rgb(cr,0,0,0);
	cairo_stroke(cr);*/
	cairo_destroy(cr);
	g_object_unref(textbuff);

	return FALSE;
}

/*
 * Enable / disable "previous" and "next image" menu buttons
 */
void toggle_prev_next_menu_button(const gchar *mode,app_data_t *myapp) {
	GList *menuitems = NULL;

	menuitems = hildon_app_menu_get_items(hildon_window_get_app_menu(myapp->mainwin));
	if (menuitems != NULL) {
		gtk_widget_set_sensitive(GTK_WIDGET (g_list_nth_data(menuitems,0)),((g_strcmp0(mode,"enable") == 0) ? TRUE : FALSE));
		gtk_widget_set_sensitive(GTK_WIDGET (g_list_nth_data(menuitems,1)),((g_strcmp0(mode,"enable") == 0) ? TRUE : FALSE));
		g_list_free(menuitems);
	}
}

/*
 * Toggle main window's fullscreen status on or off
 */
void toggle_fullscreen_mode(GtkWidget *callerobj,app_data_t *myapp) {
	GList *menuitems = NULL;

	menuitems = hildon_app_menu_get_items(hildon_window_get_app_menu(myapp->mainwin));
	if (menuitems != NULL) {
		if (!(gdk_window_get_state(gtk_widget_get_window(GTK_WIDGET (myapp->mainwin))) & GDK_WINDOW_STATE_FULLSCREEN)) {
			gtk_window_fullscreen(GTK_WINDOW (myapp->mainwin));
			gtk_widget_set_sensitive(GTK_WIDGET (g_list_nth_data(menuitems,4)),FALSE);
		} else {
			gtk_window_unfullscreen(GTK_WINDOW (myapp->mainwin));
			gtk_widget_set_sensitive(GTK_WIDGET (g_list_nth_data(menuitems,4)),TRUE);
		}
		g_list_free(menuitems);
	}
}

/*
 * Manage button presses on image to load previous / next image in the same directory
 */
gboolean detect_button_press_on_image (GtkWidget *eventbox,GdkEventButton *event,app_data_t *myapp) {
	gint dispwidth;
	action_t act;
	glong butprtime;
	gdouble butprx, butpry;
	GTimeVal now;

	if (g_list_length(myapp->allimgindir) > 1) {
		g_get_current_time(&now);
		/* DEBUG */
		/* g_print ("Event box clicked at coordinates (%f,%f) at time %ld\n",event->x_root,event->y_root,now.tv_sec); */
		switch (event->type) {
			case GDK_BUTTON_PRESS:
				myapp->pressbuttontime = now.tv_sec;
				myapp->pressbuttonx = event->x_root;
				myapp->pressbuttony = event->y_root;
				dispwidth = gdk_screen_get_width(GTK_WINDOW (myapp->mainwin)->screen);
				if (event->x_root >= ((gdouble) dispwidth / (gdouble) 2)) myapp->actioninprog = NEXT;
				else myapp->actioninprog = PREVIOUS;
				g_timeout_add(100,(GSourceFunc) draw_prev_next_arrow,(gpointer) myapp);
				return TRUE;
			case GDK_BUTTON_RELEASE:
				/* Store actual values */
				act = myapp->actioninprog;
				butprtime = myapp->pressbuttontime;
				butprx = myapp->pressbuttonx;
				butpry = myapp->pressbuttony;
				/* Reset values */
				myapp->pressbuttontime = 0;
				myapp->pressbuttonx = 0;
				myapp->pressbuttony = 0;
				myapp->actioninprog = NONE;
				if (((now.tv_sec - butprtime) >= MINBUTPRESSTIME) &
					(abs(round(event->x_root - butprx)) <= MAXBUTPRESSDISTANCE) &
					(abs(round(event->y_root - butpry)) <= MAXBUTPRESSDISTANCE))
				{
					if (act == NEXT) load_next_image(eventbox,myapp);
					else load_prev_image(eventbox,myapp);
				} else g_source_remove_by_user_data((gpointer) myapp);
				return TRUE;
			default:
				break;
		}
	}
	return FALSE;
}

/*
 * Manage key presses
 */
gboolean detect_hardware_key_press (GtkWidget *widget,GdkEventKey *event,app_data_t *myapp) {
	/* DEBUG */
	/* g_print ("Detected key press: %d\n",event->keyval); */
	switch (event->keyval) {
		case HILDON_HARDKEY_FULLSCREEN:
			toggle_fullscreen_mode(widget,myapp);
			return TRUE;
		case GDK_space:
			if (event->state & GDK_CONTROL_MASK) {
				toggle_fullscreen_mode(widget,myapp);
				return TRUE;
			}
			break;
		case HILDON_HARDKEY_RIGHT:
			load_next_image(widget,myapp);
			return TRUE;
		case HILDON_HARDKEY_LEFT:
			load_prev_image(widget,myapp);
			return TRUE;
		case HILDON_HARDKEY_INCREASE:
			myapp->zoomlevel = (myapp->zoomlevel + 1) % 4;
			scale_image(widget,myapp);
			return TRUE;
		case HILDON_HARDKEY_DECREASE:
			myapp->zoomlevel = (myapp->zoomlevel + 3) % 4;
			scale_image(widget,myapp);
			return TRUE;
		default:
			break;
	}
	return FALSE;
}

/*
 * Manage main window size changes
 */
gboolean detect_main_window_changes (GObject *object,GdkEvent *event,app_data_t *myapp) {
	/* DEBUG */
	/* g_print ("Detected event: %d\n",event->type); */
	switch (event->type) {
		case GDK_CONFIGURE:
			scale_image(GTK_WIDGET (myapp->mainwin),myapp);
			break;
		case GDK_EXPOSE:
		default:
			break;
	}
	return FALSE;
}

/*
 * Detect DBus MCE signals
 */
DBusHandlerResult detect_mce_signal (DBusConnection *conn,DBusMessage *msg,app_data_t *myapp) {
	DBusMessageIter iter;
	const gchar *mode = NULL;

	if (dbus_message_is_signal(msg,MCE_SIGNAL_IF,MCE_DEVICE_ORIENTATION_SIG)) {
		if (dbus_message_iter_init(msg,&iter)) {
			dbus_message_iter_get_basic(&iter,&mode);
			/* Rotate main window */
			if (!strcmp(mode,MCE_ORIENTATION_PORTRAIT)) hildon_gtk_window_set_portrait_flags(GTK_WINDOW (myapp->mainwin),HILDON_PORTRAIT_MODE_REQUEST);
			else hildon_gtk_window_set_portrait_flags(GTK_WINDOW (myapp->mainwin),HILDON_PORTRAIT_MODE_SUPPORT);
		}
	}
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

/*
 * Quit application
 */
void destroy_app (GtkWidget *callerobj,app_data_t *myapp) {
	/* Save configuration */
	save_configuration(myapp);
	/* Quit GTK main loop */
	gtk_main_quit();
	/* Disable accelerometers */
	enable_accelerometers(myapp,FALSE);
	/* Deinitialize OSSO context */
    osso_deinitialize(myapp->ossocont);
    /* Release memory allocated for application's data */
    app_free(myapp);
}
