/********************
 * Callback functions
 ********************/
/*
 * Include declarations
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <glob.h>

#include <dbus/dbus.h>
#include <mce/mode-names.h>
#include <mce/dbus-names.h>
#include <hildon/hildon.h>
#include <hildon/hildon-fm.h>
#include <hildon/hildon-defines.h>
#include <gtk/gtk.h>
#include <gdk/gdkpixbuf.h>
#include <gdk/gdkx.h>
#include <X11/Xatom.h>
#include <glib/gprintf.h>
#include <glib/gfileutils.h>
#include <pango/pango.h>
#include <gconf/gconf-client.h>

#include <libraw/libraw.h>

#include "interface.h"
#include "callbacks.h"
#include "../config.h"

/*
 * Load image from filename
 */
void load_image(app_data_t *myapp, gchar *imgfname) {
	GtkWidget *infobanner;
	GtkWidget *errdialog;
	GdkPixbufLoader *imgloader;
	libraw_data_t *rawdata;
	gboolean imgerr = FALSE;
	gint shutternum = 0;
	gchar *shutter;
	const gchar *paramformat = "<span size=\"medium\" face=\"arial\">Filename: <b>%s</b>\nCamera model: <b>%s %s</b>\nShutter speed: <b>%s s</b>\nAperture: <b>f/%-.1f</b>\nISO: <b>%-.0f</b>\nFocal length: <b>%-.0f mm</b></span>";

	g_free(myapp->currfname);
	myapp->currfname = g_strndup(g_path_get_basename(imgfname),MAXFNLEN);
	infobanner = hildon_banner_show_informationf(GTK_WIDGET(myapp->image),NULL,"Loading image '%s' ...",myapp->currfname);
	hildon_banner_set_timeout(HILDON_BANNER(infobanner),1000);
	rawdata = libraw_init(0);
	/* DEBUG */
	/* g_print("Supported RAW formats: %s\n",g_strjoinv("\n",(gchar**) libraw_cameraList())); */
	if (libraw_open_file(rawdata,imgfname) == LIBRAW_SUCCESS) {
		if (libraw_unpack_thumb(rawdata) == LIBRAW_SUCCESS) {
			imgloader = gdk_pixbuf_loader_new();
			if (gdk_pixbuf_loader_write(imgloader,(guchar *) rawdata->thumbnail.thumb,(gsize) rawdata->thumbnail.tlength,NULL)) {
				switch(rawdata->sizes.flip) {
					case 3:
						myapp->imgbuffer = gdk_pixbuf_rotate_simple(gdk_pixbuf_loader_get_pixbuf(imgloader),GDK_PIXBUF_ROTATE_UPSIDEDOWN);
						break;
					case 5:
						myapp->imgbuffer = gdk_pixbuf_rotate_simple(gdk_pixbuf_loader_get_pixbuf(imgloader),GDK_PIXBUF_ROTATE_COUNTERCLOCKWISE);
						break;
					case 6:
						myapp->imgbuffer = gdk_pixbuf_rotate_simple(gdk_pixbuf_loader_get_pixbuf(imgloader),GDK_PIXBUF_ROTATE_CLOCKWISE);
						break;
					case 0:
					default:
						myapp->imgbuffer = gdk_pixbuf_loader_get_pixbuf(imgloader);
						break;
				}
			} else imgerr = TRUE;
			gdk_pixbuf_loader_close(imgloader,NULL);
		} else imgerr = TRUE;
	} else imgerr = TRUE;
	g_free(myapp->imgparam);
	if (!imgerr) {
		shutternum = (gint) ((rawdata->other.shutter < 1) ? round(1 / rawdata->other.shutter) : round(rawdata->other.shutter));
		shutter = g_strdup_printf(((rawdata->other.shutter < 1) ? "1/%d" : "%d"),shutternum);
		myapp->imgparam = g_strdup_printf(paramformat,myapp->currfname,rawdata->idata.make,rawdata->idata.model,shutter,rawdata->other.aperture,rawdata->other.iso_speed,rawdata->other.focal_len);
		g_free(shutter);
		scale_image(GTK_WIDGET (myapp->mainwin),myapp);
	} else {
		g_object_unref(myapp->imgbuffer);
		gtk_image_set_from_pixbuf(GTK_IMAGE (myapp->image),NULL);
		errdialog = gtk_message_dialog_new(GTK_WINDOW (myapp->mainwin),
											GTK_DIALOG_DESTROY_WITH_PARENT,
											GTK_MESSAGE_ERROR,
											GTK_BUTTONS_CLOSE,
											"Error loading image '%s'",
											g_path_get_basename(imgfname));
		 gtk_dialog_run(GTK_DIALOG (errdialog));
		 gtk_widget_destroy(errdialog);
	}
	libraw_close(rawdata);
}

gboolean scale_image(GtkWidget *callerobj,app_data_t *myapp) {
	GdkPixbuf *tmpbuffer = NULL;
	gint dispwidth, dispheight, imgwidth, imgheight;
	guint scaledwidth, scaledheight;
	gdouble dispratio, imgratio;

	/* DEBUG */
	/* g_print("Scale image with buffer: %d (%d)\n",myapp->imgbuffer,G_IS_OBJECT (G_OBJECT (myapp->imgbuffer))); */
	if (myapp->imgbuffer != NULL) {
		/* Get actual image widget dimensions */
		gtk_window_get_size(GTK_WINDOW (myapp->mainwin),&dispwidth,&dispheight);
		dispratio = (gdouble) dispwidth / (gdouble) dispheight;
		imgwidth = gdk_pixbuf_get_width(myapp->imgbuffer);
		imgheight = gdk_pixbuf_get_height(myapp->imgbuffer);
		imgratio = (gdouble) imgwidth / (gdouble) imgheight;
		scaledwidth = scaledheight = 0;
		/* Eventually scale image */
		if ((imgwidth > dispwidth) || (imgheight > dispheight)) {
			switch (myapp->zoomlevel) {
				case FIT_SCREEN:
					if (imgratio >= dispratio) {
						scaledwidth = dispwidth;
						scaledheight = (guint) round((double) dispwidth * (1 / imgratio));
					} else {
						scaledwidth = (guint) round((double) (dispheight * imgratio));
						scaledheight = dispheight;
					}
					break;
				case ONE_THIRD_RESOLUTION:
					scaledwidth = (guint) round((double) imgwidth * 0.33);
					scaledheight = (guint) round((double) imgheight * 0.33);
					break;
				case HALF_RESOLUTION:
					scaledwidth = (guint) round((double) imgwidth * 0.5);
					scaledheight = (guint) round((double) imgheight * 0.5);
					break;
				case FULL_RESOLUTION:
				default:
					break;
			}
			if ((scaledwidth > 0) & (scaledheight > 0))
				tmpbuffer = gdk_pixbuf_scale_simple(myapp->imgbuffer,scaledwidth,scaledheight,GDK_INTERP_BILINEAR);
		}
		gtk_image_set_from_pixbuf(GTK_IMAGE (myapp->image), (tmpbuffer ? tmpbuffer : myapp->imgbuffer));
		if (tmpbuffer) g_object_unref(tmpbuffer);
		/* DEBUG */
		/* g_print("\n\nOriginal dim.: %dx%d (%f) - Window dim.: %dx%d (%f) - Scaled dim.: %dx%d\n\n",imgwidth,imgheight,imgratio,dispwidth,dispheight,dispratio,scaledwidth,scaledheight); */
	}
	return TRUE;
}

gboolean draw_image_details(GtkWidget *self,GdkEventExpose *event,app_data_t *myapp) {
	PangoLayout *textbuff;
	cairo_t *cr;
	gint x0, y0, r, txtwidth, txtheight;

	/* Get the Cairo context */
	cr = gdk_cairo_create(GDK_DRAWABLE (self->window));
	/* Clear the surface first */
	cairo_set_operator(cr,CAIRO_OPERATOR_CLEAR);
	cairo_paint(cr);
	/* Create the text layout */
	textbuff = pango_cairo_create_layout(cr);
	pango_layout_set_markup(textbuff,myapp->imgparam,-1);
	pango_layout_get_pixel_size(textbuff,&txtwidth,&txtheight);
	/* Draw a rounded rectangle */
	x0 = y0 = r = 10;
	cairo_set_operator(cr,CAIRO_OPERATOR_OVER);
	cairo_set_source_rgba(cr,0,0,0,0.4);
	cairo_move_to(cr,x0,y0 - r);
	cairo_line_to(cr,x0 + txtwidth,y0 -r);
	cairo_curve_to(cr,x0 + txtwidth + r,y0 -r,x0 + txtwidth + r,y0 -r,x0 + txtwidth + r,y0);
	cairo_line_to(cr,x0 + txtwidth + r,y0 + txtheight);
	cairo_curve_to(cr,x0 + txtwidth + r,y0 + txtheight + r,x0 + txtwidth + r,y0 + txtheight + r,x0 + txtwidth,y0 + txtheight + r);
	cairo_line_to(cr,x0,y0 + txtheight + r);
	cairo_curve_to(cr,x0 -r,y0 + txtheight + r,x0 -r,y0 + txtheight + r,x0 -r,y0 + txtheight);
	cairo_line_to(cr,x0 - r,y0);
	cairo_curve_to(cr,x0 - r,y0 -r,x0 -r,y0 -r,x0,y0 - r);
	cairo_close_path(cr);
	cairo_fill(cr);
	/* Layout the text on it */
	cairo_new_path(cr);
	cairo_move_to(cr,x0,y0);
	cairo_set_line_width(cr,1);
	cairo_set_source_rgb(cr,0.8,0.8,0.8);
	pango_cairo_update_layout(cr,textbuff);
	pango_cairo_layout_path(cr,textbuff);
	cairo_fill_preserve(cr);
	/*cairo_set_line_width(cr,0.5);
	cairo_set_source_rgb(cr,0,0,0);
	cairo_stroke(cr);*/
	cairo_destroy(cr);
	g_object_unref(textbuff);

	return FALSE;
}

gboolean draw_prev_next_arrow(app_data_t *myapp) {
	GtkImageType imgtype;
	cairo_t *cr;
	GdkPixmap *tmppm;
	GdkBitmap *tmpbm;
	gint winwidth, winheight, pointerx, pointery, x0, y0, r, m, w, h;
	GTimeVal now;

	g_get_current_time(&now);
	gdk_window_get_pointer(gtk_widget_get_root_window(GTK_WIDGET (myapp->mainwin)),&pointerx,&pointery,NULL);
	if (myapp->actioninprog != NONE) {
		if (((now.tv_sec - myapp->pressbuttontime) >= MINBUTPRESSTIME) &
			(abs(round((double)(pointerx - myapp->pressbuttonx))) <= MAXBUTPRESSDISTANCE) &
			(abs(round((double)(pointery - myapp->pressbuttony))) <= MAXBUTPRESSDISTANCE))
		{
			imgtype = gtk_image_get_storage_type(GTK_IMAGE (myapp->image));
			if (imgtype != GTK_IMAGE_EMPTY) {
				/* If the image is zoomed in, zoom out to FIT_SCREEN before drawing the arrow */
				if (myapp->zoomlevel != FIT_SCREEN) {
					myapp->zoomlevel = FIT_SCREEN;
					scale_image(GTK_WIDGET (myapp->mainwin),myapp);
				}
				if (imgtype == GTK_IMAGE_PIXBUF) gdk_pixbuf_render_pixmap_and_mask(gtk_image_get_pixbuf(GTK_IMAGE (myapp->image)),&tmppm,&tmpbm,0);
				else if (imgtype == GTK_IMAGE_PIXMAP) {
					gtk_image_get_pixmap(GTK_IMAGE (myapp->image),&tmppm,&tmpbm);
					tmppm = g_object_ref(tmppm);
					tmpbm = g_object_ref(tmpbm);
				} else return FALSE;
				gdk_drawable_get_size(GDK_DRAWABLE (tmppm),&winwidth,&winheight);
				r = 5;
				m = 15;
				w = h = 40;
				if (myapp->actioninprog == NEXT) x0 = (winwidth - w - m);
				else x0 = m;
				y0 = round((double)((winheight - h) / 2));
				cr = gdk_cairo_create(GDK_DRAWABLE (tmppm));

				/* Draw a rounded rectangle */
				cairo_set_source_rgba(cr,0,0,0,0.4);
				cairo_move_to(cr,x0,y0 - r);
				cairo_line_to(cr,x0 + w,y0 -r);
				cairo_curve_to(cr,x0 + w + r,y0 -r,x0 + w + r,y0 -r,x0 + w + r,y0);
				cairo_line_to(cr,x0 + w + r,y0 + h);
				cairo_curve_to(cr,x0 + w + r,y0 + h + r,x0 + w + r,y0 + h + r,x0 + w,y0 + h + r);
				cairo_line_to(cr,x0,y0 + h + r);
				cairo_curve_to(cr,x0 -r,y0 + h + r,x0 -r,y0 + h + r,x0 -r,y0 + h);
				cairo_line_to(cr,x0 - r,y0);
				cairo_curve_to(cr,x0 - r,y0 -r,x0 -r,y0 -r,x0,y0 - r);
				cairo_close_path(cr);
				cairo_fill(cr);
				/* Draw an arrow according to the action being performed*/
				cairo_new_path(cr);
				cairo_set_source_rgba(cr,1,1,1,0.6);
				if (myapp->actioninprog == NEXT) {
					cairo_move_to(cr,x0,y0);
					cairo_line_to(cr,x0 + w,y0 + (h / 2));
					cairo_line_to(cr,x0,y0 + h);
					cairo_line_to(cr,x0,y0);
				} else {
					cairo_move_to(cr,x0,y0 + (h / 2));
					cairo_line_to(cr,x0 + w,y0);
					cairo_line_to(cr,x0 + w,y0 + h);
					cairo_line_to(cr,x0,y0 + (h / 2));
				}
				cairo_close_path(cr);
				cairo_fill(cr);

				cairo_destroy(cr);
				gtk_image_set_from_pixmap(GTK_IMAGE (myapp->image),tmppm,tmpbm);
				g_object_unref(tmppm);
				/*
				 * Un-referencing tmpbm is not needed
				 * g_object_unref(tmpbm);
				 *
				 */
			}
			return FALSE;
		} else return TRUE;
	} else return FALSE;
}

/*
 * Image file selector
 */
void open_image_file(GtkWidget *callerobj,app_data_t *myapp) {
	GtkWidget *dialog;
	GtkFileFilter *filter;
	gint retval;
	gchar *filename = NULL;
	gchar *filterext;
	guint i;
	GList *li = NULL;

	dialog = hildon_file_chooser_dialog_new(GTK_WINDOW (myapp->mainwin),GTK_FILE_CHOOSER_ACTION_OPEN);
	if (myapp->currdir != NULL) gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER (dialog),myapp->currdir);
	/* Show only supported RAW files */
	filter = gtk_file_filter_new();
	for(i = 0;myapp->rawext[i] != NULL;i++) {
		filterext = g_strconcat("*.",myapp->rawext[i],NULL);
		gtk_file_filter_add_pattern(filter,filterext);
		g_free(filterext);
	}
	gtk_file_chooser_set_filter(GTK_FILE_CHOOSER (dialog),filter);
	retval = gtk_dialog_run(GTK_DIALOG (dialog));
	if (retval == GTK_RESPONSE_ACCEPT || retval == GTK_RESPONSE_OK)
		filename = g_strndup(gtk_file_chooser_get_filename(GTK_FILE_CHOOSER (dialog)),(MAXDIRLEN + MAXFNLEN + 1));
	gtk_widget_destroy(dialog);
	if (filename != NULL) {
		g_free(myapp->currdir);
		myapp->currdir = g_strndup(g_path_get_dirname(filename),MAXDIRLEN);
		store_filenames_in_dir(myapp);
		li = g_list_find_custom(myapp->allimgindir,(gconstpointer) g_path_get_basename(filename),(GCompareFunc) g_strcmp0);
		myapp->currfindex = g_list_position(myapp->allimgindir,li);
		/* DEBUG */
		/* g_print("Current directory: %s - Current file: %s - Current index in list: %d\n",myapp->currdir,g_path_get_basename(filename),myapp->currfindex); */
		load_image(myapp,filename);
	}
	g_free(filename);
}

/*
 * Store image file names in current directory into application's data structure
 */
void store_filenames_in_dir(app_data_t *myapp) {
	gchar *tmpstr;
	glob_t allimages;
	guint i;
	GList *li = NULL;

	/* First free the allocated memory in the list */
	if (myapp->allimgindir != NULL) {
		for(li = myapp->allimgindir;li != NULL;li = g_list_next(li))
			g_free(li->data);
		g_list_free(li);
		g_list_free(myapp->allimgindir);
		myapp->allimgindir = NULL;
	}
	/* Then search for other image files */
	tmpstr = g_strconcat(myapp->currdir,"/*.{",g_strjoinv(",",(gchar**) myapp->rawext),"}",NULL);
	if (glob(tmpstr,GLOB_ERR | GLOB_BRACE,NULL,&allimages) == 0) {
		for(i = 0;i < allimages.gl_pathc;i++) {
			myapp->allimgindir = g_list_prepend(myapp->allimgindir,g_strndup(g_path_get_basename((gchar*) allimages.gl_pathv[i]),MAXFNLEN));
		}
		if (myapp->allimgindir != NULL) myapp->allimgindir = g_list_reverse(myapp->allimgindir);
	}
	if (allimages.gl_pathc <= 1) toggle_prev_next_menu_button("disable",myapp);
	else toggle_prev_next_menu_button("enable",myapp);
	globfree(&allimages);
	g_free(tmpstr);
	/* DEBUG */
	/* for(li = myapp->allimgindir;li != NULL;li = g_list_next(li))
		g_print("File in list: %s\n",li->data);
	g_list_free(li);*/
}

/*
 * Open next image in the same directory (if any)
 */
void load_next_image(GtkWidget *callerobj,app_data_t *myapp) {
	gchar *nextfn;
	GtkWidget *dialog;

	if (myapp->allimgindir != NULL) {
		if (g_list_length(myapp->allimgindir) > 1) {
			nextfn = (gchar*) g_list_nth_data(myapp->allimgindir,++myapp->currfindex);
			if (nextfn == NULL) {
				myapp->currfindex = 0;
				nextfn = (gchar*) g_list_nth_data(myapp->allimgindir,myapp->currfindex);
			}
			nextfn = g_strconcat(myapp->currdir,"/",nextfn,NULL);
			if (!g_file_test(nextfn,G_FILE_TEST_EXISTS)) {
				g_free(nextfn);
				g_free(g_list_nth_data(myapp->allimgindir,myapp->currfindex));
				myapp->allimgindir = g_list_delete_link(myapp->allimgindir,g_list_nth(myapp->allimgindir,myapp->currfindex));
				myapp->currfindex = ((myapp->currfindex > 0) ? (myapp->currfindex - 1) : (g_list_length(myapp->allimgindir) - 1));
				load_next_image(callerobj,myapp);
			} else {
				load_image(myapp,nextfn);
				g_free(nextfn);
			}
		} else {
			toggle_prev_next_menu_button("disable",myapp);
			dialog = gtk_message_dialog_new(GTK_WINDOW (myapp->mainwin),
											GTK_DIALOG_DESTROY_WITH_PARENT,
											GTK_MESSAGE_WARNING,
											GTK_BUTTONS_CLOSE,
											"No more images in directory '%s'",
											myapp->currdir);
			 gtk_dialog_run(GTK_DIALOG (dialog));
			 gtk_widget_destroy(dialog);
		}
	}
}

/*
 * Open previous image in the same directory (if any)
 */
void load_prev_image(GtkWidget *callerobj,app_data_t *myapp) {
	gchar *prevfn;
	GtkWidget *dialog;

	if (myapp->allimgindir != NULL) {
		if (g_list_length(myapp->allimgindir) > 1) {
			prevfn = (gchar*) g_list_nth_data(myapp->allimgindir,--myapp->currfindex);
			if (prevfn == NULL) {
				myapp->currfindex = g_list_length(myapp->allimgindir) - 1;
				prevfn = (gchar*) g_list_nth_data(myapp->allimgindir,myapp->currfindex);
			}
			prevfn = g_strconcat(myapp->currdir,"/",prevfn,NULL);
			if (!g_file_test(prevfn,G_FILE_TEST_EXISTS)) {
				g_free(prevfn);
				g_free(g_list_nth_data(myapp->allimgindir,myapp->currfindex));
				myapp->allimgindir = g_list_delete_link(myapp->allimgindir,g_list_nth(myapp->allimgindir,myapp->currfindex));
				myapp->currfindex = ((myapp->currfindex != (g_list_length(myapp->allimgindir) - 1)) ? (myapp->currfindex + 1) : 0);
				load_next_image(callerobj,myapp);
			} else {
				load_image(myapp,prevfn);
				g_free(prevfn);
			}
		} else {
			toggle_prev_next_menu_button("disable",myapp);
			dialog = gtk_message_dialog_new(GTK_WINDOW (myapp->mainwin),
											GTK_DIALOG_DESTROY_WITH_PARENT,
											GTK_MESSAGE_WARNING,
											GTK_BUTTONS_CLOSE,
											"No more images in directory '%s'",
											myapp->currdir);
			 gtk_dialog_run(GTK_DIALOG (dialog));
			 gtk_widget_destroy(dialog);
		}
	}
}

/*
 * Enable / disable "previous" and "next image" menu buttons
 */
void toggle_prev_next_menu_button(const gchar *mode,app_data_t *myapp) {
	GList *menuitems = NULL;

	menuitems = hildon_app_menu_get_items(hildon_window_get_app_menu(myapp->mainwin));
	if (menuitems != NULL) {
		gtk_widget_set_sensitive(GTK_WIDGET (g_list_nth_data(menuitems,0)),((g_strcmp0(mode,"enable") == 0) ? TRUE : FALSE));
		gtk_widget_set_sensitive(GTK_WIDGET (g_list_nth_data(menuitems,1)),((g_strcmp0(mode,"enable") == 0) ? TRUE : FALSE));
		g_list_free(menuitems);
	}
}

/*
 * Toggle main window's fullscreen status on or off
 */
void toggle_fullscreen_mode(GtkWidget *callerobj,app_data_t *myapp) {
	GList *menuitems = NULL;

	menuitems = hildon_app_menu_get_items(hildon_window_get_app_menu(myapp->mainwin));
	if (menuitems != NULL) {
		if (!(gdk_window_get_state(gtk_widget_get_window(GTK_WIDGET (myapp->mainwin))) & GDK_WINDOW_STATE_FULLSCREEN)) {
			gtk_window_fullscreen(GTK_WINDOW (myapp->mainwin));
			gtk_widget_set_sensitive(GTK_WIDGET (g_list_nth_data(menuitems,4)),FALSE);
		} else {
			gtk_window_unfullscreen(GTK_WINDOW (myapp->mainwin));
			gtk_widget_set_sensitive(GTK_WIDGET (g_list_nth_data(menuitems,4)),TRUE);
		}
		g_list_free(menuitems);
	}
}

/*
 * Manage button presses on image to load previous / next image in the same directory
 */
gboolean detect_button_press_on_image (GtkWidget *eventbox,GdkEventButton *event,app_data_t *myapp) {
	gint dispwidth;
	action_t act;
	glong butprtime;
	gdouble butprx, butpry;
	GTimeVal now;

	if (g_list_length(myapp->allimgindir) > 1) {
		g_get_current_time(&now);
		/* DEBUG */
		/* g_print ("Event box clicked at coordinates (%f,%f) at time %ld\n",event->x_root,event->y_root,now.tv_sec); */
		switch (event->type) {
			case GDK_BUTTON_PRESS:
				myapp->pressbuttontime = now.tv_sec;
				myapp->pressbuttonx = event->x_root;
				myapp->pressbuttony = event->y_root;
				dispwidth = gdk_screen_get_width(GTK_WINDOW (myapp->mainwin)->screen);
				if (event->x_root >= ((gdouble) dispwidth / (gdouble) 2)) myapp->actioninprog = NEXT;
				else myapp->actioninprog = PREVIOUS;
				g_timeout_add(100,(GSourceFunc) draw_prev_next_arrow,(gpointer) myapp);
				return TRUE;
			case GDK_BUTTON_RELEASE:
				/* Store actual values */
				act = myapp->actioninprog;
				butprtime = myapp->pressbuttontime;
				butprx = myapp->pressbuttonx;
				butpry = myapp->pressbuttony;
				/* Reset values */
				myapp->pressbuttontime = 0;
				myapp->pressbuttonx = 0;
				myapp->pressbuttony = 0;
				myapp->actioninprog = NONE;
				if (((now.tv_sec - butprtime) >= MINBUTPRESSTIME) &
					(abs(round(event->x_root - butprx)) <= MAXBUTPRESSDISTANCE) &
					(abs(round(event->y_root - butpry)) <= MAXBUTPRESSDISTANCE))
				{
					if (act == NEXT) load_next_image(eventbox,myapp);
					else load_prev_image(eventbox,myapp);
				} else g_source_remove_by_user_data((gpointer) myapp);
				return TRUE;
			default:
				break;
		}
	}
	return FALSE;
}

/*
 * Manage key presses
 */
gboolean detect_hardware_key_press (GtkWidget *widget,GdkEventKey *event,app_data_t *myapp) {
	/* DEBUG */
	/* g_print ("Detected key press: %d\n",event->keyval); */
	switch (event->keyval) {
		case HILDON_HARDKEY_FULLSCREEN:
			toggle_fullscreen_mode(widget,myapp);
			return TRUE;
		case GDK_space:
			if (event->state & GDK_CONTROL_MASK) {
				toggle_fullscreen_mode(widget,myapp);
				return TRUE;
			}
			break;
		case HILDON_HARDKEY_RIGHT:
			load_next_image(widget,myapp);
			return TRUE;
		case HILDON_HARDKEY_LEFT:
			load_prev_image(widget,myapp);
			return TRUE;
		case HILDON_HARDKEY_INCREASE:
			myapp->zoomlevel = (myapp->zoomlevel + 1) % 4;
			scale_image(widget,myapp);
			return TRUE;
		case HILDON_HARDKEY_DECREASE:
			myapp->zoomlevel = (myapp->zoomlevel + 3) % 4;
			scale_image(widget,myapp);
			return TRUE;
		default:
			break;
	}
	return FALSE;
}

/*
 * Manage main window size changes
 */
gboolean detect_main_window_changes (GObject *object,GdkEvent *event,app_data_t *myapp) {
	/* DEBUG */
	/* g_print ("Detected event: %d\n",event->type); */
	switch (event->type) {
		case GDK_CONFIGURE:
			scale_image(GTK_WIDGET (myapp->mainwin),myapp);
			break;
		case GDK_EXPOSE:
		default:
			break;
	}
	return FALSE;
}

/*
 * Detect DBus MCE signals
 */
DBusHandlerResult detect_mce_signal (DBusConnection *conn,DBusMessage *msg,app_data_t *myapp) {
	DBusMessageIter iter;
	const gchar *mode = NULL;

	if (dbus_message_is_signal(msg,MCE_SIGNAL_IF,MCE_DEVICE_ORIENTATION_SIG)) {
		if (dbus_message_iter_init(msg,&iter)) {
			dbus_message_iter_get_basic(&iter,&mode);
			/* Rotate main window */
			if (!strcmp(mode,MCE_ORIENTATION_PORTRAIT)) hildon_gtk_window_set_portrait_flags(GTK_WINDOW (myapp->mainwin),HILDON_PORTRAIT_MODE_REQUEST);
			else hildon_gtk_window_set_portrait_flags(GTK_WINDOW (myapp->mainwin),HILDON_PORTRAIT_MODE_SUPPORT);
		}
	}
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

/*
 * Quit application
 */
void destroy_app (GtkWidget *callerobj,app_data_t *myapp) {
	/* Save configuration & quit application */
	save_configuration(myapp);
	gtk_main_quit();
}

/*
 * Ungrab volume keys
 */
void ungrab_volume_keys (GtkWidget *widget) {
    /* Tell maemo-status-volume daemon to ungrab keys */
    unsigned long val = 1; /* ungrab, use 0 to grab */
    Atom atom;
    GdkDisplay *display = NULL;
    display = gtk_widget_get_display(widget);
    atom = gdk_x11_get_xatom_by_name_for_display(display,"_HILDON_ZOOM_KEY_ATOM");
    XChangeProperty (GDK_DISPLAY_XDISPLAY (display),
                     GDK_WINDOW_XID (GDK_WINDOW (widget->window)),atom,XA_INTEGER,32,
                     PropModeReplace,(unsigned char *)&val,1);
}

/*
 * Store configuration values into GConf
 */
void save_configuration (app_data_t *myapp) {
	GConfClient* gcClient = NULL;

	gcClient = gconf_client_get_default();
	/* Store the values. */
	if (GCONF_IS_CLIENT (gcClient)) {
		if (!gconf_client_set_string(gcClient,g_strconcat(GC_ROOT,"currdir",NULL),(const gchar *)myapp->currdir,NULL)) {
			g_warning("Failed to set %scurrdir to %s\n",GC_ROOT,myapp->currdir);
		}
		if (!gconf_client_set_string(gcClient,g_strconcat(GC_ROOT,"currfname",NULL),(const gchar *)myapp->currfname,NULL)) {
			g_warning("Failed to set %scurrfname to %s\n",GC_ROOT,myapp->currfname);
		}
		if (!gconf_client_set_int(gcClient,g_strconcat(GC_ROOT,"currfindex",NULL),myapp->currfindex,NULL)) {
			g_warning("Failed to set %scurrfindex to %d\n",GC_ROOT,myapp->currfindex);
		}
		if (!gconf_client_set_int(gcClient,g_strconcat(GC_ROOT,"zoomlevel",NULL),myapp->zoomlevel,NULL)) {
			g_warning("Failed to set %szoomlevel to %d\n",GC_ROOT,myapp->zoomlevel);
		}
		g_object_unref(gcClient);
	} else g_warning("Failed to get default GConf client.\n");
}

/*
 * Restore configuration from GConf values
 */
void load_configuration (app_data_t *myapp) {
	GConfClient* gcClient = NULL;

	gcClient = gconf_client_get_default();
	/* Load the values. */
	if (GCONF_IS_CLIENT (gcClient)) {
		myapp->currdir = gconf_client_get_string(gcClient,g_strconcat(GC_ROOT,"currdir",NULL),NULL);
		myapp->currfname = gconf_client_get_string(gcClient,g_strconcat(GC_ROOT,"currfname",NULL),NULL);
		myapp->currfindex = gconf_client_get_int(gcClient,g_strconcat(GC_ROOT,"currfindex",NULL),NULL);
		myapp->zoomlevel = gconf_client_get_int(gcClient,g_strconcat(GC_ROOT,"zoomlevel",NULL),NULL);
		g_object_unref(gcClient);
		store_filenames_in_dir(myapp);
	} else g_warning("Failed to get default GConf client.\n");
}
