//
// RSAManagedTest.cs: NUnit Test Cases for RSAManaged
//	- based on RSACryptoServiceProviderTest
//
// Author:
//	Sebastien Pouliot <sebastien@ximian.com>
//
// (C) 2002, 2003 Motus Technologies Inc. (http://www.motus.com)
// Copyright (C) 2004,2006 Novell, Inc (http://www.novell.com)
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

using NUnit.Framework;
using System;
using System.IO;
using System.Security.Cryptography;
using Mono.Security.Cryptography;

namespace MonoTests.Mono.Security.Cryptography {

	[TestFixture]
	public class RSAManagedTest {

		static int minKeySize = 384;

		[Test]
		public void ConstructorEmpty () 
		{
			// under Mono:: a new key pair isn't generated
			RSAManaged rsa = new RSAManaged ();
			// test default key size
			Assert.AreEqual (1024, rsa.KeySize, "Default key size");
		}

		[Test]
		public void ConstructorKeySize () 
		{
			RSAManaged rsa = new RSAManaged (minKeySize);
			// test default key size
			Assert.AreEqual (minKeySize, rsa.KeySize, "Minimum key size");
		}

		[Test]
		[ExpectedException (typeof (CryptographicException))]
		public void TooSmallKeyPair () 
		{
			RSAManaged rsa = new RSAManaged (256);
		}

		[Test]
		[ExpectedException (typeof (CryptographicException))]
		public void TooBigKeyPair () 
		{
			RSAManaged rsa = new RSAManaged (32768);
		}

		[Test]
		public void Properties () 
		{
			RSAManaged rsa = new RSAManaged (minKeySize);
			Assert.AreEqual (1, rsa.LegalKeySizes.Length, "LegalKeySize");
			Assert.AreEqual (minKeySize, rsa.LegalKeySizes [0].MinSize, "LegalKeySize.MinSize");
			Assert.AreEqual (16384, rsa.LegalKeySizes [0].MaxSize, "LegalKeySize.MaxSize");
			Assert.AreEqual (8, rsa.LegalKeySizes [0].SkipSize, "LegalKeySize.SkipSize");
			Assert.AreEqual ("RSA-PKCS1-KeyEx", rsa.KeyExchangeAlgorithm, "KeyExchangeAlgorithm");
			Assert.AreEqual ("http://www.w3.org/2000/09/xmldsig#rsa-sha1", rsa.SignatureAlgorithm, "SignatureAlgorithm");
			rsa.Clear ();
			Assert.AreEqual (1, rsa.LegalKeySizes.Length, "LegalKeySize(disposed)");
			Assert.AreEqual (minKeySize, rsa.LegalKeySizes [0].MinSize, "LegalKeySize.MinSize(disposed)");
			Assert.AreEqual (16384, rsa.LegalKeySizes [0].MaxSize, "LegalKeySize.MaxSize(disposed)");
			Assert.AreEqual (8, rsa.LegalKeySizes [0].SkipSize, "LegalKeySize.SkipSize(disposed)");
			Assert.AreEqual ("RSA-PKCS1-KeyEx", rsa.KeyExchangeAlgorithm, "KeyExchangeAlgorithm(disposed)");
			Assert.AreEqual ("http://www.w3.org/2000/09/xmldsig#rsa-sha1", rsa.SignatureAlgorithm, "SignatureAlgorithm(disposed)");
		}

		// all keypairs generated by CryptoAPI on Windows
		string CapiXml384 = "<RSAKeyValue><Modulus>vWi7cHIntTcrwIoD0zj/fxoJCDfUHtC5xkMe8pJri7G+T6nKs4zRcLDWRDA0cNhf</Modulus><Exponent>AQAB</Exponent><P>5DwRzr4EPAk1NOUSwI/z1yBJzG2EyrhR</P><Q>1HOEBwbXvsrPQvV7C0MWtHJ22UWgwgmv</Q><DP>qQrEtbePM1gujErOJMl59O/5OOw02mDB</DP><DQ>bUFGqXJsavLTaaTidSU4PO4MjqnPBVqD</DQ><InverseQ>Y+XUwMmlI2G63reScAyc9PHvTPX2fwCg</InverseQ><D>o1Ku5cwZf0IegPzBRZ5NeYy6oxJ430V8n4BLZ8G8/N6kRHnXe70OyzwAylPtxjGh</D></RSAKeyValue>";
		string CapiXml448 = "<RSAKeyValue><Modulus>slPik6EfNCiN2MwhFHngEOqqRlSXeBjmKoLyMVgLCPIQJQqe/BMcyA1gQg4mgionngIgXCMuSOU=</Modulus><Exponent>AQAB</Exponent><P>1mUfmQlTK+Q8sLCCWpsngp7gNc6RANI8mP4v0w==</P><Q>1O77OcIvsaCC6KmemMILqXqm8Bzb3Ud8G9bpZw==</Q><DP>h7L/2fRuAUT4KPm/uCumSWXYEhaJ7xQiqM+SYw==</DP><DQ>fcs1Vbj3ritSSxsx27L/ar9P8hhKd55snpHHTw==</DQ><InverseQ>U7txqL9bAt+BAjqqDfQOAVKTfx3wnAcK5dFx0w==</InverseQ><D>UTV8SEimoiUZu7HyGpYJ4QpMsqyRMgL8xj1Nt0JLKBObONObUBEPAPQKknuE98cM9j/2ufRlrx0=</D></RSAKeyValue>";
		string CapiXml512 = "<RSAKeyValue><Modulus>0ci216d7QTTKkh2Zknwbqlp/Eva1IfzzIA6JZKSI4lcYWMWKAadU7cJtJ5JKGe1aeMi3Y1CDWCpr8fUL28QEkQ==</Modulus><Exponent>AQAB</Exponent><P>/LakSOv8LQUmt3LXA9kMQPAWGk1l2QrwGFL3/k+T+Wc=</P><Q>1IMl0X1/dOqGbEL3agnvgBctWcQaqC2cTzTp7zJqv0c=</Q><DP>B2FmJuMNeFsgKFdoRCqAmxzn6Fi+UrppDKzPpVO5pJ0=</DP><DQ>gWMwPXJpjefU3EmRljBib9yssgDiMg9DIj6XSBmsQLU=</DQ><InverseQ>/IggsX0boXE6UJoWppUEqw6VT13WXxb8mAds5ebxFmE=</InverseQ><D>i+23mA0Mt6fA3smDoCPMSEroq/uHQk541Q8dMdZVv7JmKBJKk1PCppwISyv7fWKVmRTsnGuEnxZFl4N9IR/qyQ==</D></RSAKeyValue>";
		string CapiXml576 = "<RSAKeyValue><Modulus>si995MicOWrr2/VlH1rUAhyFMPVLj5PvYHU8P8I8R77kh1ePn66zuJ0jRsMCpenUzYGuVtqgn0VDjiaYjo7cE2PZYESLkKJl</Modulus><Exponent>AQAB</Exponent><P>3wDFrI6eUAh72YxjdZD0b93/a/NEV26pp1c8j5fN3UUzA14r</P><Q>zI0O3NKXCgn3cC3Pe2ZUYEPbQaQDyMdSAiF06L1miwVHu0mv</Q><DP>uQaC/LE4WV8wo0gAHcOvjEG9c2vcAE8pJFcVQG7LuBN6SAkR</DP><DQ>THffOoYvmL0pF3LIweT7XhGAAgYCtChvbAR95BQKJaaPrmBZ</DQ><InverseQ>KTOfvWHp363Cl6r2lXc/gINaRqNo+iWRLeoBa6/vk0Kt5sWn</InverseQ><D>QZ9L9h4Lqvm9s1xEya9htQVb6BPuqAoLdrK2ZaTbHnEnKNEN7oLMr1Ca5zh/E6xXKCmwClWyULeKiYq7vYLUXhaemtT1rN9h</D></RSAKeyValue>";
		string CapiXml640 = "<RSAKeyValue><Modulus>vqS9CNxiRdSc6x3nJOWnPi7gVA4Vsnhd8cTAaMBKuvI0FRhhDsTRsjr8PeH4Y7yMg8XSucr3I8jkYgapBW8s5hmSvdWZhPHRGEe1MiEIeMM=</Modulus><Exponent>AQAB</Exponent><P>7Ojjp/PqeacLRM/MP76wDnp4k+BpUSbitNwkPBAxL5KcBFVwhp+YVw==</P><Q>zgFxY3BFx/ISEciI3ONux4iXd/cIwxrmq9CS3tBIecHTq0JgjXNPdQ==</Q><DP>TstCdGjOsnlZaSCHuSfN3HLlSaGYxZHeUvLo5kUoZr8nPtW/4DaSbw==</DP><DQ>DCIhn425TnF/hvScuwXaPH5bDYHLTlKDS5NZUy5JVoKICQo7zZkBEQ==</DQ><InverseQ>Xhn+9dXsK07s51n6SyRs3wRJEPwuIaNDq5EQJUQNb2Gbdb4HgdpM2w==</InverseQ><D>JhKALeTVO1zaeZnfL18tpx11b1PgwWOIX2ALjN/aDLuR6ySTPX/Q4I6zBOzVa/KB6VMNNk4abIKUvOp6MBNYIMRMK8veaIJxYrn+JAOShwE=</D></RSAKeyValue>";
		string CapiXml704 = "<RSAKeyValue><Modulus>wbclY53TAIx2E6uPkYeYNVFPYnHgLDJGO1ZoSQcUm6ccySsJvVWAP9TWLIOTjJh2zaQn5gore87ONk50L/+FMZ5uaNrbxxPCxORBLzBofFaEbFxEfu9g6w==</Modulus><Exponent>AQAB</Exponent><P>80o27emwkBtzYHg/0yoYjwjcgqhtVSktbZ8IZ+tdBEbt7gtnpyDd3rpTjc8=</P><Q>y9XqvMruAcvSJjV4rHnCX5Na4XahLhbrvuSsATirpSV+YaD8LrTd5VHwfiU=</Q><DP>aYoSVhcAPyOJY5oGpgYm7TW84vlJpJ3eVSWeMeaKTWavpIpSBtBdL1fJE0U=</DP><DQ>DDbCPRPP1S24Zu+3TmZxXja/MFliaReYgrTDFcOmjVvEyebSlZ3i2fXh+j0=</DQ><InverseQ>O9RaZjZGqG0stVjpDDUaHGk9JNbFChHL9KQ3Kyb6qBLmDNvwobxPnHY1+k4=</InverseQ><D>Wfgbz2Zvp2OVO5GEvoyBbtHy0sAc46n94HVDPhehxKEax3vLrDnXtM2/IzCxXibWqBBmF8tw156Yo4Av2/EpGxvolidOpTXVxNX4aCb3dy05VQ/6WEem0Q==</D></RSAKeyValue>";
		string CapiXml768 = "<RSAKeyValue><Modulus>3VQEd4kKl0vQG/OzLDTVcgVqaBG9KAz6NzJOu5a6mlHOYSIG5ybYwOJJpXDzZuXewBChqQNpmxgGAae/SuTy+EDQNzAQ+mzeYrEaD6C8vqwMYVanRcbNYtDC9cxAH6Ef</Modulus><Exponent>AQAB</Exponent><P>/UKWrRmudV9RYnelAK2QqJ0DSE/oKHgVtHQ2AxyJueDnMelyzVrCv7ICnL1ymogp</P><Q>37j+EpSqJRP6JRyGMDXG2mj5UXnEqgqeYXaHHNHNYsFcZWAKitRt8uUQtLz9k6gH</Q><DP>fLkVYUwbeZwmhVqsvOe9LPyeSDdp+rwg3Ey66M9eGFdlJFR2gCFpdWRiGvTFgsr5</DP><DQ>1PuxbZD4NdpB3si7+vOHPvCGwgkRr+vyAcksMcSOKgD47E21W5uGnKFz+Qyev85L</DQ><InverseQ>dnez9inW0taRSo0QzoQ3CNGTfm9qzh9Mlx6GKjnyzLLBJ87cOfzW2qVksBlmkDJx</InverseQ><D>RK5ayIFFQRfsl4/zTMeEaOKXV34Rtcj5KIG6/ulSNKsoIOo/PCHI52oRMn6veYHhHrKclBeqtAlqOJG0xrEy/ZjorBJnCB/ZSoPPuS4h7budjF2zYiBHJkRwZcxhe+nx</D></RSAKeyValue>";
		string CapiXml832 = "<RSAKeyValue><Modulus>wv35yrTrD3gjLxJEJ6HMOJsUfshcCZKGLwlxZO42TlAHK90vJPT0Q7SekeHqSVi1DJ7AAB7XW7MqoSE4SKVVioQQZSFz/HkA1fklTOaubfwRwv0vy/A3js1xPhwTs/SbDpNlXMoiEAM=</Modulus><Exponent>AQAB</Exponent><P>87j8swLApJhpXG4kXLIN86xkdbh/ccqGlpeVE1EGC0yL9OJhbkKC87nvhmQZKMRqcreaAw==</P><Q>zNCRE+qsVtVPD5o8Kf50HFdTQlZY56IA3DLRLHd0qkpNkrCP3gPJNNhwt5kqewX1dlzSAQ==</Q><DP>VXli5kTo2tC44rmd9wRa8EJdWQvDZlzoppeyqHuZ6wyFaPSbxtd8pY1n+3HPgQShcGoDkQ==</DP><DQ>QK6JcqnFHXMmEb5ay8RRNPYbFDMixpwx+1iNGKbtEC6BCxd5h9rYOzkzd7gGY23BVE0CAQ==</DQ><InverseQ>FMqDOpFyGkiJ74uTwdqFABR9dBcN4x8rpYw5trf5DBOVC1Desh+mRDAZwKPVDLCBpqrg9A==</InverseQ><D>YuJPUbfr2J1xIkbeH0cS/MXQX/cVjZkrySC9y5RuH8q/yEPTy6cZVFh9bbemi2mb7vl2nfWO+VXLN+3OStN0SU3/2XQXfhZm4dqk5iU+R5n5g9kjPBNLH8DIWYjGVgCvBqH59DHhLAE=</D></RSAKeyValue>";
		string CapiXml896 = "<RSAKeyValue><Modulus>nNiWWLAg2htBUkoBD7DTDIALapuiIt45qHJdPVbRQE5PXTcJUJ2SAEHCtQYtIeQam0WL2D78PIfAf8hGFx/72Ghn9b+NcZ8b7UHTdna+5O6aBOHyaD80FOmyfEjAjI1jS95XnGlqoD0QaDpkk4xnMQ==</Modulus><Exponent>AQAB</Exponent><P>zgkddb3AAG9p8LeF+TbcJ2Q3xvvODITJRmUe55IDOb/BhRF1/JTuCAeuHiOtVWKuxcmsJp464Bc=</P><Q>wuG8338Tyk658L7cByQVoV5/AJx+5RKN6rO0CGrPSGjOkIPDyNzwgpmotelo+NiTEiEiNoa49/c=</Q><DP>Dgl+8VOhLiZpEFZgkU8UhraEOlFTg3TUhbBD/8Dp6VhQJfG/mRrIcNGdIj6KA6Q6hg0sZmEnX7c=</DP><DQ>MuxTsz78h9+8fKkSy5blRA5yN1GtYuRPSyX8BDsMwQoJ9/9GWKVK/4VxbV95e5T0EUexLfhUOw8=</DQ><InverseQ>NbWl5XrJ6hxvE8gpRcj2oc+hAUT96ej6eysD1JTQyV1eKzc9+Mpaoo22tGSrR52s9EZRqHWmKxM=</InverseQ><D>LGd9GRq0EkuJELz20/Rhq7ZMhSAOpQR5GmFWWFlN4IDLIz7DmlkhzoTPlORsvp2Pksn7r3sVeiUbL3S1rXfIpwEjeseHfBYR54d6EqmPABqPj9UoADXDoYA1VYz+Ni2uFYlMC7Wh4kzlT0zYRc1XKQ==</D></RSAKeyValue>";
		string CapiXml960 = "<RSAKeyValue><Modulus>3nb1VtiEuSii1rnDR+yrarcAHDlvbgLUTJGUvuVEMcvQ3vYNNabwBsM46X8ab78Ac2ZBk2jI2S4736KL2S8C2e8rXva7Jb/j+fyVt6N+NrI3dkNE2NoWZ8VM+rUEdKNvPANbf6n7hvDVfJLCvdCctdtEYACkujsb</Modulus><Exponent>AQAB</Exponent><P>/nWeSPagl8TOHzSKB+CRx4MbxOlIR9RNw3RAbBZq/1r4YZtTvvSutV6e1/2yIWYblHPwiALb/FyB4XUj</P><Q>38/Agt+WwnMNeyv4ngVk0r6ffValkVCMlbPdueEqEXCFqOguTvnneINg7+T1Q1YLPXDtjR6NFGKCLG2p</Q><DP>Kebkcc9rEpLt/mWAdVudpeUJJZvksy9avtzd3u6yH+qzDB+v4roYKvWx4o98TdOqpv+QlFUkNKJnIOFR</DP><DQ>SPH9XKpjCJ1XF34NWfOIGOfoM4G6FNKb/27QJXUtsOFrrtF9xl/NAYpQXd/R0FCK+UuFISmD8dDpfHGB</DQ><InverseQ>9YlJCPfSUw8ZEauBdt9hsAybMD20we2uoNdHdDRKvdTtK8m6spS4ohG2ovzfRkiKSxQVzB3jQoYcK9L6</InverseQ><D>jrmHQYZ78Ebv4g8gCD8A4uAxg+odYVkTV2R3J9nzXHdEtCbr5qYJjG0nUDapgVPrOB48qBQr95o/84RWPdIz3I+Jzpme1c4FpWs1/iXAbA/+xnzDGtXVTnJrMfkOsXt2MYBF7+f7zWGNlOGs+zkt/fP/jeOu76Nh</D></RSAKeyValue>";
		string CapiXml1024 = "<RSAKeyValue><Modulus>uNXUqvMdjnmd36IS0Gr/lTAp63mZqoayYcN/YHnGs82eVrEWS5vAYqPWx1488BhUBh7kHvVhP5kkKh+i6pfLKAEssy5jnw9h09lgCClBHyntqFT82AlmiaFK8uVkV8rCjDol2nBpvVgNonF+JhQyexWTEI94Phjwvj/6yZRqaLM=</Modulus><Exponent>AQAB</Exponent><P>3GIqx8V+yrPI8jsFd1NYTNxJjodtLNufqmYbn7q6fIaPEdC0QG0fcfYPREfKka8xpZ4K7mqgLjm0ivDWi9Y5Iw==</P><Q>1rTxeoBb9aQcoGLhUEeZQNhY+JhhrGsMtZH4fPxy/30JUtggQmZRR12XdjBZ0KcKmsOMbxCauYD8Y6ozFhKzMQ==</Q><DP>lBMZb3TRRl0aDTd+6rgDQlFY0v1Ha7Z9Rz6oHOCX4IeApZW3JvqrACU2CMi74Lr3/rF74smdqrF3D0vWu8pKRQ==</DP><DQ>SBWZ2UoNFcySe9qW0PAo6Nd6D4SBjnSmYLNwXO4Y4eQl5DWBpylY8n/eoSSckuvyKIGsvYEyoUNH+WIkIq4GkQ==</DQ><InverseQ>M3KkaBwusTLiKBlu048fx1WyRfLLEa7clsU7a4JN5wZ66h7wElKj77r9BCXz5UZFo4j3AiBBC/701S0mVQYXOA==</InverseQ><D>NVIqaa58xk87RfphZxKW7JjaXv3TYKg+6YkWQ+Sdd91HYkbv4Zvq4gnVuenrtm+uPZ3HvU6YYVpyXlyGCRsFFeXMwJ8Y7zrF6KEs+6hJ4jqBELWUoEPrzU4889v8aIoS+BCMYptpkmzvxjfEWdTPbqjd/tJ25ObbavggoST6ScE=</D></RSAKeyValue>";
		string CapiXml1536 = "<RSAKeyValue><Modulus>1bkuWuGsKLkvSvrYYhOPVD37bARRUw0U5/yRCjam50+gLGKmxAjzaRjhGQUaaeC2S5DioGckqiqr1zkfSLyQ7KQte5oynxkTdT0QCXHJeVWgOWAknX8sNrAVNt97LOrMXXYZ2JsYMmYiVNqJRTaURWXxdJ5XibXMCRL1pfHbwnL4iqUge1tuiQbFod//7pjl90TukIutHVEyf8BFn5LRQJI4xroQdMJg0/vvRSp8jgy0LoeJ8LkTG+RjUe3XsgmT</Modulus><Exponent>AQAB</Exponent><P>8Vz3+1ixDfMPPNENDw17Rmdt8IRM32fRBJ/TX8LYfVkxlCB6m25Pd4SxH7mcdF1sgD3qgC+uKekJZb69Q0XE/CoJxPD6mYq0o4XEsz0rwy4NU/u//YpxZp84v5cDKEoR</P><Q>4q8eodQoHbiuBH5f/WelCnm2Ym22AmFIEIWpwKGY8MlnvpK/pN+N4iXrkeNv7OVofOdE7g0lw5cTbDMZV2eMyEIYKUi7YOXA8TIlSdTuDqW/5mPg0inqNLEg2PZGahVj</Q><DP>lHw7Wve/RPOpBiMdw4rpsfBjZDogCLiXkB67LQhzovnCVHx+sSx12vNY/El2BOiMnYB5yY6LuODSlTN4v/AmNXOvOud9ZAQ/CPJ8hkA1sgecz3PrMxF+nkGJ6eP/X0Ph</DP><DQ>p2cHyh6xGXHfIPZq0OqPmTLVG89FkHBjFcB/4f/0wC0cbkJVQN7PGulCFFTPvTSVe1gXMW2IK+8PquH5nvCbqPAWg7ZwmlhRqk2L+ABFZY/GLdAooUvO5+a/CTqmOYVd</DQ><InverseQ>x3EJeS7EP1h5zuKKITblv0V9R3z6KEQAbKPp3zfckHX1fvuR+3VP4FLU63D0yxd2BOjhGTvyJvjrVSDCP+Ndk+h7ZT9MkzW/i6ifiHG/5L/xaEpxjJGq35OOOuHqCmKc</InverseQ><D>rtTUab3QImQSnuiCmABeMFCf2qXRjPnXj0qZr1wzvmbxpT1yJE0aKXATu27kQ5ZyKXC1IvgdEyLi/aWZxNuURjCrkD/8hw9xTmeMNd2iLaJw9l6CtV/x4C68u+2nCoBq/Gv/ht9RqYRS6ODUTk1aOL1mPNSHT/MeFNK+06lyQNSaf95fMxGPIcnnJ1vtdkHgk2vi5ALSQf5tGIgVY1EFr55M9ABTg1p98XwIQgjIxCZcL4BygfMk1YbESaQFvzth</D></RSAKeyValue>";
		string CapiXml2048 = "<RSAKeyValue><Modulus>9TFEYeTahPGOXtcRZrLQu1gHm875jv1jgxy0R9V3E/vDyEhNbsRK76rvEs3ljw6spQBQE1zd5wfUvFnhVETOuS0z9aXSAljxSIbyhkL/jT7/4vHEHe/GnBQZPdvJ9PtPr0ssLROy8xcsUEBDwxNRPNHSpmub3FHZzHsoMlJV0cOGmn0qqeMuMrP8zVDycXhXZNsn8v5RPC38qk6VDHdCqGz/90IRYonpFr5my9LaguJHVNCq8BRVa1oJkDtNUFhFXjVHnmxmhS9Mqm2obEpTp/jUk+bRqnygKB3up9OGYdLO1rAv2IoH/K2sxcvnedDzieRmPE/9Y9JBP/UvM+VULw==</Modulus><Exponent>AQAB</Exponent><P>//j7DyPW0E5t1/xvT9iojXQZfu9t/t0bdJMkRQVAYzJL5+UUILOA+70hMtPIvrU4gdeK0CPKfbbkY0UmD4QTb9nFq4aOsHruMZgWIeqHLlEo+VOYX3+hLv+p/aXX+Wj5PvQWLYZBH1sdOLE21nEVHFXZIVev3PcIYU+P0L8R3JE=</P><Q>9Tf9pWu4aWF16h2/nCmY8HAOPNMgcHoC6jtVuJJX+8pio6VLbA862e6ptB7fZdCDDXB2y/52tmG6uAcmOIqUDZB4eLHcOWxnYctcWrvjODMIABno7EuhR7zemelN89ME97zuY7nQSnVRroWD7l3oU5pN3yIe9Y5bnMKgr1nnhL8=</Q><DP>e4AhgZiFGFP+42rEOf4KtNUDSB81LvZ3PLORmEuEWVf3D5eTMoPpA4yo6+EKxhihfuQD8ZCTLjyDzPGb/3h1+E1V9gAh8DwfmIYMh6ikOFCoOEOBDPKDTi3EUsElhwyC1UDnQme4G+zWGHhIQQambNluvYuVKkN2I51Dgi/t6kE=</DP><DQ>ZIzd9Au0pXlyOVqTbDxeWxEHtYc5AQX21gcYgkN30mZNhh7MS3X/QserTJFwNzaF1mfsPn+MPALc5oL/+CVSyjEYRR1hWSaLSb1ylD4A0NWUDT6SlPn6GwlmGaRh833uxorxEXFq6G0s3iwfSgm1rzpRfhJmsXf7Ns9TxjNOTM8=</DQ><InverseQ>uGCs/TRjtVGBXsTV/9fPMMel0L+WbrdqIPu1GBSIiXk8tWWi+P1lACSTf0iCJ6SIKB26N57zAE9r7zaocVX0R/3NJ3+eDDztV9i6AOsK6X/AitxXetPQDIflbbjAEzeZUUJf2onSAITIw6TaAs620a8HpPw3W7Yd0Oe/lt2M+GA=</InverseQ><D>ao8oyuqs1U4ts6YAWAOql2DgnaRL7QrObrLQ3s802yh1o9tYW6VPc+1zzVZSR+P2wBbsth2MCtXqbJkbRoZI2U194WpZZM/GvOB5EkSXz4jrqHOt6dzFEhviBHI6yQ9XSDWBU23WPbN6fL4RNPx2N9pwlAb8S7n+z9FOXOBPj8NyMMtykGqgPyHEAQnuhfLuf/J6OGrHX8dzsZAo6o/bnk49Rv0C6nijE74TeAwJ36UuI9Lw+h6hwtSerEN6o1vy6mIjrjdDM07mKKUMFwkEhKJc6ckBb0xVp/3VRlJb1ocYQHbfjCAPjecR8OVfjaIy9WPMj8WXHegJpiBugWK9wQ==</D></RSAKeyValue>";
		// limit to 2048 bits (else key won't fit in a line for Visual Studio)

		// same keypairs but without CRT
		string CapiXml384woCRT = "<RSAKeyValue><Modulus>vWi7cHIntTcrwIoD0zj/fxoJCDfUHtC5xkMe8pJri7G+T6nKs4zRcLDWRDA0cNhf</Modulus><Exponent>AQAB</Exponent><D>o1Ku5cwZf0IegPzBRZ5NeYy6oxJ430V8n4BLZ8G8/N6kRHnXe70OyzwAylPtxjGh</D></RSAKeyValue>";
		string CapiXml448woCRT = "<RSAKeyValue><Modulus>slPik6EfNCiN2MwhFHngEOqqRlSXeBjmKoLyMVgLCPIQJQqe/BMcyA1gQg4mgionngIgXCMuSOU=</Modulus><Exponent>AQAB</Exponent><D>UTV8SEimoiUZu7HyGpYJ4QpMsqyRMgL8xj1Nt0JLKBObONObUBEPAPQKknuE98cM9j/2ufRlrx0=</D></RSAKeyValue>";
		string CapiXml512woCRT = "<RSAKeyValue><Modulus>0ci216d7QTTKkh2Zknwbqlp/Eva1IfzzIA6JZKSI4lcYWMWKAadU7cJtJ5JKGe1aeMi3Y1CDWCpr8fUL28QEkQ==</Modulus><Exponent>AQAB</Exponent><D>i+23mA0Mt6fA3smDoCPMSEroq/uHQk541Q8dMdZVv7JmKBJKk1PCppwISyv7fWKVmRTsnGuEnxZFl4N9IR/qyQ==</D></RSAKeyValue>";
		string CapiXml576woCRT = "<RSAKeyValue><Modulus>si995MicOWrr2/VlH1rUAhyFMPVLj5PvYHU8P8I8R77kh1ePn66zuJ0jRsMCpenUzYGuVtqgn0VDjiaYjo7cE2PZYESLkKJl</Modulus><Exponent>AQAB</Exponent><D>QZ9L9h4Lqvm9s1xEya9htQVb6BPuqAoLdrK2ZaTbHnEnKNEN7oLMr1Ca5zh/E6xXKCmwClWyULeKiYq7vYLUXhaemtT1rN9h</D></RSAKeyValue>";
		string CapiXml640woCRT = "<RSAKeyValue><Modulus>vqS9CNxiRdSc6x3nJOWnPi7gVA4Vsnhd8cTAaMBKuvI0FRhhDsTRsjr8PeH4Y7yMg8XSucr3I8jkYgapBW8s5hmSvdWZhPHRGEe1MiEIeMM=</Modulus><Exponent>AQAB</Exponent><D>JhKALeTVO1zaeZnfL18tpx11b1PgwWOIX2ALjN/aDLuR6ySTPX/Q4I6zBOzVa/KB6VMNNk4abIKUvOp6MBNYIMRMK8veaIJxYrn+JAOShwE=</D></RSAKeyValue>";
		string CapiXml704woCRT = "<RSAKeyValue><Modulus>wbclY53TAIx2E6uPkYeYNVFPYnHgLDJGO1ZoSQcUm6ccySsJvVWAP9TWLIOTjJh2zaQn5gore87ONk50L/+FMZ5uaNrbxxPCxORBLzBofFaEbFxEfu9g6w==</Modulus><Exponent>AQAB</Exponent><D>Wfgbz2Zvp2OVO5GEvoyBbtHy0sAc46n94HVDPhehxKEax3vLrDnXtM2/IzCxXibWqBBmF8tw156Yo4Av2/EpGxvolidOpTXVxNX4aCb3dy05VQ/6WEem0Q==</D></RSAKeyValue>";
		string CapiXml768woCRT = "<RSAKeyValue><Modulus>3VQEd4kKl0vQG/OzLDTVcgVqaBG9KAz6NzJOu5a6mlHOYSIG5ybYwOJJpXDzZuXewBChqQNpmxgGAae/SuTy+EDQNzAQ+mzeYrEaD6C8vqwMYVanRcbNYtDC9cxAH6Ef</Modulus><Exponent>AQAB</Exponent><D>RK5ayIFFQRfsl4/zTMeEaOKXV34Rtcj5KIG6/ulSNKsoIOo/PCHI52oRMn6veYHhHrKclBeqtAlqOJG0xrEy/ZjorBJnCB/ZSoPPuS4h7budjF2zYiBHJkRwZcxhe+nx</D></RSAKeyValue>";
		string CapiXml832woCRT = "<RSAKeyValue><Modulus>wv35yrTrD3gjLxJEJ6HMOJsUfshcCZKGLwlxZO42TlAHK90vJPT0Q7SekeHqSVi1DJ7AAB7XW7MqoSE4SKVVioQQZSFz/HkA1fklTOaubfwRwv0vy/A3js1xPhwTs/SbDpNlXMoiEAM=</Modulus><Exponent>AQAB</Exponent><D>YuJPUbfr2J1xIkbeH0cS/MXQX/cVjZkrySC9y5RuH8q/yEPTy6cZVFh9bbemi2mb7vl2nfWO+VXLN+3OStN0SU3/2XQXfhZm4dqk5iU+R5n5g9kjPBNLH8DIWYjGVgCvBqH59DHhLAE=</D></RSAKeyValue>";
		string CapiXml896woCRT = "<RSAKeyValue><Modulus>nNiWWLAg2htBUkoBD7DTDIALapuiIt45qHJdPVbRQE5PXTcJUJ2SAEHCtQYtIeQam0WL2D78PIfAf8hGFx/72Ghn9b+NcZ8b7UHTdna+5O6aBOHyaD80FOmyfEjAjI1jS95XnGlqoD0QaDpkk4xnMQ==</Modulus><Exponent>AQAB</Exponent><D>LGd9GRq0EkuJELz20/Rhq7ZMhSAOpQR5GmFWWFlN4IDLIz7DmlkhzoTPlORsvp2Pksn7r3sVeiUbL3S1rXfIpwEjeseHfBYR54d6EqmPABqPj9UoADXDoYA1VYz+Ni2uFYlMC7Wh4kzlT0zYRc1XKQ==</D></RSAKeyValue>";
		string CapiXml960woCRT = "<RSAKeyValue><Modulus>3nb1VtiEuSii1rnDR+yrarcAHDlvbgLUTJGUvuVEMcvQ3vYNNabwBsM46X8ab78Ac2ZBk2jI2S4736KL2S8C2e8rXva7Jb/j+fyVt6N+NrI3dkNE2NoWZ8VM+rUEdKNvPANbf6n7hvDVfJLCvdCctdtEYACkujsb</Modulus><Exponent>AQAB</Exponent><D>jrmHQYZ78Ebv4g8gCD8A4uAxg+odYVkTV2R3J9nzXHdEtCbr5qYJjG0nUDapgVPrOB48qBQr95o/84RWPdIz3I+Jzpme1c4FpWs1/iXAbA/+xnzDGtXVTnJrMfkOsXt2MYBF7+f7zWGNlOGs+zkt/fP/jeOu76Nh</D></RSAKeyValue>";
		string CapiXml1024woCRT = "<RSAKeyValue><Modulus>uNXUqvMdjnmd36IS0Gr/lTAp63mZqoayYcN/YHnGs82eVrEWS5vAYqPWx1488BhUBh7kHvVhP5kkKh+i6pfLKAEssy5jnw9h09lgCClBHyntqFT82AlmiaFK8uVkV8rCjDol2nBpvVgNonF+JhQyexWTEI94Phjwvj/6yZRqaLM=</Modulus><Exponent>AQAB</Exponent><D>NVIqaa58xk87RfphZxKW7JjaXv3TYKg+6YkWQ+Sdd91HYkbv4Zvq4gnVuenrtm+uPZ3HvU6YYVpyXlyGCRsFFeXMwJ8Y7zrF6KEs+6hJ4jqBELWUoEPrzU4889v8aIoS+BCMYptpkmzvxjfEWdTPbqjd/tJ25ObbavggoST6ScE=</D></RSAKeyValue>";
		string CapiXml1536woCRT = "<RSAKeyValue><Modulus>1bkuWuGsKLkvSvrYYhOPVD37bARRUw0U5/yRCjam50+gLGKmxAjzaRjhGQUaaeC2S5DioGckqiqr1zkfSLyQ7KQte5oynxkTdT0QCXHJeVWgOWAknX8sNrAVNt97LOrMXXYZ2JsYMmYiVNqJRTaURWXxdJ5XibXMCRL1pfHbwnL4iqUge1tuiQbFod//7pjl90TukIutHVEyf8BFn5LRQJI4xroQdMJg0/vvRSp8jgy0LoeJ8LkTG+RjUe3XsgmT</Modulus><Exponent>AQAB</Exponent><D>rtTUab3QImQSnuiCmABeMFCf2qXRjPnXj0qZr1wzvmbxpT1yJE0aKXATu27kQ5ZyKXC1IvgdEyLi/aWZxNuURjCrkD/8hw9xTmeMNd2iLaJw9l6CtV/x4C68u+2nCoBq/Gv/ht9RqYRS6ODUTk1aOL1mPNSHT/MeFNK+06lyQNSaf95fMxGPIcnnJ1vtdkHgk2vi5ALSQf5tGIgVY1EFr55M9ABTg1p98XwIQgjIxCZcL4BygfMk1YbESaQFvzth</D></RSAKeyValue>";
		string CapiXml2048woCRT = "<RSAKeyValue><Modulus>9TFEYeTahPGOXtcRZrLQu1gHm875jv1jgxy0R9V3E/vDyEhNbsRK76rvEs3ljw6spQBQE1zd5wfUvFnhVETOuS0z9aXSAljxSIbyhkL/jT7/4vHEHe/GnBQZPdvJ9PtPr0ssLROy8xcsUEBDwxNRPNHSpmub3FHZzHsoMlJV0cOGmn0qqeMuMrP8zVDycXhXZNsn8v5RPC38qk6VDHdCqGz/90IRYonpFr5my9LaguJHVNCq8BRVa1oJkDtNUFhFXjVHnmxmhS9Mqm2obEpTp/jUk+bRqnygKB3up9OGYdLO1rAv2IoH/K2sxcvnedDzieRmPE/9Y9JBP/UvM+VULw==</Modulus><Exponent>AQAB</Exponent><D>ao8oyuqs1U4ts6YAWAOql2DgnaRL7QrObrLQ3s802yh1o9tYW6VPc+1zzVZSR+P2wBbsth2MCtXqbJkbRoZI2U194WpZZM/GvOB5EkSXz4jrqHOt6dzFEhviBHI6yQ9XSDWBU23WPbN6fL4RNPx2N9pwlAb8S7n+z9FOXOBPj8NyMMtykGqgPyHEAQnuhfLuf/J6OGrHX8dzsZAo6o/bnk49Rv0C6nijE74TeAwJ36UuI9Lw+h6hwtSerEN6o1vy6mIjrjdDM07mKKUMFwkEhKJc6ckBb0xVp/3VRlJb1ocYQHbfjCAPjecR8OVfjaIy9WPMj8WXHegJpiBugWK9wQ==</D></RSAKeyValue>";

		// import/export XML keypairs
		// so we know that Mono can use keypairs generated by CryptoAPI
		[Test]
		public void CapiXmlImportExport () 
		{
			RSAManaged rsa = new RSAManaged ();

			rsa.FromXmlString (CapiXml384);
			Assert.AreEqual (CapiXml384, rsa.ToXmlString (true), "Capi-Xml384");

			rsa.FromXmlString (CapiXml448);
			Assert.AreEqual (CapiXml448, rsa.ToXmlString (true), "Capi-Xml448");

			rsa.FromXmlString (CapiXml512);
			Assert.AreEqual (CapiXml512, rsa.ToXmlString (true), "Capi-Xml512");

			rsa.FromXmlString (CapiXml576);
			Assert.AreEqual (CapiXml576, rsa.ToXmlString (true), "Capi-Xml576");

			rsa.FromXmlString (CapiXml640);
			Assert.AreEqual (CapiXml640, rsa.ToXmlString (true), "Capi-Xml640");

			rsa.FromXmlString (CapiXml704);
			Assert.AreEqual (CapiXml704, rsa.ToXmlString (true), "Capi-Xml704");

			rsa.FromXmlString (CapiXml768);
			Assert.AreEqual (CapiXml768, rsa.ToXmlString (true), "Capi-Xml768");

			rsa.FromXmlString (CapiXml832);
			Assert.AreEqual (CapiXml832, rsa.ToXmlString (true), "Capi-Xml832");

			rsa.FromXmlString (CapiXml896);
			Assert.AreEqual (CapiXml896, rsa.ToXmlString (true), "Capi-Xml896");

			rsa.FromXmlString (CapiXml960);
			Assert.AreEqual (CapiXml960, rsa.ToXmlString (true), "Capi-Xml960");

			rsa.FromXmlString (CapiXml1024);
			Assert.AreEqual (CapiXml1024, rsa.ToXmlString (true), "Capi-Xml1024");

			rsa.FromXmlString (CapiXml1536);
			Assert.AreEqual (CapiXml1536, rsa.ToXmlString (true), "Capi-Xml1536");

			rsa.FromXmlString (CapiXml2048);
			Assert.AreEqual (CapiXml2048, rsa.ToXmlString (true), "Capi-Xml2048");
		}

		[Test]
		public void CapiXmlImportWithoutCRT () 
		{
			RSAManaged rsa = new RSAManaged ();

			rsa.FromXmlString (CapiXml384woCRT);
			Assert.AreEqual (CapiXml384woCRT, rsa.ToXmlString (true), "Capi-Xml384-WithoutCRT");

			rsa.FromXmlString (CapiXml448woCRT);
			Assert.AreEqual (CapiXml448woCRT, rsa.ToXmlString (true), "Capi-Xml448-WithoutCRT");

			rsa.FromXmlString (CapiXml512woCRT);
			Assert.AreEqual (CapiXml512woCRT, rsa.ToXmlString (true), "Capi-Xml512-WithoutCRT");

			rsa.FromXmlString (CapiXml576woCRT);
			Assert.AreEqual (CapiXml576woCRT, rsa.ToXmlString (true), "Capi-Xml576-WithoutCRT");

			rsa.FromXmlString (CapiXml640woCRT);
			Assert.AreEqual (CapiXml640woCRT, rsa.ToXmlString (true), "Capi-Xml640-WithoutCRT");

			rsa.FromXmlString (CapiXml704woCRT);
			Assert.AreEqual (CapiXml704woCRT, rsa.ToXmlString (true), "Capi-Xml704-WithoutCRT");

			rsa.FromXmlString (CapiXml768woCRT);
			Assert.AreEqual (CapiXml768woCRT, rsa.ToXmlString (true), "Capi-Xml768-WithoutCRT");

			rsa.FromXmlString (CapiXml832woCRT);
			Assert.AreEqual (CapiXml832woCRT, rsa.ToXmlString (true), "Capi-Xml832-WithoutCRT");

			rsa.FromXmlString (CapiXml896woCRT);
			Assert.AreEqual (CapiXml896woCRT, rsa.ToXmlString (true), "Capi-Xml896-WithoutCRT");

			rsa.FromXmlString (CapiXml960woCRT);
			Assert.AreEqual (CapiXml960woCRT, rsa.ToXmlString (true), "Capi-Xml960-WithoutCRT");

			rsa.FromXmlString (CapiXml1024woCRT);
			Assert.AreEqual (CapiXml1024woCRT, rsa.ToXmlString (true), "Capi-Xml1024-WithoutCRT");

			rsa.FromXmlString (CapiXml1536woCRT);
			Assert.AreEqual (CapiXml1536woCRT, rsa.ToXmlString (true), "Capi-Xml1536-WithoutCRT");

			rsa.FromXmlString (CapiXml2048woCRT);
			Assert.AreEqual (CapiXml2048woCRT, rsa.ToXmlString (true), "Capi-Xml2048-WithoutCRT");
		}

		private void EncryptDecrypt (string msg, RSAManaged rsa) 
		{
			RSAParameters param = rsa.ExportParameters (false);

			byte[] data = { 0xFF, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13 };
			// we don't need the private key to encrypt
			RSAManaged pubkey = new RSAManaged ();
			pubkey.ImportParameters (param);
			byte[] enc = pubkey.EncryptValue (data);

			byte[] dec = rsa.DecryptValue (enc);
			// note: the decrypted value is now right padded with zeros
			Assert.IsTrue (BitConverter.ToString (dec).EndsWith (BitConverter.ToString (data)), msg);
		}

		// Validate that we can sign with every keypair and verify the signature
		// With Mono this means that we can use CAPI keypair to sign and verify.
		// For Windows this doesn't mean much.
		[Test]
		public void CapiEncryptDecrypt () 
		{
			RSAManaged rsa = new RSAManaged ();

			rsa.FromXmlString (CapiXml384);
			EncryptDecrypt ("Capi-384", rsa);

			rsa.FromXmlString (CapiXml448);
			EncryptDecrypt ("Capi-448", rsa);

			rsa.FromXmlString (CapiXml512);
			EncryptDecrypt ("Capi-512", rsa);

			rsa.FromXmlString (CapiXml576);
			EncryptDecrypt ("Capi-576", rsa);

			rsa.FromXmlString (CapiXml640);
			EncryptDecrypt ("Capi-640", rsa);

			rsa.FromXmlString (CapiXml704);
			EncryptDecrypt ("Capi-704", rsa);

			rsa.FromXmlString (CapiXml768);
			EncryptDecrypt ("Capi-768", rsa);

			rsa.FromXmlString (CapiXml832);
			EncryptDecrypt ("Capi-832", rsa);

			rsa.FromXmlString (CapiXml896);
			EncryptDecrypt ("Capi-896", rsa);

			rsa.FromXmlString (CapiXml960);
			EncryptDecrypt ("Capi-960", rsa);

			rsa.FromXmlString (CapiXml1024);
			EncryptDecrypt ("Capi-1024", rsa);

			rsa.FromXmlString (CapiXml1536);
			EncryptDecrypt ("Capi-1536", rsa);

			rsa.FromXmlString (CapiXml2048);
			EncryptDecrypt ("Capi-2048", rsa);
		}

		[Test]
		public void CapiEncryptDecryptWithoutCRT () 
		{
			RSAManaged rsa = new RSAManaged ();

			rsa.FromXmlString (CapiXml384woCRT);
			EncryptDecrypt ("Capi-384-WithoutCRT", rsa);

			rsa.FromXmlString (CapiXml448woCRT);
			EncryptDecrypt ("Capi-448-WithoutCRT", rsa);

			rsa.FromXmlString (CapiXml512woCRT);
			EncryptDecrypt ("Capi-512-WithoutCRT", rsa);

			rsa.FromXmlString (CapiXml576woCRT);
			EncryptDecrypt ("Capi-576-WithoutCRT", rsa);

			rsa.FromXmlString (CapiXml640woCRT);
			EncryptDecrypt ("Capi-640-WithoutCRT", rsa);

			rsa.FromXmlString (CapiXml704woCRT);
			EncryptDecrypt ("Capi-704-WithoutCRT", rsa);

			rsa.FromXmlString (CapiXml768woCRT);
			EncryptDecrypt ("Capi-768-WithoutCRT", rsa);

			rsa.FromXmlString (CapiXml832woCRT);
			EncryptDecrypt ("Capi-832-WithoutCRT", rsa);

			rsa.FromXmlString (CapiXml896woCRT);
			EncryptDecrypt ("Capi-896-WithoutCRT", rsa);

			rsa.FromXmlString (CapiXml960woCRT);
			EncryptDecrypt ("Capi-960-WithoutCRT", rsa);

			rsa.FromXmlString (CapiXml1024woCRT);
			EncryptDecrypt ("Capi-1024-WithoutCRT", rsa);

			rsa.FromXmlString (CapiXml1536woCRT);
			EncryptDecrypt ("Capi-1536-WithoutCRT", rsa);

			rsa.FromXmlString (CapiXml2048woCRT);
			EncryptDecrypt ("Capi-2048-WithoutCRT", rsa);
		}

		// all keypairs generated by Mono on Windows
		string MonoXml384 = "<RSAKeyValue><Modulus>i9COhlY/aGJD+B0fjUtOR+bbDkKyrwDzXzvwbBE8TJmtFYNrjfLHOj/gjdKCEcqb</Modulus><Exponent>EQ==</Exponent><P>tJ2yzv36kToJOy2vURzL9X+W0YsKiHZr</P><Q>xitaV4u8NRFVtIuFvYWArakoLHFeUiiR</Q><DP>dN6CwiveXfhgU3fp6TDANXCs4fCOWEyf</DP><DQ>aOnGapVFhYGl2A2hKBmAW/AkU8OMSZ0B</DQ><InverseQ>bU13UMhy7nsqfFtm7AhyjkJT4GCd/OPT</InverseQ><D>GKxVYwApTqft4H2cJ/46/aE1tzjyWx5IzfYoG1Qmnzr+u1/rf+EGkyIzGWIEcyXR</D></RSAKeyValue>";
		string MonoXml448 = "<RSAKeyValue><Modulus>rGhbXQawiO5muvnbq7FuVZc3QuFp2CZNqHekmRA/fdUzmUP6jFH7usaVh1kZ+Fk0em0fJzjOYgU=</Modulus><Exponent>EQ==</Exponent><P>rp5tO6WKHonnIRHxS8e566iNe20quz30u3R2Yw==</P><Q>/MIc/2TV6WPi9ddM5NzLeaukuv7pmZf/VfFedw==</Q><DP>j83DXkwXZHGRKksC8xzzdscLGlnm9I1gIedwjQ==</DP><DQ>hdAteCZTITThkTW/S/xruOJmROFOfn2lS53mtw==</DQ><InverseQ>L0fk3nG45E5WbtKkFODlQjMQGIdSEwlsvt/TYA==</InverseQ><D>ZWqQGJqF9jHiE6IIv1lP9hy3GEhcYQd4+a/KOvAKrWm95W6eF+aDWFgzao9lR1/BLP86HhcucSk=</D></RSAKeyValue>";
		string MonoXml512 = "<RSAKeyValue><Modulus>jq1Bst+LWM5ScL29w+C/GVULDAebcT0Kl9ERGp/I2A2qJ4jHuzr7ltoQqtVGRquQwwDZQ/o1fH99bWraz5Fi7w==</Modulus><Exponent>EQ==</Exponent><P>1O1TVs0UsUSx7Vjg7H/nJnZyfteeLNmUSAPylutaXmk=</P><Q>q4nwfMxpskF8TIRVmcTl1yHcu9rlaMt50Tb/skhFu5c=</Q><DP>ica9dGaU6yxzIRtkXMs7NwFZJOXe0bn2iPN+2h/RD+k=</DP><DQ>MnPdUeHEu/Ukjvm+0t+O5OvXgous4pYy4y5LNG+cCf8=</DQ><InverseQ>fAvQlMRGT5B/FPjQBeArxevr9DN59L0JfhN5qO84pLs=</InverseQ><D>bRsUH1+my3CZZUXNWY2/T5tizPbCKWrp+5/f5y7k4XK2d7xCN+qSZxn+nT0patdhewBMFOI9/xp5zD4U/seSIQ==</D></RSAKeyValue>";
		string MonoXml576 = "<RSAKeyValue><Modulus>h18QgR8YiVXKVmcTRuPfHljm9Yz5vPt9UXg25GU2WztihEihEyZYMJaRzEb8/iS7eGg7jLXW8SrxhyLUnXfgWvtn/9uNGinj</Modulus><Exponent>EQ==</Exponent><P>x3NP/zn0CxxHMoce7+3UbMtWbtZCxxZZGMID5OCJKGeJUYZF</P><Q>rcC07OWzbx2h827VnJlAWlmxmRRGIaXFOp10tNG8C1bE4GYH</Q><DP>UiBsO+qvyFbwI91I+WHuDq4Uh/3fQusVoMhb9NTtLsE4ivsN</DP><DQ>mU+usujLgCk0i3/pqEr8i/TJ4Ww94XQmfwNm+eY8gnm8xf+r</DQ><InverseQ>KqPD9+C3p6VF6KLOhkyEOLtBlImFMY3kK79+1bmJLjEdyzUH</InverseQ><D>b3t3AOxujzeXknMA0Pfk69C+M6FGIyl2Qxe0vBcdtItCEpYpGKlxv+l1rPu1/VOy4F8Q3u+5dMuhc2OlLfX7F6WkL5+dN7pB</D></RSAKeyValue>";
		string MonoXml640 = "<RSAKeyValue><Modulus>4YELOUiQPs7Nw6AM2m14LxuKDk1dXGYhvKczD6mB2//U4sitp+MFReJ0ot840hQNwwCssw8hyyniFqIvSjKJknv+HFL4Lc3AyCP3pL5iFf0=</Modulus><Exponent>EQ==</Exponent><P>9Q+nEGCtAF5MIk3lFGn0YtMYSYGM9JI2Ai16dS10Dx6Wxeq1bnVHLw==</P><Q>65Ho89urLekWoDLeZb2ManjaLQS7se6A0eM+Sj54cFnHj8UDe0uqkw==</Q><DP>rPv9dPjyltkmrs2SpP9/VNE+UgEYNCr48np0jvLogySmqdLabBaMmQ==</DP><DQ>UyRwVhFLeZ2Pg9W356xPrRuYTB/JxlQtd0ElCyUbcvKCqzZ5swulnQ==</DQ><InverseQ>5OZh1s2cw3Ydxa+abpooyXC08yJwNS9omtFIl9ct2HZkK4EKvhqJvg==</InverseQ><D>rHG9SewyEelwLC8Y40SnMxUPOB0LKIpWCLwX/OsI84dmj04qcVNAQxB0O5hmt5GXswJd1VPOdRmf/YrMiLZhIHFbLgdrQP+gyWGCSu3koz0=</D></RSAKeyValue>";
		string MonoXml704 = "<RSAKeyValue><Modulus>mUDm0P3S9oPB06NJT5oDFS5+Xy/ACBvGdkvNjCiZo9bOLvVpOsiih9ipWZw9zQInjfP0uhGqDBkl9jw61yEHIqed151b9hkO2cmfb2o3GvPkbXkoMcoxBQ==</Modulus><Exponent>EQ==</Exponent><P>sMV0t0i+ucYTdpvFTL9lzhxTC902auu5vMDf5Q45WAiu57e9WcGNGbx3PqU=</P><Q>3fEKYrtMZmXoL/vf4xnrRDKpt106zSyrwaVnq5vwrQUZGqj/zPx+umBj+uE=</Q><DP>FMvvnRed97zzOyFigX/t3ANVEHRgwUjorMtlonok3S4ysdlhkhbFTlJoYbk=</DP><DQ>nKolct6QSEfvMO4HcyFaxrpZrpwpgcUt8hpnS/Wa1HwRuHdLJ0jR7Pi/C3E=</DQ><InverseQ>Sy1FwjQNv201hd5uEVIDrxuCXHAnCA5G8p2XDTGee+mlGnMqFvrNKdR57VM=</InverseQ><D>WiYta+CaNqfbbW8cELTywSpodDo0uXnACVnENFQeJCQA0FQfyDnJBJ2Qy0vuhbaAbz2qNXZcYp6QxbeBQQZGEB//nahkNns7KxKXDEFMMU1ghU9eoubr8Q==</D></RSAKeyValue>";
		string MonoXml768 = "<RSAKeyValue><Modulus>pnVBGKuCV1sdnZP7yHMKlMnyc+4MZicSRDu6ESWPKUx/Ab2fbwwOoHDE5pSW/wCGziaEA2Agfwt6YKAr/JyPmcBcHichUjhzjDvExBrdd3jmQ1WjjNzV829+qSEq0kUl</Modulus><Exponent>EQ==</Exponent><P>u3CSNJ8PqWVfZkLnDBKFGk9EhGnY2poXTBR8aWqLDoa/nG1OPB4Zx0CjwNyR5mSH</P><Q>41gBCJ7TC5URUbKIqzfxQKcoDbz45tJ/1/h0nhGGXehGHYFWlNeKDMcqLtPDB5/z</Q><DP>NyEb8Vv1jCzf0shiEpwJB7z2CNPWXmmOYausHwE39Taw06etmTYHlPTk3l79vDuv</DP><DQ>uzlqQ1Wevj6Gu8A0UMSoj5i3ktfcCWIO7hfnkTub1N1m6x84PlcmZOBA20T69zht</DQ><InverseQ>sJSC1ons7852EcNt7Cctp8ryAvWZxndAz2/W5vcLVi4sk5+6EQ8Kfyw66hYNPvTO</InverseQ><D>kt/uJNORH+b8A4KS3gsndDm3z7Ps0prx//h3ABIU6DRwEJhBYfuUb3KPnkbflbUqONprCS1FlC0mo+Hxald/r99a5PE3JWiVTN4GHuZ3zYEviEvRtNsOG7XYKydTUO3F</D></RSAKeyValue>";
		string MonoXml832 = "<RSAKeyValue><Modulus>3dVclQYr1ks5x8VXLcO0H0BRNlmg/XT01IFso0aQY5AwU/kFbDUwTSrmSZCuRBk2blp14BYvpwpviuP9WkIcjhTSuPwPD1XWSl33WHSWEc6uYEHxjn0gieHK3SiyVAgOrcVXtOaNJck=</Modulus><Exponent>EQ==</Exponent><P>5BX1qdEJBAhdRC3YXMur5Hs2AfznVNExYbkj7wrK3qLxpQOLQk+TSk8PzWFoOx4xnDG98w==</P><Q>+PuDZFrHTTsSLXrrbS7PhQTWd5BHEqVifUttqI+YSx3pRTxFmfulj5m8uccDE775VrUwUw==</Q><DP>KEAcSyTjeS6nDAgXH288c51FxB2SPCTqmMZRoqeNNlj9d3kYkzs4HCwR2PMSZMkX0EUDZw==</DP><DQ>K/AmPuLX4HPWCAaiBDVv6kwl2N05t/8RYWep8JHPlMkLDDfQDB1Zc7G35IyIEosc8S8Ihw==</DQ><InverseQ>p7fE4OUIq/uvESb5mExGPNxh7BZBLQ3GRxq67QvHhhhaw4KAiISMk/MJY2TxD+DZ2F25hQ==</InverseQ><D>aGRnzaiNGYzP5YoK6FwYhy1TZN7iWSf6vlsF8nuPPemAJ4Q+yYJxFUFdT8udTTkKjkiv8BrhBpTUlAjP9ge+BNftSibZ4zp8wd1pxewIPFYnhimAVIFasWv/1NnIP8QwH0bQQOshCxE=</D></RSAKeyValue>";
		string MonoXml896 = "<RSAKeyValue><Modulus>8SR9zUf+X33JO4Gen620JE7yqQhvP0RSnU+XGE5fSHTPmy/HtuP5/IsCbfDsKOa5aAa4uj6PpxsG/6GgrbgmkxSfJmj8OfYP7Eo7wjXheJmv87Iw8y0rJ3PhkKbF82GFU/CmYQO3A77QVK74v9NMRw==</Modulus><Exponent>EQ==</Exponent><P>+q7kK0s1I10ZVAXqqnTzZ8T7L+Q82UW658IUeH1aL8OteL9qbh/05vhqknbptbZ6Xjsf9UtoTT8=</P><Q>9kHMYmw0rlmWcxQOAnTkn1g6ZCyMDR05Yhvf7qQtbZQ92GKWrs0+MTl9N+RkYREqhskY5S5TVvk=</Q><DP>hLbxRBjBx22F4TBPDvKe+rOUCkunvlIXp984A42oN2eYEr+w0OO947Cw5CDWFOgiqluJY76Cgz8=</DP><DQ>n1fAe+upnf27s9/M1GnBG8+tT+CW208lIV1UqXlKkjK+mxKdvGavxX+NQkh9L7/QORi1wXhUCxk=</DQ><InverseQ>lsGlEhb8TteuqHjnPULwYF+naVMMThqPa9+sAMflXAHm6zQ7YB6JotrhnSnSVdxuri0bhZwbMt4=</InverseQ><D>Ko34FSrSiVJuzkQM7wCJM5V2HdRP3fz/hSwpuP7FhUHKSI/2ETdKO6APfNAprN1sA0x687Cv8E+4mTmZ7zr/6qhiXIxKO1CYJIssptb/IfAC9McjnXeRfvn7XtHwprzmFRCfTE06ZPWD4QXJG22WIQ==</D></RSAKeyValue>";
		string MonoXml960 = "<RSAKeyValue><Modulus>tsPl3wQ+EYH4tuzDokt/CknZUVNZMr9AEwZCGMzpXORmkHaBA5Dh5N24OtQwO1YQ4gXCosU9YpdQujIvki48txiL+r6/KtK1hIOm4h/LCZRZ0ztYZvufH2ysv6Sv22iEbv8/ABOmaAu7A0RueGI1dX8aZ+dIlqzz</Modulus><Exponent>EQ==</Exponent><P>1kRCrCRrHd9STbYjSUZ0bD/Sqcx44aXwEjn06z6starjsKQ/RAboA/VCalDWlA66UC8CPQxN2Sd0Sn09</P><Q>2lzq2YjgLtwcQzW7GdAXf5yAoXO7mghh8rgfRvpVtAN6wkTSR1XQXPdXYJRVDYyGyrA5bBOnheotuNXv</Q><DP>WDo5ki0dDEzlp4c7tMKoaM78ZAjmewgXjwjOQr90Ss3lSLwaDPPI8pIqaAMrLefyXUCIc32Jd4i3afdV</DP><DQ>QDlyP/sUwnz5QPGvgBAG6Uwl1SIKACCVR2NUfkmgvHl+dV+JJAouOXXshdFGMSlU0jPUp1ETJ2L+Y4o3</DQ><InverseQ>gssEimyKQEsy0cnyZlhx4h6Ua/owWnlz4bRac6tVYQRK05LG78sAbgtysi/aZ2rXqG6+ZBTrY0FHuR5+</InverseQ><D>rAOrLEA6atSt2VdOtt2kvmOfW5m9XPA8TiQB+TlUGzFReOgA9Ewf5nZTCjEeVfamel/GPtfBTbuXRdTf41eV1Ci1WHuXJ2/ZiN8ob06O0UAjJTlva1RGhOVVBZyI2tJOA2vJ/zPPovcdiReyOeSu3neqYqr3IUVx</D></RSAKeyValue>";
		string MonoXml1024 = "<RSAKeyValue><Modulus>i+O73a/wy4fMi6V9dF9Us4oyAUvNwrJcOS69r7edDjw0MDd5vz8PcaAdL3BOwMEmItI1i8Z39HxqvuuwdB0GGCNKGh4Ft680nIAgsLugztJ4Shjm+xgFM3cFSGxJpSWkWlTpXs2ARhOSIbr5/+UwON4EvpkYxC5GbRIsrqDHHV8=</Modulus><Exponent>EQ==</Exponent><P>88rUpdL/RNdeuYTReHSK+MmRWnc2S79Z67uHtMIWO0BsG/gMUIKnFSpJiEaFpnHXe881cdUvNvo9NGuTz5Jh1Q==</P><Q>kuT52+3qfpuQ539A08+DiPSfyNHsq2vwOg8vxd3pBfVUyLSPkaLHOmXumk7BDvWAsQXQPOblsgeByunEO1ooYw==</Q><DP>ZGKTy89aDUmfeZEK9V0qKjThf5p/xNZSM+POd5s2VKIOZd6bqK5EzHrS++DNnuOU9r67p1fIJbJVb/APr9Lc/Q==</DP><DQ>ikDrKVhkOuzEnaTxuErWJovhrfLAoVaHvixpMrK9MsjIRG20TNVwNvaGNuC1s7niiH3xKkK6EPgBr+sTCq81EQ==</DQ><InverseQ>N3di8j2v4ZxE0Y/WFST/wSuAvTfTevx3F6I9jjYReUH8Yy88aKoKr+mgRSl+94/glwbCQ3Q+jP79puiBmcRrPA==</InverseQ><D>OZoCD/0m6mUX/UQkmVRuK89+AIigBOAH2097SFqqFOudBMuMe+zKH7pmXtPkMUB5HWWdk+hPgsnRmeiT85N6+kB7xMfgGI5AsJga19OAi2yI3RnXlTq0FMcYHZCvRE8AXUxVQTO7/n3iq3r8TEDaPmcTpn6eooXvGoA6jRCHLYk=</D></RSAKeyValue>";
		string MonoXml1536 = "<RSAKeyValue><Modulus>jzYjq6zGOWfmpnzDf5SNrSifeYbFFgbh0/tO2UkLWBpbFqHdLo8CZJt1j3UtSN82ZuV6IBMJnmIsxOH3JQ677IPb76Hz/F0V07jCM88kcz3RRtCahXj/14p2EwTY3297yylWx8+zGmpQDPOxejbaGyjn2TyPx6Nq54SVSAfBUgcXuwwpksogwv/neZLyOlu/JBG5Ej11FOfJ8vi7gb/OCqK1EdzRthf9e4FS5KoxC81TiTUXyfesYpSsHI6fQ5i1</Modulus><Exponent>EQ==</Exponent><P>r0edPMfgFzP3gOrrFIS83DteP5B9l0Y0c9ONSdM3il1Gm/NtU1YTmEHAdAaNHS/p8xqO2ojnYLTK7EUADQeeEUnl3f+BYcYkVSOy6Fq8KKJlDP/PUNF97Yp42TgUL2bD</P><Q>0SngrKVGg1QV5paow1TV6IBTaxVflPcHRdkk7H3WkzSz8XGmpVWYumRWQoDT7NwdsMZnvXImlavUX4v0fNqTvHwN1HwTU8g64RFqfePZK48fN3EjIZDKcTSEQxrFdDsn</Q><DP>e7oUo1/pecpUWwAtd+U6BN6c4ZMredc0FYZFnYYJFl/1m0JrSeJoLz14yl76Mrho59aC9Jzfj46tW3wACTKNsdnPb4cuJuY3w6C6heWx/pC/zPDOk2azPkOgmVTC9EiJ</DP><DQ>uI5c1JHUzjsiYgx2yngICWIrfJpjZVJv1DgRhV/5kPJEa6CD+0uGwpTElQhChbMpQaABLrAD7Xl/JyEjBMDctV5mjk9cWPv3t4fWbxRWNX5IuHLiw0OFcvIaWVPbZo6L</DQ><InverseQ>p52gqtyki7MRn0fFTOkSCee/2QQAZyM2cYt3DtgDQbRcqrMT6Q9WXgyXOGdadqFFvUpGxysCQ/rppLttCG9S8wG8lD9I6hulda3TkiKCsYNuNLco0epmjXpWc0YmCcys</InverseQ><D>OvgszjgVYu6MJm+byx8NKS7YQRlgNj8RsaO3Di0iyey8GGDEfJU9OHw/d05d8NRhseYFHEQTBPs/nF0LaZynu73TCFG+0VOBdUwTuvrw5CiDWWT0VRO0pAvWRBEOAaZfiC2ViL8qyz8MYj4MM8wOUMOOx+Oy9LGa9MJOcLSkJKixqWLqxv17PVIoyMiHE+SW2m5uMleT32TVU+Mzz3lec2n1RWRGS4P14EyOf1l8WqmRlMk8jXm/xVf8w9xgbwtF</D></RSAKeyValue>";
		string MonoXml2048 = "<RSAKeyValue><Modulus>ppiz9wru8QdjxD1IZ8Ouc4rpKMjR51jDq/mPu7gILbto1sD+01r/P84zFd2n9Jbv+vHJ8ClTr/9vh3dy6OzN/ASUoAMzmGnHQPsFHyaZ7TxDmF2EaF45F0rDbsPsOxlHUKkc33EuHwzXRZRdD4c2vCY8D+qrdpldkC4KH4oWjBHPk2MlqM1mPGJIvzz9nANhEl9YHHHZmAV2auZIWMWyO9zPHoE0YYMeFTHIuOZmxbu6iKX0pWcHzukx6FCqxN9zOAz5xl9D0f//0vhsiZUINM4y4ypS9C+jF8WAUthj29R3s1Hxr5I32kr3+nhtkHu7ljqLzYqCBq6hixZEhHlUlQ==</Modulus><Exponent>EQ==</Exponent><P>w5gi5/IS9jFz8tAVpUgmCBBvUCMWG6VmzvrKxf+Z6VQ/Uz3iirTDyMHbjfThlRLik39EbCs2woN9HYeIbEoQRKvbPVewvQZzV6nD5vAIR/wMzL/T14kStaMI7oYUXTBsTrfz/YwNFXPgANV1QEjFMT+S7zGacduP9PG3Kx5dqu8=</P><Q>2gv5mTL6RDAitvOdFoI8Ke/PrQFne1Gx8oKrMBVoRUL9Jzzp4oWlNoDjU4N4USET3KsuR8OBgH8KH/caJC1NuDG+CQ/Bkf/iB82LuoOvaBSAKFCXRMng6LyoDFxb1ptwFavFogTrxPs3GpHYzIq4quhuP18X9bjHkiNzZ3qqCLs=</Q><DP>ihEJlKrgNVAVnFal3hTPjTjHC2QPmwtXoSmAE0sDO0qHDZUYYekCq9QiglKBHe82hju30+JExYn99roF8hYpmeK44AGp7tdgeh17OZpgMtAJCP/g1GDB6aBCikCG2F5qketC0ReQw9lS02l/8SRO9ZZJmcipQU+w6SMX4jONaZk=</DP><DQ>TPUq6sayrqeTyBm++OKr0pDf4rUzlO+oN3lpmIAGrwiVlWDLBKejuOIyHXmx/oQlILTjKGMepdJ8C0gnWA/9UBGOXY0XJHht5KLmBZfjjiVaaJTqGEdAUiR3i+Rcphi+JcQnophTNnbIJ37E/OWqlqxjJWzbR6qgyiqhM5S0e40=</DQ><InverseQ>dYOYJxCnJUZLastml8Fu9/HoW+DoX3OQ/I2b3aRBYByh7dpeRPaMdtycHgSNtMAocYDXMypl+f92vHFXENc4WFtEVAQSkxgD2IWTgQEj92kmaCA8Jrs8boH3eXfLb6ZlHf1fas5lIreq5Gzj/0IC0zDlYKBWOQf/3d+mxI9kbyw=</InverseQ><D>f2W2y/lNTueIhwGv1uDu0NOjPU5GKWH/CwoinpvKBNqbdwwOKSd35X+QeiH46Dcv/CJPIRCLSlnr4BAMlABhSD+88tVFkqs+BIO4n1nBALWdGillQMCGAsCzkPAtHiJjtib39ilfgSfv6ensz6OiU6TEhKRlAFcpbkFTCQ8+axtxer2M3TiaAarE3njI65lgOwmu+djnqDyKWzfGcA37QafIQPOnWiMIgtBWfKetnKXtk1ReqdDw2IckUNBud4EeRStM7jy7MlVNc0TXiTDK7ow+Vd2EIPGpQXkav167V4F4Lc7VYrGDrFTV2fM66uBBcmgaH3gUOVGq0qUttAuZLQ==</D></RSAKeyValue>";

		// same keypairs but without CRT
		string MonoXml384woCRT = "<RSAKeyValue><Modulus>i9COhlY/aGJD+B0fjUtOR+bbDkKyrwDzXzvwbBE8TJmtFYNrjfLHOj/gjdKCEcqb</Modulus><Exponent>EQ==</Exponent><D>GKxVYwApTqft4H2cJ/46/aE1tzjyWx5IzfYoG1Qmnzr+u1/rf+EGkyIzGWIEcyXR</D></RSAKeyValue>";
		string MonoXml448woCRT = "<RSAKeyValue><Modulus>rGhbXQawiO5muvnbq7FuVZc3QuFp2CZNqHekmRA/fdUzmUP6jFH7usaVh1kZ+Fk0em0fJzjOYgU=</Modulus><Exponent>EQ==</Exponent><D>ZWqQGJqF9jHiE6IIv1lP9hy3GEhcYQd4+a/KOvAKrWm95W6eF+aDWFgzao9lR1/BLP86HhcucSk=</D></RSAKeyValue>";
		string MonoXml512woCRT = "<RSAKeyValue><Modulus>jq1Bst+LWM5ScL29w+C/GVULDAebcT0Kl9ERGp/I2A2qJ4jHuzr7ltoQqtVGRquQwwDZQ/o1fH99bWraz5Fi7w==</Modulus><Exponent>EQ==</Exponent><D>bRsUH1+my3CZZUXNWY2/T5tizPbCKWrp+5/f5y7k4XK2d7xCN+qSZxn+nT0patdhewBMFOI9/xp5zD4U/seSIQ==</D></RSAKeyValue>";
		string MonoXml576woCRT = "<RSAKeyValue><Modulus>h18QgR8YiVXKVmcTRuPfHljm9Yz5vPt9UXg25GU2WztihEihEyZYMJaRzEb8/iS7eGg7jLXW8SrxhyLUnXfgWvtn/9uNGinj</Modulus><Exponent>EQ==</Exponent><D>b3t3AOxujzeXknMA0Pfk69C+M6FGIyl2Qxe0vBcdtItCEpYpGKlxv+l1rPu1/VOy4F8Q3u+5dMuhc2OlLfX7F6WkL5+dN7pB</D></RSAKeyValue>";
		string MonoXml640woCRT = "<RSAKeyValue><Modulus>4YELOUiQPs7Nw6AM2m14LxuKDk1dXGYhvKczD6mB2//U4sitp+MFReJ0ot840hQNwwCssw8hyyniFqIvSjKJknv+HFL4Lc3AyCP3pL5iFf0=</Modulus><Exponent>EQ==</Exponent><D>rHG9SewyEelwLC8Y40SnMxUPOB0LKIpWCLwX/OsI84dmj04qcVNAQxB0O5hmt5GXswJd1VPOdRmf/YrMiLZhIHFbLgdrQP+gyWGCSu3koz0=</D></RSAKeyValue>";
		string MonoXml704woCRT = "<RSAKeyValue><Modulus>mUDm0P3S9oPB06NJT5oDFS5+Xy/ACBvGdkvNjCiZo9bOLvVpOsiih9ipWZw9zQInjfP0uhGqDBkl9jw61yEHIqed151b9hkO2cmfb2o3GvPkbXkoMcoxBQ==</Modulus><Exponent>EQ==</Exponent><D>WiYta+CaNqfbbW8cELTywSpodDo0uXnACVnENFQeJCQA0FQfyDnJBJ2Qy0vuhbaAbz2qNXZcYp6QxbeBQQZGEB//nahkNns7KxKXDEFMMU1ghU9eoubr8Q==</D></RSAKeyValue>";
		string MonoXml768woCRT = "<RSAKeyValue><Modulus>pnVBGKuCV1sdnZP7yHMKlMnyc+4MZicSRDu6ESWPKUx/Ab2fbwwOoHDE5pSW/wCGziaEA2Agfwt6YKAr/JyPmcBcHichUjhzjDvExBrdd3jmQ1WjjNzV829+qSEq0kUl</Modulus><Exponent>EQ==</Exponent><D>kt/uJNORH+b8A4KS3gsndDm3z7Ps0prx//h3ABIU6DRwEJhBYfuUb3KPnkbflbUqONprCS1FlC0mo+Hxald/r99a5PE3JWiVTN4GHuZ3zYEviEvRtNsOG7XYKydTUO3F</D></RSAKeyValue>";
		string MonoXml832woCRT = "<RSAKeyValue><Modulus>3dVclQYr1ks5x8VXLcO0H0BRNlmg/XT01IFso0aQY5AwU/kFbDUwTSrmSZCuRBk2blp14BYvpwpviuP9WkIcjhTSuPwPD1XWSl33WHSWEc6uYEHxjn0gieHK3SiyVAgOrcVXtOaNJck=</Modulus><Exponent>EQ==</Exponent><D>aGRnzaiNGYzP5YoK6FwYhy1TZN7iWSf6vlsF8nuPPemAJ4Q+yYJxFUFdT8udTTkKjkiv8BrhBpTUlAjP9ge+BNftSibZ4zp8wd1pxewIPFYnhimAVIFasWv/1NnIP8QwH0bQQOshCxE=</D></RSAKeyValue>";
		string MonoXml896woCRT = "<RSAKeyValue><Modulus>8SR9zUf+X33JO4Gen620JE7yqQhvP0RSnU+XGE5fSHTPmy/HtuP5/IsCbfDsKOa5aAa4uj6PpxsG/6GgrbgmkxSfJmj8OfYP7Eo7wjXheJmv87Iw8y0rJ3PhkKbF82GFU/CmYQO3A77QVK74v9NMRw==</Modulus><Exponent>EQ==</Exponent><D>Ko34FSrSiVJuzkQM7wCJM5V2HdRP3fz/hSwpuP7FhUHKSI/2ETdKO6APfNAprN1sA0x687Cv8E+4mTmZ7zr/6qhiXIxKO1CYJIssptb/IfAC9McjnXeRfvn7XtHwprzmFRCfTE06ZPWD4QXJG22WIQ==</D></RSAKeyValue>";
		string MonoXml960woCRT = "<RSAKeyValue><Modulus>tsPl3wQ+EYH4tuzDokt/CknZUVNZMr9AEwZCGMzpXORmkHaBA5Dh5N24OtQwO1YQ4gXCosU9YpdQujIvki48txiL+r6/KtK1hIOm4h/LCZRZ0ztYZvufH2ysv6Sv22iEbv8/ABOmaAu7A0RueGI1dX8aZ+dIlqzz</Modulus><Exponent>EQ==</Exponent><D>rAOrLEA6atSt2VdOtt2kvmOfW5m9XPA8TiQB+TlUGzFReOgA9Ewf5nZTCjEeVfamel/GPtfBTbuXRdTf41eV1Ci1WHuXJ2/ZiN8ob06O0UAjJTlva1RGhOVVBZyI2tJOA2vJ/zPPovcdiReyOeSu3neqYqr3IUVx</D></RSAKeyValue>";
		string MonoXml1024woCRT = "<RSAKeyValue><Modulus>i+O73a/wy4fMi6V9dF9Us4oyAUvNwrJcOS69r7edDjw0MDd5vz8PcaAdL3BOwMEmItI1i8Z39HxqvuuwdB0GGCNKGh4Ft680nIAgsLugztJ4Shjm+xgFM3cFSGxJpSWkWlTpXs2ARhOSIbr5/+UwON4EvpkYxC5GbRIsrqDHHV8=</Modulus><Exponent>EQ==</Exponent><D>OZoCD/0m6mUX/UQkmVRuK89+AIigBOAH2097SFqqFOudBMuMe+zKH7pmXtPkMUB5HWWdk+hPgsnRmeiT85N6+kB7xMfgGI5AsJga19OAi2yI3RnXlTq0FMcYHZCvRE8AXUxVQTO7/n3iq3r8TEDaPmcTpn6eooXvGoA6jRCHLYk=</D></RSAKeyValue>";
		string MonoXml1536woCRT = "<RSAKeyValue><Modulus>jzYjq6zGOWfmpnzDf5SNrSifeYbFFgbh0/tO2UkLWBpbFqHdLo8CZJt1j3UtSN82ZuV6IBMJnmIsxOH3JQ677IPb76Hz/F0V07jCM88kcz3RRtCahXj/14p2EwTY3297yylWx8+zGmpQDPOxejbaGyjn2TyPx6Nq54SVSAfBUgcXuwwpksogwv/neZLyOlu/JBG5Ej11FOfJ8vi7gb/OCqK1EdzRthf9e4FS5KoxC81TiTUXyfesYpSsHI6fQ5i1</Modulus><Exponent>EQ==</Exponent><D>OvgszjgVYu6MJm+byx8NKS7YQRlgNj8RsaO3Di0iyey8GGDEfJU9OHw/d05d8NRhseYFHEQTBPs/nF0LaZynu73TCFG+0VOBdUwTuvrw5CiDWWT0VRO0pAvWRBEOAaZfiC2ViL8qyz8MYj4MM8wOUMOOx+Oy9LGa9MJOcLSkJKixqWLqxv17PVIoyMiHE+SW2m5uMleT32TVU+Mzz3lec2n1RWRGS4P14EyOf1l8WqmRlMk8jXm/xVf8w9xgbwtF</D></RSAKeyValue>";
		string MonoXml2048woCRT = "<RSAKeyValue><Modulus>ppiz9wru8QdjxD1IZ8Ouc4rpKMjR51jDq/mPu7gILbto1sD+01r/P84zFd2n9Jbv+vHJ8ClTr/9vh3dy6OzN/ASUoAMzmGnHQPsFHyaZ7TxDmF2EaF45F0rDbsPsOxlHUKkc33EuHwzXRZRdD4c2vCY8D+qrdpldkC4KH4oWjBHPk2MlqM1mPGJIvzz9nANhEl9YHHHZmAV2auZIWMWyO9zPHoE0YYMeFTHIuOZmxbu6iKX0pWcHzukx6FCqxN9zOAz5xl9D0f//0vhsiZUINM4y4ypS9C+jF8WAUthj29R3s1Hxr5I32kr3+nhtkHu7ljqLzYqCBq6hixZEhHlUlQ==</Modulus><Exponent>EQ==</Exponent><D>f2W2y/lNTueIhwGv1uDu0NOjPU5GKWH/CwoinpvKBNqbdwwOKSd35X+QeiH46Dcv/CJPIRCLSlnr4BAMlABhSD+88tVFkqs+BIO4n1nBALWdGillQMCGAsCzkPAtHiJjtib39ilfgSfv6ensz6OiU6TEhKRlAFcpbkFTCQ8+axtxer2M3TiaAarE3njI65lgOwmu+djnqDyKWzfGcA37QafIQPOnWiMIgtBWfKetnKXtk1ReqdDw2IckUNBud4EeRStM7jy7MlVNc0TXiTDK7ow+Vd2EIPGpQXkav167V4F4Lc7VYrGDrFTV2fM66uBBcmgaH3gUOVGq0qUttAuZLQ==</D></RSAKeyValue>";

		// import/export XML keypairs
		// so we know that Windows (original MS Framework) can use keypairs generated by Mono
		[Test]
		public void MonoXmlImportExport () 
		{
			RSAManaged rsa = new RSAManaged ();

			rsa.FromXmlString (MonoXml384);
			Assert.AreEqual (MonoXml384, rsa.ToXmlString (true), "Mono-Xml384");

			rsa.FromXmlString (MonoXml448);
			Assert.AreEqual (MonoXml448, rsa.ToXmlString (true), "Mono-Xml448");

			rsa.FromXmlString (MonoXml512);
			Assert.AreEqual (MonoXml512, rsa.ToXmlString (true), "Mono-Xml512");

			rsa.FromXmlString (MonoXml576);
			Assert.AreEqual (MonoXml576, rsa.ToXmlString (true), "Mono-Xml576");

			rsa.FromXmlString (MonoXml640);
			Assert.AreEqual (MonoXml640, rsa.ToXmlString (true), "Mono-Xml640");

			rsa.FromXmlString (MonoXml704);
			Assert.AreEqual (MonoXml704, rsa.ToXmlString (true), "Mono-Xml704");

			rsa.FromXmlString (MonoXml768);
			Assert.AreEqual (MonoXml768, rsa.ToXmlString (true), "Mono-Xml768");

			rsa.FromXmlString (MonoXml832);
			Assert.AreEqual (MonoXml832, rsa.ToXmlString (true), "Mono-Xml832");

			rsa.FromXmlString (MonoXml896);
			Assert.AreEqual (MonoXml896, rsa.ToXmlString (true), "Mono-Xml896");

			rsa.FromXmlString (MonoXml960);
			Assert.AreEqual (MonoXml960, rsa.ToXmlString (true), "Mono-Xml960");

			rsa.FromXmlString (MonoXml1024);
			Assert.AreEqual (MonoXml1024, rsa.ToXmlString (true), "Mono-Xml1024");

			rsa.FromXmlString (MonoXml1536);
			Assert.AreEqual (MonoXml1536, rsa.ToXmlString (true), "Mono-Xml1536");

			rsa.FromXmlString (MonoXml2048);
			Assert.AreEqual (MonoXml2048, rsa.ToXmlString (true), "Mono-Xml2048");
		}

		[Test]
		public void MonoXmlImportExportWithoutCRT () 
		{
			RSAManaged rsa = new RSAManaged ();

			rsa.FromXmlString (MonoXml384woCRT);
			Assert.AreEqual (MonoXml384woCRT, rsa.ToXmlString (true), "Mono-Xml384-WithoutCRT");

			rsa.FromXmlString (MonoXml448woCRT);
			Assert.AreEqual (MonoXml448woCRT, rsa.ToXmlString (true), "Mono-Xml448-WithoutCRT");

			rsa.FromXmlString (MonoXml512woCRT);
			Assert.AreEqual (MonoXml512woCRT, rsa.ToXmlString (true), "Mono-Xml512-WithoutCRT");

			rsa.FromXmlString (MonoXml576woCRT);
			Assert.AreEqual (MonoXml576woCRT, rsa.ToXmlString (true), "Mono-Xml576-WithoutCRT");

			rsa.FromXmlString (MonoXml640woCRT);
			Assert.AreEqual (MonoXml640woCRT, rsa.ToXmlString (true), "Mono-Xml640-WithoutCRT");

			rsa.FromXmlString (MonoXml704woCRT);
			Assert.AreEqual (MonoXml704woCRT, rsa.ToXmlString (true), "Mono-Xml704-WithoutCRT");

			rsa.FromXmlString (MonoXml768woCRT);
			Assert.AreEqual (MonoXml768woCRT, rsa.ToXmlString (true), "Mono-Xml768-WithoutCRT");

			rsa.FromXmlString (MonoXml832woCRT);
			Assert.AreEqual (MonoXml832woCRT, rsa.ToXmlString (true), "Mono-Xml832-WithoutCRT");

			rsa.FromXmlString (MonoXml896woCRT);
			Assert.AreEqual (MonoXml896woCRT, rsa.ToXmlString (true), "Mono-Xml896-WithoutCRT");

			rsa.FromXmlString (MonoXml960woCRT);
			Assert.AreEqual (MonoXml960woCRT, rsa.ToXmlString (true), "Mono-Xml960-WithoutCRT");

			rsa.FromXmlString (MonoXml1024woCRT);
			Assert.AreEqual (MonoXml1024woCRT, rsa.ToXmlString (true), "Mono-Xml1024-WithoutCRT");

			rsa.FromXmlString (MonoXml1536woCRT);
			Assert.AreEqual (MonoXml1536woCRT, rsa.ToXmlString (true), "Mono-Xml1536-WithoutCRT");

			rsa.FromXmlString (MonoXml2048woCRT);
			Assert.AreEqual (MonoXml2048woCRT, rsa.ToXmlString (true), "Mono-Xml2048-WithoutCRT");
		}

		[Test]
		public void MonoEncryptDecrypt () 
		{
			RSAManaged rsa = new RSAManaged ();

			rsa.FromXmlString (MonoXml384);
			EncryptDecrypt ("Mono-384", rsa);

			rsa.FromXmlString (MonoXml448);
			EncryptDecrypt ("Mono-448", rsa);

			rsa.FromXmlString (MonoXml512);
			EncryptDecrypt ("Mono-512", rsa);

			rsa.FromXmlString (MonoXml576);
			EncryptDecrypt ("Mono-576", rsa);

			rsa.FromXmlString (MonoXml640);
			EncryptDecrypt ("Mono-640", rsa);

			rsa.FromXmlString (MonoXml704);
			EncryptDecrypt ("Mono-704", rsa);

			rsa.FromXmlString (MonoXml768);
			EncryptDecrypt ("Mono-768", rsa);

			rsa.FromXmlString (MonoXml832);
			EncryptDecrypt ("Mono-832", rsa);

			rsa.FromXmlString (MonoXml896);
			EncryptDecrypt ("Mono-896", rsa);

			rsa.FromXmlString (MonoXml960);
			EncryptDecrypt ("Mono-960", rsa);

			rsa.FromXmlString (MonoXml1024);
			EncryptDecrypt ("Mono-1024", rsa);

			rsa.FromXmlString (MonoXml1536);
			EncryptDecrypt ("Mono-1536", rsa);

			rsa.FromXmlString (MonoXml2048);
			EncryptDecrypt ("Mono-2048", rsa);
		}

		[Test]
		public void MonoEncryptDecryptWithoutCRT () 
		{
			RSAManaged rsa = new RSAManaged ();

			rsa.FromXmlString (MonoXml384woCRT);
			EncryptDecrypt ("Mono-384-WithoutCRT", rsa);

			rsa.FromXmlString (MonoXml448woCRT);
			EncryptDecrypt ("Mono-448-WithoutCRT", rsa);

			rsa.FromXmlString (MonoXml512woCRT);
			EncryptDecrypt ("Mono-512-WithoutCRT", rsa);

			rsa.FromXmlString (MonoXml576woCRT);
			EncryptDecrypt ("Mono-576-WithoutCRT", rsa);

			rsa.FromXmlString (MonoXml640woCRT);
			EncryptDecrypt ("Mono-640-WithoutCRT", rsa);

			rsa.FromXmlString (MonoXml704woCRT);
			EncryptDecrypt ("Mono-704-WithoutCRT", rsa);

			rsa.FromXmlString (MonoXml768woCRT);
			EncryptDecrypt ("Mono-768-WithoutCRT", rsa);

			rsa.FromXmlString (MonoXml832woCRT);
			EncryptDecrypt ("Mono-832-WithoutCRT", rsa);

			rsa.FromXmlString (MonoXml896woCRT);
			EncryptDecrypt ("Mono-896-WithoutCRT", rsa);

			rsa.FromXmlString (MonoXml960woCRT);
			EncryptDecrypt ("Mono-960-WithoutCRT", rsa);

			rsa.FromXmlString (MonoXml1024woCRT);
			EncryptDecrypt ("Mono-1024-WithoutCRT", rsa);

			rsa.FromXmlString (MonoXml1536woCRT);
			EncryptDecrypt ("Mono-1536-WithoutCRT", rsa);

			rsa.FromXmlString (MonoXml2048woCRT);
			EncryptDecrypt ("Mono-2048-WithoutCRT", rsa);
		}

		[Test]
		[ExpectedException (typeof (CryptographicException))]
		public void Bug79269 ()
		{
	                RSAManaged rsa = new RSAManaged ();
			rsa.FromXmlString ("<RSAKeyValue><Modulus>iSObDmmhDgrl4NiLaviFcpv4NdysBWJcqiVz3AQbPdajtXaQQ8VJdfRkixah132yKOFGCWZhHS3EuPMh8dcNwGwta2nh+m2IV6ktzI4+mZ7CSNAsmlDY0JI+H8At1vKvNArlC5jkVGuliYroJeSU/NLPLNYgspi7TtXGy9Rfug8=</Modulus><Exponent>EQ==</Exponent></RSAKeyValue>");
			Assert.IsTrue (rsa.PublicOnly, "PublicOnly");
        	        string b64 = @"YgyAhscnTTIcDeLJTZcOYYyHVxNhV6d03jeZYjq0sPMEsfCCbE/NcFyYHD9BTuiduqPplCLbGpfZIZYJ6vAP9m5z4Q9eEw79kmEFCsm8wSKEo/gKiptVpwQ78VOPrWd/wEkTTeeg2nVim3JIsTKGFlV7rKxIWQhGN9aAqgP8nZI=";
        	        byte [] bytes = Convert.FromBase64String (b64);
	                rsa.DecryptValue (bytes);
		}
	}
}
