/*
 *  Microfeed - Backend for accessing feed-based services (utils)
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "config.h"

#define _GNU_SOURCE

#include <microfeed-common/microfeedconfiguration.h>
#include <microfeed-common/microfeedprotocol.h>
#include <microfeed-common/microfeedmisc.h>
#include <string.h>

#include "configuration.h"


typedef struct {
	int just_created;
	const char** wizard_settings;
	int wizard_mode;
	GtkWidget* dialog;
	GtkWidget* edit_button;
	GtkListStore* list_store;
} EditDialog;

typedef struct {
	int column;
	const char* value;
	GtkTreeIter iter;
	int found;
} TreeModelSearchData;

static gboolean tree_model_search_function(GtkTreeModel* model, GtkTreePath* path, GtkTreeIter* iter, gpointer user_data) {
	gboolean found = FALSE;
	TreeModelSearchData* search_data;
	GValue value = { 0 };
	const char* s;
	
	search_data = (TreeModelSearchData*)user_data;
	gtk_tree_model_get_value(model, iter, search_data->column, &value);
	if ((s = g_value_get_string(&value)) && !strcmp(search_data->value, s)) {
		found = TRUE;
		search_data->found = 1;
		search_data->iter = *iter;
	}
	g_value_unset(&value);
	
	return found;
}

static void error_occured(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const char* error_name, const char* error_message, void* user_data) {
	EditDialog* edit_dialog;
	GtkWidget* dialog;
	
	edit_dialog = (EditDialog*)user_data;
	if (error_name) {
		dialog = gtk_message_dialog_new(GTK_WINDOW(edit_dialog->dialog), GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "Failed to edit settings for %s: %s", publisher, (error_message ? error_message : error_name));
		gtk_window_set_title(GTK_WINDOW(dialog), "Error");
		gtk_dialog_run(GTK_DIALOG(dialog));
		gtk_widget_destroy(dialog);
		gtk_dialog_response(GTK_DIALOG(edit_dialog->dialog), GTK_RESPONSE_NONE);
	}
}

static void setting(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedItem* item, void* user_data) {
	EditDialog* edit_dialog;
	const char* uid;
	const char* text;
	const char* value;
	const char* unit;
	const char* type;
	char* value_unit;
	const char* length_string;
	int length;	
	TreeModelSearchData* search_data;

	edit_dialog = (EditDialog*)user_data;
	
	uid = microfeed_item_get_uid(item);
	
	if (strcmp(uid, MICROFEED_ITEM_UID_FEED_METADATA) &&
	    uid[0] != '.') {
		if (!(text = microfeed_item_get_property(item, "setting.text"))) {
			text = uid;
		}
		if (!(value = microfeed_item_get_property(item, "setting.value"))) {
			value = "";
		}
		if (!(unit = microfeed_item_get_property(item, "setting.unit"))) {
			unit = "";
		}
		if (!(type = microfeed_item_get_property(item, "setting.type"))) {
			type = "string";
		}
		if (!(length_string = microfeed_item_get_property(item, "setting.length")) || (length = atoi(length_string)) <= 0) {
			length = 256;
		}
		if (!strcmp(type, "password")) {
			value_unit = g_strdup("<hidden>");
		} else {
			value_unit = g_strconcat(value, " ", unit, NULL);
		}
		search_data = microfeed_memory_allocate(TreeModelSearchData);
		search_data->column = 1;
		search_data->value = uid;
		gtk_tree_model_foreach(GTK_TREE_MODEL(edit_dialog->list_store), tree_model_search_function, search_data);
		if (!search_data->found && !edit_dialog->wizard_mode) {
			gtk_list_store_append(edit_dialog->list_store, &search_data->iter);
			search_data->found = 1;
		}
		if (search_data->found) {
			gtk_list_store_set(edit_dialog->list_store, &search_data->iter, 0, value_unit, 1, uid, 2, text, 3, value, 4, unit, 5, type, 6, length, -1);
		}
		microfeed_memory_free(search_data);
		free(value_unit);
	}
}

static MicrofeedSubscriberCallbacks callbacks = {
	NULL, /* error_occured */
	NULL, /* feed_update_started */
	NULL, /* feed_update_ended */
	NULL, /* feed_republishing_started */
	NULL, /* feed_republishing_ended */
	setting,
	setting,
	setting,
	NULL, /* item_removed */
	NULL, /* item_status_changed */
	NULL /* item_data_received */
};

static void update_list(MicrofeedSubscriber* subscriber, const char* publisher, EditDialog* edit_dialog) {
	const char** s;
	const char* space;
	char* uid;
	MicrofeedItem* item;
	GtkTreeIter iter;
	
	gtk_list_store_clear(edit_dialog->list_store);

	if (edit_dialog->wizard_settings && edit_dialog->wizard_mode) {
		for (s = edit_dialog->wizard_settings; *s; s++) {
			if ((space = strchr(*s, ' '))) {
	printf("*  %s  %d  *\n", *s, edit_dialog->just_created);
				if (edit_dialog->just_created) {
					uid = g_strndup(*s, space - *s);
					item = microfeed_item_new(uid, time(NULL));
					microfeed_item_set_property(item, "setting.value", space + 1);
					free(uid);
					microfeed_subscriber_modify_item(subscriber, publisher, MICROFEED_FEED_URI_SETTINGS, item, NULL, NULL);
					microfeed_item_free(item);
				}
			} else {
				gtk_list_store_append(edit_dialog->list_store, &iter);
				gtk_list_store_set(edit_dialog->list_store, &iter, 1, *s, -1);
			}
		}
	}
	edit_dialog->just_created = 0;
	microfeed_subscriber_republish_items(subscriber, publisher, MICROFEED_FEED_URI_SETTINGS, NULL, NULL, 1024, error_occured, edit_dialog);
}

static void settings_subscribed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const char* error_name, const char* error_message, void* user_data) {
	if (error_name) {
		error_occured(subscriber, publisher, uri, uid, error_name, error_message, user_data);
	} else {
		update_list(subscriber, publisher, user_data);	
	}
}

#if MAEMO_CHANGES

static void on_changed(HildonTouchSelector* touch_selector, gint column, gpointer user_data) {
	EditDialog* edit_dialog;
	gboolean selected;
	
	edit_dialog = (EditDialog*)user_data;
	selected = (hildon_touch_selector_get_active(touch_selector, 0) == -1 ? FALSE : TRUE);
	gtk_widget_set_sensitive(edit_dialog->edit_button, selected);
}

#else

static void on_row_activated(GtkTreeView* tree_view, GtkTreePath* path, GtkTreeViewColumn* column, gpointer user_data) {
	EditDialog* edit_dialog;
	
	edit_dialog = (EditDialog*)user_data;
	gtk_dialog_response(GTK_DIALOG(edit_dialog->dialog), 2);
}

static void on_selection_changed(GtkTreeSelection* selection, gpointer user_data) {
	EditDialog* edit_dialog;
	gboolean selected;

	edit_dialog = (EditDialog*)user_data;
	selected = gtk_tree_selection_get_selected(selection, NULL, NULL);
	gtk_widget_set_sensitive(edit_dialog->edit_button, selected);
}

#endif

void show_edit_dialog(MicrofeedSubscriber* subscriber, GtkWindow* parent, const char* publisher, int just_created) {
	MicrofeedConfiguration* configuration;
	char* publisher_part;
	char* provider_part;
	EditDialog* edit_dialog;
	GtkWidget* table;
	GtkWidget* label;
	GtkWidget* entry;
	char* text;
#ifdef MAEMO_CHANGES
	GtkWidget* touch_selector;
	HildonTouchSelectorColumn* column;
#else
	GtkWidget* button;
	GtkWidget* tree_view;
	GtkTreeViewColumn* column;
	GtkWidget* scrolled_window;
	GtkTreeSelection* selection;
#endif
	GtkCellRenderer* renderer;
	int response;
	GtkTreeModel* tree_model;
	GtkTreeIter iter;
	GValue uid_value = { 0 };
	GValue text_value = { 0 };
	GValue value_value = { 0 };
	GValue unit_value = { 0 };
	GValue type_value = { 0 };
	GValue length_value = { 0 };
	GtkWidget* mode_button;
	GtkLabel* mode_label;
	
	configuration = microfeed_configuration_new();
	
	if ((provider_part = strchr(publisher, MICROFEED_PUBLISHER_IDENTIFIER_SEPARATOR_CHAR)) && provider_part != publisher && provider_part[1] != 0) {
		publisher_part = strndup(publisher, provider_part - publisher);
		provider_part = strdup(provider_part + 1);
	
		edit_dialog = microfeed_memory_allocate(EditDialog);
		edit_dialog->just_created = just_created;
		edit_dialog->dialog = gtk_dialog_new_with_buttons("Configure Microfeed Publisher", parent, 0, NULL);
		edit_dialog->wizard_settings = microfeed_configuration_get_wizard_settings(configuration, publisher);
		if (edit_dialog->wizard_settings) {
			edit_dialog->wizard_mode = 1;
			mode_button = gtk_button_new_with_label("Advanced");
			gtk_dialog_add_action_widget(GTK_DIALOG(edit_dialog->dialog), mode_button, 1);
		} else {
			mode_button = gtk_button_new_with_label("Simple");
			gtk_widget_set_sensitive(mode_button, FALSE);
			gtk_dialog_add_action_widget(GTK_DIALOG(edit_dialog->dialog), mode_button, 1);
		}
		edit_dialog->edit_button = gtk_button_new_from_stock(GTK_STOCK_EDIT);
		gtk_widget_set_sensitive(edit_dialog->edit_button, FALSE);
		gtk_dialog_add_action_widget(GTK_DIALOG(edit_dialog->dialog), edit_dialog->edit_button, 2);
#ifndef MAEMO_CHANGES
		button = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
		gtk_dialog_add_action_widget(GTK_DIALOG(edit_dialog->dialog), button, GTK_RESPONSE_NONE);
#endif
	
		table = gtk_table_new(3, 2, FALSE);
	 	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(edit_dialog->dialog)->vbox), table);
		label = gtk_label_new("Name:");
		gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, 0, 3, 3);
		entry = gtk_entry_new();
		gtk_entry_set_text(GTK_ENTRY(entry), publisher_part);
		gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);
		gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 0, 1, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 3, 3);
		label = gtk_label_new("Provider:");
		gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, 0, 3, 3);
		entry = gtk_entry_new();
		text = g_strdup_printf("%s (%s)", microfeed_configuration_get_provider_name(configuration, provider_part), provider_part);
		gtk_entry_set_text(GTK_ENTRY(entry), text);
		free(text);
		gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);
		gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 1, 2, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 3, 3);
	
		edit_dialog->list_store = gtk_list_store_new(7, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INT);		

#ifdef MAEMO_CHANGES
		touch_selector = hildon_touch_selector_new();
		gtk_widget_set_size_request(touch_selector, -1, 250);
		g_signal_connect(touch_selector, "changed", G_CALLBACK(on_changed), edit_dialog);
		column = hildon_touch_selector_append_column(HILDON_TOUCH_SELECTOR(touch_selector), GTK_TREE_MODEL(edit_dialog->list_store), NULL, NULL);

		renderer = gtk_cell_renderer_text_new();
		g_object_set(renderer, "xpad", 4, NULL);
		gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(column), renderer, FALSE);
		gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(column), renderer, "text", 2);

		renderer = gtk_cell_renderer_text_new();
		g_object_set(renderer, "xpad", 4, NULL);
		gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(column), renderer, TRUE);
		gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(column), renderer, "text", 0);

		gtk_table_attach(GTK_TABLE(table), touch_selector, 0, 2, 2, 3, GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 3, 3);
#else
		tree_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(edit_dialog->list_store));
		gtk_widget_set_size_request(tree_view, 400, 150);
		g_signal_connect(tree_view, "row-activated", G_CALLBACK(on_row_activated), edit_dialog);
		g_object_unref(G_OBJECT(edit_dialog->list_store));
	
		renderer = gtk_cell_renderer_text_new ();
		column = gtk_tree_view_column_new_with_attributes ("Setting", renderer, "text", 2, NULL);
		gtk_tree_view_append_column (GTK_TREE_VIEW(tree_view), column);
	
		renderer = gtk_cell_renderer_text_new ();
		column = gtk_tree_view_column_new_with_attributes ("Value", renderer, "text", 0, NULL);
		gtk_tree_view_append_column (GTK_TREE_VIEW(tree_view), column);
	
		scrolled_window = gtk_scrolled_window_new(NULL, NULL);
		gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	 	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled_window), GTK_SHADOW_IN);
	 	gtk_container_add(GTK_CONTAINER(scrolled_window), tree_view);
		gtk_table_attach(GTK_TABLE(table), scrolled_window, 0, 2, 2, 3, GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 3, 3);
				
		selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree_view));
		g_signal_connect(selection, "changed", G_CALLBACK(on_selection_changed), edit_dialog);
#endif
	
		if (microfeed_subscriber_subscribe_feed(subscriber, publisher, MICROFEED_FEED_URI_SETTINGS, &callbacks, settings_subscribed, edit_dialog)) {
			gtk_widget_show_all(edit_dialog->dialog);
			
			while ((response = gtk_dialog_run(GTK_DIALOG(edit_dialog->dialog))) != GTK_RESPONSE_DELETE_EVENT && response != GTK_RESPONSE_NONE) {
				if (response == 1) {
					mode_label = GTK_LABEL(gtk_bin_get_child(GTK_BIN(mode_button)));
					if (edit_dialog->wizard_mode) {
						edit_dialog->wizard_mode = 0;
						gtk_label_set_text(mode_label, "Simple");
						update_list(subscriber, publisher, edit_dialog);
					} else {
						edit_dialog->wizard_mode = 1;
						gtk_label_set_text(mode_label, "Advanced");
						update_list(subscriber, publisher, edit_dialog);
					}
				} else if (response == 2) {
#ifdef MAEMO_CHANGES
					if (hildon_touch_selector_get_selected(HILDON_TOUCH_SELECTOR(touch_selector), 0, &iter)) {
						tree_model = GTK_TREE_MODEL(edit_dialog->list_store);
#else
					if (gtk_tree_selection_get_selected(selection, &tree_model, &iter)) {
#endif
						gtk_tree_model_get_value(tree_model, &iter, 1, &uid_value);
						gtk_tree_model_get_value(tree_model, &iter, 2, &text_value);
						gtk_tree_model_get_value(tree_model, &iter, 3, &value_value);
						gtk_tree_model_get_value(tree_model, &iter, 4, &unit_value);
						gtk_tree_model_get_value(tree_model, &iter, 5, &type_value);
						gtk_tree_model_get_value(tree_model, &iter, 6, &length_value);
						show_value_dialog(subscriber, GTK_WINDOW(edit_dialog->dialog), publisher,
						                  g_value_get_string(&uid_value),
						                  g_value_get_string(&text_value),
						                  g_value_get_string(&value_value),
						                  g_value_get_string(&unit_value),
						                  g_value_get_string(&type_value),
						                  g_value_get_int(&length_value));
						g_value_unset(&uid_value);
						g_value_unset(&text_value);
						g_value_unset(&value_value);
						g_value_unset(&unit_value);
						g_value_unset(&type_value);
						g_value_unset(&length_value);
					}
				}
			}
		}
		gtk_widget_destroy(edit_dialog->dialog);
		free(publisher_part);
		free(provider_part);
	
		microfeed_subscriber_unsubscribe_feed(subscriber, publisher, MICROFEED_FEED_URI_SETTINGS, NULL, NULL);
	}
	
	microfeed_configuration_free(configuration);
}
