
#include "config.h"

#include <microfeed-common/microfeedconfiguration.h>
#include <microfeed-common/microfeedprotocol.h>
#include <microfeed-common/microfeedmisc.h>
#include <string.h>

#include "configuration.h"

typedef struct {
	MicrofeedSubscriber* subscriber;
	char* publisher_identifier;
	GtkWidget* vbox;
	GtkWidget* table;
	GHashTable* setting_entries;
} WizardPage;

typedef struct {
	GtkWidget* dialog;
	GArray* pages;
} WizardDialog;


static void error_occured(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const char* error_name, const char* error_message, void* user_data) {
	GtkWidget* dialog;
	
	if (error_name) {
		dialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "Failed to edit settings for %s: %s", publisher, (error_message ? error_message : error_name));
		gtk_window_set_title(GTK_WINDOW(dialog), "Error");
		gtk_dialog_run(GTK_DIALOG(dialog));
		gtk_widget_destroy(dialog);
	}
}

static void setting(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedItem* item, void* user_data) {
	WizardPage* wizard_page;
	MicrofeedConfiguration* configuration;
	int index;
	int found;
	const char** settings;
	const char* uid;
	const char* text;
	const char* value;
	const char* unit;
	const char* type;
	char* value_unit;
	const char* length_string;
	int length;
	GtkWidget* widget;

	wizard_page = (WizardPage*)user_data;
	configuration = microfeed_configuration_new();
		
	uid = microfeed_item_get_uid(item);
	index = 0;
	found = 0;
	for (settings = microfeed_configuration_get_wizard_settings(configuration, wizard_page->publisher_identifier);
	     settings && *settings; settings++) {
		if (!strcmp(uid, *settings)) {
			found = 1;
			break;
		}
		index++;
	}
	if (found) {
		if (!(text = microfeed_item_get_property(item, "setting.text"))) {
			text = uid;
		}
		if (!(value = microfeed_item_get_property(item, "setting.value"))) {
			value = "";
		}
		if (!(unit = microfeed_item_get_property(item, "setting.unit"))) {
			unit = "";
		}
		if (!(type = microfeed_item_get_property(item, "setting.type"))) {
			type = "string";
		}
		if (!(length_string = microfeed_item_get_property(item, "setting.length")) || (length = atoi(length_string)) <= 0) {
			length = 256;
		}
		if (!strcmp(type, "password")) {
			value_unit = g_strdup("<hidden>");
		} else {
			value_unit = g_strconcat(value, " ", unit, NULL);
		}

		widget = gtk_label_new(text);
		gtk_table_attach(GTK_TABLE(wizard_page->table), widget, 0, 1, index, index + 1, GTK_FILL, 0, 3, 3);
		widget = gtk_entry_new();
		gtk_entry_set_text(GTK_ENTRY(widget), value);
		gtk_entry_set_max_length(GTK_ENTRY(widget), length);
		if (length < 16) {
			gtk_entry_set_width_chars(GTK_ENTRY(widget), length);
		}
		if (!strcmp(type, "password")) {
			gtk_entry_set_visibility(GTK_ENTRY(widget), FALSE);
		}
//		g_signal_connect(widgety, "changed", G_CALLBACK(on_entry_changed), value_dialog);
		gtk_table_attach(GTK_TABLE(wizard_page->table), widget, 1, 2, index, index + 1, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 3, 3);
		widget = gtk_label_new(unit);
		gtk_misc_set_alignment(GTK_MISC(widget), 0.0, 0.5);
		gtk_table_attach(GTK_TABLE(wizard_page->table), widget, 2, 3, index, index + 1, GTK_FILL, 0, 3, 3);
		
		free(value_unit);
	}
	
	microfeed_configuration_free(configuration);
}

static void show_settings(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, void* user_data) {
	WizardPage* wizard_page;

	wizard_page = (WizardPage*)user_data;

	gtk_box_pack_start(GTK_BOX(wizard_page->vbox), wizard_page->table, TRUE, TRUE, 4);
	gtk_widget_show_all(wizard_page->table);
}

static MicrofeedSubscriberCallbacks callbacks = {
	NULL, /* error_occured */
	NULL, /* feed_update_started */
	NULL, /* feed_update_ended */
	NULL, /* feed_republishing_started */
	show_settings,
	NULL, /* item_added */
	NULL, /* item_changed */
	setting,
	NULL, /* item_removed */
	NULL, /* item_status_changed */
	NULL /* item_data_received */
};

static void settings_subscribed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const char* error_name, const char* error_message, void* user_data) {
	WizardPage* wizard_page;
	MicrofeedConfiguration* configuration;
	const char** settings;
	int count;

	wizard_page = (WizardPage*)user_data;
	configuration = microfeed_configuration_new();

	if (error_name) {
		error_occured(subscriber, publisher, uri, uid, error_name, error_message, user_data);
	} else {
		count = 0;
		for (settings = microfeed_configuration_get_wizard_settings(configuration, wizard_page->publisher_identifier);
		     settings && *settings; settings++) {
			count++;
		}
		wizard_page->table = gtk_table_new(count, 3, FALSE);
		wizard_page->setting_entries = g_hash_table_new(g_str_hash, g_str_equal);
		microfeed_subscriber_republish_items(subscriber, publisher, uri, NULL, NULL, 1024, error_occured, user_data);	
	}

	microfeed_configuration_free(configuration);
}

static void publisher_created(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const char* error_name, const char* error_message, void* user_data) {
	if (error_name) {
		error_occured(subscriber, publisher, uri, uid, error_name, error_message, user_data);
	} else {
		microfeed_subscriber_subscribe_feed(subscriber, publisher, MICROFEED_FEED_URI_SETTINGS, &callbacks, settings_subscribed, user_data);
	}
}

static void on_check_button_toggled(GtkToggleButton* toggle_button, gpointer user_data) {
	WizardPage* wizard_page;
	MicrofeedConfiguration* configuration;

	wizard_page = (WizardPage*)user_data;
	configuration = microfeed_configuration_new();

	if (gtk_toggle_button_get_active(toggle_button)) {
		if (microfeed_configuration_get_publisher_directory(configuration, wizard_page->publisher_identifier)) {
			microfeed_subscriber_subscribe_feed(wizard_page->subscriber, wizard_page->publisher_identifier, MICROFEED_FEED_URI_SETTINGS, &callbacks, settings_subscribed, wizard_page);
		} else {
			microfeed_subscriber_create_publisher(wizard_page->subscriber, wizard_page->publisher_identifier, publisher_created, wizard_page);
		}
	} else if (wizard_page->table) {
		microfeed_subscriber_unsubscribe_feed(wizard_page->subscriber, wizard_page->publisher_identifier, MICROFEED_FEED_URI_SETTINGS, NULL, NULL);
		gtk_widget_destroy(wizard_page->table);
		wizard_page->table = NULL;
	}

	microfeed_configuration_free(configuration);
}

void show_wizard_dialog(MicrofeedSubscriber* subscriber, GtkWindow* parent, const char* identifier) {
	MicrofeedConfiguration* configuration;
	WizardDialog* wizard_dialog;
	GtkWidget* button;
	const char** wizards;
	const char* colon;
	char* publisher_part;
	char* name;
	WizardPage* wizard_page;
	GtkWidget* check_button;
	GtkWidget* label;
	
	configuration = microfeed_configuration_new();
	
	wizard_dialog = microfeed_memory_allocate(WizardDialog);
	wizard_dialog->dialog = gtk_assistant_new();
	gtk_window_set_title(GTK_WINDOW(wizard_dialog->dialog), (identifier ? "Subscribe Microfeed Publishers" : "Configure Microfeed Publishers"));
	button = gtk_button_new_with_label("Advanced...");
	gtk_assistant_add_action_widget(GTK_ASSISTANT(wizard_dialog->dialog), button);
	gtk_widget_show_all(button);

	wizard_dialog->pages = g_array_new(FALSE, FALSE, sizeof(WizardPage*));
	for (wizards = microfeed_configuration_get_wizards(configuration); wizards && *wizards; wizards++) {
		if ((colon = strchr(*wizards, ':'))) {
			publisher_part = g_strndup(*wizards, colon - *wizards);
		} else {
			publisher_part = g_strdup(*wizards);
		}
		name = microfeed_util_string_concatenate(publisher_part, " (", microfeed_configuration_get_wizard_name(configuration, *wizards), ")", NULL);
		wizard_page = microfeed_memory_allocate(WizardPage);
		wizard_page->subscriber = subscriber;
		wizard_page->publisher_identifier = g_strdup(*wizards);
		wizard_page->vbox = gtk_vbox_new(FALSE, 8);
		check_button = gtk_check_button_new_with_label((identifier ? "Subscribe" : "Create"));
		g_signal_connect(check_button, "toggled", G_CALLBACK(on_check_button_toggled), wizard_page);
		gtk_box_pack_start(GTK_BOX(wizard_page->vbox), check_button, FALSE, TRUE, 8);
		if (microfeed_configuration_get_publisher_directory(configuration, *wizards) &&
		    (!identifier || microfeed_configuration_is_subscribed(configuration, identifier, *wizards))) {
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_button), TRUE);
		}
		gtk_assistant_append_page(GTK_ASSISTANT(wizard_dialog->dialog), wizard_page->vbox);
		gtk_assistant_set_page_title(GTK_ASSISTANT(wizard_dialog->dialog), wizard_page->vbox, name);
		gtk_assistant_set_page_type(GTK_ASSISTANT(wizard_dialog->dialog), wizard_page->vbox, GTK_ASSISTANT_PAGE_CONTENT);
		gtk_assistant_set_page_complete(GTK_ASSISTANT(wizard_dialog->dialog), wizard_page->vbox, TRUE);

		g_free(publisher_part);
		microfeed_memory_free(name);
	}
	label = gtk_label_new("You have configured all predefined publishers.\nThat is usually enough. However, if you want\nto do more advanced configuring, use the\nAdvanced... button.");
	gtk_misc_set_padding(GTK_MISC(label), 16, 16);
	gtk_assistant_append_page(GTK_ASSISTANT(wizard_dialog->dialog), label);
	gtk_assistant_set_page_title(GTK_ASSISTANT(wizard_dialog->dialog), label, "Configuration done");
	gtk_assistant_set_page_type(GTK_ASSISTANT(wizard_dialog->dialog), label, GTK_ASSISTANT_PAGE_CONFIRM);
	gtk_assistant_set_page_complete(GTK_ASSISTANT(wizard_dialog->dialog), label, TRUE);
	
	gtk_widget_show_all(wizard_dialog->dialog);
	
	gtk_main();

	gtk_widget_destroy(wizard_dialog->dialog);
	microfeed_configuration_free(configuration);
}
