from com import msgs
from mediabox.MediaWidget import MediaWidget
from mediabox import media_bookmarks
from mediabox import config as mb_config
from ui.EventBox import EventBox
from ui.ImageButton import ImageButton
from ui.Image import Image
from ui.ProgressBar import ProgressBar
from ui.Slider import Slider
import mediaplayer
from utils import maemo
from utils import logging
from theme import theme

import gtk


class VideoWidget(MediaWidget):
    """
    Media widget for viewing videos.
    """

    def __init__(self):
    
        self.__layout = None
        self.__player = None
        mediaplayer.add_observer(self.__on_observe_player)

        self.__aspect_ratio = 1.0

        self.__current_file = None
        self.__uri = ""
        self.__context_id = 0

        MediaWidget.__init__(self)
      
        # video screen
        self.__screen = gtk.DrawingArea()
        self.__screen.set_double_buffered(False)
        self.__screen.modify_bg(gtk.STATE_NORMAL, gtk.gdk.color_parse("#ff0000"))
        self.__screen.connect("expose-event", self.__on_expose)
        self.__screen.set_events(gtk.gdk.BUTTON_PRESS_MASK |
                                 gtk.gdk.KEY_PRESS_MASK)

        scr = self.__screen.get_screen()
        cmap = scr.get_rgb_colormap()
        self.__screen.set_colormap(cmap)

        
        self.__ebox = EventBox()
        self.add(self.__ebox)
        self.__ebox.connect_button_pressed(self.__on_click)


        # volume slider
        self.__volume = Slider(theme.mb_slider_gauge)
        self.__volume.set_mode(Slider.HORIZONTAL)
        self.__volume.set_background_color("#000000")
        self.add(self.__volume)
        self.__volume.connect_value_changed(self.__on_change_volume)


        # controls
        self.__btn_play = ImageButton(theme.mb_btn_play_1,
                                      theme.mb_btn_play_2)
        self.__btn_play.connect_clicked(self.__on_play_pause)

        self.__progress = ProgressBar()
        self.__progress.connect_changed(self.__on_set_position)
        self.__progress.connect_bookmark_changed(self.__on_change_bookmark)
        
        btn_bookmark = ImageButton(theme.mb_btn_bookmark_1,
                                   theme.mb_btn_bookmark_2)
        btn_bookmark.connect_clicked(self.__on_add_bookmark)
        
        self._set_controls(Image(theme.mb_toolbar_space_1),
                           self.__btn_play,
                           Image(theme.mb_progress_left),
                           self.__progress,
                           Image(theme.mb_progress_right),
                           btn_bookmark,
                           Image(theme.mb_toolbar_space_1))        


    def handle_message(self, msg, *args):
    
        if (msg == msgs.INPUT_EV_NAV_UP):
            handled = self.__player.send_key(self.__player.KEY_UP)
            
        elif (msg == msgs.INPUT_EV_NAV_DOWN):
            handled = self.__player.send_key(self.__player.KEY_DOWN)

        elif (msg == msgs.INPUT_EV_NAV_LEFT):
            handled = self.__player.send_key(self.__player.KEY_LEFT)

        elif (msg == msgs.INPUT_EV_NAV_RIGHT):
            handled = self.__player.send_key(self.__player.KEY_RIGHT)

        elif (msg == msgs.INPUT_EV_NAV_ENTER):
            handled = self.__player.send_key(self.__player.KEY_SELECT)

        elif (msg == msgs.INPUT_EV_NAV_MENU):
            handled = self.__player.send_key(self.__player.KEY_MENU1)

        elif (msg == msgs.MEDIA_EV_DOWNLOAD_PROGRESS):
            f, amount, total = args
            if (f == self.__current_file and total > 0):
                self.__progress.set_amount(float(amount) / total)
                

    def _visibility_changed(self):

        if (not self.may_render()):
            self.__screen.hide()
        else:
            self.__screen.show()
            

    def render_this(self):

        x, y = self.get_screen_pos()
        w, h = self.get_size()
        if (w == 0 or h == 0): return
        screen = self.get_screen()    

        self.__volume.set_geometry(0, h - 42, w, 42)
        h -= 42

        self.__ebox.set_geometry(0, 0, w, h)

        fx, fy, fw, fh = x, y, w, h
        vx, vy, vw, vh = x + 1, y + 1, w - 2, h - 2
        #vx, vy, vw, vh = x, y, w, h

        screen.fill_area(fx, fy, fw, fh, "#000000")

        # scale video screen
        w2 = int(self.__aspect_ratio * h)
        h2 = int(w / self.__aspect_ratio)

        if (w % 2 == 1): w -= 1
        if (w2 % 2 == 1): w2 -= 1
         
	"""
        if (w2 > w):
            if (w > 0): self.__screen.set_size_request(w, h2)
            print "SET", w, h2
            w2, h2 = w, h2
        else:
            if (w2 > 0): self.__screen.set_size_request(w2, h)
            print "SET", w2, h
            w2, h2 = w2, h
	"""
        self.__screen.set_size_request(w, h)
        w2, h2 = w, h

        if (not self.__layout):
            self.__layout = self.get_window()
            self.__layout.put(self.__screen, x + (w - w2) / 2, y + (h - h2) / 2)
            if (self.is_visible()):
                self.__screen.show()
        else:
            self.__layout.move(self.__screen, x + (w - w2) / 2, y + (h - h2) / 2)

        
    def __on_expose(self, src, ev):

        if (self.__player):
            win = self.__screen.window
            gc = win.new_gc()
            cmap = win.get_colormap()
            gc.set_foreground(cmap.alloc_color("#000000"))
            x, y, w, h = ev.area
            self.__player.handle_expose(win, gc, x, y, w, h)
            
            
    def __on_click(self, px, py):
            
        if (self.__player):
            handled = self.__player.send_key(self.__player.KEY_SELECT)
            
            if (not handled):
                self.emit_event(self.EVENT_FULLSCREEN_TOGGLED)


    def __on_observe_player(self, src, cmd, *args):
           
        if (cmd == src.OBS_POSITION):
            ctx, pos, total = args
            if (ctx == self.__context_id):
                pos_m = pos / 60
                pos_s = pos % 60
                if (total > 0.001):
                    total_m = total / 60
                    total_s = total % 60
                    info = "%d:%02d / %d:%02d" % (pos_m, pos_s, total_m, total_s)
                else:
                    info = "%d:%02d" % (pos_m, pos_s)            

                #self.emit_message(msgs.MEDIA_EV_POSITION,
                #                  pos * 1000, total * 1000)
                self.__progress.set_position(pos, total)
                self.emit_event(self.EVENT_MEDIA_POSITION,
                                pos * 1000, total * 1000)

            # bad mplayer shows its screen even when not asked for...
            #if (not self.may_render()):
            #    self.__hide_video_screen()

        elif (cmd == src.OBS_STARTED):
            #print "Started Player"
            self.__progress.set_message("")
            self.__progress.set_amount(0)
            #self.__btn_play.set_images(theme.btn_play_1,
            #                           theme.btn_play_2)
            
        elif (cmd == src.OBS_KILLED):
            #print "Killed Player"
            self.__uri = ""
            self.__progress.set_message("")
            self.__progress.set_amount(0)
            self.__btn_play.set_images(theme.mb_btn_play_1,
                                       theme.mb_btn_play_2)

        elif (cmd == src.OBS_CONNECTING):
            ctx = args[0]
            if (ctx == self.__context_id):
                self.__progress.set_message("... connecting ...")
                self.__progress.set_amount(0)

        elif (cmd == src.OBS_BUFFERING):
            ctx, value = args
            if (ctx == self.__context_id):
                self.__progress.set_message("... buffering (%d%%)..." % value)

        elif (cmd == src.OBS_ERROR):
            ctx, err = args
            if (ctx == self.__context_id):
                self.__current_file = None
                self.__btn_play.set_images(theme.mb_btn_play_1,
                                           theme.mb_btn_play_2)
                self.__progress.set_message("error")
                self.__show_error(err)
                self.emit_message(msgs.MEDIA_EV_PAUSE)
            
        elif (cmd == src.OBS_PLAYING):
            ctx = args[0]
            if (ctx == self.__context_id):
                #print "Playing"
                self.__progress.set_message("")
                self.__btn_play.set_images(theme.mb_btn_pause_1,
                                           theme.mb_btn_pause_2)
                self.emit_message(msgs.MEDIA_EV_PLAY)
            
        elif (cmd == src.OBS_STOPPED):
            ctx = args[0]
            if (ctx == self.__context_id):
                #print "Stopped"
                self.__progress.set_message("")
                self.__btn_play.set_images(theme.mb_btn_play_1,
                                           theme.mb_btn_play_2)
                self.emit_message(msgs.MEDIA_EV_PAUSE)
            
        elif (cmd == src.OBS_EOF):
            ctx = args[0]
            if (ctx == self.__context_id):        
                self.__uri = ""

                # unfullscreen
                #if (self.__is_fullscreen): self.__on_fullscreen()
                
                self.__btn_play.set_images(theme.mb_btn_play_1,
                                           theme.mb_btn_play_2)                
                self.emit_message(msgs.MEDIA_EV_PAUSE)
                self.emit_event(self.EVENT_MEDIA_EOF)


        elif (cmd == src.OBS_ASPECT):
            ctx, ratio = args
            if (ctx == self.__context_id):
                self.__set_aspect_ratio(ratio)



    def __show_error(self, errcode):
    
        if (errcode == self.__player.ERR_INVALID):
            self.call_service(msgs.DIALOG_SVC_ERROR,
                              "Invalid Stream",
                              "The media stream is invalid and cannot be loaded.")
            #dialogs.error("Invalid Stream", "Cannot load this stream.")
        elif (errcode == self.__player.ERR_NOT_FOUND):
            self.call_service(msgs.DIALOG_SVC_ERROR,
                              "Media Not Found",
                              "Cannot find a media stream to play.")
            #dialogs.error("Not found", "Cannot find a stream to play.")
        elif (errcode == self.__player.ERR_CONNECTION_TIMEOUT):
            self.call_service(msgs.DIALOG_SVC_ERROR,
                              "Connection Timeout",
                              "A connection timeout occured while loading the media stream.")
            #dialogs.error("Timeout", "Connection timed out.")       
        elif (errcode == self.__player.ERR_NOT_SUPPORTED):
            self.call_service(msgs.DIALOG_SVC_ERROR,
                              "Unsupported Media Format",
                              "The format of this media stream is not supported.")
            #dialogs.error("Not supported", "The media format is not supported.")


    def __on_set_position(self, pos):
    
        self.__player.set_volume(mb_config.volume())
        self.__player.seek_percent(pos)


    def __on_play_pause(self):
    
        self.__player.set_volume(mb_config.volume())
        self.__player.pause()
        
   
    def __on_add_bookmark(self):
    
        if (self.__current_file):
            self.__progress.add_bookmark()


    def __on_change_bookmark(self):
    
        if (self.__current_file):
            bookmarks = self.__progress.get_bookmarks()
            media_bookmarks.set_bookmarks(self.__current_file, bookmarks)
            self.emit_message(msgs.MEDIA_EV_BOOKMARKED,
                              self.__current_file, bookmarks)


    def __on_change_volume(self, v):

        self.set_scaling(v)


    def __set_aspect_ratio(self, ratio):
        """
        Sets the aspect ratio of the screen to the given value.
        """
    
        self.__aspect_ratio = ratio or 1.0
        self.render()


    def load(self, item, direction = MediaWidget.DIRECTION_NEXT):
        
        gtk.gdk.window_process_all_updates()
        
        uri = item.get_resource()
        if (not uri.startswith("/") and
            not "://localhost" in uri and
            not "://127.0.0.1" in uri):                    
            maemo.request_connection()
        #end if

        self.__player = mediaplayer.get_player_for_mimetype(item.mimetype)
        self.__player.set_window(self.__screen.window.xid)
        try:
            self.__context_id = self.__player.load_video(uri)
        except:
            logging.error("could not load video '%s'\n%s" \
                            % (uri, logging.stacktrace()))
            return
                        
        self.__player.set_volume(mb_config.volume())
        bookmarks = media_bookmarks.get_bookmarks(item)
        self.__progress.set_bookmarks(bookmarks)
        self.__uri = uri
        self.__current_file = item
                

    def play_pause(self):
    
        if (self.__player):
            self.__on_play_pause()


    def stop(self):
    
        if (self.__player):
            self.__player.stop()


    def close(self):
        
        if (self.__player):
            self.__player.close()
            

    def increment(self):

        vol = mb_config.volume()
        vol = min(100, vol + 5)
        mb_config.set_volume(vol)        
        if (self.__player):
            self.__player.set_volume(vol)
        self.__volume.set_value(vol / 100.0)
        self.emit_event(self.EVENT_MEDIA_SCALE, vol / 100.0)

       
    def decrement(self):

        vol = mb_config.volume()
        vol = max(0, vol - 5)
        mb_config.set_volume(vol)        
        if (self.__player):
            self.__player.set_volume(vol)
        self.__volume.set_value(vol / 100.0)
        self.emit_event(self.EVENT_MEDIA_SCALE, vol / 100.0)


    def set_scaling(self, v):

        self.__volume.set_value(v)
        self.__set_volume(v)


    def __set_volume(self, v):

        vol = int(v * 100)
        mb_config.set_volume(vol)
        if (self.__player):
            self.__player.set_volume(vol)
        self.send_event(self.EVENT_MEDIA_SCALE, v)


    def seek(self, pos):
    
        if (self.__player):
            self.__player.seek(pos)


    def rewind(self):
    
        if (self.__player):
            self.__player.rewind()
        
        
    def forward(self):
    
        if (self.__player):
            self.__player.forward()

