# -*- coding: utf-8 -*-
'''
MediaBox Widget UI Progress Bar
2010(c) Kirill Plyashkevich <ru.spirit@gmail.com>
'''

import gobject, gtk
from advpowcommon.util.format import *
from mediabox_widget.ui.Widgets import ImageWidget
from mediabox_widget.ui.Cairo import Cairo

class AbstractProgressBar(ImageWidget):

  def __init__(self, images, name, size):
    ImageWidget.__init__(self, name, size, images)
    self.__progress_pos, self.__progress_total = 0, 0
    self.set_events(gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.BUTTON_RELEASE_MASK | gtk.gdk.LEAVE_NOTIFY)
    self.__progress_percentage = [0,0]
    self.__redraw_progress = True
    self.current_progress_pixbuf = None

  def draw(self, image):
    raise NotImplementedError('AbstractProgressBar.draw() must be implemented')

  def load_image(self, file):
    raise NotImplementedError('AbstractProgressBar.load_image() must be implemented')

  def get_progress(self):
    return self.__progress_pos, self.__progress_total

  def load_progress(self, pos, total):
    self.__progress_total = max(total, 0)
    self.__progress_pos = min(max(pos, 0), self.__progress_total)
    tmp = [0,0]
    if self.__progress_total != 0:
      tmp = [int(self.size_width*(float(self.__progress_pos)/self.__progress_total)),int(self.size_height*(float(self.__progress_pos)/self.__progress_total))]
    else:
      self.current_progress_pixbuf = None
    if self.__progress_percentage != tmp:
      self.__progress_percentage = tmp
      self.__redraw_progress = True
    else:
      self.__redraw_progress = False

  def update_progress(self, pos, total):
    self.load_progress(pos, total)
    self.redraw()

  def increase_progress(self, incr):
    self.update_progress(min(self.__progress_pos + incr, self.__progress_total), self.__progress_total)
    return self.get_wstate() != self.DISABLED and self.__progress_total != 0

  def redraw_progress(self):
    return self.__redraw_progress

  def handle_active_change():
    self.update_progress(self.__progress_pos, self.__progress_total)

class HProgressBar(AbstractProgressBar):

  def __init__(self, name, size):
    images = ('cursor_frame', 'progress_1', 'progress_2')
    AbstractProgressBar.__init__(self, images, name, size)

  def draw(self, cr):
    if not self.current_progress_pixbuf:
      self.current_progress_pixbuf = self.images[self.get_wstate()].copy()
    progress_pos, progress_total = self.get_progress()
    if self.redraw_progress():
      image = self.images[self.get_wstate()].copy()
      if image:
        if self.get_wstate() == self.ENABLED and progress_total != 0:
          self.images[self.PRESSED].copy_area(0, 0, int(self.size_width*(float(progress_pos)/progress_total)), self.size_height, image, 0, 0)
          self.current_progress_pixbuf = image
    Cairo.draw_add_pixbuf(cr, self.current_progress_pixbuf)
    if progress_total != 0:
      Cairo.draw_text(cr, self.draw_text(), (self.size_width, self.size_height), self.size_height/2, text_color=self._theme.get_color('mb_button_text'))
    return False

  def draw_text(self):
    progress_pos, progress_total = self.get_progress()
    if progress_total > 0:
      return '%d%' % (float(progress_pos)/progress_total)*100
    else:
      return ''

  def load_image(self, file):
    pix1 = self.cacher.get(file).scale_simple(self.size_height, self.size_height, gtk.gdk.INTERP_BILINEAR)
    pix = pix1.scale_simple(self.size_width, self.size_height, gtk.gdk.INTERP_BILINEAR)
    pix2 = pix1.copy()
    pix3 = pix1.copy()
    pix1.copy_area(0, 0, self.size_height/2, self.size_height, pix, 0, 0)
    for x in range(self.size_height/2, self.size_width - self.size_height/2):
      pix2.copy_area(self.size_height/2, 0, 1, self.size_height, pix, x, 0)
    pix3.copy_area(self.size_height/2, 0, self.size_height/2, self.size_height, pix, self.size_width - self.size_height/2, 0)
    return pix

class HClickableProgressBar(HProgressBar):

  def __init__(self, name, size, progress_cb=None):
    HProgressBar.__init__(self, name, size)
    self.progress_cb = progress_cb
    self.connect('button-press-event', self.__do_button_press_event)
    self.connect('button-release-event', self.__do_button_release_event)
    self.connect('leave-notify-event', self.__do_leave_notify_event)

  def __do_button_press_event(self, widget, event):
    if self.get_wstate() == self.ENABLED and event.type == gtk.gdk.BUTTON_PRESS:
      self.enable()
    return False

  def __do_button_release_event(self, widget, event):
    if self.get_wstate() != self.DISABLED and self.get_property('visible'):
      progress_pos, progress_total = self.get_progress()
      if event.type == gtk.gdk.BUTTON_RELEASE and progress_pos:
        x, y = event.get_coords()
        self.load_progress(int((float(x)/self.size_width)*progress_total), progress_total)
        if self.progress_cb:
          progress_pos, progress_total = self.get_progress()
          self.progress_cb(progress_pos)
      self.enable()
    return False

  def __do_leave_notify_event(self, widget, event):
    return self.__do_button_release_event(widget, event)

class SeekBar(HClickableProgressBar):

  def __init__(self, name, size, progress_cb=None):
    HClickableProgressBar.__init__(self, name, size, progress_cb)
    self.__timer = None
    self.connect('destroy', self.stop_timer)

  def draw_text(self):
    progress_pos, progress_total = self.get_progress()
    return '%s / %s' % (get_formatted_time(progress_pos), get_formatted_time(progress_total))

  def update_progress(self, pos, total):
    self.stop_timer()
    self.load_progress(pos, total)
    self.start_timer()
    self.redraw()

  def start_timer(self):
    if not self.__timer:
      self.__timer = gobject.timeout_add(1000, self.increase_progress, 1)

  def stop_timer(self, widget=None):
    if self.__timer:
      gobject.source_remove(self.__timer)
      self.__timer = None
