# -*- coding: utf-8 -*-
'''
MediaBox Widget UI Cairo Drawing
2010(c) Kirill Plyashkevich <ru.spirit@gmail.com>
'''

import cairo, gtk

class Cairo():
  pi = 3.1415926
  rgb_norm = 65535.0

  ALIGN_LEFT = 0
  ALIGN_CENTER = 1
  ALIGN_RIGHT = 2

  ALIGN_TOP = 0
  ALIGN_MIDDLE = 1
  ALIGN_BOTTOM = 2

  @staticmethod
  def draw_background(cr, has_alpha=False):
    if has_alpha:
      cr.set_source_rgba(1.0, 1.0, 1.0, 0.0)
    else:
      cr.set_source_rgb(1.0, 1.0, 1.0)
    cr.set_operator(cairo.OPERATOR_SOURCE)
    cr.paint()

  @staticmethod
  def draw_rectangle(widget, bgcolor='#000000', has_alpha=False, alpha=0.5, draw_background=False):
    cr = None
    try:
      cr = widget.window.cairo_create()
      if draw_background:
        Cairo.draw_background(cr, has_alpha)
      else:
        cr.set_operator(cairo.OPERATOR_SOURCE)
      width, height = widget.allocation[2], widget.allocation[3]
      cr.rectangle(0, 0, width, height)
      bg_color = gtk.gdk.color_parse(bgcolor)
      cr.set_source_rgba(bg_color.red / Cairo.rgb_norm, bg_color.green/Cairo.rgb_norm, bg_color.blue/Cairo.rgb_norm, alpha)
      cr.fill_preserve()
    except:
      pass
    return cr

  @staticmethod
  def draw_rounded_rectangle(widget, bgcolor='#000000', has_alpha=False, alpha=0.5, radius=10, draw_background=False):
    cr = None
    try:
      bg_color = gtk.gdk.color_parse(bgcolor)
      cr = widget.window.cairo_create()
      if draw_background:
        Cairo.draw_background(cr, has_alpha)
      else:
        cr.set_operator(cairo.OPERATOR_SOURCE)
      width, height = widget.allocation[2], widget.allocation[3]
      cr.move_to(0, radius)
      cr.arc(radius, radius, radius, Cairo.pi, 1.5 * Cairo.pi)
      cr.line_to(width - radius, 0)
      cr.arc(width - radius, radius, radius, 1.5 * Cairo.pi, 0.0)
      cr.line_to(width, height - radius)
      cr.arc(width - radius, height - radius, radius, 0.0, 0.5 * Cairo.pi)
      cr.line_to(radius, height)
      cr.arc(radius, height - radius, radius, 0.5 * Cairo.pi, Cairo.pi)
      cr.close_path()
      cr.set_source_rgba(bg_color.red / Cairo.rgb_norm, bg_color.green/Cairo.rgb_norm, bg_color.blue/Cairo.rgb_norm, alpha)
      cr.fill_preserve()
    except:
      pass
    return cr

  @staticmethod
  def draw_text(cr, text, rect_size, font_size, align=1, valign=1, text_color='#ffffff'):
    if cr and text:
      x_bearing, y_bearing, t_width, t_height, x_advance, y_advance = Cairo.get_text_size(cr, text, font_size)
      text_x, text_y = 0, -y_bearing
      if align == Cairo.ALIGN_CENTER:
        text_x = (rect_size[0] - t_width)/2
      elif align == Cairo.ALIGN_RIGHT:
        text_x = rect_size[0] - t_width
      if valign == Cairo.ALIGN_MIDDLE:
        text_y = (rect_size[1] - font_size)/2 - y_bearing
      elif valign == Cairo.ALIGN_BOTTOM:
        text_y = rect_size[1] - font_size - y_bearing
      Cairo.draw_text_at_pos(cr, text, font_size, text_x, text_y, text_color)
    return cr

  @staticmethod
  def draw_text_at_pos(cr, text, font_size, x=0, y=0, text_color='#ffffff'):
    if cr and text:
      txt_color = gtk.gdk.color_parse(text_color)
      cr.set_operator(cairo.OPERATOR_OVER)
      cr.set_font_size(font_size)
      cr.set_antialias(cairo.ANTIALIAS_GRAY)
      cr.move_to(x, y)
      cr.set_source_rgb(txt_color.red / Cairo.rgb_norm, txt_color.green/Cairo.rgb_norm, txt_color.blue/Cairo.rgb_norm)
      cr.show_text(text)
    return cr

  @staticmethod
  def draw_text_with_icon(cr, icon, text, rect_size, font_size, align=1, valign=1, text_color='#ffffff', padding=0):
    if cr:
      if icon:
        icon_x, icon_y, t_width, t_height, text_x, text_y = 0, (rect_size[1] - icon.get_height())/2, 0, 0, 0, 0
        if text:
          x_bearing, y_bearing, t_width, t_height, x_advance, y_advance = Cairo.get_text_size(cr, text, font_size)
          text_y = 0
        if align == Cairo.ALIGN_CENTER:
          icon_x += (rect_size[0] - t_width - icon.get_width())/2
          if text:
            icon_x -= padding/2
            text_x += icon.get_width() + icon_x + padding
        elif align == Cairo.ALIGN_RIGHT:
          text_x += rect_size[0] - t_width
          icon_x += text_x - icon.get_width()
          if not(text):
            icon_x -= padding
        cr = Cairo.draw_add_pixbuf(cr, icon, icon_x, icon_y)
        if text:
          if valign == Cairo.ALIGN_MIDDLE:
            text_y += (rect_size[1] - font_size)/2 - y_bearing
          elif valign == Cairo.ALIGN_BOTTOM:
            text_y += rect_size[1] - font_size - y_bearing
          cr = Cairo.draw_text_at_pos(cr, text, font_size, x=text_x, y=text_y, text_color=text_color)
      else:
        cr = Cairo.draw_text(cr, text, rect_size, font_size, align, valign, text_color)
    return cr

  @staticmethod
  def get_text_size(cr, text, font_size):
    cr.set_font_size(font_size)
    return cr.text_extents(text)

  @staticmethod
  def draw_add_pixbuf(cr, pixbuf, x=0, y=0):
    cr.set_operator(cairo.OPERATOR_OVER)#ADD)
    cr.set_source_pixbuf(pixbuf, x, y)
    cr.paint()
    return cr

  def draw_frame(cr, x, y, width, height, frame_width=1, frame_color='#ffffff'):
    cr.set_operator(cairo.OPERATOR_OVER)
    cr.set_line_width(frame_width)
    fr_color = gtk.gdk.color_parse(frame_color)
    cr.set_source_rgb(fr_color.red / Cairo.rgb_norm, fr_color.green/Cairo.rgb_norm, fr_color.blue/Cairo.rgb_norm)
    cr.rectangle(x, y, width, height)
    cr.stroke_preserve()
    return cr
