#! /usr/bin/env python
# -*- coding: UTF-8 -*-

import os, sys, time

import gobject
import dbus, dbus.glib

MAFW_GSTREAMER_PLUGIN = 'com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer'
MAFW_RENDERER_PATH    = '/com/nokia/mafw/renderer/gstrenderer'
MAFW_RENDERER_IFACE   = 'com.nokia.mafw.renderer'

MAFW_RENDERER_STATE_STOPPED       = 0
MAFW_RENDERER_STATE_STARTED       = 1
MAFW_RENDERER_STATE_PAUSED        = 2
MAFW_RENDERER_STATE_TRANSITIONING = 3

MAFW_RENDERER_STATE_STRINGS       = ['stopped', 'started', 'paused', 'transitioning']

DEBUG = False

def debug(message):
  if DEBUG and sys.stderr.isatty():
    sys.stderr.write('[%f] %s\n' % (time.time(), message))

class IRendererMonitor:
  def on_state_stopped(self):
    pass

  def on_state_started(self):
    pass

  def on_state_paused(self):
    pass

  def on_state_transitioning(self):
    pass

  def on_media_changed(self, index, id):
    pass

  def on_playlist_changed(self, playlist):
    pass

  def on_buffering_info(self, status):
    pass

  def on_metadata(self, key, value):
    pass

class MafwRendererMonitor(dbus.Interface, IRendererMonitor):
  STATE_CHANGE_CALLBACKS = [
    'on_state_stopped',
    'on_state_started',
    'on_state_paused',
    'on_state_transitioning'
  ]

  def __init__(self, object):
    dbus.Interface.__init__(self, object, MAFW_RENDERER_IFACE)

    self.connect_to_signal('state_changed',    self.__state_changed)
    self.connect_to_signal('metadata_changed', self.__metadata_changed)
    self.connect_to_signal('media_changed',    self.__media_changed)
    self.connect_to_signal('playlist_changed', self.__playlist_changed)
    self.connect_to_signal('buffering_info',   self.__buffering_info)

  def __state_changed(self, state):
    if state < len(MafwRendererMonitor.STATE_CHANGE_CALLBACKS):
      symbol    = MafwRendererMonitor.STATE_CHANGE_CALLBACKS[state]
      callback  = self.__getattribute__(symbol)
      callback()
    else:
      debug('INFO: MafwRendererInterface: Unknown state %d' % (state))
      return

    debug('INFO: MafwRendererInterface: state changed: %s' % MAFW_RENDERER_STATE_STRINGS[state])

  def __media_changed(self, index, id):
    debug("INFO: MafwRendererInterface: media changed: %d: %s" % (index, id))
    self.on_media_changed(index, id)

  def __playlist_changed(self, playlist):
    debug("INFO: MafwRendererInterface: playlist changed: " + str(playlist))
    self.on_playlist_changed(playlist)

  def __metadata_changed(self, key, state, value):
    debug("INFO: MafwRendererInterface: metadata: %s='%s'" % (key, value))
    self.on_metadata(key, value)

  def __buffering_info(self, status):
    debug("INFO: MafwRendererInterface: buffering: %3d%%" % (status * 100))
    self.on_buffering_info(status)

if __name__ == '__main__':
  DEBUG = True

  session   = dbus.SessionBus()
  main      = gobject.MainLoop()
  renderer  = session.get_object(MAFW_GSTREAMER_PLUGIN, MAFW_RENDERER_PATH)
  monitor   = MafwRendererMonitor(renderer)

  p,c,s,i = monitor.get_status()

  print "playlist: %d" % p
  print "   index: %d" % c
  print "   state: %s" % MAFW_RENDERER_STATE_STRINGS[s]
  print "      id: %s" % i 

  print

  main.run()

