
#define _GNU_SOURCE

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cairo.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <microfeed-common/microfeedconfiguration.h>
#include <microfeed-subscriber/microfeedsubscriber.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeedprotocol.h>
#include <string.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>

#include "mauku-widget.h"

#define CYCLES 30
#define SUBSCRIBER_ID "com.innologies.mauku_widget"
#define HORIZONTAL_PADDING 8
#define VERTICAL_PADDING 16

HD_DEFINE_PLUGIN_MODULE(MaukuWidget, mauku_widget, HD_TYPE_HOME_PLUGIN_ITEM);

typedef struct _Badge Badge;

struct _Badge {
	Badge* next;
	Badge* previous;
	
	char* publisher;
	time_t timestamp;
	char* uid;
	char* uri;
	char* text;
	char* info;
	char* sender_image;

	int shown : 1;
	int placeholder : 1;
};

struct _MaukuWidgetPrivate {
	MicrofeedSubscriber* subscriber;
	Badge* current_badge;
	GtkWidget* window;
	GtkWidget* label;
	guint timeout;
	GdkPixmap* pixmap;
	cairo_surface_t* cairo_surface;
	int hide_show_counter;
	guint hide_show_timeout;
	double alpha;
	int buttons_visible;
	int button_previous_activated;
	int button_next_activated;
	int button_open_activated;
	cairo_surface_t* background;
	GHashTable* image_cache;
};

static Badge* badge_new(const char* publisher, time_t timestamp, const char* uid, const char* uri, const char* text, const char* info, const char* sender_image) {
	Badge* badge;
	
	badge = microfeed_memory_allocate(Badge);
	badge->publisher = strdup(publisher);
	badge->uid = strdup(uid);
	badge->uri = strdup(uri);
	badge->text = strdup(text);
	badge->info = strdup(info);
	if (sender_image) {
		badge->sender_image = strdup(sender_image);
	}
	
	return badge;
}

static void badge_free(Badge* badge) {
	free(badge->uid);
	free(badge->uri);
	free(badge->text);
	microfeed_memory_free(badge);
}

static void add_badge_into_list(MaukuWidget* mauku_widget, Badge* badge) {
	Badge* i;
	
	if (!mauku_widget->priv->current_badge) {
		mauku_widget->priv->current_badge = badge;
	} else {
		for (i = mauku_widget->priv->current_badge; i->previous; i = i->previous) {
		}
		badge->next = i;
		i->previous = badge;
	}
}

static remove_badge_from_list(MaukuWidget* mauku_widget, Badge* badge) {
	if (badge->previous) {
		badge->previous->next = badge->next;
	}
	if (badge->next) {
		badge->next->previous = badge->previous;
	}
	badge->previous = badge->next = NULL;
}

GdkPixbuf* image_cache_get_image(MaukuWidget* mauku_widget, const char* uid) {
	
	return g_hash_table_lookup(mauku_widget->priv->image_cache, uid);
}

void image_cache_set_image(MaukuWidget* mauku_widget, const char* uid, const void* data, size_t data_size) {
	GdkPixbufLoader* loader;
	GdkPixbuf* pixbuf;
	
	loader = gdk_pixbuf_loader_new();
	if (gdk_pixbuf_loader_write(loader, data, data_size, NULL) &&
	    gdk_pixbuf_loader_close(loader, NULL) &&
	    (pixbuf = gdk_pixbuf_loader_get_pixbuf(loader))) {
		g_object_ref(pixbuf);
		g_hash_table_insert(mauku_widget->priv->image_cache, g_strdup(uid), pixbuf);
	}
	g_object_unref(loader);
}

static gboolean view_hide_show_timeout(void* user_data) {
	gboolean continue_animation = TRUE;
	MaukuWidget* mauku_widget;
	int width;
	int height;

	mauku_widget = MAUKU_WIDGET(user_data);
	
	if (mauku_widget->priv->pixmap) {
		gdk_drawable_get_size(mauku_widget->priv->pixmap, &width, &height);

		if (mauku_widget->priv->hide_show_counter < CYCLES) {
			mauku_widget->priv->alpha = 1.0 - (mauku_widget->priv->hide_show_counter / (double)CYCLES);
		} else if (mauku_widget->priv->hide_show_counter == CYCLES) {
			mauku_widget->priv->alpha = 0.0;
			if (mauku_widget->priv->cairo_surface) {
				cairo_surface_destroy(mauku_widget->priv->cairo_surface);
				mauku_widget->priv->cairo_surface = NULL;
			}
		} else if (mauku_widget->priv->hide_show_counter == CYCLES * 2) {
			mauku_widget->priv->alpha = 1.0;
			continue_animation = FALSE;
			mauku_widget->priv->hide_show_timeout = 0;
		} else {
			mauku_widget->priv->alpha = (mauku_widget->priv->hide_show_counter - CYCLES) / (double)CYCLES;
		}
		mauku_widget->priv->hide_show_counter++;

		gtk_widget_queue_draw_area(GTK_WIDGET(mauku_widget), 0, 0, width, height);
	}
	
	return continue_animation;
}

static void view_hide_show(MaukuWidget* mauku_widget) {
	if (!mauku_widget->priv->hide_show_timeout) {
		mauku_widget->priv->hide_show_counter = 0;
		mauku_widget->priv->hide_show_timeout = g_timeout_add(15, view_hide_show_timeout, mauku_widget);
	} else if (mauku_widget->priv->hide_show_counter > CYCLES) {
		mauku_widget->priv->hide_show_counter = (CYCLES * 2) - mauku_widget->priv->hide_show_counter;
	}
}

static gboolean view_show_next_timeout(void* user_data) {
	gboolean more_unshown = FALSE;
	MaukuWidget* mauku_widget;
	Badge* b;
	
	mauku_widget = MAUKU_WIDGET(user_data);
	
	for (b = mauku_widget->priv->current_badge; b->shown && b->previous; b = b->previous) {
	}

	if (!b->shown) {	
		mauku_widget->priv->current_badge = b;
		view_hide_show(mauku_widget);
		if (mauku_widget->priv->current_badge->previous) {
			more_unshown = TRUE;
			if (!mauku_widget->priv->timeout) {
				mauku_widget->priv->timeout = g_timeout_add(5000, view_show_next_timeout, mauku_widget);
			}
		}
	}
	if (!more_unshown) {
		mauku_widget->priv->timeout = 0;
	}

	return more_unshown;
}

static void view_show_next(MaukuWidget* mauku_widget) {
	if (!mauku_widget->priv->timeout) {
		view_show_next_timeout(mauku_widget);	
	}
}

static void view_show_next_delayed(MaukuWidget* mauku_widget) {
	if (!mauku_widget->priv->timeout) {
		mauku_widget->priv->timeout = g_timeout_add(5000, view_show_next_timeout, mauku_widget);
	}
}

static void feed_subscribed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const char* error_name, const char* error_message, void* user_data) {
	MaukuWidget* mauku_widget;
	
	mauku_widget = MAUKU_WIDGET(user_data);
	
	if (error_name) {
		printf("Subscribing failed: %s %s: %s (%s)\n", publisher, uri, error_name, error_message);
	} else {

		printf("Subscribed: %s %s\n", publisher, uri);
	
		microfeed_subscriber_republish_items(subscriber, publisher, uri, NULL, NULL, 1, NULL, NULL);
	}
}

static void error_occured(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const char* error_name, const char* error_message, void* user_data) {
	MaukuWidget* mauku_widget;
	const char* separator;
	char* publisher_part;
	GtkWidget* banner;
	
	mauku_widget = MAUKU_WIDGET(user_data);
	if ((separator = strchr(publisher, MICROFEED_PUBLISHER_IDENTIFIER_SEPARATOR_CHAR))) {
		publisher_part = g_strndup(publisher, separator - publisher);
	} else {
		publisher_part = g_strdup(publisher);
	}
	banner = hildon_banner_show_informationf(GTK_WIDGET(mauku_widget), "", "%s: %s", publisher_part, error_message);
	hildon_banner_set_timeout(HILDON_BANNER(banner), 10000);
	g_free(publisher_part);
	printf("Failed: %s %s (%s): %s (%s)\n", publisher, uri, uid, error_name, error_message);
}

static void item_added(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedItem* item, void* user_data) {
	MaukuWidget* mauku_widget;
	const char* text;
	const char* thread_uri;
	const char* sender_image;
	const char* sender_nick;
	char* s;
	const char* publisher_part;
	char* info;
	Badge* badge;
	
	mauku_widget = MAUKU_WIDGET(user_data);

	printf("Added: %s %s %s %ld %u\n", publisher, uri, microfeed_item_get_uid(item), (unsigned long)microfeed_item_get_timestamp(item), (unsigned int)microfeed_item_get_status(item));
	/*if (microfeed_item_get_status(item) & MICROFEED_ITEM_STATUS_UNREAD || !current_badge) {*/

	if (strcmp(microfeed_item_get_uid(item), MICROFEED_ITEM_UID_FEED_METADATA)) {
		if (!(text = microfeed_item_get_property(item, "content.text"))) {
			text = "<no content>";
		}
		if (!(thread_uri = microfeed_item_get_property(item, "thread.uri")) &&
		    !(thread_uri = microfeed_item_get_property(item, "feed.uri")) &&
		    !(thread_uri = microfeed_item_get_property(item, "user.uri"))) {
			thread_uri = uri;
		}
		if ((sender_image = microfeed_item_get_property(item, "user.image")) &&
		    !image_cache_get_image(mauku_widget, sender_image)) {
			microfeed_subscriber_send_item_data(subscriber, publisher, MICROFEED_FEED_URI_IMAGES, sender_image, NULL, NULL);
		}
		if (!(sender_nick = microfeed_item_get_property(item, "user.nick"))) {
			sender_nick = "<unknown>";
		}
		if ((s = strchr(publisher, MICROFEED_PUBLISHER_IDENTIFIER_SEPARATOR_CHAR))) {
			publisher_part = strndup(publisher, s - publisher);
		} else {
			publisher_part = strdup("<not specified>");
		}
		info = microfeed_util_string_concatenate("By ", sender_nick, " in ", publisher_part, " ", NULL);		
		badge = badge_new(publisher, microfeed_item_get_timestamp(item), microfeed_item_get_uid(item), uri, text, info, sender_image);
		add_badge_into_list(mauku_widget, badge);
		view_show_next(mauku_widget);
	}
/*	}*/
}

static void item_changed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedItem* item, void* user_data) {
	MaukuWidget* mauku_widget;
	
	mauku_widget = MAUKU_WIDGET(user_data);
	
	printf("Changed: %s %s %s %ld %u\n", publisher, uri, microfeed_item_get_uid(item), (unsigned long)microfeed_item_get_timestamp(item), (unsigned int)microfeed_item_get_status(item));
}

static void item_removed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, void* user_data) {
	MaukuWidget* mauku_widget;
	
	mauku_widget = MAUKU_WIDGET(user_data);
	
	printf("Removed: %s %s %s\n", publisher, uri, uid);
}

static void item_status_changed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const MicrofeedItemStatus status, void* user_data) {
	MaukuWidget* mauku_widget;
	
	mauku_widget = MAUKU_WIDGET(user_data);
	
	printf("Status changed: %s %s %s %u\n", publisher, uri, uid, (unsigned int)status);
}

static void item_data(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const void* data, const size_t length, void* user_data) {
	MaukuWidget* mauku_widget;
	int width;
	int height;
	
	mauku_widget = MAUKU_WIDGET(user_data);
	
	printf("Data: %s %s %s (length: %u)\n", publisher, uri, uid, length);
	image_cache_set_image(mauku_widget, uid, data, length);
	if (!mauku_widget->priv->current_badge->sender_image || !strcmp(mauku_widget->priv->current_badge->sender_image, uid)) {
		if (mauku_widget->priv->cairo_surface) {
			cairo_surface_destroy(mauku_widget->priv->cairo_surface);
			mauku_widget->priv->cairo_surface = NULL;
		}
		
		if (mauku_widget->priv->pixmap) {
			gdk_drawable_get_size(mauku_widget->priv->pixmap, &width, &height);
			gtk_widget_queue_draw_area(GTK_WIDGET(mauku_widget), 0, 0, width, height);
		}
	}
}

static MicrofeedSubscriberCallbacks images_callbacks = {
	error_occured,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	item_data
};

static MicrofeedSubscriberCallbacks overview_callbacks = {
	error_occured,
	NULL,
	NULL,
	NULL,
	NULL,
	item_added,
	item_changed,
	item_added, /* item_republished */
	item_removed,
	item_status_changed,
	NULL
};

gboolean on_configure_event(GtkWidget* widget, GdkEventConfigure* event, gpointer user_data) {
	MaukuWidget* mauku_widget;
	static int width = 0;
	static int height = 0;
	GdkPixmap* new_pixmap;
	
	mauku_widget = MAUKU_WIDGET(widget);

	if (mauku_widget->priv->pixmap) {
		gdk_drawable_get_size(mauku_widget->priv->pixmap, &width, &height);
	}
	if (width != event->width || height != event->height){
		new_pixmap = gdk_pixmap_new(widget->window, event->width,  event->height, -1);
		if (mauku_widget->priv->pixmap) {
			gdk_draw_drawable(new_pixmap, widget->style->fg_gc[GTK_WIDGET_STATE(widget)], mauku_widget->priv->pixmap, 0, 0, 0, 0, (width < event->width ? width : event->width), (height < event->height ? height : event->height));
			g_object_unref(mauku_widget->priv->pixmap); 
		}
		mauku_widget->priv->pixmap = new_pixmap;
		width = event->width;
		height = event->height;
		if (mauku_widget->priv->cairo_surface) {
			cairo_surface_destroy(mauku_widget->priv->cairo_surface);
			mauku_widget->priv->cairo_surface = NULL;
		}
		gtk_widget_queue_draw_area(widget, 0, 0, width, height);
	}

	return TRUE;
}

gboolean on_expose_event(GtkWidget* widget, GdkEventExpose* event, gpointer user_data) {
	MaukuWidget* mauku_widget;
	int x = 0;
	int y;
	int width;
	int height;
	cairo_t* cairo;
	PangoLayout* layout;
	PangoFontDescription* font_description;
	GdkPixbuf* sender_pixbuf = NULL;
	GdkPixbuf* content_pixbuf;
	GdkPixbuf* pixbuf1;
	GdkPixbuf* pixbuf2;

	mauku_widget = MAUKU_WIDGET(widget);	

	if (mauku_widget->priv->pixmap) {
		gdk_drawable_get_size(mauku_widget->priv->pixmap, &width, &height);
		if (!mauku_widget->priv->cairo_surface) {
			mauku_widget->priv->cairo_surface = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, width, height);
			cairo = cairo_create(mauku_widget->priv->cairo_surface);
			cairo_translate(cairo, HORIZONTAL_PADDING, VERTICAL_PADDING);
			cairo_set_source_rgba(cairo, 0.0, 0.0, 0.0, 0.0);
			cairo_set_operator(cairo, CAIRO_OPERATOR_SOURCE);
			cairo_paint(cairo);
			if (mauku_widget->priv->current_badge) {
				mauku_widget->priv->current_badge->shown = 1;

				cairo_set_operator(cairo, CAIRO_OPERATOR_OVER);
				cairo_set_source_rgba(cairo, 0.0, 0.0, 0.0, 1.0);
				if (mauku_widget->priv->current_badge->sender_image &&
				    (sender_pixbuf = image_cache_get_image(mauku_widget, mauku_widget->priv->current_badge->sender_image))) {
					x += gdk_pixbuf_get_width(sender_pixbuf) + 8;
				}

				cairo_move_to(cairo, x, 0);
				layout = pango_cairo_create_layout(cairo);
				font_description = pango_font_description_from_string("Sans 18");
				pango_layout_set_font_description(layout, font_description);
				pango_font_description_free(font_description);
				pango_layout_set_width(layout, (width - x - 2 * HORIZONTAL_PADDING) * PANGO_SCALE);
				pango_layout_set_height(layout, (height - 24) * PANGO_SCALE);
				pango_layout_set_wrap(layout, PANGO_WRAP_WORD_CHAR);
				pango_layout_set_ellipsize(layout, PANGO_ELLIPSIZE_END);
				pango_layout_set_text(layout, mauku_widget->priv->current_badge->text, -1);
				pango_cairo_show_layout(cairo, layout);
				pango_layout_get_size(layout, NULL, &y);
				y /= PANGO_SCALE;
				g_object_unref(layout);

				cairo_move_to(cairo, x, y + 4);
				layout = pango_cairo_create_layout(cairo);
				font_description = pango_font_description_from_string("Sans 14");
				pango_layout_set_font_description(layout, font_description);
				pango_font_description_free(font_description);
				pango_layout_set_width(layout, (width - x) * PANGO_SCALE);
				pango_layout_set_height(layout, (height - y - 4) * PANGO_SCALE);
				pango_layout_set_wrap(layout, PANGO_WRAP_WORD_CHAR);
				pango_layout_set_ellipsize(layout, PANGO_ELLIPSIZE_END);
				pango_layout_set_text(layout, mauku_widget->priv->current_badge->info, -1);
				pango_cairo_show_layout(cairo, layout);
				g_object_unref(layout);

				if (sender_pixbuf) {
					gdk_cairo_set_source_pixbuf(cairo, sender_pixbuf, 1, 1);
					cairo_paint(cairo);
					x = gdk_pixbuf_get_width(sender_pixbuf);
					y = gdk_pixbuf_get_height(sender_pixbuf);
					cairo_set_source_rgba(cairo, 0.5, 0.5, 0.5, 0.9);
					cairo_set_line_width(cairo, 2);
					cairo_move_to(cairo, x + 2, 0);
					cairo_line_to(cairo, 0, 0);
					cairo_line_to(cairo, 0, y + 2);
					cairo_stroke(cairo); 
					cairo_set_source_rgba(cairo, 0.8, 0.8, 0.8, 0.9);
					cairo_set_line_width(cairo, 2);
					cairo_move_to(cairo, x + 2, 0);
					cairo_line_to(cairo, x + 2, y + 2);
					cairo_line_to(cairo, 0, y + 2);
					cairo_stroke(cairo); 
					if ((pixbuf1 = gdk_pixbuf_scale_simple(sender_pixbuf, x, y / 3, GDK_INTERP_BILINEAR))) {
						if ((pixbuf2 = gdk_pixbuf_flip(pixbuf1, FALSE))) {
							gdk_cairo_set_source_pixbuf(cairo, pixbuf2, 1, y + 2);
							cairo_paint_with_alpha(cairo, 0.2);					
							gdk_pixbuf_unref(pixbuf2);
						}
						gdk_pixbuf_unref(pixbuf1);
					}
				}
			}
			cairo_destroy(cairo);
		}

		cairo = gdk_cairo_create(mauku_widget->priv->pixmap);
		if (mauku_widget->priv->background) {
			cairo_set_source_surface(cairo, mauku_widget->priv->background, 0.0, 0.0);
		} else {
			cairo_set_source_rgba(cairo, 1.0, 1.0, 1.0, 0.9);
		}
		cairo_set_operator(cairo, CAIRO_OPERATOR_SOURCE);
		cairo_paint(cairo);
		cairo_set_source_surface(cairo, mauku_widget->priv->cairo_surface, 0, 0);
		cairo_set_operator(cairo, CAIRO_OPERATOR_OVER);
		cairo_paint_with_alpha(cairo, mauku_widget->priv->alpha);
		if (mauku_widget->priv->buttons_visible && mauku_widget->priv->current_badge) {
			if (mauku_widget->priv->current_badge->next) {
				if (mauku_widget->priv->button_next_activated) {
					cairo_set_source_rgba(cairo, 1.0, 0.0, 0.0, 0.6);
				} else {
					cairo_set_source_rgba(cairo, 0.0, 0.0, 0.0, 0.6);
				}
				cairo_move_to(cairo, width / 4 - HORIZONTAL_PADDING, VERTICAL_PADDING);
				cairo_line_to(cairo, HORIZONTAL_PADDING, height / 2);
				cairo_line_to(cairo, width / 4 - HORIZONTAL_PADDING, height - VERTICAL_PADDING);
				cairo_line_to(cairo, width / 4 - HORIZONTAL_PADDING, VERTICAL_PADDING);
				cairo_fill(cairo);
			}
			if (mauku_widget->priv->current_badge->previous) {
				if (mauku_widget->priv->button_previous_activated) {
					cairo_set_source_rgba(cairo, 1.0, 0.0, 0.0, 0.6);
				} else {
					cairo_set_source_rgba(cairo, 0.0, 0.0, 0.0, 0.6);
				}
				cairo_move_to(cairo, 3 * width / 4 + HORIZONTAL_PADDING, VERTICAL_PADDING);
				cairo_line_to(cairo, width - HORIZONTAL_PADDING, height / 2);
				cairo_line_to(cairo, 3 * width / 4 + HORIZONTAL_PADDING, height - VERTICAL_PADDING);
				cairo_line_to(cairo, 3 * width / 4 + HORIZONTAL_PADDING, VERTICAL_PADDING);
				cairo_fill(cairo);			
			}
			if (mauku_widget->priv->button_open_activated) {
				cairo_set_source_rgba(cairo, 1.0, 0.0, 0.0, 0.6);
			} else {
				cairo_set_source_rgba(cairo, 0.0, 0.0, 0.0, 0.6);
			}
			cairo_move_to(cairo, width / 2, VERTICAL_PADDING);
			cairo_line_to(cairo, 5 * width / 8, height - VERTICAL_PADDING);
			cairo_line_to(cairo, 3 * width / 8, height - VERTICAL_PADDING);
			cairo_line_to(cairo, width / 2, VERTICAL_PADDING);
			cairo_fill(cairo);			
			
		}
		cairo_destroy(cairo);

		gdk_draw_drawable(widget->window, widget->style->fg_gc[GTK_WIDGET_STATE(widget)], mauku_widget->priv->pixmap, event->area.x, event->area.y, event->area.x, event->area.y, event->area.width, event->area.height);
	}	

	return TRUE;
}

gboolean on_button_press_event(GtkWidget* widget, GdkEventButton* event, gpointer user_data) {
	MaukuWidget* mauku_widget;
	int width;
	int height;

	mauku_widget = MAUKU_WIDGET(widget);	

	mauku_widget->priv->buttons_visible = 1;
	mauku_widget->priv->button_next_activated = mauku_widget->priv->button_previous_activated = mauku_widget->priv->button_open_activated = 0;

	if (mauku_widget->priv->pixmap) {
		gdk_drawable_get_size(mauku_widget->priv->pixmap, &width, &height);
		if (event->x > 0 && event->y > 0 && event->x < width && event->y < height) {
			if (event->x < width / 4) {
				mauku_widget->priv->button_next_activated = 1;
			} else if (event->x > 3 * width / 4) {
				mauku_widget->priv->button_previous_activated = 1;
			} else if (event->x > 3 * width / 8 && event->x < 5 * width / 8) {
				mauku_widget->priv->button_open_activated = 1;
			}
		}
		gtk_widget_queue_draw_area(widget, 0, 0, width, height);
	}

	return FALSE;
}

gboolean on_button_release_event(GtkWidget* widget, GdkEventButton* event, gpointer user_data) {
	MaukuWidget* mauku_widget;
	int width;
	int height;

	mauku_widget = MAUKU_WIDGET(widget);	

	mauku_widget->priv->buttons_visible = 0;

	if (mauku_widget->priv->pixmap) {
		gdk_drawable_get_size(mauku_widget->priv->pixmap, &width, &height);
		if (event->x > 0 && event->y > 0 && event->x < width && event->y < height) {
			if (event->x < width / 4 && mauku_widget->priv->current_badge && mauku_widget->priv->current_badge->next) {
				if (mauku_widget->priv->timeout) {
					g_source_remove(mauku_widget->priv->timeout);
					mauku_widget->priv->timeout = 0;
				}
				mauku_widget->priv->current_badge = mauku_widget->priv->current_badge->next;
				view_hide_show(mauku_widget);
				view_show_next_delayed(mauku_widget);
			} else if (event->x > 3 * width / 4 && mauku_widget->priv->current_badge && mauku_widget->priv->current_badge->previous) {
				if (mauku_widget->priv->timeout) {
					g_source_remove(mauku_widget->priv->timeout);
					mauku_widget->priv->timeout = 0;
				}
				mauku_widget->priv->current_badge = mauku_widget->priv->current_badge->previous;
				view_hide_show(mauku_widget);
				view_show_next_delayed(mauku_widget);
			} else if (event->x > 3 * width / 8 && event->x < 5 * width / 8 && mauku_widget->priv->current_badge) {
/* TODO */
			}
		}

		gtk_widget_queue_draw_area(widget, 0, 0, width, height);
	}

	return FALSE;
}

static int is_string_in_array(const char** array, const char* string) {
	int retvalue = 0;
	while (*array) {
		if (!strcmp(*array, string)) {
			retvalue = 1;
			break;
		}
		array++;
	}
	
	return retvalue;
}

static void configured_subscribe(MicrofeedSubscriber* subscriber, const char* publisher, void* user_data) {
	microfeed_subscriber_subscribe_feed(subscriber, publisher, MICROFEED_FEED_URI_IMAGES, &images_callbacks, NULL, user_data);
	microfeed_subscriber_subscribe_feed(subscriber, publisher, MICROFEED_FEED_URI_OVERVIEW, &overview_callbacks, feed_subscribed, user_data);
}
	
static void configured_unsubscribe(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, void* user_data) {
	microfeed_subscriber_unsubscribe_feed(subscriber, publisher, uri, NULL, NULL);
}

static void on_child_exit(GPid pid, gint status, gpointer user_data) {
	MaukuWidget* mauku_widget;
	
	mauku_widget = MAUKU_WIDGET(user_data);
	microfeed_subscriber_handle_configured_subscriptions(mauku_widget->priv->subscriber, configured_subscribe, configured_unsubscribe, mauku_widget);

	g_spawn_close_pid(pid);
}

static void on_show_settings(HDHomePluginItem* hd_home_plugin_item, gpointer user_data) {
	MaukuWidget* mauku_widget;
	GtkWidget* about_dialog;
	GtkWidget* label;
	const char** subscriptions;
	gchar* args[3];
	GPid pid;
	GtkWidget* message_dialog;
	
	mauku_widget = MAUKU_WIDGET(hd_home_plugin_item);
	about_dialog = hildon_dialog_new_with_buttons("Mauku Widget", NULL, GTK_DIALOG_MODAL, "Configure...", GTK_RESPONSE_ACCEPT, NULL);
	label = gtk_label_new("");
	gtk_label_set_markup(GTK_LABEL(label), "<big>Mauku Widget</big>\nCopyright 2009 Henrik Hedberg\n&lt;henrik.hedberg@innologies.fi&gt;");
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(about_dialog)->vbox), label);
	gtk_widget_show_all(about_dialog);
	if (gtk_dialog_run(GTK_DIALOG(about_dialog)) == GTK_RESPONSE_ACCEPT) {
		args[0] = "microfeed-configuration";
		args[1] = SUBSCRIBER_ID;
		args[2] = NULL;
		if (g_spawn_async(NULL, args, NULL, G_SPAWN_DO_NOT_REAP_CHILD | G_SPAWN_SEARCH_PATH, NULL, NULL, &pid, NULL)) {
			g_child_watch_add(pid, on_child_exit, mauku_widget);
		} else {
			message_dialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, "Could not launch the microfeed-configure application. Maybe the package is missing?");
			gtk_dialog_run(GTK_DIALOG(message_dialog));
			gtk_widget_destroy(message_dialog);
		}
	}
	gtk_widget_destroy(about_dialog);
}

static void mauku_widget_init(MaukuWidget* mauku_widget) {
	GdkColormap* colormap;
	DBusError error;
	DBusConnection* connection;
	const char** subscriptions;

	mauku_widget->priv = G_TYPE_INSTANCE_GET_PRIVATE(mauku_widget, MAUKU_TYPE_WIDGET, MaukuWidgetPrivate);

	microfeed_thread_init();

	if ((colormap = gdk_screen_get_rgba_colormap(gdk_screen_get_default()))) {
		gtk_widget_set_colormap(GTK_WIDGET(mauku_widget), colormap);
	}
	gtk_widget_set_double_buffered(GTK_WIDGET(mauku_widget), FALSE);

	mauku_widget->priv->background = cairo_image_surface_create_from_png("/home/user/.mauku-widget/background.png");
	if (cairo_surface_status(mauku_widget->priv->background) != CAIRO_STATUS_SUCCESS) {
		cairo_surface_destroy(mauku_widget->priv->background);
		mauku_widget->priv->background = cairo_image_surface_create_from_png("/usr/share/images/mauku-widget-background.png");
		if (cairo_surface_status(mauku_widget->priv->background) != CAIRO_STATUS_SUCCESS) {
			cairo_surface_destroy(mauku_widget->priv->background);
			mauku_widget->priv->background = NULL;
		}
	}
	if (mauku_widget->priv->background) {
		gtk_widget_set_size_request(GTK_WIDGET(mauku_widget),
		                            cairo_image_surface_get_width(mauku_widget->priv->background),
					    cairo_image_surface_get_height(mauku_widget->priv->background));
	} else {
		gtk_widget_set_size_request(GTK_WIDGET(mauku_widget), 760, 175);
	}
	
	hd_home_plugin_item_set_settings(HD_HOME_PLUGIN_ITEM(mauku_widget), TRUE);
	g_signal_connect(mauku_widget, "show-settings", G_CALLBACK(on_show_settings), NULL);

	g_signal_connect(mauku_widget, "destroy", G_CALLBACK(gtk_main_quit), NULL);
	g_signal_connect(mauku_widget, "expose_event", G_CALLBACK(on_expose_event), NULL);
	g_signal_connect(mauku_widget, "configure_event", G_CALLBACK(on_configure_event), NULL);
	g_signal_connect(mauku_widget, "button_press_event", G_CALLBACK(on_button_press_event), NULL);
	g_signal_connect(mauku_widget, "motion_notify_event", G_CALLBACK(on_button_press_event), NULL);
	g_signal_connect(mauku_widget, "button_release_event", G_CALLBACK(on_button_release_event), NULL);
	gtk_widget_add_events(GTK_WIDGET(mauku_widget), GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK | GDK_BUTTON_MOTION_MASK);
	
	mauku_widget->priv->image_cache = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_object_unref);

	dbus_error_init(&error);
	connection = dbus_bus_get(DBUS_BUS_SESSION, &error);
	dbus_connection_setup_with_g_main(connection, NULL);
	mauku_widget->priv->subscriber = microfeed_subscriber_new(SUBSCRIBER_ID, "/com/innologies/mauku_widget", connection);
	microfeed_subscriber_handle_configured_subscriptions(mauku_widget->priv->subscriber, configured_subscribe, configured_unsubscribe, mauku_widget);
}

static void mauku_widget_class_init(MaukuWidgetClass* klass) {
	g_type_class_add_private(G_OBJECT_CLASS(klass), sizeof(MaukuWidgetPrivate));
}

static void mauku_widget_class_finalize(MaukuWidgetClass* klass) {
}
