/*
  MathJinni - A simple formular calculator
  Copyright (C) 2008  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Formula.h"

#include <Lum/Base/String.h>

#include <Lum/Combo.h>
#include <Lum/Dialog.h>
#include <Lum/Label.h>
#include <Lum/String.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>
#include <Lum/View.h>
#include <Lum/Wizard.h>

#include "Util.h"

#define LEADING_PAGES 1
#define TRAILING_PAGES 1

class FormulaFillDialog : public Lum::Dialog
{
private:
  Parser                             parser;

  Formula                            *formula;
  std::vector<Lum::Model::SizeTRef>  srcUnits;
  std::vector<Lum::Model::StringRef> srcValues;
  Lum::Model::SizeTRef               resUnit;
  std::vector<Lum::Model::ActionRef> stepActions;
  Lum::Model::StringRef              result;

public:
  FormulaFillDialog(Formula* formula)
  : formula(formula),
    result(new Lum::Model::String(L""))
  {
    Observe(GetClosedAction());
  }

  void PreInit()
  {
    Lum::Object                *oneOfMany;
    Lum::Label                 *label;
    Lum::Panel                 *panel;
    Lum::String                *string;
    Lum::Text                  *text;
    Lum::TextValue             *textValue;
    Lum::Wizard                *wizard;

    Lum::Model::ActionRef      stepAction;
    Lum::Model::StringTableRef resUnitList;
    Unit                       *unit;

    wizard=new Lum::Wizard();
    wizard->SetCancelAction(GetClosedAction());

    // Initial page

    panel=Lum::VPanel::Create(true,true);

    text=new Lum::Text(L"You are now calculating the following formula:");
    text->SetFlex(true,false);
    panel->Add(text);

    panel->AddSpace();

    label=Lum::Label::Create(true,false);
    label->AddLabel(L"Formula:",formula->GetFormula());
    label->AddLabel(L"Description:",formula->GetDescription());

    resUnit=new Lum::Model::SizeT(1);
    resUnitList=new Lum::Model::StringTable();
    unit=unitMap[formula->GetUnit()];

    for (size_t j=0; j<unit->GetAlternativeCount(); j++) {
      resUnitList->Append(unit->GetAlternative(j)->GetName());
    }

    Lum::Def::OneOfMany resUnitDef(Lum::Def::Desc(L"Unit"),
                                resUnit,resUnitList);

    oneOfMany=Lum::OS::display->GetBehaviour()->GetOneOfManyControl(resUnitDef);
    oneOfMany->SetFlex(true,false);
    label->AddLabel(L"Unit:",oneOfMany);

    panel->Add(label);

    stepAction=new Lum::Model::Action();
    Observe(stepAction);
    stepActions.push_back(stepAction);

    wizard->AddPage(L"Unit of Result",panel,stepAction);

    for (size_t i=0; i<formula->GetVariables().size(); i++) {
      Lum::Model::SizeTRef       srcUnit=new Lum::Model::SizeT(1);
      Lum::Model::StringTableRef srcUnitList=new Lum::Model::StringTable();

      Lum::Model::StringRef value=new Lum::Model::String(L"");

      Observe(value);

      unit=unitMap[formula->GetVariables()[i]->GetUnit()];

      for (size_t j=0; j<unit->GetAlternativeCount(); j++) {
        srcUnitList->Append(unit->GetAlternative(j)->GetName());
      }

      panel=Lum::VPanel::Create(true,true);

      label=Lum::Label::Create(true,false);
      label->AddLabel(L"Formula:",formula->GetFormula());
      label->AddLabel(L"Variable name:",formula->GetVariables()[i]->GetName());
      label->AddLabel(L"Description:",formula->GetVariables()[i]->GetDescription());

      string=new Lum::String();
      string->SetFlex(true,false);
      string->SetAlignment(Lum::String::right);
      string->SetInputHelper(new Lum::InputHelperNumber(Lum::InputHelperNumber::typeFloat));
      string->SetModel(value);
      label->AddLabel(L"Value:",string);

      Lum::Def::OneOfMany srcUnitDef(Lum::Def::Desc(L"Unit"),
                                     srcUnit,srcUnitList);

      oneOfMany=Lum::OS::display->GetBehaviour()->GetOneOfManyControl(resUnitDef);
      oneOfMany->SetFlex(true,false);
      label->AddLabel(L"Unit:",oneOfMany);

      panel->Add(label);

      stepAction=new Lum::Model::Action();
      Observe(stepAction);
      stepAction->Disable();
      stepActions.push_back(stepAction);

      wizard->AddPage(L"Value and unit of variable",panel,stepAction);

      srcValues.push_back(value);
      srcUnits.push_back(srcUnit);
    }

    panel=Lum::VPanel::Create(true,true);

    label=Lum::Label::Create(true,false);
    label->AddLabel(L"Formula:",formula->GetFormula());

    textValue=new Lum::TextValue();
    textValue->SetFlex(true,false);
    textValue->SetModel(result);

    label->AddLabel(L"Result:",textValue);

    panel->Add(label);

    wizard->AddPage(L"Result",panel,GetClosedAction());

    SetMain(wizard);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      Exit();
    }
    else {
      for (size_t i=0; i<srcValues.size(); i++) {
        if (model==srcValues[i]) {
          Parser::Expression *expression=NULL;
          Parser::ReturnCode returnCode;
          size_t             pos;

          if (!srcValues[i]->Get().empty()) {
            expression=parser.Parse(Lum::Base::WStringToString(srcValues[i]->Get()),returnCode,pos,false);
          }

          if (expression==NULL) {
            stepActions[LEADING_PAGES+i]->Disable();
          }
          else {
            stepActions[LEADING_PAGES+i]->Enable();
          }

          delete expression;

          break;
        }
      }

      for (size_t i=LEADING_PAGES; i<stepActions.size(); i++) {
        if (model==stepActions[i] && stepActions[i]->IsFinished()) {

          for (size_t j=0; j<i; j++) {
            Unit               *unit;
            Parser::Expression *expression;
            Parser::ReturnCode returnCode;
            size_t             pos;
            bool               success;
            double             factor;
            double             value;

            // Factor
            unit=unitMap[formula->GetVariables()[j]->GetUnit()];
            expression=parser.Parse(Lum::Base::WStringToString(unit->GetAlternative(srcUnits[j]->Get()-1)->GetFactor()),returnCode,pos,false);
            assert(expression!=NULL);
            success=expression->Calculate(factor,returnCode);
            assert(success);
            delete expression;

            // Value
            expression=parser.Parse(Lum::Base::WStringToString(srcValues[j]->Get()),returnCode,pos,false);
            assert(expression!=NULL);
            success=expression->Calculate(value,returnCode);
            assert(success);
            parser.AddConstant(Lum::Base::WStringToString(formula->GetVariables()[j]->GetName()),value/factor);
            delete expression;
          }

          if (i==stepActions.size()-1) {
            Unit               *unit;
            Parser::Expression *expression;
            Parser::ReturnCode returnCode;
            size_t             pos;
            bool               success;
            double             factor;
            double             value;

            //Factor
            unit=unitMap[formula->GetUnit()];
            expression=parser.Parse(Lum::Base::WStringToString(unit->GetAlternative(resUnit->Get()-1)->GetFactor()),returnCode,pos,false);
            assert(expression!=NULL);
            success=expression->Calculate(factor,returnCode);
            assert(success);
            delete expression;

            // Value
            expression=parser.Parse(Lum::Base::WStringToString(formula->GetFormula()),returnCode,pos,false);
            assert(expression!=NULL);
            success=expression->Calculate(value,returnCode);
            assert(success);
            result->Set(DoubleToWStringFloat(factor*value)+L" "+unit->GetAlternative(resUnit->Get()-1)->GetName());
            delete expression;
          }

          break;
        }
      }
    }

    Lum::Dialog::Resync(model,msg);
  }
};

class FormulaItem : public Lum::Model::Tree::Item
{
private:
  Formula *formula;

public:
  FormulaItem(Formula *formula)
  : formula(formula)
  {
    // no code
  }

  std::wstring GetText() const
  {
    return formula->GetName();
  }

  Formula* GetFormula() const
  {
    return formula;
  }
};

FormulaCalc::FormulaCalc()
 : selectAction(new Lum::Model::Action()),
   executeAction(new Lum::Model::Action()),
   functions(new Lum::Model::Tree()),
   notation(new Lum::Model::String(L"")),
   description(new Lum::Model::String(L"")),
   tree(NULL)
{
  Observe(selectAction);
  Observe(executeAction);
}

void FormulaCalc::CalcSize()
{
  //Lum::Combo               *combo;
  Lum::Label               *label;
  //Lum::String              *string;
  Lum::TextValue           *textValue;
  Lum::View                *view;

  label=new Lum::Label();

  view=new Lum::View();
  view->SetFlex(true,true);

  tree=new Lum::Tree();
  tree->SetFlex(true,true);
  tree->SetSelectionChangedAction(selectAction);
  tree->SetDoubleClickAction(executeAction);
  view->SetObject(tree);
  view->SetModel(functions);
  label->AddLabel(L"Functions:",view);

  textValue=new Lum::TextValue();
  textValue->SetFlex(true,false);
  textValue->SetModel(notation);
  label->AddLabel(L"Formula:",textValue);

  textValue=new Lum::TextValue();
  textValue->SetFlex(true,false);
  textValue->SetModel(description);
  label->AddLabel(L"Description:",textValue);

  container=label;

  Component::CalcSize();
}

void FormulaCalc::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (tree!=NULL && model==selectAction && selectAction->IsFinished()) {
    FormulaItem *item=dynamic_cast<FormulaItem*>(tree->GetSelection());

    if (item!=NULL) {
      description->Set(item->GetFormula()->GetDescription());
      notation->Set(item->GetFormula()->GetFormula());
    }
    else {
      description->Set(L"");
      notation->Set(L"");
    }
  }
  else if (model==executeAction && executeAction->IsFinished()) {
    FormulaItem *item=dynamic_cast<FormulaItem*>(tree->GetSelection());

    if (item!=NULL) {
      FormulaFillDialog *dialog;

      dialog=new FormulaFillDialog(item->GetFormula());
      dialog->SetParent(GetWindow());
      if (dialog->Open()) {
        dialog->EventLoop();
        dialog->Close();
      }

      delete dialog;
    }
  }
}

void ListCategories(const std::wstring& category, std::vector<std::wstring>& subs)
{
  size_t start,pos;

  start=0;
  pos=0;

  subs.clear();

  while (pos<category.length()) {
    if (category[pos]=='/') {
      subs.push_back(category.substr(start,pos-start));
      start=pos+1;
    }
    pos++;
  }

  if (start<category.length()) {
    subs.push_back(category.substr(start,pos-start));
  }
}

void FormulaCalc::LoadConfig(Lum::Config::Node* /*top*/)
{
  Lum::Model::Tree::Item                         *top;
  std::map<std::wstring,Lum::Model::Tree::Item*> categories;

  // Prepare tree structure

  top=new Lum::Model::Tree::TextItem(L"Formula");

  functions->SetTop(top);

  for (size_t i=0; i<formulas.size(); i++) {
    std::vector<std::wstring> subs;
    Lum::Model::Tree::Item    *t=top;

    ListCategories(formulas[i]->GetCategory(),subs);

    for (size_t j=0; j<subs.size(); j++) {
      std::wstring c;

      for (size_t k=0; k<=j; k++) {
        if (k>0) {
          c.append(1,'/');
        }
        c.append(subs[k]);
      }

      std::map<std::wstring,Lum::Model::Tree::Item*>::const_iterator iter;

      iter=categories.find(c);

      if (iter!=categories.end()) {
        t=iter->second;
      }
      else {
        Lum::Model::Tree::Item *s=new Lum::Model::Tree::TextItem(subs[j]);

        s->Collaps();
        t->Append(s);
        categories[c]=s;

        t=s;
      }
    }
  }

  for (size_t i=0; i<formulas.size(); i++) {
    Formula                                                        *f=formulas[i];
    std::map<std::wstring,Lum::Model::Tree::Item*>::const_iterator iter;

    iter=categories.find(f->GetCategory());

    iter->second->Append(new FormulaItem(f));
  }

}

void FormulaCalc::StoreConfig(Lum::Config::Node *top)
{
  // no code
}

