/*
 * POI definitions
 *
 * See docs for table definitions.
 */

#ifndef _MAPPER_POI_H
#define _MAPPER_POI_H

#include <glib.h>
#include <sqlite3.h>

#include "osm.h"

/** Category list **/
typedef enum {
	CAT_ID,
	CAT_ENABLED,
	CAT_LABEL,
	CAT_DESC,
	CAT_POI_CNT,
	CAT_ICON,
	CAT_NUM_COLUMNS
} CategoryList;

typedef enum {
	POI_TYPE_POINT=0,
	POI_TYPE_LANDMARK=1,
} poi_types_e;

typedef enum {
	POI_SEARCH_NEAR=0,
	POI_SEARCH_TEXT,
	POI_SEARCH_TEXT_CAT,
	POI_SEARCH_CAT,
} poi_search_type;

typedef enum {
	POI_SOURCE_USER=0,
	POI_SOURCE_OSM=1,
	POI_SOURCE_GPX_IMPORT=2,
} poi_source;

typedef struct _poi_db poi_db;
struct _poi_db {
	sqlite3 *db;
	gchar *file;
	guint zoom;
};

/** 
 * Data to describe a POI. 
 */
typedef struct _poi_info poi_info;
struct _poi_info {
	guint poi_id;
	guint cat_id;
	guint is_in_c;
	guint is_in_p;
	guint source;
	guint osm_id;
	time_t addtime;
	gdouble lat;
	gdouble lon;
	gboolean public;
	gchar *label;
	gchar *desc;
	gchar *cat_label;
	gchar *cat_desc;
	gchar *postal_code;
	gchar *url;
};

/** 
 * Data used during action: add or edit category/poi 
 */
typedef struct _delete_poi delete_poi;
struct _delete_poi {
	GtkWidget *dialog;
	gchar *txt_label;
	guint id;
};

/**
 * Data describing a single POI category 
 */
typedef struct _poi_category poi_category;
struct _poi_category {
	guint id;
	gboolean enabled;
	gboolean warning;
	gchar *label;
	gchar *desc;
};

/* POI SQL */
struct sql_poi_stmt {
	sqlite3_stmt *select_poi;
	sqlite3_stmt *select_poi_fast;
	sqlite3_stmt *select_poi_by_id;
	sqlite3_stmt *select_nearest_poi;
	sqlite3_stmt *select_poi_search;
	sqlite3_stmt *select_poi_search_cat;
	sqlite3_stmt *select_poi_by_cat;
	sqlite3_stmt *insert_poi;
	sqlite3_stmt *update_poi;
	sqlite3_stmt *delete_poi;
	sqlite3_stmt *delete_poi_by_catid;
	sqlite3_stmt *select_cat;
	sqlite3_stmt *insert_cat;
	sqlite3_stmt *update_cat;
	sqlite3_stmt *delete_cat;
	sqlite3_stmt *toggle_cat;
	sqlite3_stmt *selall_cat;
	sqlite3_stmt *selall_cat_fast;
};
struct sql_poi_stmt poisql;

guint _poi_zoom;

gboolean poi_init(void);
gboolean poi_init_db(sqlite3 **db);
void poi_deinit(void);
void poi_deinit_db(sqlite3 *db);

poi_info *poi_new(void);
void poi_free(poi_info *p);

gboolean poi_get_list_near_unit(guint unitx, guint unity, guint range, GtkListStore **store, guint *num_poi);
gboolean poi_get_list_inside(gdouble lat1, gdouble lon1, gdouble lat2, gdouble lon2, GtkListStore **store, guint *num_poi);

poi_info *poi_get_by_id(guint id);

gboolean poi_update_or_add(poi_info *p);
gboolean poi_update(poi_info *p);
gboolean poi_add(poi_info *p);
gboolean poi_delete(guint id);

gboolean poi_search(poi_search_type pst, gdouble lat, gdouble lon, gchar *text, guint cat, GtkListStore **store);
poi_info *poi_find_nearest(gdouble lat, gdouble lon);

gboolean poi_get_list_near_unit(guint unitx, guint unity, guint range, GtkListStore **store, guint *num_poi);

poi_category *poi_category_new(void);
void poi_category_free(poi_category *c);

gboolean poi_category_get_icon_and_color(guint cat_id, gchar **icon, gchar **color);
gboolean poi_category_get(guint cat_id, poi_category **c);
gboolean poi_category_delete(guint id);
gboolean poi_category_update(guint cat_id, poi_category *c);
gboolean poi_category_toggle(guint cat_id, gboolean cat_enabled);

GdkPixbuf *poi_get_icon(const gchar *icon, gboolean big);
const gchar *poi_get_icon_from_type(node_type_t t);
void poi_icon_hash_clear(void);

GtkListStore *poi_list_store_new(void);
GtkListStore *poi_category_generate_store(gboolean do_count);

gint poi_get_icon_path(gchar *buffer, size_t size, gboolean big, const gchar *icon);

#endif
