/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
#include <config.h>

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stddef.h>
#include <locale.h>
#include <math.h>
#include <errno.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <fcntl.h>
#include <gconf/gconf-client.h>
#include <libxml/parser.h>
#include <libintl.h>
#include <locale.h>
#include <sqlite3.h>
#include <image-cache/image-cache.h>

#include "utils.h"
#include "poi.h"
#include "gps.h"
#include "map.h"
#include "mapper-types.h"
#include "settings.h"
#include "db.h"
#include "osm.h"
#include "osm-db.h"
#include "latlon.h"
#include "osm-sql-tables.h"

static sqlite3 *poidb;

/* POI Icon theme. "classic" or "square". Should be made into a configuration option */
static gchar *theme="square";
static gchar *theme_base=DATADIR "/icons/map-icons";

static ImageCache *poi_ic=NULL;

struct _poi_categories {
	node_type_t type;
	const gchar *name, *desc, *icon, *color;
};

/* The default POI categories */
static struct _poi_categories default_poi_categories[] = {
	{ NODE_AMENITY_PARKING, "Parking", "Parking place for vehicles."  , "vehicle/parking", "#2020ff" },
	{ NODE_AMENITY_FUEL, "Fuel", "Stations for purchasing fuel for vehicles."  , "vehicle/fuel", "#4040f0" },
	{ NODE_AMENITY_SPEEDCAM, "Speed Cam", "Automatic speed cameras."  , "vehicle/restrictions/speed_trap", "#ff0000" },
	{ NODE_AMENITY_HOSPITAL, "Hospital", "Get medical help"  , "health/hospital", "#ff4040" },
	{ NODE_AMENITY_PHARMACY, "Pharmacy", "Place to get drugs."  , "health/pharmacy", "#40f040" },
	{ NODE_AMENITY_POLICE, "Police", "Police station" , "public/police", "#8570ff" },
	{ NODE_TOURISM_HOTEL, "Hotel", "Places to stay temporarily or for the night." , "accommodation/hotel", "#ba20ba" },
	{ NODE_TOURISM_HOSTEL, "Hostel", "Places to stay temporarily or for the night. Cheap." , "accommodation/hostel", "#ba30ba" },
	{ NODE_TOURISM_MOTEL, "Motel", "Places to stay temporarily or for the night. Cheap." , "accommodation/motel", "#ba40ba" },
	{ NODE_AMENITY_ATM, "ATM", "Automatic Teller Machine/Cashpoint." , "money/atm", "#40a040" },
	{ NODE_AMENITY_BANK, "Bank", "Place to take care of your money." , "money/bank", "#50b550" },
	{ NODE_AMENITY_BUREAU_DE_CHANGE, "Bureau de Change", "Place to change currency" , "money/bank", "#20b580" },
	{ NODE_AMENITY_POST, "Post office", "Place to handle mail." , "service/post_office", "#ff6868" },
	{ NODE_AMENITY_POST_BOX, "Post box", "Send letters." , "service/post_box", "#ff6060" },
	{ NODE_TOURISM_INFO, "Tourism info", "A place for tourists to get information." , "misc/information", "#4af04a" },
	{ NODE_AMENITY_TAXI, "Taxi station", "Get a Taxi here." , "transport/taxi", "#50ffff" },
	{ NODE_AMENITY_EMBASSY, "Embassy", "An embassy of a foreign country" , "misc", "#80f080" },
	{ NODE_AMENITY_BROTHEL, "Brothel", "An establishment specifically dedicated to prostitution" , "misc", "#bf1010" },
	{ NODE_RAILWAY_STATION, "Railway station", "Transportation by train." , "transport/railway_station", "#fa7070" },
	{ NODE_RAILWAY_HALT, "Railway halt", "Transportation by train." , "transport/railway_small", "#f27777" },
	{ NODE_AMENITY_BUS_STATION, "Bus station", "Transportation by bus." , "transport/bus", "#7070f7" },
	{ NODE_HIGHWAY_BUS_STOP, "Bus stop", "" , "transport/bus_small", "#9090f9" },
	{ NODE_AMENITY_BOAT, "Harbour", "Transportation by boat." , "transport/ferry", "#9090ee" },
	{ NODE_AIRPORT_TERMINAL, "Airport", "Transportation by air." , "transport/airport", "#909099" },
	{ NODE_TOURISM_CAMP_SITE, "Camp site", "Place to go camping" , "accommodation/camping", "#90f080" },
	{ NODE_TOURISM_CARAVAN_SITE, "Caravan site", "" , "accommodation/camping/caravan", "#90ff80" },
	{ NODE_TOURISM_PICNIC_SITE, "Picnic", "Place to have a Picnic" , "recreation/picnic", "#a5f8e5" },
	{ NODE_AMENITY_FOOD, "Fast food", "Places to eat or drink." , "food/fastfood", "#e5960d" },
	{ NODE_AMENITY_RESTAURANT, "Restaurant", "Fancy places to eat or drink." , "food/restaurant", "#e5960d" },
	{ NODE_AMENITY_PUB, "Pub", "Place to drink." , "food/pub", "#f5960d" },
	{ NODE_AMENITY_NIGHTCLUB, "Disco, Club", "Place to drink, party and dance." , "leisure/nightclub", "#f59c1d" },
	{ NODE_AMENITY_CAFE, "Cafe", "Place to drink coffe or tee and eat." , "food/cafe", "#ff960d" },
	{ NODE_AMENITY_CINEMA, "Cinema", "Place to see movies" , "recreation/cinema", "#9090a0" },
	{ NODE_AMENITY_THEATRE, "Theatre", "Place to see people performing" , "recreation/theater", "#9595a5" },
	{ NODE_AMENITY_SHOP, "Shopping", "Places to shop or acquire services." , "shop", "#61ef1b" },
	{ NODE_AMENITY_SHOP_ADULT, "Adult shop", "Shops providing adult items, such as sex toys,pornography and fetish clothing" , "leisure/stripclub", "#ff0000" },
	{ NODE_AMENITY_SHOP_COMPUTER, "Computer shop", "Shops providing computer hardware, software and services" , "shop/computers", "#f10000" },
	{ NODE_AMENITY_POW, "Place of Worchip", "Religious building." , "religion/church", "#7c5b0b" },
	{ NODE_TOURISM_THEME_PARK, "Theme Park", "Place to have fun and ride for example rollercoasters." , "recreation/theme_park", "#8eef1b" },
	{ NODE_AMENITY_COLLEGE, "College Campus/Building", "" , "education/college", "#813fdc" },
	{ NODE_AMENITY_SCHOOL, "School", "" , "education/school", "#813fdc" },
	{ NODE_AMENITY_LIBRARY, "Library", "Place to read and borrow books and magazines" , "shop/rental/library", "#803090" },
	{ NODE_AMENITY_TOWNHALL, "Townhall", "" , "public", "#408090" },
	{ NODE_AMENITY_WC, "WC/Toilets", "" , "service/toilets", "#e1d62c" },
	{ NODE_AMENITY_RECYCLING, "Recycling", "" , "service/recycling", "#4ce04c" },
	{ NODE_AMENITY_TELEPHONE, "Telephone", "Public telephone" , "service/telephone", "#208060" },
	{ NODE_TOURISM_ATTRACTION, "Attraction", "Something interesting" , "misc", "#005000" },

	{ NODE_HISTORIC_MUSEUM, "Museum", "A place where objects of historical, artistic, or scientific interest are exhibited, preserved or studied." , "sightseeing/museum", "#202020" },
	{ NODE_HISTORIC_CASTLE, "Castle", "Historical building or group of building used for defense by military forces, whose main structures are walls and towers." , "sightseeing/castle", "#404040" },

	{ NODE_SPORT_CENTER, "Sport Center", "" , "sport/centre", "#101080" },
	{ NODE_SPORT_STADIUM, "Sport Stadium", "" , "sport/stadium", "#101080" },
	{ NODE_SPORT_SKIING, "Skiing", "" , "sport/skiing", "#5050A0" },
	{ NODE_SPORT_SWIMMING, "Swimming", "" , "sport/swimming", "#102080" },
	{ NODE_SPORT_FOOTBALL, "Football", "" , "sport/football", "#102080" },
	{ NODE_SPORT_SOCCER, "Soccer", "" , "sport/soccer", "#102080" },
	{ NODE_SPORT_GOLF, "Golf", "" , "sport/golf", "#102080" },
	{ NODE_SPORT_TENNIS, "Tennis", "" , "sport/tennis", "#101080" },
	{ NODE_SPORT_BOWLING, "Bowling", "" , "sport", "#101080" },
	{ NODE_SPORT_RUGBY, "Rugby", "" , "sport", "#101080" },
	{ NODE_SPORT_CLIMBING, "Climbing", "" , "sport/climbing", "#101080" },
	{ NODE_SPORT_CYCLING, "Cycling", "" , "sport/cycling", "#101080" },
	{ NODE_SPORT_MOTOR, "Motor sport", "" , "sport", "#101080" },
	{ NODE_SPORT_HOCKEY, "Hockey", "" , "sport/hockey", "#5050A0" },
	{ NODE_SPORT_SKATING, "Skating", "" , "sport", "#5050A0" },
	{ NODE_SPORT_SKATEBOARD, "Skateboard", "" , "sport", "#101080" },
	{ NODE_SPORT_HORSES, "Horses", "Horse riding or racing" , "sports/riding", "#101080" },
	{ NODE_SPORT_DOG, "Dog racing", "" , "sport", "#101080" },
	{ NODE_SPORT_BASKETBALL, "Basketball", "" , "sport", "#101080" },
	{ NODE_SPORT_BASEBALL, "Baseball", "" , "sport", "#101080" },
	{ NODE_SPORT_CANOE, "Canoe", "" , "sport/canoe", "#101080" },
	{ NODE_SPORT_CROQUET, "Croquet", "" , "sport/croquet", "#101080" },
	{ NODE_SPORT_CRICKET, "Cricket", "" , "sport/cricket", "#101080" },
	{ NODE_SPORT_SHOOTING, "Shooting", "Shooting range" , "sports", "#101080" },
	{ NODE_SPORT_PAINTBALL, "Paintball", "Run around and shoot people with paintballs" , "sports", "#101080" },
	{ NODE_SPORT_TABLE_TENNIS, "Table tennis", "" , "sport/table_tennis", "#101080" },
	{ NODE_SPORT_PELOTA, "Pelota", "" , "sport", "#101080" },
	{ NODE_SPORT_RACQUET, "Racquet", "" , "sport/racquetball", "#101080" },
	{ NODE_SPORT_BOWLS, "Lawn Bowls", "" , "sport", "#101080" },
	{ NODE_SPORT_ATHLETICS, "Athletics", "" , "sport", "#101080" },
	{ NODE_SPORT_OTHER, "Other Sports", "" , "sport/stadium", "#101077" },

	{ NODE_BUILDING, "Buildings", "Various buildings." , "misc/landmark/building", "#000000" },

	{ NODE_AMENITY_GENERIC, "Other", "Miscellaneous category for everything else." , "misc", "#002000" },
	{ NODE_POI_END, NULL, NULL, NULL, NULL }
};

typedef enum {
	PS_LAT=0,
	PS_LON,
	PS_ID,
	PS_LABEL,
	PS_DESC,
	PS_CAT_ID,
	PS_CAT_LABEL,
	PS_CAT_DESC,
	PS_CAT_ICON,
	PS_CAT_COLOR,
	PS_SOURCE,
	PS_PUBLIC,
	PS_URL,
	PS_POSTAL_CODE,
	PS_OSM_ID,
} poi_sql_column;

#define POI_BASE_SQL_FIELDS "p.lat, p.lon, p.poi_id, p.label, p.desc, p.cat_id, c.label, c.desc, c.icon, c.color, p.source, p.public, p.url, p.postal_code, p.osm_id"
#define POI_MINI_SQL_FIELDS "p.lat, p.lon, p.poi_id, p.label, p.desc, p.cat_id, c.label, c.desc"
#define POI_FAST_SQL_FIELDS "p.lat, p.lon, p.poi_id, p.cat_id"

static gboolean
poi_populate_categories(sqlite3 *db)
{
sqlite3_stmt *sql_cat;
guint i;

DB_PREP(db,"insert or replace into category (cat_id, label, desc, enabled, priority, icon, color) values (?, ?, ?, 1, ?, ?, ?)", sql_cat);

for (i=0; default_poi_categories[i].name; i++) {
	gint pri;

	/* Force specific priorities for some types */
	switch (default_poi_categories[i].type) {
		case NODE_AMENITY_PARKING:
			pri=2;
		break;
		case NODE_HIGHWAY_BUS_STOP:
			pri=10;
		break;
		case NODE_BUILDING:
			pri=11;
		break;
		default:
			pri=default_poi_categories[i].type/100;
	}

	sqlite3_bind_int(sql_cat, 1, default_poi_categories[i].type);
	sqlite3_bind_text(sql_cat, 2, default_poi_categories[i].name, -1, SQLITE_STATIC);
	sqlite3_bind_text(sql_cat, 3, default_poi_categories[i].desc, -1, SQLITE_STATIC);
	sqlite3_bind_int(sql_cat, 4, pri);
	sqlite3_bind_text(sql_cat, 5, default_poi_categories[i].icon, -1, SQLITE_STATIC);
	sqlite3_bind_text(sql_cat, 6, default_poi_categories[i].color, -1, SQLITE_STATIC);
	if (sqlite3_step(sql_cat)==SQLITE_OK)
		g_warning("Failed to update category: %d [%d:%s]", i, default_poi_categories[i].type, default_poi_categories[i].name);
	sqlite3_reset(sql_cat);
	sqlite3_clear_bindings(sql_cat);
}

DB_FINALIZE(db, sql_cat);

return TRUE;
}

/**
 * XXX: Make it faster!!
 */
gboolean poi_category_get_icon_and_color(guint cat_id, gchar **icon, gchar **color)
{
gint i;

for (i=0; default_poi_categories[i].name; i++) {
	if (default_poi_categories[i].type==cat_id) {
		*icon=default_poi_categories[i].icon;
		*color=default_poi_categories[i].color;
		return TRUE;
	}
}
return FALSE;
}

const gchar *
poi_get_icon_from_type(node_type_t t) 
{
guint i;

for (i=0; default_poi_categories[i].name; i++) {
	if (t==default_poi_categories[i].type)
		return default_poi_categories[i].icon;
}
return NULL;
}

static gboolean
poi_db_create(sqlite3 *db)
{
gchar **pszResult;
gint nRow, nColumn;

g_return_val_if_fail(db, FALSE);

if (SQLITE_OK==sqlite3_get_table(db, "select label from poi limit 1", &pszResult, &nRow, &nColumn, NULL)) {
	sqlite3_free_table(pszResult);
	poi_populate_categories(db);
	return TRUE;
}

if (!db_exec_sql(db, OSM_TABLE_POI) || !db_exec_sql(db, OSM_TABLE_POI_CATEGORY)) {
	g_warning("%s",_("Failed to open or create POI database table"));
	db_close(&db);
	return FALSE;
}

/* Make sure default categories exists */
poi_populate_categories(db);
return TRUE;
}

static gboolean
poi_db_prepare(sqlite3 *db)
{
/* Select POIs inside given minmax lat,lon */
DB_PREP(db,
		"select " 
		POI_BASE_SQL_FIELDS
		" from poi p, category c "
		" where p.lat between ? and ? "
		" and p.lon between ? and ? "
		" and c.enabled=1 and p.cat_id=c.cat_id order by c.priority limit 2000",
		poisql.select_poi);

DB_PREP(db,
		"select " 
		POI_FAST_SQL_FIELDS
		" from poi p, category c "
		" where p.lat between ? and ? "
		" and p.lon between ? and ? "
		" and c.enabled=1 and p.cat_id=c.cat_id "
		" order by c.priority,(($LAT-p.lat)*($LAT-p.lat)+($LON-p.lon)*($LON-p.lon)) "
		" limit 30000",
		poisql.select_poi_fast);

/* Get POI with given ID */
DB_PREP(db,
		"select "
		POI_BASE_SQL_FIELDS
		" from poi p, category c "
		" where p.poi_id = ? "
		" and p.cat_id=c.cat_id",
		poisql.select_poi_by_id);

/* Search POIs by label and any category */
DB_PREP(db,
		"select "
		POI_BASE_SQL_FIELDS
		" from poi p, category c "
		" where p.lat between ? and ? "
		" and p.lon between ? and ? "
		" and c.enabled=1 and p.cat_id = c.cat_id and (p.label like $NAME or p.postal_code like $NAME) order by p.label, c.label",
		poisql.select_poi_search);

/* Search POI by label and category */
DB_PREP(db,
		"select "
		POI_BASE_SQL_FIELDS
		" from poi p, category c "
		" where p.lat between ? and ? "
		" and p.lon between ? and ? "
		" and c.enabled=1 and p.cat_id = c.cat_id and (p.label like $NAME or p.postal_code like $NAME) and c.cat_id = ? order by p.label",
		poisql.select_poi_search_cat);

/* Search POIs by category */
DB_PREP(db,
		"select "
		POI_BASE_SQL_FIELDS
		" from poi p, category c "
		" where p.lat between ? and ? "
		" and p.lon between ? and ? "
		" and c.enabled=1 and p.cat_id=c.cat_id and c.cat_id=? order by p.label",
		poisql.select_poi_by_cat);

/* Select any nearest POI */
DB_PREP(db,
		"select "
		POI_MINI_SQL_FIELDS
		" from poi p, category c "
		" where c.enabled = 1 and p.cat_id = c.cat_id "
		" and p.lat between $LAT-0.10 and $LAT+0.10 "
		" and p.lon between $LON-0.10 and $LAT+0.10 "
		" order by (($LAT-p.lat) * ($LAT-p.lat)+($LON-p.lon)*($LON-p.lon)) "
		" limit 1",
		poisql.select_nearest_poi);

/* Insert POI */
DB_PREP(db,  "insert into poi (lat, lon, label, desc, url, postal_code, cat_id, addtime, public, source)"
						" values (?, ?, ?, ?, ?, ?, ?, ?, 1, ?)", poisql.insert_poi);
/* update poi */
DB_PREP(db, "update poi set label=?, desc=?, cat_id=? where poi_id=?", poisql.update_poi);
/* delete from poi */
DB_PREP(db, "delete from poi where poi_id=?", poisql.delete_poi);
/* delete from poi by cat_id */
DB_PREP(db, "delete from poi where cat_id=?", poisql.delete_poi_by_catid);

/* select from category */
DB_PREP(db, "select c.label, c.desc, c.enabled from category c where c.cat_id = ?", poisql.select_cat);
/* insert into category */
DB_PREP(db, "insert into category (label, desc, enabled) values (?, ?, ?)", poisql.insert_cat);
/* update category */
DB_PREP(db, "update category set label = ?, desc = ?, enabled = ? where cat_id = ?", poisql.update_cat);
/* delete from category */
DB_PREP(db,"delete from category where cat_id = ?", poisql.delete_cat);

/* enable category */
DB_PREP(db,
		"update category set enabled = ?"
		" where cat_id = ?", poisql.toggle_cat);
/* select all category */
DB_PREP(db,
		"select c.cat_id, c.label, c.desc, c.enabled, c.icon, c.color,"
		" count(p.poi_id)"
		" from category c"
		" left outer join poi p on c.cat_id = p.cat_id"
		" group by c.cat_id, c.label, c.desc, c.enabled "
		" order by c.priority,c.label", poisql.selall_cat);

/* select all categories, fast */
DB_PREP(db,
		"select c.cat_id, c.label, c.desc, c.enabled, c.icon, c.color, 0"
		" from category c order by c.priority,c.label", poisql.selall_cat_fast);

return TRUE;
}

void
poi_icon_hash_clear(void)
{
image_cache_clear(poi_ic);
}

void
poi_deinit(void)
{
if (!poi_ic)
	return;
g_object_unref(poi_ic);
poi_ic=NULL;
}

void
poi_deinit_db(sqlite3 *db)
{
g_return_if_fail(db);
DB_FINALIZE(db, poisql.selall_cat);
DB_FINALIZE(db, poisql.selall_cat_fast);
DB_FINALIZE(db, poisql.toggle_cat);
DB_FINALIZE(db, poisql.delete_cat);
DB_FINALIZE(db, poisql.update_cat);
DB_FINALIZE(db, poisql.insert_cat);
DB_FINALIZE(db, poisql.select_cat);
DB_FINALIZE(db, poisql.insert_poi);
DB_FINALIZE(db, poisql.update_poi);
DB_FINALIZE(db, poisql.delete_poi);
DB_FINALIZE(db, poisql.delete_poi_by_catid);
DB_FINALIZE(db, poisql.select_nearest_poi);
DB_FINALIZE(db, poisql.select_poi);
DB_FINALIZE(db, poisql.select_poi_fast);
DB_FINALIZE(db, poisql.select_poi_search);
DB_FINALIZE(db, poisql.select_poi_search_cat);
}

gboolean
poi_init(void)
{
if (!poi_ic)
	poi_ic=image_cache_new(256, TRUE);
return TRUE;
}

gboolean
poi_init_db(sqlite3 **db)
{
if (!db || !*db)
	return FALSE;

poidb=*db;
if (poi_db_create(poidb)==FALSE)
	return FALSE;
if (poi_db_prepare(poidb)==FALSE)
	return FALSE;
return TRUE;
}

poi_info *
poi_new(void)
{
poi_info *p;

p=g_slice_new0(poi_info);
p->source=POI_SOURCE_USER;
return p;
}

void
poi_free(poi_info *p)
{
if (p->label)
	g_free(p->label);
if (p->desc)
	g_free(p->desc);
if (p->url)
	g_free(p->url);
if (p->postal_code)
	g_free(p->postal_code);
if (p->cat_label)
	g_free(p->cat_label);
if (p->cat_desc)
	g_free(p->cat_desc);
g_slice_free(poi_info, p);
}

GtkListStore *
poi_list_store_new(void) {
return gtk_list_store_new(ITEM_NUM_COLUMNS, 
			G_TYPE_INT,	/* POI ID */
			G_TYPE_INT,	/* Category ID */
			G_TYPE_DOUBLE,	/* Latitude */
			G_TYPE_DOUBLE,	/* Longitude */
			G_TYPE_DOUBLE,	/* Dist */
			G_TYPE_STRING,	/* Lat/Lon */
			G_TYPE_STRING,	/* Label */
			G_TYPE_STRING,	/* Desc. */
			G_TYPE_STRING,	/* Category Label */
			G_TYPE_STRING,	/* Icon */
			G_TYPE_STRING);	/* Color */
}

/*************************************
 * POI Category functions
 *
 */

poi_category *
poi_category_new(void)
{
return g_slice_new0(poi_category);
}

void
poi_category_free(poi_category *c)
{
if (c->label)
	g_free(c->label);
if (c->desc)
	g_free(c->desc);
g_slice_free(poi_category, c);
}

gboolean
poi_category_toggle(guint cat_id, gboolean cat_enabled) 
{
g_return_val_if_fail(poisql.toggle_cat, FALSE);

if (SQLITE_OK != sqlite3_bind_int(poisql.toggle_cat, 1, cat_enabled) ||
    SQLITE_OK != sqlite3_bind_int(poisql.toggle_cat, 2, cat_id) ||
    SQLITE_DONE != sqlite3_step(poisql.toggle_cat)) {
		return FALSE;
	}
return TRUE;
}

gboolean
poi_category_get(guint cat_id, poi_category **c)
{
poi_category *cc;

g_return_val_if_fail(poisql.select_cat, FALSE);
if (SQLITE_OK != sqlite3_bind_int(poisql.select_cat, 1, cat_id) || SQLITE_ROW != sqlite3_step(poisql.select_cat)) {
	sqlite3_reset(poisql.select_cat);
	return FALSE;
}

cc=poi_category_new();
cc->id=cat_id;
cc->label = g_strdup(sqlite3_column_text(poisql.select_cat, 0));
cc->desc = g_strdup(sqlite3_column_text(poisql.select_cat, 1));
cc->enabled = sqlite3_column_int(poisql.select_cat, 2);

sqlite3_reset(poisql.select_cat);
sqlite3_clear_bindings(poisql.select_cat);
*c=cc;
return TRUE;
}

gboolean
poi_category_update(guint cat_id, poi_category *c)
{
gboolean results=TRUE;

if (!_db)
	return FALSE;

g_return_val_if_fail(poisql.update_cat, FALSE);
g_return_val_if_fail(poisql.insert_cat, FALSE);

if (cat_id > 0) {
/* edit category */
		if (SQLITE_OK != sqlite3_bind_text(poisql.update_cat, 1, c->label, -1, SQLITE_STATIC)
		    || SQLITE_OK != sqlite3_bind_text(poisql.update_cat, 2, c->desc, -1, SQLITE_STATIC)
		    || SQLITE_OK != sqlite3_bind_int(poisql.update_cat, 3, c->enabled)
		    || SQLITE_OK != sqlite3_bind_int(poisql.update_cat, 4, c->id)
		    || SQLITE_DONE != sqlite3_step(poisql.update_cat)) {
			results = FALSE;
		}
		sqlite3_reset(poisql.update_cat);
		sqlite3_clear_bindings(poisql.update_cat);
	} else {
		/* add category */
		if (SQLITE_OK != sqlite3_bind_text(poisql.insert_cat, 1, c->label, -1, SQLITE_STATIC)
		    || SQLITE_OK != sqlite3_bind_text(poisql.insert_cat, 2, c->desc, -1, SQLITE_STATIC)
		    || SQLITE_OK != sqlite3_bind_int(poisql.insert_cat, 3, c->enabled)
		    || SQLITE_DONE != sqlite3_step(poisql.insert_cat)) {
			results = FALSE;
		}
		sqlite3_reset(poisql.insert_cat);
		sqlite3_clear_bindings(poisql.insert_cat);
	}
return results;
}

gboolean 
poi_category_delete(guint id)
{
if (!poidb)
	return FALSE;

g_return_val_if_fail(poisql.delete_poi_by_catid, FALSE);
g_return_val_if_fail(poisql.delete_cat, FALSE);

if (SQLITE_OK != sqlite3_bind_int(poisql.delete_poi_by_catid, 1, id) || SQLITE_DONE != sqlite3_step(poisql.delete_poi_by_catid)) {
	sqlite3_reset(poisql.delete_poi_by_catid);
	return FALSE;
}
sqlite3_reset(poisql.delete_poi_by_catid);
sqlite3_clear_bindings(poisql.delete_poi_by_catid);

if (SQLITE_OK != sqlite3_bind_int(poisql.delete_cat, 1, id) || SQLITE_DONE != sqlite3_step(poisql.delete_cat)) {
	sqlite3_reset(poisql.delete_cat);
	return FALSE;
}
sqlite3_reset(poisql.delete_cat);
sqlite3_clear_bindings(poisql.delete_cat);
return TRUE;
}


gboolean 
poi_delete(guint id)
{
g_return_val_if_fail(poidb, FALSE);
g_return_val_if_fail(poisql.delete_poi, FALSE);

if (SQLITE_OK != sqlite3_bind_int(poisql.delete_poi, 1, id) || SQLITE_DONE != sqlite3_step(poisql.delete_poi)) {
	sqlite3_reset(poisql.delete_poi);
	return FALSE;
}
sqlite3_reset(poisql.delete_poi);
sqlite3_clear_bindings(poisql.delete_poi);

return TRUE;
}

gboolean
poi_search(poi_search_type pst, gdouble lat, gdouble lon, gchar *text, guint cat, GtkListStore **store)
{
GtkTreeIter iter;
sqlite3_stmt *sql=NULL;
gchar *ltext=NULL;
guint rows=0;
gchar tmp1[16], tmp2[16];
gdouble range=1.0;

g_return_val_if_fail(poidb, FALSE);
g_return_val_if_fail(poisql.select_poi, FALSE);
g_return_val_if_fail(poisql.select_poi_search, FALSE);
g_return_val_if_fail(poisql.select_poi_search_cat, FALSE);
g_return_val_if_fail(poisql.select_poi_by_cat, FALSE);

switch (pst) {
	case POI_SEARCH_NEAR:
		range=0.5;
		sql=poisql.select_poi;
	break;
	case POI_SEARCH_TEXT:
		ltext=g_strdup_printf("%s%%", text);
		
		if (SQLITE_OK != sqlite3_bind_text(poisql.select_poi_search, 5, ltext, -1, SQLITE_TRANSIENT)) {
				g_printerr("Failed to bind values for poisql.select_poi_search\n");
				sqlite3_clear_bindings(poisql.select_poi_search);
				g_free(ltext);
				return FALSE;
		}
		g_free(ltext);
		sql=poisql.select_poi_search;
	break;
	case POI_SEARCH_TEXT_CAT:
		ltext=g_strdup_printf("%s%%", text);

		if (SQLITE_OK != sqlite3_bind_int(poisql.select_poi_search_cat, 6, cat) ||
			SQLITE_OK != sqlite3_bind_text(poisql.select_poi_search_cat, 5, ltext, -1, SQLITE_TRANSIENT)) {
				g_printerr("Failed to bind values for poisql.select_poi_search_cat\n");
				sqlite3_clear_bindings(poisql.select_poi_search_cat);
				g_free(ltext);
				return FALSE;
		}
		g_free(ltext);
		sql=poisql.select_poi_search_cat;
	break;
	case POI_SEARCH_CAT:
		if (SQLITE_OK != sqlite3_bind_int(poisql.select_poi_by_cat, 5, cat)) {
				g_printerr("Failed to bind values for poisql.select_poi_by_cat\n");
				sqlite3_clear_bindings(poisql.select_poi_by_cat);
				return FALSE;
		}
		sql=poisql.select_poi_by_cat;
	break;
	default:
		g_assert_not_reached();
		return FALSE;
	break;
}

/* XXX: Use common bind for common variables */
if (SQLITE_OK != sqlite3_bind_double(sql, 1, lat-range) ||
    SQLITE_OK != sqlite3_bind_double(sql, 2, lat+range) ||
    SQLITE_OK != sqlite3_bind_double(sql, 3, lon-range) ||
    SQLITE_OK != sqlite3_bind_double(sql, 4, lon+range)) {
	g_printerr("Failed to bind common variables for POI search\n");
	sqlite3_clear_bindings(sql);
	return FALSE;
}

if (*store==NULL) {
	*store=poi_list_store_new();
} else {
	gtk_list_store_clear(*store);
}

while (SQLITE_ROW == sqlite3_step(sql)) {
	gdouble rlat, rlon, dist;
	gchar *sl;

	rlat=sqlite3_column_double(sql, 0);
	rlon=sqlite3_column_double(sql, 1);
	lat_format(_degformat, rlat, tmp1);
	lon_format(_degformat, rlon, tmp2);
	dist=calculate_distance(lat, lon, rlat, rlon) * UNITS_CONVERT[_units];

	sl=g_strdup_printf("%s, %s", tmp1, tmp2);

	gtk_list_store_append(*store, &iter);
	gtk_list_store_set(*store, &iter,
		ITEM_ID, sqlite3_column_int(sql, 2),
		ITEM_CATID, sqlite3_column_int(sql, 5),
		ITEM_LAT, rlat, 
		ITEM_LON, rlon, 
		ITEM_DIST, dist, 
		ITEM_LATLON, sl,
		ITEM_LABEL, sqlite3_column_text(sql, 3),
		ITEM_DESC, sqlite3_column_text(sql, 4),
		ITEM_CATLAB, sqlite3_column_text(sql, 6),
		-1);
	g_free(sl);
	rows++;
}

sqlite3_reset(sql);
sqlite3_clear_bindings(sql);

return TRUE;
}

gboolean
poi_get_list_inside(gdouble lat1, gdouble lon1, gdouble lat2, gdouble lon2, GtkListStore **store, guint *num_poi)
{
static gboolean active=FALSE;
GtkTreeIter iter;
gdouble dist=-1, lat, lon;

g_return_val_if_fail(_db, FALSE);
g_return_val_if_fail(poisql.select_poi, FALSE);

if (active)
	return FALSE;

*num_poi=0;

lat1=CLAMP(lat1, -90.0, 90.0);
lat2=CLAMP(lat2, -90.0, 90.0);
lon1=CLAMP(lon1, -180.0, 180.0);
lon2=CLAMP(lon2, -180.0, 180.0);

unit2latlon(_center.unitx, _center.unity, lat, lon);

active=TRUE;
sqlite3_reset(poisql.select_poi);

if (SQLITE_OK != sqlite3_bind_double(poisql.select_poi_fast, 1, lat1) ||
    SQLITE_OK != sqlite3_bind_double(poisql.select_poi_fast, 2, lat2) ||
    SQLITE_OK != sqlite3_bind_double(poisql.select_poi_fast, 3, lon1) ||
    SQLITE_OK != sqlite3_bind_double(poisql.select_poi_fast, 4, lon2) ||
    SQLITE_OK != sqlite3_bind_double(poisql.select_poi_fast, 5, lat) ||
    SQLITE_OK != sqlite3_bind_double(poisql.select_poi_fast, 6, lon)) {
	g_warning("Failed to bind values for poisql.select_poi_fast");
	sqlite3_clear_bindings(poisql.select_poi_fast);
	active=FALSE;
	return FALSE;
}

if (*store==NULL) {
	*store=poi_list_store_new();
} else {
	gtk_list_store_clear(*store);
}

while (sqlite3_step(poisql.select_poi_fast)==SQLITE_ROW) {
	gtk_list_store_append(*store, &iter);
	gtk_list_store_set(*store, &iter,
		ITEM_ID, sqlite3_column_int(poisql.select_poi_fast, 2),
		ITEM_CATID, sqlite3_column_int(poisql.select_poi_fast, 3),
		ITEM_LAT, sqlite3_column_double(poisql.select_poi_fast, 0),
		ITEM_LON, sqlite3_column_double(poisql.select_poi_fast, 1),
		ITEM_DIST, dist, 
		-1);
	(*num_poi)++;
}
sqlite3_reset(poisql.select_poi_fast);
sqlite3_clear_bindings(poisql.select_poi_fast);
active=FALSE;
return TRUE;
}

gboolean
poi_get_list_near_unit(guint unitx, guint unity, guint range, GtkListStore **store, guint *num_poi)
{
gdouble lat1, lon1, lat2, lon2;
guint x, y;

x=unitx-pixel2unit(3*range);
y=unity+pixel2unit(3*range);
unit2latlon(x, y, lat1, lon1);

x=unitx+pixel2unit(3*range);
y=unity-pixel2unit(3*range);
unit2latlon(x, y, lat2, lon2); 

return poi_get_list_inside(lat1, lon1, lat2, lon2, store, num_poi);
}

poi_info *
poi_get_by_id(guint id)
{
poi_info *p=NULL;

g_return_val_if_fail(poisql.select_poi_by_id, FALSE);
g_return_val_if_fail(id>0, FALSE);

if (SQLITE_OK!=sqlite3_bind_int(poisql.select_poi_by_id, 1, id))
	return NULL;

if (SQLITE_ROW==sqlite3_step(poisql.select_poi_by_id)) {
	p=poi_new();
	p->poi_id=sqlite3_column_int(poisql.select_poi_by_id, PS_ID);
	p->lat=sqlite3_column_double(poisql.select_poi_by_id, PS_LAT);
	p->lon=sqlite3_column_double(poisql.select_poi_by_id, PS_LON);
	p->source=sqlite3_column_int(poisql.select_poi_by_id, PS_SOURCE);
	p->public=sqlite3_column_int(poisql.select_poi_by_id, PS_PUBLIC)==1 ? TRUE : FALSE;
	p->label=g_strdup(sqlite3_column_text(poisql.select_poi_by_id, PS_LABEL));
	p->desc=g_strdup(sqlite3_column_text(poisql.select_poi_by_id, PS_DESC));
	p->url=g_strdup(sqlite3_column_text(poisql.select_poi_by_id, PS_URL));
	p->postal_code=g_strdup(sqlite3_column_text(poisql.select_poi_by_id, PS_POSTAL_CODE));

	p->osm_id=sqlite3_column_int(poisql.select_poi_by_id, PS_OSM_ID);

	p->cat_id=sqlite3_column_int(poisql.select_poi_by_id, PS_CAT_ID);
	p->cat_label=g_strdup(sqlite3_column_text(poisql.select_poi_by_id, PS_CAT_LABEL));
	p->cat_desc=g_strdup(sqlite3_column_text(poisql.select_poi_by_id, PS_CAT_DESC));
}

sqlite3_reset(poisql.select_poi_by_id);
sqlite3_clear_bindings(poisql.select_poi_by_id);

return p;
}

gboolean
poi_update_or_add(poi_info *p)
{
g_return_val_if_fail(p, FALSE);
return (p->poi_id==0) ? poi_add(p) : poi_update(p);
}

gboolean
poi_update(poi_info *p)
{
if (!poidb)
	return FALSE;

g_return_val_if_fail(p, FALSE);
g_return_val_if_fail(poisql.update_poi, FALSE);
g_return_val_if_fail(p->poi_id!=0, FALSE);

if (SQLITE_OK != sqlite3_bind_text(poisql.update_poi, 1, p->label, -1, SQLITE_STATIC)
   || SQLITE_OK != sqlite3_bind_text(poisql.update_poi, 2, p->desc, -1, SQLITE_STATIC)
   || SQLITE_OK != sqlite3_bind_int(poisql.update_poi, 3, p->cat_id)
   || SQLITE_OK != sqlite3_bind_int(poisql.update_poi, 4, p->poi_id)
   || SQLITE_DONE != sqlite3_step(poisql.update_poi)) {
		DB_ERROR(poidb);
		sqlite3_reset(poisql.insert_poi);
		sqlite3_clear_bindings(poisql.insert_poi);
		return FALSE;
	}
sqlite3_reset(poisql.update_poi);
sqlite3_clear_bindings(poisql.update_poi);
return TRUE;
}

/* XXX: Add url and postal_code */
gboolean
poi_add(poi_info *p)
{
time_t t;

if (!poidb)
	return FALSE;

g_return_val_if_fail(p, FALSE);
g_return_val_if_fail(poisql.insert_poi, FALSE);
g_return_val_if_fail(p->poi_id==0, FALSE);

t=time(NULL);

if (SQLITE_OK != sqlite3_bind_double(poisql.insert_poi, 1, p->lat)
    || SQLITE_OK != sqlite3_bind_double(poisql.insert_poi, 2, p->lon)
    || SQLITE_OK != sqlite3_bind_text(poisql.insert_poi, 3, p->label, -1, SQLITE_STATIC)
    || SQLITE_OK != sqlite3_bind_text(poisql.insert_poi, 4, p->desc, -1, SQLITE_STATIC)
    || SQLITE_OK != sqlite3_bind_text(poisql.insert_poi, 5, p->url, -1, SQLITE_STATIC)
    || SQLITE_OK != sqlite3_bind_text(poisql.insert_poi, 6, p->postal_code, -1, SQLITE_STATIC)
    || SQLITE_OK != sqlite3_bind_int(poisql.insert_poi, 7, p->cat_id) 
    || SQLITE_OK != sqlite3_bind_int(poisql.insert_poi, 8, t) 
    || SQLITE_OK != sqlite3_bind_int(poisql.insert_poi, 9, p->source)
	|| SQLITE_DONE != sqlite3_step(poisql.insert_poi)) {
		DB_ERROR(poidb);
		sqlite3_reset(poisql.insert_poi);
		sqlite3_clear_bindings(poisql.insert_poi);
		return FALSE;
	}
sqlite3_reset(poisql.insert_poi);
sqlite3_clear_bindings(poisql.insert_poi);
return TRUE;
}

poi_info *
poi_find_nearest(gdouble lat, gdouble lon) 
{
poi_info *p;

if (!poidb)
	return FALSE;

g_return_val_if_fail(poisql.select_nearest_poi, FALSE);

sqlite3_reset(poisql.select_nearest_poi);
sqlite3_clear_bindings(poisql.select_nearest_poi);

if (SQLITE_OK == sqlite3_bind_double(poisql.select_nearest_poi, 1, lat)
	&& SQLITE_OK == sqlite3_bind_double(poisql.select_nearest_poi, 2, lon)
	&& SQLITE_ROW == sqlite3_step(poisql.select_nearest_poi)) {

	p=poi_new();
	p->lat=sqlite3_column_double(poisql.select_nearest_poi, 0);
	p->lon=sqlite3_column_double(poisql.select_nearest_poi, 1);
	p->poi_id=sqlite3_column_double(poisql.select_nearest_poi, 2);
	p->label=g_strdup(sqlite3_column_text(poisql.select_nearest_poi, 3));
	p->desc=g_strdup(sqlite3_column_text(poisql.select_nearest_poi, 4));
	p->cat_id=sqlite3_column_double(poisql.select_nearest_poi, 5);
	p->cat_desc=g_strdup(sqlite3_column_text(poisql.select_nearest_poi, 6));
	sqlite3_reset(poisql.select_nearest_poi);
	sqlite3_clear_bindings(poisql.select_nearest_poi);
	return p;
}
return NULL;
}

gint
poi_get_icon_path(gchar *buffer, size_t size, gboolean big, const gchar *icon)
{
return g_snprintf(buffer, size, "%s/%s.%s/%s.png", theme_base, theme, (big==TRUE) ? "big" : "small", icon);
}

GdkPixbuf *
poi_get_icon(const gchar *icon, gboolean big)
{
GdkPixbuf *s;
gchar buffer[128];
gchar key[32];

g_return_val_if_fail(poi_ic, NULL);

if (icon==NULL)
	return NULL;

if (strlen(icon)==0)
	return NULL;

g_snprintf(key, sizeof(key), "%s:%s:%s", theme, (big==TRUE) ? "b" : "s", icon);

s=image_cache_get(poi_ic, key);
if (s)
	return s;

poi_get_icon_path(buffer, sizeof(buffer), big, icon);
s=gdk_pixbuf_new_from_file(buffer, NULL);
if (s)
	image_cache_put(poi_ic, g_strdup(key), s, NULL);
return s;
}

GtkListStore *
poi_category_generate_store(gboolean do_counts)
{
GtkTreeIter iter;
GtkListStore *store;
sqlite3_stmt *sc;

g_return_val_if_fail(poidb, NULL);
g_return_val_if_fail(poisql.selall_cat, FALSE);
g_return_val_if_fail(poisql.selall_cat_fast, FALSE);

sc=(do_counts) ? poisql.selall_cat : poisql.selall_cat_fast;

store=gtk_list_store_new(CAT_NUM_COLUMNS, /* pixbuf */
				G_TYPE_UINT,
				G_TYPE_BOOLEAN,
				G_TYPE_STRING, 
				G_TYPE_STRING, 
				G_TYPE_UINT,
				GDK_TYPE_PIXBUF);

while (SQLITE_ROW == sqlite3_step(sc)) {
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter,
		CAT_ID, sqlite3_column_int(sc, 0), 
		CAT_ENABLED, sqlite3_column_int(sc, 3),
		CAT_LABEL, sqlite3_column_text(sc, 1),
		CAT_DESC, sqlite3_column_text(sc, 2),
		CAT_POI_CNT, sqlite3_column_int(sc, 6), 
		CAT_ICON, poi_get_icon(sqlite3_column_text(sc, 4),TRUE),
		-1);
}

sqlite3_reset(sc);
sqlite3_clear_bindings(sc);

return store;
}
