/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
#include <config.h>

#include <gtk/gtk.h>

#include "mapper-types.h"
#include "path.h"
#include "utils.h"
#include "map.h"
#include "track.h"
#include "route.h"
#include "settings.h"
#include "latlon.h"
#include "gpx.h"

Path *
path_new(void)
{
Path *p;

p=g_slice_new0(Path);
MACRO_PATH_INIT(*p);
p->store=gtk_list_store_new(ROUTE_NUM_COLUMNS, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_DOUBLE, G_TYPE_DOUBLE);
return p;
}

void
path_clear(Path *p)
{
g_return_if_fail(p);
MACRO_PATH_FREE(*p);
MACRO_PATH_INIT(*p);
p->points=0;
p->length=0;
p->avgspeed=0;
p->maxspeed=0;
gtk_list_store_clear(p->store);
}

void
path_free(Path *p)
{
g_return_if_fail(p);
MACRO_PATH_FREE(*p);
gtk_list_store_clear(p->store);
g_object_unref(p->store);
p->store=NULL;
g_slice_free(Path, p);
}

void 
path_resize(Path *path, guint size)
{
if (path->head + size != path->cap) {
	Point *old_head = path->head;
	WayPoint *curr;
	path->head = g_renew(Point, old_head, size);
	path->cap = path->head + size;
	if (path->head != old_head) {
		path->tail = path->head + (path->tail - old_head);

		/* Adjust all of the waypoints. */
		for (curr = path->whead - 1; curr++ != path->wtail;)
			curr->point = path->head + (curr->point - old_head);
	}
}
}

void 
path_wresize(Path *path, guint wsize)
{
if (path->whead + wsize != path->wcap) {
	WayPoint *old_whead = path->whead;
	path->whead = g_renew(WayPoint, old_whead, wsize);
	path->wtail = path->whead + (path->wtail - old_whead);
	path->wcap = path->whead + wsize;
}
}

gboolean 
path_insert_break(Path *path)
{
g_return_val_if_fail(path, FALSE);

if (!path->tail)
	return FALSE;

if (path->tail->unity) {
	/* To mark a "waypoint" in a track, we'll add a (0, 0) point and then
	 * another instance of the most recent track point. */
	MACRO_PATH_INCREMENT_TAIL(*path);
	*path->tail=_point_null;
	MACRO_PATH_INCREMENT_TAIL(*path);
	*path->tail=path->tail[-2];
	return TRUE;
}
return FALSE;
}

gboolean
path_load(Path *path, const gchar *dir, const gchar *file)
{
gchar *pfile;
gchar *bytes;
gint size;

pfile = gnome_vfs_uri_make_full_from_relative(dir, file);
if (gnome_vfs_read_entire_file(pfile, &size, &bytes)==GNOME_VFS_OK)
	gpx_parse(path, bytes, size, GPX_PATH_NEW);
g_free(pfile);
return TRUE;
}

gboolean 
path_save(Path *path, const gchar *dir, const gchar *file)
{
GnomeVFSHandle *handle;
gchar *tfile;

tfile=gnome_vfs_uri_make_full_from_relative(dir, file);
if (gnome_vfs_create(&handle, tfile, GNOME_VFS_OPEN_WRITE, FALSE, 0600)==GNOME_VFS_OK) {
	gpx_write(path, handle);
	gnome_vfs_close(handle);
}
g_free(tfile);
return TRUE;
}

/**
 * Add a text description at current point
 * 
 */
gboolean
path_insert_mark_text(Path *p, gchar *text)
{
Point *pp;

g_return_val_if_fail(p, FALSE);
g_return_val_if_fail(text, FALSE);

MACRO_PATH_INCREMENT_WTAIL(*p);
if (p->head==p->tail)
	return FALSE;

/* Find last non-zero point. */
for (pp=p->tail; !pp->unity && !pp->unitx && pp>=p->head; pp--) {
}

p->wtail->point=pp;
p->wtail->desc=text;
path_update_store(p);
return TRUE;
}

gdouble
path_calculate_distance_from(Path *path, Point *point, gdouble lat, gdouble lon)
{
gdouble lat1, lon1, lat2, lon2;
gdouble sum=0.0;
Point *curr;

lat1=lat;
lon1=lon;

/* Skip _track->tail because that should be _gps. */
for (curr = path->tail; curr > point; --curr) {
	if (curr->unity) {
		unit2latlon(curr->unitx, curr->unity, lat2, lon2);
		sum+=calculate_distance(lat1, lon1, lat2, lon2);
		lat1=lat2;
		lon1=lon2;
	}
}
return sum;
}

gboolean
path_update_store(Path *path)
{
WayPoint *wcurr;
GtkTreeIter iter;
gchar buffer1[80];
gchar buffer2[32];
gchar tmp1[16], tmp2[16];
gdouble lat1=0, lon1=0, lat2=0, lon2=0;
gdouble sum=0.0;
gboolean init=FALSE;

g_return_val_if_fail(path, FALSE);
g_return_val_if_fail(path->store, FALSE);

gtk_list_store_clear(path->store);

wcurr=path->whead;

if (!wcurr)
	return FALSE;

do {
	if (!wcurr)
		break;

	if (!wcurr->point) {
		g_debug("No point for waypoint (%s) skipping", wcurr->desc);
		wcurr++;
		continue;
	}
	if (!init) {
		init=TRUE;
		unit2latlon(wcurr->point->unitx, wcurr->point->unity, lat1, lon1);
	}

	unit2latlon(wcurr->point->unitx, wcurr->point->unity, lat2, lon2);
	lat_format(_degformat, lat2, tmp1);
	lon_format(_degformat, lon2, tmp2);

	g_snprintf(buffer1, sizeof(buffer1), "%s,%s", tmp1, tmp2);
	sum += calculate_distance(lat1, lon1, lat2, lon2);
	g_snprintf(buffer2, sizeof(buffer2), "%.02f %s", sum * UNITS_CONVERT[_units], UNITS_TEXT[_units]);

	gtk_list_store_append(path->store, &iter);
	gtk_list_store_set(path->store, &iter,
		ROUTE_LATLON, buffer1,
		ROUTE_DISTANCE, buffer2,
		ROUTE_WAYPOINT, wcurr->desc,
		ROUTE_LAT, lat2,
		ROUTE_LON, lon2,
		-1);

	lat1=lat2;
	lon1=lon2;

	wcurr++;
} while (wcurr<=path->wtail);

return TRUE;
}

void
position_set(Position *pos, gboolean valid, gdouble lat, gdouble lon)
{
pos->valid=valid;
if (valid) {
	pos->lat=lat;
	pos->lon=lon;
} else {
	pos->lat=NAN;
	pos->lon=NAN;
}
}

void
position_update(Position *pos, GpsData *data)
{
position_set(pos, data->fix!=FIX_NOFIX, data->lat, data->lon);
}

void
position_swap(Position *p1, Position *p2)
{
Position tmp;

tmp.valid=p1->valid;
tmp.lat=p1->lat;
tmp.lon=p1->lon;

p1->valid=p2->valid;
p1->lat=p2->lat;
p1->lon=p2->lon;

p2->valid=tmp.valid;
p2->lat=tmp.lat;
p2->lon=tmp.lon;
}
