/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

/*
 * Routines to read OSM planet XML file and store it in a sqlite3 database.
 * Reads in all nodes (if used, skips nodes outside bounding box)
 * Special POI nodes are stored in POI table.
 * Place POI nodes are stored in place table.
 *
 * Ways are read in and their data (name, type, etc) are stored 
 * in way, way_name and way_ref tables.
 * 
 * Nodes used by they ways are stored in way_n2n table.
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <math.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <sqlite3.h>
#include <expat.h>
#include <bzlib.h>

#include "osm.h"
#include "latlon.h"
#include "db.h"
#include "osm-db-import.h"

#if 0
#define VERBOSE
#endif
/* #define VERBOSE_KEYS */

#define FILE_BUFFER (128*1024)

static guint node_cnt=0;		/* Nodes */
static guint node_skip_cnt=0;	/* Skipped nodes */
static guint noded_cnt=0;		/* Nodes with (usable) data */
static guint way_cnt=0;			/* Ways */
static guint way_names=0;		/* Ways with name */
static guint way_refs=0;		/* Ways with ref or int_ref */

static guint dbnode_cnt=0;
static guint dbnoded_cnt=0;
static guint dbway_cnt=0;

/* For threaded importing */
static GThread* import_thread=NULL;
static GSourceFunc osm_import_progress_cb=NULL;
static osm_import_data_req osm_import_req;
static guint import_sid=0;
static XML_Parser xp;

/* XML tag IDs */
typedef enum {
	START,
	IN_OSM_TAG,
	IN_NODE_TAG,
	IN_WNODE_TAG,
	IN_WAY_TAG,
	IN_KEY_TAG,
	IN_BOUND_TAG,
	IN_RELATION_TAG,
	IN_MEMBER_TAG,
	END,
	ERROR
} tag_state_t;

/* Parent tag type */
typedef enum {
	IS_NONE,
	IS_NODE,
	IS_WAY,
	IS_RELATION
} tag_parent_t;

/* Node types table */
/* XXX: Add support for parent category */
struct _nodeinfo {
	gchar *k, *v;
	node_type_t type;
} nodeinfo[] = {
	{ "amenity", "fuel", 		NODE_AMENITY_FUEL },
	{ "amenity", "parking",		NODE_AMENITY_PARKING },

	{ "amenity", "pub",			NODE_AMENITY_PUB },
	{ "amenity", "nightclub",	NODE_AMENITY_NIGHTCLUB },
	{ "amenity", "biergarten",	NODE_AMENITY_PUB },
	{ "amenity", "cafe",		NODE_AMENITY_CAFE },
	{ "amenity", "fast_food",	NODE_AMENITY_FOOD },
	{ "amenity", "restaurant",	NODE_AMENITY_RESTAURANT },

	{ "amenity", "telephone",	NODE_AMENITY_TELEPHONE },
	{ "amenity", "toilets",		NODE_AMENITY_WC },

	{ "amenity", "hospital",	NODE_AMENITY_HOSPITAL },
	{ "amenity", "doctors",		NODE_AMENITY_HOSPITAL },
	{ "amenity", "pharmacy",	NODE_AMENITY_PHARMACY },

	{ "amenity", "post_office",	NODE_AMENITY_POST },
	{ "amenity", "post_box",	NODE_AMENITY_POST_BOX },

	{ "amenity", "cinema",		NODE_AMENITY_CINEMA },
	{ "amenity", "theatre",		NODE_AMENITY_THEATRE },

	{ "amenity", "atm",			NODE_AMENITY_ATM },
	{ "amenity", "bank",		NODE_AMENITY_BANK },
	{ "amenity", "bureau_de_change",		NODE_AMENITY_BUREAU_DE_CHANGE },

	{ "amenity", "embassy",		NODE_AMENITY_EMBASSY },

	{ "amenity", "taxi",	NODE_AMENITY_TAXI },
	{ "highway", "bus_stop",	NODE_HIGHWAY_BUS_STOP },
	{ "amenity", "bus_station",	NODE_AMENITY_BUS_STATION },

	{ "amenity", "police",		NODE_AMENITY_POLICE },
	{ "amenity", "speed_trap",	NODE_AMENITY_SPEEDCAM },
	{ "amenity", "speed_camera",	NODE_AMENITY_SPEEDCAM },
	{ "amenity", "speed camera",	NODE_AMENITY_SPEEDCAM },
	{ "highway", "speed_trap",	NODE_AMENITY_SPEEDCAM },
	{ "highway", "speed_camera",	NODE_AMENITY_SPEEDCAM },
	{ "highway", "speed camera",	NODE_AMENITY_SPEEDCAM },

	{ "amenity", "place_of_worship",NODE_AMENITY_POW },

	{ "amenity", "brothel",		NODE_AMENITY_BROTHEL },

	{ "amenity", "recycling",	NODE_AMENITY_RECYCLING },

	{ "amenity", "school",		NODE_AMENITY_SCHOOL },
	{ "amenity", "college",		NODE_AMENITY_COLLEGE },
	{ "amenity", "university",	NODE_AMENITY_COLLEGE },

	{ "amenity", "library",	NODE_AMENITY_LIBRARY },
	{ "amenity", "townhall",	NODE_AMENITY_TOWNHALL },

	{ "amenity", "supermarket",	NODE_AMENITY_SHOP },
	{ "amenity", "shopping_centre",	NODE_AMENITY_SHOP },
	{ "amenity", "shop",		NODE_AMENITY_SHOP },
	{ "amenity", "shops",		NODE_AMENITY_SHOP },
	{ "amenity", "shopping",	NODE_AMENITY_SHOP },
	{ "amenity", "shopping_mall",NODE_AMENITY_SHOP },
	{ "amenity", "cycle_shop",	NODE_AMENITY_SHOP },
	{ "amenity", "bike_shop",	NODE_AMENITY_SHOP },
	{ "amenity", "coffee_shop",	NODE_AMENITY_SHOP },
	{ "amenity", "indoor_shopping_centre",	NODE_AMENITY_SHOP },
	{ "amenity", "farm_shop",	NODE_AMENITY_SHOP },
	{ "amenity", "tea_shop",	NODE_AMENITY_SHOP },

	/* Shops */
	{ "shop", 	 "supermarket",	NODE_AMENITY_SHOP },
	{ "shop", 	 "bakery",		NODE_AMENITY_SHOP },
	{ "shop", 	 "alcohol",		NODE_AMENITY_SHOP }, 
	{ "shop", 	 "butcher",		NODE_AMENITY_SHOP },
	{ "shop", 	 "flowers",		NODE_AMENITY_SHOP },
	{ "shop", 	 "clothing",	NODE_AMENITY_SHOP },
	{ "shop", 	 "souvenir",	NODE_AMENITY_SHOP },
	{ "shop", 	 "bicycles",	NODE_AMENITY_SHOP },
	{ "shop", 	 "grocers",		NODE_AMENITY_SHOP },
	{ "shop", 	 "newsagents",	NODE_AMENITY_SHOP },
	{ "shop", 	 "convenience",	NODE_AMENITY_SHOP },
	{ "shop", 	 "bakers",		NODE_AMENITY_SHOP },
	{ "shop", 	 "garden_centre",NODE_AMENITY_SHOP },
	{ "shop", 	 "photography",	NODE_AMENITY_SHOP },
	{ "shop", 	 "general_store",NODE_AMENITY_SHOP },
	{ "shop", 	 "food",		NODE_AMENITY_SHOP },
	{ "shop", 	 "drinks",		NODE_AMENITY_SHOP },
	{ "shop", 	 "sex",			NODE_AMENITY_SHOP_ADULT },
	{ "shop", 	 "adult",		NODE_AMENITY_SHOP_ADULT },
	{ "shop", 	 "shoes",		NODE_AMENITY_SHOP },
	{ "shop", 	 "computer",		NODE_AMENITY_SHOP_COMPUTER },
	{ "shop", 	 "computers",		NODE_AMENITY_SHOP_COMPUTER },
	{ "shop", 	 "electronics",		NODE_AMENITY_SHOP },
	{ "shop", 	 "furniture",		NODE_AMENITY_SHOP },
	{ "shop", 	 "pharmacy",	NODE_AMENITY_PHARMACY },

	/* Sport */
	{ "sport"  , "skiing",		NODE_SPORT_SKIING },
	{ "sport"  , "swimming",	NODE_SPORT_SWIMMING },
	{ "sport"  , "golf",		NODE_SPORT_GOLF },
	{ "sport"  , "tennis",		NODE_SPORT_TENNIS },
	{ "sport"  , "football",	NODE_SPORT_FOOTBALL },
	{ "sport"  , "soccer",		NODE_SPORT_SOCCER },
	{ "sport"  , "baskteball",	NODE_SPORT_BASKETBALL },
	{ "sport"  , "rugby",		NODE_SPORT_RUGBY },
	{ "sport"  , "skating",		NODE_SPORT_SKATING },
	{ "sport"  , "hockey",		NODE_SPORT_HOCKEY },
	{ "sport"  , "skateboard",	NODE_SPORT_SKATEBOARD },
	{ "sport"  , "bowling",		NODE_SPORT_BOWLING },
	{ "sport"  , "10pin",		NODE_SPORT_BOWLING },
	{ "sport"  , "motor",		NODE_SPORT_MOTOR },
	{ "sport"  , "shooting_range",NODE_SPORT_SHOOTING },
	{ "sport"  , "paintball",	NODE_SPORT_PAINTBALL },
	{ "sport"  , "horse_racing",NODE_SPORT_HORSES },
	{ "sport"  , "horse",		NODE_SPORT_HORSES },
	{ "sport"  , "horses",		NODE_SPORT_HORSES },
	{ "sport"  , "dog_racing",	NODE_SPORT_DOG },
	{ "sport"  , "pelota",		NODE_SPORT_PELOTA },
	{ "sport"  , "racquet",		NODE_SPORT_RACQUET },
	{ "sport"  , "equestrian",	NODE_SPORT_HORSES },
	{ "sport"  , "baseball",	NODE_SPORT_BASEBALL },
	{ "sport"  , "cricket",		NODE_SPORT_CRICKET },
	{ "sport"  , "croquet",		NODE_SPORT_CROQUET },
	{ "sport"  , "cycling",		NODE_SPORT_CYCLING },
	{ "sport"  , "bowls",		NODE_SPORT_BOWLS },
	{ "sport"  , "athletics",	NODE_SPORT_ATHLETICS },
	{ "sport"  , "gymnastics",	NODE_SPORT_GYMNASTICS },
	{ "sport"  , "multi",		NODE_SPORT_OTHER },
	{ "leisure", "sport_centre",NODE_SPORT_CENTER },

	/* Tourism */
	{ "tourism", "information",	NODE_TOURISM_INFO },
	{ "tourism", "camp_site",	NODE_TOURISM_CAMP_SITE },
	{ "tourism", "caravan_site",NODE_TOURISM_CARAVAN_SITE },
	{ "tourism", "picnic_site",	NODE_TOURISM_PICNIC_SITE },
	{ "tourism", "theme_park",	NODE_TOURISM_THEME_PARK },
	{ "tourism", "hotel",		NODE_TOURISM_HOTEL },
	{ "tourism", "motel",		NODE_TOURISM_MOTEL },
	{ "tourism", "hostel",		NODE_TOURISM_HOSTEL },
	{ "tourism", "attraction",	NODE_TOURISM_ATTRACTION },
	{ "tourism", "zoo",			NODE_TOURISM_ATTRACTION },
	{ "tourism", "museum",		NODE_HISTORIC_MUSEUM },

	{ "historic", "ruins",		NODE_TOURISM_ATTRACTION },
	{ "historic", "monument",	NODE_TOURISM_ATTRACTION },
	{ "historic", "memorial",	NODE_TOURISM_ATTRACTION },
	{ "historic", "battlefield",	NODE_TOURISM_ATTRACTION },
	{ "historic", "museum",		NODE_HISTORIC_MUSEUM },
	{ "historic", "castle",		NODE_HISTORIC_CASTLE },

	{ "railway", "station",		NODE_RAILWAY_STATION },
	{ "railway", "halt",		NODE_RAILWAY_HALT },

	{ "aeroway", "terminal",	NODE_AIRPORT_TERMINAL },

	/* Places */	
	{ "place", "city",			NODE_PLACE_CITY },
	{ "place", "town",			NODE_PLACE_TOWN },
	{ "place", "village",		NODE_PLACE_VILLAGE },
	{ "place", "hamlet",		NODE_PLACE_HAMLET },
	{ "place", "locality",		NODE_PLACE_LOCALITY },
	{ "place", "suburb",		NODE_PLACE_SUBURB },
	{ "place", "island",		NODE_PLACE_ISLAND },

	{ "highway", "traffic_signals",	NODE_TRAFFIC_SIGNALS },
	{ "highway", "motorway_junction",	NODE_JUNCTION },
	{ "highway", "services",	NODE_AMENITY_PARKING },
	{ "highway", "toll_booth",	NODE_TOLLBOOTH },
	{ "highway", "gate",		NODE_GATE },

	{ NULL, NULL, NODE_PLAIN }
};

/* Array to get id number and defaults for ways of different types */
struct _wayinfo {
	gchar *k, *v;
	guint defspeed;
	way_type_t type;
	gboolean oneway, link, area, car, foot;
} wayinfo[] = {
	{ "highway", "motorway",120,WAY_MOTORWAY, 		TRUE, FALSE, FALSE, TRUE, FALSE },
	{ "highway", "motorway_link",120,WAY_MOTORWAY,	TRUE, TRUE, FALSE, TRUE, FALSE },
	{ "highway", "trunk",100,WAY_TRUNK,				FALSE, FALSE, FALSE, TRUE, FALSE },
	{ "highway", "trunk_link",100,WAY_TRUNK,		FALSE, TRUE, FALSE, TRUE, FALSE },
	{ "highway", "primary",80,WAY_PRIMARY,			FALSE, FALSE, FALSE, TRUE, TRUE },
	{ "highway", "primary_link",60,WAY_PRIMARY,		FALSE, TRUE, FALSE, TRUE, TRUE },
	{ "highway", "secondary",80,WAY_SECONDARY, 		FALSE, FALSE, FALSE, TRUE, TRUE },
	{ "highway", "secondary_link",60,WAY_SECONDARY,	FALSE, TRUE, FALSE, TRUE, TRUE },
	{ "highway", "tertiary",60,WAY_TERTIARY,		FALSE, FALSE, FALSE, TRUE, TRUE },
	{ "highway", "unclassified",50,WAY_UNCLASSIFIED,	FALSE, FALSE, FALSE, TRUE, TRUE },
	{ "highway", "road",30,WAY_UNCLASSIFIED,	FALSE, FALSE, FALSE, TRUE, TRUE },
	{ "highway", "byway",40,WAY_UNCLASSIFIED,	FALSE, FALSE, FALSE, TRUE, TRUE },
	{ "highway", "residential",40,WAY_RESIDENTIAL,	FALSE, FALSE, FALSE, TRUE, TRUE },
	{ "highway", "service",20,WAY_SERVICE,			FALSE, FALSE, FALSE, TRUE, TRUE },
	{ "highway", "track",30,WAY_TRACK,				FALSE, FALSE, FALSE, TRUE, TRUE },
	{ "highway", "unsurfaced",60,WAY_TRACK,			FALSE, FALSE, FALSE, TRUE, TRUE },
	{ "highway", "minor",60,WAY_TRACK,			FALSE, FALSE, FALSE, TRUE, TRUE },

	{ "highway", "pedestrian",20,WAY_FOOTWAY,		FALSE, FALSE, FALSE, FALSE, TRUE },
	{ "highway", "living_street",20,WAY_FOOTWAY,		FALSE, FALSE, FALSE, TRUE, TRUE },
	{ "highway", "footway",5,WAY_FOOTWAY,			FALSE, FALSE, FALSE, FALSE, TRUE },
	{ "highway", "steps",2,WAY_FOOTWAY,				FALSE, FALSE, FALSE, FALSE, TRUE},
	{ "highway", "path",5,WAY_FOOTWAY,				FALSE, FALSE, FALSE, FALSE, TRUE},
	{ "highway", "bridleway",10,WAY_FOOTWAY,		FALSE, FALSE, FALSE, FALSE, TRUE },
	{ "highway", "cycleway",10,WAY_CYCLEWAY,		FALSE, FALSE, FALSE, FALSE, TRUE },

	{ "railway", "rail",0,WAY_RAIL,					FALSE, FALSE, FALSE, FALSE, FALSE },

	{ "aeroway", "runway",0,WAY_RUNWAY,				FALSE, FALSE, FALSE, FALSE, FALSE },
	{ "aeroway", "taxiway",0,WAY_TAXIWAY,			FALSE, FALSE, FALSE, FALSE, FALSE },

	{ "natural", "water",0,WAY_WATER,				FALSE, FALSE, TRUE, FALSE, FALSE },

	{ "waterway", "river",0,WAY_WATER,				FALSE, FALSE, FALSE, FALSE, FALSE },
	{ "waterway", "canal",0,WAY_WATER,				FALSE, FALSE, FALSE, FALSE, FALSE },
	{ "waterway", "stream",0,WAY_WATER,				FALSE, FALSE, FALSE, FALSE, FALSE },
	{ "building", "*",0,WAY_UNWAYED, 				FALSE, FALSE, TRUE, FALSE, FALSE },

	{ NULL, NULL, 0, WAY_UNWAYED, FALSE, FALSE, FALSE, FALSE, FALSE }
};

static sqlite3 *db;
tag_parent_t tag_parent=IS_NONE;

static GHashTable *osm_nodes;
static GHashTable *osm_node_tags;
static GHashTable *osm_way_tags;
static GSList *osm_ways;
static GSList *osm_poi;

static GHashTable *osm_place_country;
static GHashTable *osm_place_region;
static GHashTable *osm_place_city;
static GHashTable *osm_place_suburb;
static GHashTable *osm_place_village;
static GHashTable *osm_place_island;
static GHashTable *osm_node_isin;
static GHashTable *osm_way_isin;

static node *cnode=NULL;
static way *cway=NULL;

struct sql_stmt {
	sqlite3_stmt *insert_poi;
	sqlite3_stmt *delete_osm_poi;

	sqlite3_stmt *insert_node;
	sqlite3_stmt *delete_nodes;
	sqlite3_stmt *select_node;
	sqlite3_stmt *update_node;

	sqlite3_stmt *insert_way_data;
	sqlite3_stmt *insert_way_ref;
	sqlite3_stmt *insert_way_pc;
	sqlite3_stmt *insert_way_name;
	sqlite3_stmt *insert_way_names_nls;
	sqlite3_stmt *insert_way_n2n;
	sqlite3_stmt *delete_way;
	sqlite3_stmt *delete_way_n2n;
	sqlite3_stmt *delete_way_name;
	sqlite3_stmt *delete_way_names_nls;
	sqlite3_stmt *delete_way_ref;
	sqlite3_stmt *delete_way_pc;

	sqlite3_stmt *insert_place;
	sqlite3_stmt *delete_place;
	sqlite3_stmt *select_place_near;
};
static struct sql_stmt sql;


static struct map_bbox bbox;
static gboolean use_bbox;

static void osm_free_way_data(way *w);
static void print_way(way *w);

static gboolean osm_db_prepare(sqlite3 *db);
static gboolean db_insert_node(node *n);
static guint32 osm_find_way_place(way *w, node_type_t nt);
static guint32 db_find_nearest_place(node_type_t type, gdouble lat, gdouble lon);

gboolean osm_db_create(sqlite3 *db);

/****************************************************/
/* Functions */
/****************************************************/

static void
osm_db_finalize(void)
{
sqlite3_finalize(sql.insert_poi);
sqlite3_finalize(sql.delete_osm_poi);

sqlite3_finalize(sql.insert_node);
sqlite3_finalize(sql.select_node);
sqlite3_finalize(sql.delete_nodes);
sqlite3_finalize(sql.update_node);

sqlite3_finalize(sql.insert_place);
sqlite3_finalize(sql.delete_place);
sqlite3_finalize(sql.select_place_near);

sqlite3_finalize(sql.delete_way);
sqlite3_finalize(sql.insert_way_data);

sqlite3_finalize(sql.delete_way_name);
sqlite3_finalize(sql.insert_way_name);

sqlite3_finalize(sql.delete_way_n2n);
sqlite3_finalize(sql.insert_way_n2n);

sqlite3_finalize(sql.delete_way_pc);
sqlite3_finalize(sql.insert_way_pc);

sqlite3_finalize(sql.delete_way_names_nls);
sqlite3_finalize(sql.insert_way_names_nls);
}

static void
osm_db_analyze(sqlite3 *db)
{
g_print("Optimzing database indexes\n");
sqlite3_exec(db, "ANALYZE;", NULL, NULL, NULL);
}

static gboolean
osm_db_prepare(sqlite3 *db)
{
/* Way nodes */
sqlite3_prepare_v2(db, "insert or replace into nodes (nid,ilat,ilon,l,f) values (?,?,?,0,?)", -1, &sql.insert_node, NULL);
sqlite3_prepare_v2(db, "select ilat,ilon,l from nodes where nid=?", -1, &sql.select_node, NULL);
sqlite3_prepare_v2(db, "delete from nodes", -1, &sql.delete_nodes, NULL);
sqlite3_prepare_v2(db, "update nodes set l=l+1 where nid=?", -1, &sql.update_node, NULL);

/* Places */
sqlite3_prepare_v2(db, "insert or replace into places (nid,type,name,isin_c,isin_p,lat,lon) values (?, ?, ?, ?, ?, ?, ?)", -1, &sql.insert_place, NULL);
sqlite3_prepare_v2(db, "delete from places", -1, &sql.delete_place, NULL);

DB_PREP(db, "select nid as id,(($LAT-lat)*($LAT-lat))+(($LON-lon)*($LON-lon)) as d,isin_p,isin_c "
					" from places where type=$TYPE "
					" and lat between $LAT-$RANGE and $LAT+$RANGE "
					" and lon between $LON-$RANGE and $LON+$RANGE "
					" order by d limit 1", sql.select_place_near);

/* POI nodes */
if (sqlite3_prepare_v2(db, "insert or replace into poi (osm_id, lat, lon, label, cat_id, public, source, priority, isin_c, isin_p, desc, url, postal_code) "
					   " values (?, ?, ?, ?, ?, 1, 1, ?, ?, ?, ?, ?, ?)", -1, &sql.insert_poi, NULL)!=SQLITE_OK)
	g_printf("SQL: %s\n", sqlite3_errmsg(db));

sqlite3_prepare_v2(db, "delete from poi where osm_id>0 and source=1", -1, &sql.delete_osm_poi, NULL);

/* Ways */
sqlite3_prepare_v2(db, "insert or replace into way (wid,nodes,type,flags,speed,isin_c,isin_p,lat,lon) values (?, ?, ?, ?, ?, ?, ?, ?, ?)", -1, &sql.insert_way_data, NULL);
sqlite3_prepare_v2(db, "delete from way", -1, &sql.delete_way, NULL);

/* Way nodes */
sqlite3_prepare_v2(db, "insert into way_n2n (wid,f,t) values (?,?,?)", -1, &sql.insert_way_n2n, NULL);
sqlite3_prepare_v2(db, "delete from way_n2n where wid=?", -1, &sql.delete_way_n2n, NULL);

/* Way names */
sqlite3_prepare_v2(db, "insert or replace into way_names (wid,name) values (?, ?)",  -1, &sql.insert_way_name, NULL);
sqlite3_prepare_v2(db, "delete from way_names", -1, &sql.delete_way_name, NULL);

/* Way postal codes */
sqlite3_prepare_v2(db, "insert or replace into way_pc (wid,pc) values (?, ?)",  -1, &sql.insert_way_pc, NULL);
sqlite3_prepare_v2(db, "delete from way_pc", -1, &sql.delete_way_pc, NULL);

/* Other language names for ways */
sqlite3_prepare_v2(db, "insert into way_names_nls (wid,lang,name) values (?, ?, ?)",  -1, &sql.insert_way_names_nls, NULL);
sqlite3_prepare_v2(db, "delete from way_names_nls where wid=?", -1, &sql.delete_way_names_nls, NULL);

/* Way ref and int_ref */
sqlite3_prepare_v2(db, "insert or replace into way_ref (rid,ref,int_ref) values (?, ?, ?)", -1, &sql.insert_way_ref, NULL);
sqlite3_prepare_v2(db, "delete from way_ref", -1, &sql.delete_way_ref, NULL);

return TRUE;
}

/********************************************************************/

static void
print_way(way *w)
{
#ifdef VERBOSE
g_assert(w);
g_printf("Way #%d(N:%d T:%d S:%d IS: %d/%d): %s [%s:%s:%s]\n", 
		w->id, 	
		g_slist_length(w->nodes), 
		w->type,
		w->data ? w->data->speed : 0,
		w->data ? w->data->isin_c : -1,
		w->data ? w->data->isin_p : -1,
		w->data ? w->data->name ? w->data->name : "" : "", 
		w->flags & W_ONEWAY ? "-" : "=", 
		w->flags & W_ROUNDABOUT ? "O" : "-", 
		w->flags & W_LINK ? "|" : " ");
#endif
}

static void
print_node(node *n)
{
#ifdef VERBOSE
g_assert(n);
g_printf("Node #%d: T:%d IS: %d/%d [%s]\n",
	n->id,
	n->type,
	n->data ? n->data->isin_c : -1,
	n->data ? n->data->isin_p : -1,
	n->data ? n->data->name : "");
#endif
}

/********************************************************************/

/**
 * Search for the nearest place with given type
 */
static guint32
db_find_nearest_place(node_type_t type, gdouble lat, gdouble lon)
{
guint32 id=0; 
gdouble range;

g_return_val_if_fail(sql.select_place_near, 0);

switch (type) {
	case NODE_PLACE_CITY:
		range=0.6;
	break;
	case NODE_PLACE_TOWN:
		range=0.5;
	break;
	case NODE_PLACE_HAMLET:
		range=0.4;
	break;
	case NODE_PLACE_VILLAGE:
		range=0.35;
	break;
	case NODE_PLACE_SUBURB:
		range=0.2;
	break;
	case NODE_PLACE_ISLAND:
		range=0.4;
	break;
	case NODE_PLACE_LOCALITY:
		range=0.18;
	break;
	default:
		range=0.5;
	break;
}

sqlite3_clear_bindings(sql.select_place_near);
sqlite3_reset(sql.select_place_near);

if (SQLITE_OK != sqlite3_bind_double(sql.select_place_near, 1, lat) ||
    SQLITE_OK != sqlite3_bind_double(sql.select_place_near, 2, lon) ||
    SQLITE_OK != sqlite3_bind_int(sql.select_place_near, 3, type) ||
    SQLITE_OK != sqlite3_bind_double(sql.select_place_near, 4, range)) {
	sqlite3_clear_bindings(sql.select_place_near);
	g_warning("Failed to bind values for near place");
	return 0;
}

if (SQLITE_ROW == sqlite3_step(sql.select_place_near)) {
	id=sqlite3_column_int(sql.select_place_near, 0);
}
return id;
}

/********************************************************************/

static gboolean
db_insert_node(node *n)
{
gint32 lat, lon;

g_assert(n);

lat=lat2mp_int(n->lat);
lon=lon2mp_int(n->lon);

sqlite3_bind_int(sql.insert_node, 1, n->id);

/* Projected and integerized lat/lot */
sqlite3_bind_int(sql.insert_node, 2, lat);
sqlite3_bind_int(sql.insert_node, 3, lon);
sqlite3_bind_int(sql.insert_node, 4, n->type);

db_exec(db, sql.insert_node);

return TRUE;
}

static gboolean
db_insert_place(node *n)
{
g_assert(n);
if (!n->data)
	return FALSE;
if (!n->data->name)
	return FALSE;
sqlite3_bind_int(sql.insert_place, 1, n->id);
sqlite3_bind_int(sql.insert_place, 2, n->type);
sqlite3_bind_text(sql.insert_place, 3, n->data->name, -1, SQLITE_TRANSIENT);
sqlite3_bind_int(sql.insert_place, 4, n->data->isin_p);
sqlite3_bind_int(sql.insert_place, 5, n->data->isin_c);
sqlite3_bind_double(sql.insert_place, 6, n->lat);
sqlite3_bind_double(sql.insert_place, 7, n->lon);

return db_exec(db,sql.insert_place);
}

static gboolean
db_insert_poi(node *n)
{
g_assert(n);
sqlite3_bind_int(sql.insert_poi, 1, n->id);
sqlite3_bind_double(sql.insert_poi, 2, n->lat);
sqlite3_bind_double(sql.insert_poi, 3, n->lon);
if (n->data->name)
	sqlite3_bind_text(sql.insert_poi, 4, n->data->name, -1, SQLITE_TRANSIENT);
else
	sqlite3_bind_text(sql.insert_poi, 4, "", -1, SQLITE_TRANSIENT);
sqlite3_bind_int(sql.insert_poi, 5, n->type);
sqlite3_bind_int(sql.insert_poi, 6, n->type/100);
sqlite3_bind_int(sql.insert_poi, 7, n->data->isin_c);
sqlite3_bind_int(sql.insert_poi, 8, n->data->isin_p);

if (n->data->desc)
	sqlite3_bind_text(sql.insert_poi, 9, n->data->desc, -1, SQLITE_TRANSIENT);
if (n->data->url)
	sqlite3_bind_text(sql.insert_poi, 10, n->data->url, -1, SQLITE_TRANSIENT);
if (n->data->postal_code)
	sqlite3_bind_text(sql.insert_poi, 11, n->data->postal_code, -1, SQLITE_TRANSIENT);

return db_exec(db,sql.insert_poi);
}

/**
 * Update node usage count
 */
static gboolean
db_update_node_links(node *n)
{
g_assert(n);
sqlite3_bind_int(sql.update_node, 1, n->id);

return db_exec(db,sql.update_node);
}

/**
 * Insert way,node1,node2 triplet
 */
static gboolean
db_insert_way_n2n(way *w, node *nf, node *nt)
{
if (!w) {
	g_warning("NULL WAY");
	return FALSE;
}

if (!nf) {
	g_warning("NULL NODE 1, way: %u", w->id);
	return FALSE;
}

if (!nt) {
	g_warning("NULL NODE 2, way: %u", w->id);
	return FALSE;
}

sqlite3_bind_int(sql.insert_way_n2n, 1, w->id);
sqlite3_bind_int(sql.insert_way_n2n, 2, nf->id);
sqlite3_bind_int(sql.insert_way_n2n, 3, nt->id);

db_exec(db,sql.insert_way_n2n);
db_update_node_links(nf);
db_update_node_links(nt);
return TRUE;
}

/**
 * Insert way ref and int_ref
 */
static gboolean 
db_insert_way_ref(way *w)
{
if (!w->data)
	return TRUE;

if (!w->data->ref && !w->data->int_ref)
	return TRUE;

way_refs++;

sqlite3_bind_int(sql.insert_way_ref, 1, w->id);
if (w->data->ref)
	sqlite3_bind_text(sql.insert_way_ref, 2, w->data->ref, -1, SQLITE_TRANSIENT);
if (w->data->int_ref)
	sqlite3_bind_text(sql.insert_way_ref, 3, w->data->int_ref, -1, SQLITE_TRANSIENT);

return db_exec(db,sql.insert_way_ref);
}

/**
 * Insert way name
 */
static gboolean
db_insert_way_name(way *w)
{
if (!w->data)
	return TRUE;
if (!w->data->name)
	return TRUE;

way_names++;

sqlite3_bind_int(sql.insert_way_name, 1, w->id);
sqlite3_bind_text(sql.insert_way_name, 2, w->data->name, -1, SQLITE_TRANSIENT);

return db_exec(db,sql.insert_way_name);
}

static gboolean
db_delete_way_names_nls(way *w)
{
sqlite3_bind_int(sql.delete_way_names_nls, 1, w->id);
return db_exec(db,sql.delete_way_names_nls);
}

static gboolean 
db_insert_way_pc(way *w)
{
if (!w->data)
	return TRUE;
if (!w->data->postal_code)
	return TRUE;

sqlite3_bind_int(sql.insert_way_pc, 1, w->id);
sqlite3_bind_text(sql.insert_way_pc, 2, w->data->postal_code, -1, SQLITE_TRANSIENT);

return db_exec(db,sql.insert_way_pc);
}

static gboolean
db_delete_way_pc(way *w)
{
sqlite3_bind_int(sql.delete_way_pc, 1, w->id);
return db_exec(db,sql.delete_way_pc);
}

static void
db_insert_way_names_nls_cb(gpointer key, gpointer value, gpointer user_data)
{
way *w=(way *)user_data;

sqlite3_bind_int(sql.insert_way_names_nls, 1, w->id);
sqlite3_bind_text(sql.insert_way_names_nls, 2, (gchar *)key, -1, SQLITE_TRANSIENT);
sqlite3_bind_text(sql.insert_way_names_nls, 3, (gchar *)value, -1, SQLITE_TRANSIENT);
db_exec(db,sql.insert_way_names_nls);
}

static void
db_insert_way_names_nls(way *w)
{
if (!w->data)
	return;
if (!w->data->names)
	return;

g_hash_table_foreach(w->data->names, db_insert_way_names_nls_cb, w);
}

static node *way_get_middle_node(way *w)
{
guint ncnt;

ncnt=g_slist_length(w->nodes);
return (ncnt>0) ? g_slist_nth_data(w->nodes, ncnt/2) : NULL;
}

/**
 * Insert all data for the given way
 * - name
 * - ref
 * - nodes
 * 
 */
static gboolean
db_insert_way(way *w)
{
GSList *iter;
node *wmn=NULL;

if (!w)
	return FALSE;

/* Skip things we don't use (yet) */
if (w->type==WAY_UNWAYED || w->type>WAY_ROAD_END)
	return TRUE;

/* Insert nodes */
for (iter=w->nodes; iter!=NULL; iter=iter->next) {
	if (!iter->next)
		break;
	if (db_insert_way_n2n(w, iter->data, iter->next->data)==FALSE)
		return FALSE;
}

if (w->id==0)
	return FALSE;

if (w->data) {
	w->data->isin_p=osm_find_way_place(w, NODE_PLACE_CITY);
	if (w->data->isin_p==0)
		w->data->isin_p=osm_find_way_place(w, NODE_PLACE_VILLAGE);
	if (w->data->isin_p==0)
		w->data->isin_p=osm_find_way_place(w, NODE_PLACE_SUBURB);

	/* xxx: Rare to have any country info and we don't do anything with yet so ignore */
	w->data->isin_c=0;
}

sqlite3_bind_int(sql.insert_way_data, 1, w->id);
sqlite3_bind_int(sql.insert_way_data, 2, w->ncnt);
sqlite3_bind_int(sql.insert_way_data, 3, w->type);
sqlite3_bind_int(sql.insert_way_data, 4, w->flags);

/* Get middle node, use it as a rough way location */
wmn=way_get_middle_node(w);
if (wmn) {
	sqlite3_bind_double(sql.insert_way_data, 8, wmn->lat);
	sqlite3_bind_double(sql.insert_way_data, 9, wmn->lon);
} else {
	g_debug("Way [%d] has no nodes", w->id);
}

if (w->data && w->data->isin_p==0 && wmn) {
	guint32 pid;

	pid=db_find_nearest_place(NODE_PLACE_CITY, wmn->lat, wmn->lon);
	if (pid==0)
		pid=db_find_nearest_place(NODE_PLACE_TOWN, wmn->lat, wmn->lon);
	if (pid==0)
		pid=db_find_nearest_place(NODE_PLACE_VILLAGE, wmn->lat, wmn->lon);
	if (pid==0)
		pid=db_find_nearest_place(NODE_PLACE_HAMLET, wmn->lat, wmn->lon);
	if (pid==0)
		pid=db_find_nearest_place(NODE_PLACE_SUBURB, wmn->lat, wmn->lon);
	if (pid==0)
		pid=db_find_nearest_place(NODE_PLACE_LOCALITY, wmn->lat, wmn->lon);
	w->data->isin_p=pid;
}

if (w->data) {
	sqlite3_bind_int(sql.insert_way_data, 5, w->data->speed);
	if (w->data->isin_c>0)
		sqlite3_bind_int(sql.insert_way_data, 6, w->data->isin_c);
	sqlite3_bind_int(sql.insert_way_data, 7, w->data->isin_p);
}
	
print_way(w);
db_exec(db,sql.insert_way_data);

db_insert_way_ref(w);
db_insert_way_name(w);
db_insert_way_names_nls(w);
db_insert_way_pc(w);

osm_free_way_data(w);
return TRUE;
}

/********************************************************************/

static gchar *
get_attr_key_value(const gchar **p, gchar *key)
{
gchar **d;

d=p;
while (*d!=NULL) {
	if (strncmp(*d, key, strlen(key))==0) {
		d++;
		return *d;
	}
	d++;
	d++;
}
return NULL;
}

static tag_state_t 
check_tag(const gchar *tag)
{
if (strcmp(tag,"node")==0) return IN_NODE_TAG;
else if (strcmp(tag,"nd")==0) return IN_WNODE_TAG;
else if (strcmp(tag,"way")==0) return IN_WAY_TAG;
else if (strcmp(tag,"tag")==0) return IN_KEY_TAG;
else if (strcmp(tag,"osm")==0) return IN_OSM_TAG;
else if (strcmp(tag,"bound")==0) return IN_BOUND_TAG;
else if (strcmp(tag,"relation")==0) return IN_RELATION_TAG;
else if (strcmp(tag,"member")==0) return IN_MEMBER_TAG;
else return ERROR;
}

static void
find_nls_names(gpointer key, gpointer value, gpointer data)
{
gchar *k, *v;
gchar *tmp;
way *w;
GHashTable *nls;

k=(gchar *)key;
v=(gchar *)value;
w=(way *)data;

g_return_if_fail(w);
g_return_if_fail(w->data);
g_return_if_fail(w->data->names);

nls=w->data->names;

/* Check if it is a name key, return if not. */
if (g_str_has_prefix(k, "name:")==FALSE)
	return;

/* Skip in case default name is duplicated */
if (strcmp(w->data->name, v)==0)
	return;

tmp=g_strrstr(k, ":");
if (!tmp)
	return;
tmp++; /* skip : */
if (*tmp==0)
	return;
g_hash_table_insert(nls, g_strdup(tmp), g_strstrip(g_utf8_normalize(v, -1, G_NORMALIZE_ALL_COMPOSE)));
}

/********************************************************************/

static void
node_print (node *n)
{
g_assert(n);
if (n->data) {
	g_printf("N: %d [%f:%f][%s](%d)\n", 
		n->id, n->lat, n->lon, 
		n->data->name ? n->data->name : "-", 
		n->type);
} else {
	g_printf("N: %d [%f:%f]\n",
		n->id, n->lat, n->lon);
}
}

#ifdef DEBUG
static void 
dump_array(const gchar **p)
{
char **d;

d=p;
while (*d!=NULL) {
        g_printf("[%s]", *d);
        d++;
}
g_print("\n");
}
#endif

static inline gboolean
osm_node_check_box(gdouble nlat, gdouble nlon)
{
if (use_bbox==FALSE)
	return TRUE;
return (nlat > bbox.lat_min && nlat < bbox.lat_max && nlon > bbox.lon_min && nlon < bbox.lon_max) ? TRUE : FALSE;
}

static void
osm_new_node_data(node *n)
{
if (n==NULL) 
	return;
if (n->data!=NULL) 
	return;
n->data=g_slice_new0(node_data);
n->type=NODE_PLAIN;
noded_cnt++;
}

static void
osm_free_node_data(node *n)
{
g_assert(n);
g_assert(n->data);
if (n->data->name)
	g_free(n->data->name);
if (n->data->url)
	g_free(n->data->url);
if (n->data->desc)
	g_free(n->data->desc);
if (n->data->postal_code)
	g_free(n->data->postal_code);
g_slice_free(node_data, n->data);
n->data=NULL;
noded_cnt--;
}

static node *
osm_new_node(gint id, gdouble lat, gdouble lon)
{
node *n=NULL;

n=g_slice_new(node);
g_assert(n);
n->id=id;
n->lat=lat;
n->lon=lon;
n->data=(node_data *)NULL;
return n;
}

static void
osm_free_node(node *n)
{
g_assert(n);
g_slice_free(node, n);
}

static node *
osm_find_node(guint32 nid)
{
node *n;

g_assert(osm_nodes);
return g_hash_table_lookup(osm_nodes, GINT_TO_POINTER(nid));
}

static void
osm_new_way_data(way *w)
{
g_return_if_fail(w);
g_return_if_fail(!w->data);

w->data=g_slice_new(way_data);
w->data->name=NULL;
w->data->names=NULL;
w->data->ref=NULL;
w->data->int_ref=NULL;
w->data->postal_code=NULL;
w->data->layer=0;
w->data->speed=0;
}

static void
osm_free_way_data(way *w)
{
g_return_if_fail(w);
g_return_if_fail(w->data);

if (w->data->name)
	g_free(w->data->name);
if (w->data->ref)
	g_free(w->data->ref);
if (w->data->int_ref)
	g_free(w->data->int_ref);
g_slice_free(way_data, w->data);
w->data=NULL;
}

static way *
osm_new_way(gint id)
{
way *w;

w=g_slice_new(way);
g_assert(w);
w->id=id;
w->nodes=NULL;
w->type=WAY_UNWAYED;
w->data=NULL;
w->ncnt=0;
w->flags=0;

/* Add to list of ways */
return w;
}

static void
osm_free_way(way *w)
{
g_return_if_fail(w);
if (w->nodes)
	g_slist_free(w->nodes);
g_slice_free(way, w);
}

static void
osm_way_add_to_list(way *w)
{
g_assert(w);
osm_ways=g_slist_prepend(osm_ways, w);
}

static void
osm_way_new_node(way *w, gint nid)
{
node *n;

g_assert(w);
n=osm_find_node(nid);
w->nodes=g_slist_prepend(w->nodes, n);
w->ncnt++;
}

/**
 * Search the place hash table for the location of the node.
 *
 */
static guint32 
osm_find_node_place(node *n)
{
node *t;
gchar **isin;
gchar **place;

if (!n->data)
	return 0;

isin=g_hash_table_lookup(osm_node_isin, GINT_TO_POINTER(n->id));

if (!isin)
	return 0;

place=isin;
while (*place!=NULL) {
	gchar *ps;
	ps=g_strstrip(*place);
#ifdef VERBOSE
	g_printf("Checking (%d) [%s] in [%s]\n",n->type, n->data->name, ps);
#endif
	switch (n->type) {
	case NODE_PLACE_CITY:
	case NODE_PLACE_TOWN:
		t=g_hash_table_lookup(osm_place_region, ps);
		if (t)
			return t->id;
		t=g_hash_table_lookup(osm_place_country, ps);
		if (t)
			return t->id;
	case NODE_PLACE_VILLAGE:
	case NODE_PLACE_HAMLET:
		t=g_hash_table_lookup(osm_place_village, ps);
		if (t)
			return t->id;
	case NODE_PLACE_SUBURB:
	case NODE_PLACE_LOCALITY:
		t=g_hash_table_lookup(osm_place_suburb, ps);
		if (t)
			return t->id;
	case NODE_PLACE_ISLAND:
		t=g_hash_table_lookup(osm_place_island, ps);
		if (t)
			return t->id;
	default:
		t=g_hash_table_lookup(osm_place_city, ps);
		if (t)
			return t->id;
	break;
	}
	place++;
}

return 0;
}

static guint32
osm_find_way_nearest_place(way *w)
{
}

static guint32
osm_find_way_place(way *w, node_type_t nt)
{
gchar **isin;
gchar **place;

isin=g_hash_table_lookup(osm_way_isin, GINT_TO_POINTER(w->id));
if (!isin)
	return 0;

place=isin;
while (*place!=NULL) {
	node *t;
	gchar *ps;

	ps=g_strstrip(*place);

#ifdef VERBOSE
	g_printf("Checking (%d) in [%s]\n",w->id, ps);
#endif
switch (nt) {
	case NODE_PLACE_CITY:
	case NODE_PLACE_TOWN:
		t=g_hash_table_lookup(osm_place_city, ps);
		if (t)
			return t->id;
	case NODE_PLACE_VILLAGE:
	case NODE_PLACE_HAMLET:
		t=g_hash_table_lookup(osm_place_village, ps);
		if (t)
			return t->id;
	case NODE_PLACE_SUBURB:
	case NODE_PLACE_LOCALITY:
		t=g_hash_table_lookup(osm_place_suburb, ps);
		if (t)
			return t->id;
	break;
	case NODE_PLACE_COUNTRY:
		t=g_hash_table_lookup(osm_place_country, ps);
		if (t)
			return t->id;
	break;
	default:
		g_assert_not_reached();
	break;
	}
	place++;
}

return 0;
}

/***********************************************************************/

static void
osm_node_save_node(gint key, gpointer value, gpointer user_data)
{
node *n=(node *)value;

dbnode_cnt++;
db_insert_node(n);
if (dbnode_cnt % 20000==0)
	g_printf("\rNodes: %f%% (%u/%u)\n",
		((float)dbnode_cnt/(float)(node_cnt-node_skip_cnt))*100,
		dbnode_cnt, node_cnt);
}

/**
 * Check node type and insert as POI or Place
 * Discard extra data after insert.
 */
static gboolean
osm_node_save_poi(node *n, gpointer user_data)
{
if (!n) {
	g_printerr("ERROR: null poi\n");
	return FALSE;
}

if (!n->data) {
	g_printerr("POI node with no data ?\n");
	return FALSE;
}

n->data->isin_p=osm_find_node_place(n);
n->data->isin_c=0;

if (n->type>NODE_POI_START && n->type<NODE_POI_END) {
	print_node(n);
	db_insert_poi(n);
	osm_free_node_data(n);
} else if (n->type>NODE_PLACE_START && n->type<NODE_PLACE_END) {
	print_node(n);
	db_insert_place(n);
} else {
	osm_free_node_data(n);
	return FALSE;
}

return TRUE;
}

static gboolean
osm_planet_poi_clear_nodes(void)
{
g_print("Removing old OSM POIs...\n");
db_transaction_begin(db);
sqlite3_step(sql.delete_osm_poi);
sqlite3_step(sql.delete_place);
return db_transaction_commit(db);
}

static gboolean
osm_planet_poi_save_nodes(void)
{
g_print("Storing new POIs...\n");
db_transaction_begin(db);
g_slist_foreach(osm_poi, osm_node_save_poi, NULL);
g_slist_free(osm_poi);
return db_transaction_commit(db);
}

/*********************************************************************/

static void
osm_planet_clear_nodes(void)
{
g_print("Clearing old nodes\n");
sqlite3_step(sql.delete_nodes);
}

static gboolean
osm_planet_save_nodes(void)
{
g_print("Storing nodes...\n");
db_transaction_begin(db);
g_hash_table_foreach(osm_nodes, osm_node_save_node, NULL);
return db_transaction_commit(db);
}

/*********************************************************************/

static void
osm_way_save(way *value, gpointer user_data)
{
dbway_cnt++;
db_transaction_begin(db);
db_insert_way(value);
if (dbway_cnt % 15000==0 && dbway_cnt>0) {
		g_printf("\rWays: %f%% (%u/%u)\n",
			(((float)dbway_cnt/(float)way_cnt)*100),
			dbway_cnt, way_cnt);
		print_way(value);
}
db_transaction_commit(db);
}

static void
osm_planet_clear_ways(void)
{
g_print("Clearing old data\n");
sqlite3_step(sql.delete_way);
sqlite3_step(sql.delete_way_name);
sqlite3_step(sql.delete_way_ref);
sqlite3_step(sql.delete_way_n2n);
}

static gboolean
osm_planet_save_ways(gboolean is_update)
{
g_print("Inserting new ways\n");
g_slist_foreach(osm_ways, osm_way_save, NULL);
}

/*********************************************************************/

static void
osm_planet_save_all_nodes(gboolean is_update)
{
g_printf("Saving planet nodes to database:\n");

if (!is_update) {
	osm_planet_poi_clear_nodes();
	osm_planet_clear_nodes();
	osm_planet_clear_ways();
}
osm_planet_poi_save_nodes();
osm_planet_save_nodes();
}

static void
osm_planet_save_all_ways(gboolean is_update)
{
g_printf("Saving planet way to database:\n");

osm_planet_save_ways(is_update);
}

/***********************************************************************/

static void
_osm_tag_start(void *userData, const char *name, const char **atts)
{
tag_state_t t;
gchar *k, *v;
guint32 id, ndref;
gdouble nlat, nlon;

t=check_tag(name);
switch (t) {
	case IN_OSM_TAG:
		g_printf("Starting...\n");
	break;
	case IN_NODE_TAG:
		tag_parent=IS_NODE;
		node_cnt++;

		id=atoi(get_attr_key_value(atts, "id"));
		nlat=atof(get_attr_key_value(atts, "lat"));
		nlon=atof(get_attr_key_value(atts, "lon"));

		cnode=osm_new_node(id, nlat, nlon);
		osm_node_tags=g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
	break;
	case IN_WAY_TAG:
		tag_parent=IS_WAY;
		way_cnt++;
		id=atoi(get_attr_key_value(atts, "id"));
		cway=osm_new_way(id);
		osm_new_way_data(cway);
		osm_way_tags=g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
	break;
	case IN_WNODE_TAG:
		ndref=atoi(get_attr_key_value(atts, "ref"));
		if (use_bbox==TRUE) {
			if (osm_find_node(ndref)==NULL) {
				cway->id=0;
				return;
			}
		}
		osm_way_new_node(cway, ndref);
	break;
	case IN_KEY_TAG:
		k=get_attr_key_value(atts, "k");
		if (strcmp(k,"created_by")==0)
			return;
		if (strcmp(k,"source")==0)
			return;

		v=get_attr_key_value(atts, "v");
#ifdef VERBOSE_KEYS
		g_printf("TAG: K=[%s] V=[%s]\n", k, v);
#endif

		switch (tag_parent) {
		case IS_NONE:
			g_printf("Tag key/value pair but unknown owner\n");
		break;
		case IS_NODE:
			if (!osm_node_tags)
				return;

			g_return_if_fail(cnode);
			g_hash_table_insert(osm_node_tags, g_strdup(k), g_strdup(v));
		break;
		case IS_WAY: 
			g_return_if_fail(cway);
			g_hash_table_insert(osm_way_tags, g_strdup(k), g_strdup(v));
		break;
		case IS_RELATION:

		break;
		}
	break;
	case IN_BOUND_TAG:

	break;
	case IN_RELATION_TAG:
		tag_parent=IS_RELATION;
	break;
	case IN_MEMBER_TAG:

	break;
	default:
		tag_parent=IS_NONE;
		g_printf("Unknown tag: %s\n", name);
	break;
}
}

static node_type_t
osm_find_type(GHashTable *tags)
{
guint i;
gchar *v;

g_assert(tags);

for (i=0; nodeinfo[i].k; i++) {
	v=g_hash_table_lookup(tags, nodeinfo[i].k);
	if (!v)
		continue;
	if (strcasecmp(v, nodeinfo[i].v)==0)
		return nodeinfo[i].type;
}

/* Check any generic keys and use generic type */
if (g_hash_table_lookup(tags, "amenity"))
	return NODE_AMENITY_GENERIC;
else if (g_hash_table_lookup(tags, "tourism"))
	return NODE_AMENITY_GENERIC;
else if (g_hash_table_lookup(tags, "shop"))
	return NODE_AMENITY_SHOP;
else if (g_hash_table_lookup(tags, "building"))
	return NODE_BUILDING;

return NODE_PLAIN;
}

static gboolean
node_analyze_poi_data(node *cn, GHashTable *tags)
{
gchar *v;

if (!cn->data)
	osm_new_node_data(cn);

v=g_hash_table_lookup(tags, "name");
if (v)
	cn->data->name=g_strstrip(g_utf8_normalize(v, -1, G_NORMALIZE_ALL_COMPOSE));
v=g_hash_table_lookup(tags, "note");
if (v)
	cn->data->desc=g_strstrip(g_strdup(v));
v=g_hash_table_lookup(tags, "postal_code");
if (v)
	cn->data->postal_code=g_strstrip(g_strdup(v));
/* Links */
v=g_hash_table_lookup(tags, "url");
if (v) {
	cn->data->url=g_strstrip(g_strdup(v));
} else {
	v=g_hash_table_lookup(tags, "wikipedia");
	if (v && strncmp(v,"http:", 5)==0) 
		cn->data->url=g_strstrip(g_strdup(v));
}
return TRUE;
}

static void
node_analyze(node *cn, GHashTable *tags)
{
gchar *v;

osm_new_node_data(cn);
cn->type=osm_find_type(tags);

/* Check if node is inside bounding box, if not skip it. 
 * But keep it if it's something we might need for other nodes:
 * - Places (for is_in)
 * - ...
 */
if ((osm_node_check_box(cn->lat, cn->lon)==FALSE) && (cn->type<NODE_PLACE_START)) {
	osm_free_node_data(cn);
	osm_free_node(cn);
	node_skip_cnt++;
	return;
}

g_hash_table_insert(osm_nodes, GINT_TO_POINTER(cn->id), cn);

cn->data->isin_c=0;
cn->data->isin_p=0;
v=g_hash_table_lookup(tags, "is_in");
if (v) {
	gchar **isin;
	isin=g_strsplit(v, ",", 10);
	g_hash_table_insert(osm_node_isin, GINT_TO_POINTER(cn->id), isin);
}

if (cn->type!=NODE_PLAIN && node_analyze_poi_data(cn, tags)==TRUE)
	osm_poi=g_slist_prepend(osm_poi, cn);

if (cn->type==NODE_PLAIN) {
	osm_free_node_data(cn);
} else {
	if (cn->data->name) {
		switch (cn->type) {
		case NODE_PLACE_COUNTRY:
			g_hash_table_insert(osm_place_country, cn->data->name, cn);
		break;
		case NODE_PLACE_CITY:
		case NODE_PLACE_TOWN:
			g_hash_table_insert(osm_place_city, cn->data->name, cn);
		break;
		case NODE_PLACE_SUBURB:
		case NODE_PLACE_LOCALITY:
			g_hash_table_insert(osm_place_suburb, cn->data->name, cn);
		break;
		case NODE_PLACE_VILLAGE:
		case NODE_PLACE_HAMLET:
			g_hash_table_insert(osm_place_village, cn->data->name, cn);
		break;
		case NODE_PLACE_ISLAND:
			g_hash_table_insert(osm_place_island, cn->data->name, cn);
		break;
		default:;
		}
	}
}

}

static void
_osm_tag_end(void *userData, const char *name)
{
tag_state_t t;
gchar *v;
guint i;
t=check_tag(name);
switch (t) {
	case IN_NODE_TAG:
		if (node_cnt % 25000==0) {
			g_printf("Nodes: %u of %u, POIs: %u, Outside box: %u\n", 
				node_cnt-node_skip_cnt, node_cnt, noded_cnt, node_skip_cnt);
		}
		if (!osm_node_tags)
			return;

		node_analyze(cnode, osm_node_tags);

		g_hash_table_destroy(osm_node_tags);
		cnode=NULL;
	break;
	case IN_WAY_TAG:
		if (way_cnt % 1024==0) {
			g_printf("\rWays: %d\n", way_cnt);
		}

		cway->nodes=g_slist_reverse(cway->nodes);

		for (i=0; wayinfo[i].k; i++) {
			v=g_hash_table_lookup(osm_way_tags, wayinfo[i].k);
			if (!v)
				continue;
			if (strcasecmp (v, wayinfo[i].v)==0) {
				if (wayinfo[i].link==TRUE)
					cway->flags|=W_LINK;
				if (wayinfo[i].area==TRUE)
					cway->flags|=W_AREA;
				if (wayinfo[i].oneway==TRUE)
					cway->flags|=W_ONEWAY;
				cway->type=wayinfo[i].type;
				if (cway->data->speed==0)
					cway->data->speed=wayinfo[i].defspeed;
				break;
			}
		}

		v=g_hash_table_lookup(osm_way_tags, "name");
		if (v) {
			cway->data->name=g_utf8_normalize(v, -1, G_NORMALIZE_ALL_COMPOSE);
			/* Try to find other language names */
			cway->data->names=g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
			g_hash_table_foreach(osm_way_tags, find_nls_names, cway);
			if (g_hash_table_size(cway->data->names)==0) {
				g_hash_table_destroy(cway->data->names);
				cway->data->names=NULL;
			}
		} else {
			cway->flags|=W_NONAME;
		}

		v=g_hash_table_lookup(osm_way_tags, "ref");
		if (v)
			cway->data->ref=g_strdup(v);
		v=g_hash_table_lookup(osm_way_tags, "int_ref");
		if (v)
			cway->data->int_ref=g_strdup(v);

		if (!cway->data->ref && !cway->data->int_ref)
			cway->flags|=W_NOREF;

		v=g_hash_table_lookup(osm_way_tags, "postal_code");
		if (v)
			cway->data->postal_code=g_strdup(v);

		/* XXX: somehow handle the silly -1 'reversed' oneway */
		v=g_hash_table_lookup(osm_way_tags, "oneway");
		if (v)
			cway->flags|=W_ONEWAY;

		v=g_hash_table_lookup(osm_way_tags, "noexit");
		if (v)
			cway->flags|=W_NOEXIT;
		
		v=g_hash_table_lookup(osm_way_tags, "speedlimit");
		if (v)
			cway->data->speed=atoi(v);
		v=g_hash_table_lookup(osm_way_tags, "maxspeed");
		if (v)
			cway->data->speed=atoi(v);

		v=g_hash_table_lookup(osm_way_tags, "junction");
		if (v && strcasecmp(v,"roundabout")==0) {
			cway->flags|=W_ROUNDABOUT;
			cway->flags|=W_ONEWAY;
		} else if (v && strcasecmp(v,"mini_roundabout")==0) {
			cway->flags|=W_ROUNDABOUT;
			cway->flags|=W_ONEWAY;
		}

		/* XXX: Should check keys */
		v=g_hash_table_lookup(osm_way_tags, "access");
		if (v && (strcasecmp(v, "private")==0)) {
			cway->flags|=W_NOACCESS;
		}

		v=g_hash_table_lookup(osm_way_tags, "is_in");
		if (v) {
			gchar **isin;				
			isin=g_strsplit(v, ",", 10);
			g_hash_table_insert(osm_way_isin, GINT_TO_POINTER(cway->id), isin);
		}

		if (cway->id!=0 && cway->type!=WAY_UNWAYED) {
			osm_way_add_to_list(cway);
		} else {
			/* Check if they way is a POI and use middle node as POI location if it's not already a POI */
			if (cway->type==WAY_UNWAYED) {
				node_type_t t;

				t=osm_find_type(osm_way_tags);
				if (t!=NODE_PLAIN) {
					node *n=way_get_middle_node(cway);

					if (n && n->type==NODE_PLAIN) {
						n->type=t;
						if (node_analyze_poi_data(n, osm_way_tags)==TRUE)
							osm_poi=g_slist_prepend(osm_poi, n);
					}
				}
			}
			osm_free_way(cway);
		}
		if (cway->data && cway->data->name==NULL && cway->data->ref==NULL &&
			cway->data->int_ref==NULL && cway->data->layer==0 && cway->data->speed==0)
			osm_free_way_data(cway);

		g_hash_table_destroy(osm_way_tags);
		cway=NULL;
	break;
	case IN_BOUND_TAG:
		/* */
	break;
	case IN_OSM_TAG:
		g_printf("\nPlanet loaded.\n");
	break;
	default:;
}
}

/************************************************************************/

static void
storage_init(void)
{
osm_nodes=g_hash_table_new(g_direct_hash, g_direct_equal);

osm_place_country=g_hash_table_new(g_str_hash, g_str_equal);
osm_place_city=g_hash_table_new(g_str_hash, g_str_equal);
osm_place_suburb=g_hash_table_new(g_str_hash, g_str_equal);
osm_place_village=g_hash_table_new(g_str_hash, g_str_equal);
osm_place_region=g_hash_table_new(g_str_hash, g_str_equal);
osm_place_island=g_hash_table_new(g_str_hash, g_str_equal);
osm_node_isin=g_hash_table_new(g_direct_hash, g_direct_equal);
osm_way_isin=g_hash_table_new(g_direct_hash, g_direct_equal);
}

static void
storage_free(void)
{
g_hash_table_destroy(osm_nodes);

g_hash_table_destroy(osm_place_country);
g_hash_table_destroy(osm_place_city);
g_hash_table_destroy(osm_place_suburb);
g_hash_table_destroy(osm_place_village);
g_hash_table_destroy(osm_place_region);
g_hash_table_destroy(osm_place_island);
g_hash_table_destroy(osm_node_isin);
}

/************************************************************************/

static gint
print_fail(const gchar *msg, gint ret)
{
g_printerr("ERROR: %s\n", msg);
return ret;
}

/************************************************************************/

static void
print_memory_usage(void)
{
g_print("Memory usage per item:\n");
g_printf("Node  size: %d\n", (gint)sizeof(node));
g_printf("NodeD size: %d\n", (gint)sizeof(node_data));
g_printf("Way   size: %d\n", (gint)sizeof(way));
g_printf("WayD  size: %d\n", (gint)sizeof(way_data));
}

/************************************************************************
 * Public inteface
 ************************************************************************/

void
osm_planet_parser_init(void)
{
xp=XML_ParserCreate(NULL);
XML_SetElementHandler(xp, _osm_tag_start, _osm_tag_end);
storage_init();
}

void
osm_planet_parser_deinit(void)
{
XML_ParserFree(xp);
storage_free();
}

gboolean
osm_planet_parse_buffer(gchar *buffer, size_t r)
{
if (XML_Parse(xp, buffer, r, r>0 ? 0:1) == XML_STATUS_ERROR) {
	g_printerr("Parse error at line %d:\n%s\n",
		(gint)XML_GetCurrentLineNumber(xp),
		XML_ErrorString(XML_GetErrorCode(xp)));
	return FALSE;
}
return TRUE;
}

gboolean 
osm_planet_parse_file(gchar *pfile)
{
FILE *f;
BZFILE *b;
int bzerror;
int r;
gchar buffer[FILE_BUFFER];
gboolean res=TRUE;

f=fopen(pfile, "r");
if (!f) {
	perror("fopen failed\n");
	return FALSE;
}

b=BZ2_bzReadOpen(&bzerror, f, 0, 0, NULL, 0);
if (bzerror != BZ_OK) {
	g_printf("BZ2_bzReadOpen failed\n");
	BZ2_bzReadClose(&bzerror, b);
	fclose(f);
	return FALSE;
}

do {
	r=BZ2_bzRead(&bzerror, b, buffer, FILE_BUFFER);
	if ((bzerror!=BZ_STREAM_END) && (bzerror!=BZ_OK)) {
		res=FALSE;
		break;
	}
	if (!osm_planet_parse_buffer(buffer, r)) {
		res=FALSE;
		break;
	}
} while (bzerror==BZ_OK);

BZ2_bzReadClose(&bzerror, b);
fclose(f);
return res;
}

/**
 * Set up bounding box for import.
 *
 */
void
osm_import_set_bbox(gboolean use_bb, gdouble latmin, gdouble lonmin, gdouble latmax, gdouble lonmax)
{
use_bbox=use_bb;
bbox.lat_min=latmin;
bbox.lon_min=lonmin;
bbox.lat_max=latmax;
bbox.lon_max=lonmax;
g_printf("Skipping data outside of box: %f,%f - %f,%f\n",
	bbox.lat_min, bbox.lon_min,	bbox.lat_max, bbox.lon_max);
}

static void
osm_print_import_stats(void)
{
g_printf("Total nodes %d, POIs: %d and Ways %d.\n",	node_cnt, noded_cnt, way_cnt);
g_printf("Cities/Towns: %d\n", g_hash_table_size(osm_place_city));
g_printf("Villages/Hamlets: %d\n", g_hash_table_size(osm_place_village));
g_printf("Suburbs: %d\n", g_hash_table_size(osm_place_suburb));
g_printf("Nodes: %d\n", g_hash_table_size(osm_nodes));
}

/**
 * Simple helper to do all preparations and importing from planet -> database
 *
 */
gboolean
osm_import(const gchar *planet, const gchar *database)
{
gboolean is_update=TRUE;

if (db_connect(&db, database)!=TRUE) {
	g_printerr("Database open failed: %s", database);
	return FALSE;
}

/* Try to speed up import */
sqlite3_exec(db, "PRAGMA temp_store=2", NULL, NULL, NULL);
sqlite3_exec(db, "PRAGMA journal_mode=MEMORY", NULL, NULL, NULL);

if (!osm_db_create(db)) {
	g_printerr("Failed to create osm tables or indexes\n");
	return FALSE;
}

if (!osm_db_prepare(db)) {
	g_printerr("Failed to prepare sql statements\n");
	return FALSE;
}

osm_planet_parser_init();

if (osm_planet_parse_file(planet)==FALSE) {
	g_printerr("Failed to parse file: %s\n", planet);
	return FALSE;
}

osm_print_import_stats();

osm_planet_save_all_nodes(is_update);
osm_planet_save_all_ways(is_update);
osm_planet_parser_deinit();
/* XXX: analyze destroys the db in some odd way, skip for now */
#if 0
osm_db_analyze(db);
#endif
osm_db_finalize();
db_close(&db);
g_print("All done.\n");
return TRUE;
}

static gpointer 
osm_import_thread(gpointer user_data)
{
gboolean r;
osm_import_data_req *req=(osm_import_data_req *)user_data;

g_assert(req);
g_assert(req->planet);
g_assert(req->db);

osm_import_progress_cb=req->progress_cb!=NULL ? req->progress_cb : NULL;

r=osm_import(req->planet, req->db);
g_debug("OSM import result: %d", r);

g_free(req->planet);
g_free(req->db);

if (req->done_cb!=NULL)
	g_idle_add(req->done_cb, GINT_TO_POINTER(r==TRUE ? 0 : 1));

return r==TRUE ? 0 : 1;
}

/**
 * Helper to start an import in the background using a thread.
 *
 * Two callback can be given, one for progress feedback and one when the operation is done.
 * Done callback must call the join function.
 * Only one import thread can run at a time.
 *
 */
gboolean 
osm_import_bg(const gchar *planet, const gchar *database, GSourceFunc progress_cb, GSourceFunc done_cb)
{
GError *error=NULL;

g_return_val_if_fail(import_thread==NULL, FALSE);

osm_import_req.planet=g_strdup(planet);
osm_import_req.db=g_strdup(database);
osm_import_req.progress_cb=progress_cb;
osm_import_req.done_cb=done_cb;

import_thread=g_thread_create(osm_import_thread, &osm_import_req, TRUE, &error);
if (import_thread==NULL) {
	g_free(osm_import_req.planet);
	g_free(osm_import_req.db);
	g_printerr("Import thread creation failed.\n");
	return FALSE;
}
if (osm_import_progress_cb!=NULL)
	import_sid=g_timeout_add(1000, osm_import_progress_cb, NULL);
return TRUE;
}

gint
osm_import_join_bg(void)
{
gint ret;
g_assert(import_thread!=NULL);

if (import_sid!=0)
	g_source_remove(import_sid);
ret=g_thread_join(import_thread);
import_thread=NULL;
return ret;
}
