/*
 * This file is part of mapper
 *
 * Copyright (C) 2007-2009 Kaj-Michael Lang
 * Copyright (C) 2006-2007 John Costigan.
 *
 * POI and GPS-Info code originally written by Cezary Jackiewicz.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stddef.h>
#include <math.h>
#include <errno.h>
#include <sys/wait.h>
#include <libintl.h>
#include <locale.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <libgnomevfs/gnome-vfs.h>
#include <curl/multi.h>
#include <gconf/gconf-client.h>

#include "config-gconf.h"
#include "gconf-utils.h"
#include "utils.h"
#include "mapper-types.h"
#include "latlon.h"
#include "map.h"
#include "route.h"
#include "db.h"
#include "poi.h"
#include "gps.h"
#include "ui-common.h"
#include "settings.h"
#include "gpx.h"
#include "filter.h"
#include "dialogs.h"
#include "map-repo.h"

#define PROXY_PORT (8080)

void 
config_update_proxy(void)
{
if (_http_proxy_host)
	g_free(_http_proxy_host);

_http_proxy_host=NULL;
_http_proxy_port=0;

/* Get proxy data and register for updates. */
if (gconf_client_get_bool(gconf_client, GCONF_KEY_HTTP_PROXY_ON, NULL)) {
	_http_proxy_host=gconf_client_get_string(gconf_client, GCONF_KEY_HTTP_PROXY_HOST, NULL);
	_http_proxy_port=mapper_gconf_get_int(gconf_client, GCONF_KEY_HTTP_PROXY_PORT, PROXY_PORT);
}
}

#ifndef TABLET_USER
#define TABLET_USER "user"
#endif

static gboolean
config_is_tablet_user(void)
{
const gchar *u;

/* This should do it for now */
u=g_get_user_name();
return (strcmp(u, TABLET_USER)==0 && getuid()==29999 && getgid()==29999) ? TRUE : FALSE;
}

static gchar *
config_get_repo_base(void)
{
gchar *b, *mmc=NULL;

b=g_get_home_dir();

if (!b)
	b=g_getenv("HOME");

if (config_is_tablet_user()) {
	g_debug("On tablet probably...");
	if (g_file_test("/media/mmc1/Mapper", G_FILE_TEST_IS_DIR))
		mmc="/media/mmc1";
	else if (g_file_test("/media/mmc2/Mapper", G_FILE_TEST_IS_DIR))
		mmc="/media/mmc2";
	if (mmc)
		return g_build_filename(mmc, "Mapper", "Maps", ".store", NULL);
	else
		return g_build_filename(b, "MyDocs", "Mapper", "Maps", ".store", NULL);
}
if (!b) {
	g_warning("Failed to get a directory, using temporary location!");
	b=g_get_tmp_dir();
}
return g_build_filename(b, "Mapper", "Maps", ".store", NULL);
}

static RepoData *
config_create_repo(gchar *name, gchar *url, gchar *format, gchar *base)
{
RepoData *r=g_new(RepoData, 1);

r->cache_dir=g_build_filename(base, name, NULL);
r->url=url;
r->name=name;
r->format=format;
r->dl_zoom_steps=1;
r->nextable=TRUE;
r->double_size=FALSE;
r->view_zoom_steps=1;
set_repo_type(r);

_repo_list=g_list_append(_repo_list, r);
return r;
}

void
config_create_repos(void)
{
RepoData *r;
gchar *base;

_repo_list=NULL;
base=config_get_repo_base();
g_debug("Map repo base: %s", base);

config_create_repo("Google", "http://mt.google.com/vt/lyrs=m@118&x=%d&y=%d&zoom=%d", "png", base);
config_create_repo("OpenPisteMap", "http://tiles.openpistemap.org/contours/%0d/%d/%d.png", "png", base);
config_create_repo("CycleMap", "http://andy.sandbox.cloudmade.com/tiles/cycle/%0d/%d/%d.png", "png", base);
config_create_repo("ÖPNV-Karte", "http://tile.xn--pnvkarte-m4a.de/tilegen/%0d/%d/%d.png", "png", base);
config_create_repo("TilesAtHome", "http://tah.openstreetmap.org/Tiles/tile/%0d/%d/%d.png", "png", base);
r=config_create_repo("OpenStreetMap", "http://tile.openstreetmap.org/%0d/%d/%d.png", "png", base);
repo_set_curr(r);
}

gboolean
config_load_position(Position *pos, const gchar *key)
{
gchar buffer[64];

g_assert(gconf_client);

pos->valid=gconf_client_get_pair(gconf_client, key, GCONF_VALUE_FLOAT, GCONF_VALUE_FLOAT, &pos->lat, &pos->lon, NULL);
if (!pos->valid) {
	pos->lat=NAN;
	pos->lon=NAN;
} else if (pos->lat==NAN || pos->lon==NAN) {
	pos->valid=FALSE;
} else {
	g_snprintf(buffer, sizeof(buffer), "%s_valid", key);
	pos->valid = gconf_client_get_bool(gconf_client, buffer, NULL);
}
return pos->valid;
}

gboolean
config_save_position(Position *pos, const gchar *key)
{
gchar buffer[64];

g_assert(gconf_client);

if (!gconf_client_set_pair(gconf_client, key, GCONF_VALUE_FLOAT, GCONF_VALUE_FLOAT, &pos->lat, &pos->lon, NULL))
	return FALSE;
g_snprintf(buffer, sizeof(buffer), "%s_valid", key);
return gconf_client_set_bool(gconf_client, buffer, pos->valid, NULL);
}

gboolean
config_save_filter(GpsTrackFilter *f)
{
g_assert(gconf_client);

/* Filtering */
gconf_client_set_bool(gconf_client, GCONF_KEY_GPS_FILTER, f->enabled, NULL);
gconf_client_set_float(gconf_client, GCONF_KEY_GPS_FILTER_HDOP, f->hdop, NULL);
gconf_client_set_float(gconf_client, GCONF_KEY_GPS_FILTER_VDOP, f->vdop, NULL);
gconf_client_set_float(gconf_client, GCONF_KEY_GPS_FILTER_ANGLE, f->angle, NULL);
gconf_client_set_float(gconf_client, GCONF_KEY_GPS_FILTER_OSM, f->osm, NULL);
gconf_client_set_int(gconf_client, GCONF_KEY_GPS_FILTER_MAXDROP, f->maxdrop, NULL);

gconf_client_suggest_sync(gconf_client, NULL);
return TRUE;
}

/**
 * Save all configuration data to GCONF.
 */
gboolean
config_save(void)
{
gchar *config_dir;
gchar buffer[16];
gint i;
Position gps;
Position center;

g_assert(gconf_client);

/* Initialize config_dir. */
config_dir = gnome_vfs_expand_initial_tilde(CONFIG_DIR_NAME);
g_mkdir_with_parents(config_dir, 0700);

/* Save Receiver MAC from GConf. */
if (_gps->io.address)
	gconf_client_set_string(gconf_client, GCONF_KEY_GPS_MAC, _gps->io.address, NULL);
else
	gconf_client_unset(gconf_client, GCONF_KEY_GPS_MAC, NULL);

gconf_client_set_int(gconf_client, GCONF_KEY_GPS_TYPE, _gps->io.type, NULL);
gconf_client_set_int(gconf_client, GCONF_KEY_GPS_PORT, _gps->io.port, NULL);

/* Save Auto-Download. */
gconf_client_set_bool(gconf_client, GCONF_KEY_AUTO_DOWNLOAD, _auto_download, NULL);

/* Save Auto-Center Sensitivity. */
gconf_client_set_int(gconf_client, GCONF_KEY_CENTER_SENSITIVITY, _center_ratio, NULL);

/* Save Auto-Center Lead Amount. */
gconf_client_set_int(gconf_client, GCONF_KEY_LEAD_AMOUNT, _lead_ratio, NULL);

/* Save Draw Line Width. */
gconf_client_set_int(gconf_client, GCONF_KEY_DRAW_WIDTH, _draw_width, NULL);

/* Save Announce Advance Notice Ratio. */
gconf_client_set_int(gconf_client, GCONF_KEY_ANNOUNCE_NOTICE, _announce_notice_ratio, NULL);

/* Set announcement flags */
gconf_client_set_bool(gconf_client, GCONF_KEY_ANNOUNCE_WAYPOINTS, _announce_waypoints, NULL);
gconf_client_set_bool(gconf_client, GCONF_KEY_ANNOUNCE_OVERSPEED, _announce_overspeed, NULL);
gconf_client_set_bool(gconf_client, GCONF_KEY_ANNOUNCE_DESTINATION, _announce_destination, NULL);

/* Save Enable Voice flag. */
gconf_client_set_bool(gconf_client, GCONF_KEY_ENABLE_VOICE, _enable_voice, NULL);

/* Save Keep On When Fullscreen flag. */
gconf_client_set_bool(gconf_client, GCONF_KEY_ALWAYS_KEEP_ON, _always_keep_on, NULL);

/* Save Units. */
gconf_client_set_string(gconf_client, GCONF_KEY_UNITS, UNITS_TEXT[_units], NULL);

/* Save Custom Key Actions. */
for (i = 0; i < CUSTOM_KEY_ENUM_COUNT; i++)
	gconf_client_set_string(gconf_client, CUSTOM_KEY_GCONF[i], CUSTOM_ACTION_TEXT[_action[i]], NULL);

/* Save Deg Format. */
gconf_client_set_string(gconf_client, GCONF_KEY_DEG_FORMAT,	DEG_FORMAT_TEXT[_degformat], NULL);

/* Save Speed Limit On flag. */
gconf_client_set_bool(gconf_client, GCONF_KEY_SPEED_LIMIT_ON, _speed_on, NULL);

/* Save Speed Limit. */
gconf_client_set_int(gconf_client, GCONF_KEY_SPEED_LIMIT, _speed_limit, NULL);

/* Save Info Font Size. */
gconf_client_set_string(gconf_client, GCONF_KEY_INFO_FONT_SIZE, INFO_FONT_TEXT[_info_font_size], NULL);

/* Save gps and center latitude/longiture. */
gps.valid=TRUE;
gps.lat=_gps->data.lat;
gps.lon=_gps->data.lon;
config_save_position(&gps, GCONF_KEY_POSITION_GPS);

center.valid=TRUE;
unit2latlon(_center.unitx, _center.unity, center.lat, center.lon);
config_save_position(&center, GCONF_KEY_POSITION_CENTER);

/* Save last Zoom Level. */
gconf_client_set_int(gconf_client, GCONF_KEY_ZOOM, _zoom, NULL);

/* Save Route Directory. */
if (_route_dir_uri)
	gconf_client_set_string(gconf_client, GCONF_KEY_ROUTEDIR, _route_dir_uri, NULL);

/* Save Last Track File. */
if (_track_file_uri)
	gconf_client_set_string(gconf_client, GCONF_KEY_TRACKFILE, _track_file_uri, NULL);

/* Save Auto-Center Mode. */
gconf_client_set_int(gconf_client, GCONF_KEY_AUTOCENTER_MODE, _center_mode, NULL);

/* Save Show Scale flag. */
gconf_client_set_bool(gconf_client, GCONF_KEY_SHOWSCALE, _show_scale, NULL);

/* Save Show Tracks flag. */
gconf_client_set_bool(gconf_client, GCONF_KEY_SHOWTRACKS, _show_tracks & TRACKS_MASK, NULL);

/* Save Show Routes flag. */
gconf_client_set_bool(gconf_client, GCONF_KEY_SHOWROUTES, _show_tracks & ROUTES_MASK, NULL);

/* Save Show Velocity Vector flag. */
gconf_client_set_bool(gconf_client, GCONF_KEY_SHOWVELVEC, _show_velvec, NULL);

/* Save Show POIs flag. */
gconf_client_set_bool(gconf_client, GCONF_KEY_SHOWPOIS, _show_poi, NULL);

/* Save Enable GPS flag. */
gconf_client_set_bool(gconf_client, GCONF_KEY_ENABLE_GPS, _enable_gps, NULL);

/* Save Route Locations. */
gconf_client_set_list(gconf_client, GCONF_KEY_ROUTE_LOCATIONS, GCONF_VALUE_STRING, _loc_list, NULL);

/* Save Route Download Radius. */
gconf_client_set_int(gconf_client, GCONF_KEY_ROUTE_DL_RADIUS, _route_dl_radius, NULL);
gconf_client_set_int(gconf_client, GCONF_KEY_TRACK_DL_RADIUS, _track_dl_radius, NULL);

/* Save Colors. */
for (i = 0; i < COLORABLE_ENUM_COUNT; i++) {
	g_snprintf(buffer, sizeof(buffer), "#%02x%02x%02x",
		 _color[i].red >> 8, _color[i].green >> 8, _color[i].blue >> 8);
	gconf_client_set_string(gconf_client, COLORABLE_GCONF[i], buffer, NULL);
}

/* Save Mapper database. */
if (_mapper_db)
	gconf_client_set_string(gconf_client, GCONF_KEY_MAPPER_DB, _mapper_db, NULL);
else
	gconf_client_unset(gconf_client, GCONF_KEY_MAPPER_DB, NULL);

/* Save Show POI below zoom. */
gconf_client_set_int(gconf_client, GCONF_KEY_POI_ZOOM, _poi_zoom, NULL);

gconf_client_suggest_sync(gconf_client, NULL);
g_free(config_dir);
return TRUE;
}

/**
 *
 */
gboolean
config_load_string_list(const gchar *gconf_key, GSList **list, GtkListStore **model)
{
GSList *curr;

*list=gconf_client_get_list(gconf_client, gconf_key, GCONF_VALUE_STRING, NULL);
*model=gtk_list_store_new(1, G_TYPE_STRING);
for (curr = *list; curr != NULL; curr = curr->next) {
	GtkTreeIter iter;
	gtk_list_store_insert_with_values(*model, &iter, INT_MAX, 0, curr->data, -1);
}
return TRUE;
}

/**
 * Initialize all configuration from GCONF.  This should not be called more
 * than once during execution.
 */
void
config_init(void)
{
GConfValue *value;
gchar *config_dir;
gchar *str;
gint i;
Position gps;
Position center;

gconf_client=gconf_client_get_default();
if (!gconf_client) {
	popup_error(mapp.mainwindow, _("Failed to initialize GConf. Quitting."));
	exit(1);
}

_center_mode = CENTER_LEAD;
_units = UNITS_KM;
_degformat = DDPDDDDD;
_info_font_size = INFO_FONT_MEDIUM;

/* Preload configuration */
gconf_client_add_dir (gconf_client, GCONF_KEY_PREFIX, GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);

/* Initialize config_dir. */
config_dir = gnome_vfs_expand_initial_tilde(CONFIG_DIR_NAME);
g_mkdir_with_parents(config_dir, 0700);

/* Get Receiver MAC from GConf.  Default is scanned via hci_inquiry. */
_gps->io.address=gconf_client_get_string(gconf_client, GCONF_KEY_GPS_MAC, NULL);
_gps->io.port=mapper_gconf_get_int(gconf_client, GCONF_KEY_GPS_PORT, 2947);
_gps->io.type=(GpsIOSourceType)mapper_gconf_get_bound_int(gconf_client, GCONF_KEY_GPS_TYPE, GPS_IO_SIMULATION, GPS_IO_TYPE_MIN+1, GPS_IO_TYPE_MAX-1);

/* Get Auto-Download.  Default is FALSE. */
_auto_download = gconf_client_get_bool(gconf_client, GCONF_KEY_AUTO_DOWNLOAD, NULL);

_center_ratio=mapper_gconf_get_int(gconf_client, GCONF_KEY_CENTER_SENSITIVITY, 7);
_lead_ratio=mapper_gconf_get_int(gconf_client, GCONF_KEY_LEAD_AMOUNT, 5);
_draw_width=mapper_gconf_get_int(gconf_client, GCONF_KEY_DRAW_WIDTH, 4);

/* Invalidate destination */
_dest.valid=FALSE;

/* Get Announce Advance Notice - Default is 30. */
_announce_notice_ratio=mapper_gconf_get_int(gconf_client, GCONF_KEY_ANNOUNCE_NOTICE, 8);

_announce_waypoints=mapper_gconf_get_boolean(gconf_client, GCONF_KEY_ANNOUNCE_WAYPOINTS, TRUE);
_announce_overspeed=mapper_gconf_get_boolean(gconf_client, GCONF_KEY_ANNOUNCE_OVERSPEED, FALSE);
_announce_destination=mapper_gconf_get_boolean(gconf_client, GCONF_KEY_ANNOUNCE_DESTINATION, FALSE);

/* Get Enable Voice flag.  Default is TRUE. */
_enable_voice=mapper_gconf_get_boolean(gconf_client, GCONF_KEY_ENABLE_VOICE, TRUE);
_voice_pitch=mapper_gconf_get_bound_int(gconf_client, GCONF_KEY_VOICE_PITCH, 50, 10, 99);
_voice_speed=mapper_gconf_get_bound_int(gconf_client, GCONF_KEY_VOICE_SPEED, 120, 80, 370);

/* Get Always Keep On flag.  Default is FALSE. */
_always_keep_on=mapper_gconf_get_boolean(gconf_client, GCONF_KEY_ALWAYS_KEEP_ON, FALSE);

filter.enabled=gconf_client_get_bool(gconf_client, GCONF_KEY_GPS_FILTER, NULL);
filter.hdop=gconf_client_get_float(gconf_client, GCONF_KEY_GPS_FILTER_HDOP, NULL);
filter.vdop=gconf_client_get_float(gconf_client, GCONF_KEY_GPS_FILTER_VDOP, NULL);
filter.angle=gconf_client_get_float(gconf_client, GCONF_KEY_GPS_FILTER_ANGLE, NULL);
filter.osm=gconf_client_get_float(gconf_client, GCONF_KEY_GPS_FILTER_OSM, NULL);
filter.maxdrop=gconf_client_get_int(gconf_client, GCONF_KEY_GPS_FILTER_MAXDROP, NULL);
filter.drop_cnt=0;

filter.maxdrop=CLAMP(filter.maxdrop, 0, 60);
filter.hdop=CLAMP(filter.hdop, 0, 50);
filter.vdop=CLAMP(filter.vdop, 0, 50);
filter.angle=CLAMP(filter.angle, 0, 45);

/* Get Units.  Default is UNITS_KM. */
{
	gchar *units_str=gconf_client_get_string(gconf_client, GCONF_KEY_UNITS, NULL);
	guint i = 0;
	if (units_str)
		for (i = UNITS_ENUM_COUNT - 1; i > 0; i--)
			if (!strcmp(units_str, UNITS_TEXT[i]))
				break;
	_units = i;
}

	/* Get Custom Key Actions. */
{
	gint i;
	for (i = 0; i < CUSTOM_KEY_ENUM_COUNT; i++) {
		gint j = CUSTOM_KEY_DEFAULT[i];
		gchar *str = gconf_client_get_string(gconf_client, CUSTOM_KEY_GCONF[i], NULL);
		if (str)
			for (j = CUSTOM_ACTION_ENUM_COUNT - 1; j > 0; j--)
				if (!strcmp(str, CUSTOM_ACTION_TEXT[j]))
					break;
		_action[i] = j;
	}
}

/* Get Deg format.  Default is DDPDDDDD. */
{
	gchar *degformat_key_str = gconf_client_get_string(gconf_client, GCONF_KEY_DEG_FORMAT, NULL);
	guint i = 0;
	if (degformat_key_str)
		for (i = DEG_FORMAT_ENUM_COUNT - 1; i > 0; i--)
			if (!strcmp(degformat_key_str, DEG_FORMAT_TEXT[i]))
				break;
	_degformat = i;
}

/* Get Info Font Size.  Default is INFO_FONT_MEDIUM. */
{
	gchar *info_font_size_str = gconf_client_get_string(gconf_client, GCONF_KEY_INFO_FONT_SIZE, NULL);
	gint i = -1;
	if (info_font_size_str)
		for (i = INFO_FONT_ENUM_COUNT - 1; i >= 0; i--)
			if (!strcmp(info_font_size_str, INFO_FONT_TEXT[i]))
				break;
	if (i == -1)
		i = INFO_FONT_MEDIUM;
	_info_font_size = i;
	g_debug("LOADED FONT SIZE: %d %s", _info_font_size, info_font_size_str);
}

/* Get Speed Limit On flag.  Default is FALSE. */
_speed_on = gconf_client_get_bool(gconf_client, GCONF_KEY_SPEED_LIMIT_ON, NULL);

/* Get Speed Limit */
_speed_limit = gconf_client_get_int(gconf_client, GCONF_KEY_SPEED_LIMIT, NULL);
CLAMP(_speed_limit, 10, 300);

/* Get Auto-Download.  Default is FALSE. */
_auto_download=mapper_gconf_get_boolean(gconf_client, GCONF_KEY_AUTO_DOWNLOAD, FALSE);

/* Get saved locations */
config_load_position(&gps, GCONF_KEY_POSITION_GPS);
if (!gps.valid) {
	_gps->data.lat=60.20;
	_gps->data.lon=22.20;
}

config_load_position(&center, GCONF_KEY_POSITION_CENTER);
if (!center.valid) {
	center.lat = _gps->data.lat;
	center.lon = _gps->data.lon;
}
latlon2unit(center.lat, center.lon, _center.unitx, _center.unity);

/* Special positions Home and Dest */
config_load_position(&_home, GCONF_KEY_POSITION_HOME);
config_load_position(&_dest, GCONF_KEY_POSITION_DEST);

config_create_repos();

/* Get last Zoom Level.  Default is 12. */
value = gconf_client_get(gconf_client, GCONF_KEY_ZOOM, NULL);
if (value) {
	_zoom = gconf_value_get_int(value) / _curr_repo->view_zoom_steps * _curr_repo->view_zoom_steps;
	gconf_value_free(value);
} else {
	_zoom = 12 / _curr_repo->view_zoom_steps * _curr_repo->view_zoom_steps;
}
CLAMP(_zoom, 0, MAX_ZOOM - 1);
_world_size_tiles = unit2tile(WORLD_SIZE_UNITS);

/* Get Route Directory.  Default is NULL. */
_route_dir_uri = gconf_client_get_string(gconf_client, GCONF_KEY_ROUTEDIR, NULL);

/* Get Last Track File.  Default is NULL. */
_track_file_uri = gconf_client_get_string(gconf_client, GCONF_KEY_TRACKFILE, NULL);

_center_mode=mapper_gconf_get_int(gconf_client, GCONF_KEY_AUTOCENTER_MODE, CENTER_LEAD);

/* Get Show Scale flag.  Default is TRUE. */
_show_scale=mapper_gconf_get_boolean(gconf_client, GCONF_KEY_SHOWSCALE, TRUE);

/* Get Show Tracks flag.  Default is TRUE. */
_show_tracks |= mapper_gconf_get_boolean(gconf_client, GCONF_KEY_SHOWTRACKS, TRUE) ? TRACKS_MASK : 0;

/* Get Show Routes flag.  Default is TRUE. */
_show_tracks |= mapper_gconf_get_boolean(gconf_client, GCONF_KEY_SHOWROUTES, TRUE) ? ROUTES_MASK : 0;

/* Get Show Velocity Vector flag.  Default is TRUE. */
_show_velvec=mapper_gconf_get_boolean(gconf_client, GCONF_KEY_SHOWVELVEC, TRUE);

_show_poi=mapper_gconf_get_boolean(gconf_client, GCONF_KEY_SHOWPOIS, TRUE);

/* Get Enable GPS flag. Default is FALSE. */
_enable_gps=mapper_gconf_get_boolean(gconf_client, GCONF_KEY_ENABLE_GPS, FALSE);

/* Initialize _conn_state based on _enable_gps-> */
_gps->io.conn = (_enable_gps ? RCVR_DOWN : RCVR_OFF);

config_load_string_list(GCONF_KEY_ROUTE_LOCATIONS, &_loc_list, &_loc_model);

/* Show POIs below zoom */
_poi_zoom = mapper_gconf_get_bound_int(gconf_client, GCONF_KEY_POI_ZOOM, 6, 0, 10);

/* Set Route Download URL. */
_route_dl_url=g_strdup(ROUTE_DOWNLOAD_URL);

/* Get Route Download Radius.  Default is 4. */
_route_dl_radius=mapper_gconf_get_int(gconf_client, GCONF_KEY_ROUTE_DL_RADIUS, 4);
_track_dl_radius=mapper_gconf_get_int(gconf_client, GCONF_KEY_TRACK_DL_RADIUS, 4);

/* Get Colors. */
for (i = 0; i < COLORABLE_ENUM_COUNT; i++) {
	str = gconf_client_get_string(gconf_client, COLORABLE_GCONF[i], NULL);
	if (!str || !gdk_color_parse(str, &_color[i]))
		_color[i] = COLORABLE_DEFAULT[i];
}

/* Get Mapper Database. Default is in REPO_DEFAULT_CACHE_BASE */
_mapper_db = gconf_client_get_string(gconf_client, GCONF_KEY_MAPPER_DB, NULL);
if (_mapper_db == NULL) {
	gchar *db_base = gnome_vfs_expand_initial_tilde(REPO_DEFAULT_CACHE_BASE);
	_mapper_db = gnome_vfs_uri_make_full_from_relative(db_base, "osm-planet.db");
	g_free(db_base);
}

/* Get current proxy settings. */
config_update_proxy();

gconf_client_clear_cache(gconf_client);
g_free(config_dir);
}


