/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 * Copyright (C) 2006-2007 John Costigan.
 *
 * POI and GPS-Info code originally written by Cezary Jackiewicz.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <config.h>

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stddef.h>
#include <locale.h>
#include <math.h>
#include <errno.h>
#include <sys/wait.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <fcntl.h>
#include <libintl.h>
#include <locale.h>

#include "utils.h"
#include "poi.h"
#include "gps.h"
#include "map.h"
#include "mapper-types.h"
#include "ui-common.h"
#include "settings.h"
#include "poi-gui.h"
#include "osm.h"
#include "osm-db.h"
#include "search.h"
#include "dialogs.h"
#include "latlon.h"

typedef enum {
	SBTN_INFO=1,
	SBTN_GOTO=2,
	SBTN_ROUTETO=3,
	SBTN_EDIT=4,
	SBTN_SEARCH=5,
} search_action_e;

typedef struct _search_dialog search_dialog;
struct _search_dialog {
	GtkWidget *dialog;
	GtkWidget *cmb_type;
	GtkWidget *cmb_category;
	GtkWidget *list;
	GtkWidget *search_entry;
	GtkWidget *label;
	GtkListStore *store;
	gdouble lat;
	gdouble lon;
	mapper_search_type stype;
};

static search_dialog sd;
static GSList *search_list=NULL;
static GtkListStore *search_model=NULL;

static gboolean
search_do(search_dialog *s)
{
gchar *st;
gint cid;
GtkTreeIter iter;
guint slen;
gboolean sres=FALSE;
slen=strlen(gtk_entry_get_text(GTK_ENTRY(s->search_entry)));

if (gtk_combo_box_get_active_iter(GTK_COMBO_BOX(s->cmb_category), &iter)==TRUE)
	gtk_tree_model_get(gtk_combo_box_get_model(GTK_COMBO_BOX(s->cmb_category)), &iter, 0, &cid, -1);
else
	cid=-1;
if (s->stype==SEARCH_TYPE_POI && cid==-1 && slen<2) {
#ifdef WITH_HILDON
	hildon_banner_show_information(s->dialog, NULL, _("Query string too short. Minimum 2 characters."));
#else
	popup_error(s->dialog, _("Query string too short. Minimum 2 characters."));
#endif
	return TRUE;
} else if (slen<2 && s->stype!=SEARCH_TYPE_POI) {
#ifdef WITH_HILDON
	hildon_banner_show_information(s->dialog, NULL, _("Query string too short. Minimum 2 characters."));
#else
	popup_error(s->dialog, _("Query string too short. Minimum 2 characters."));
#endif
	return TRUE;
}

gtk_widget_set_sensitive(s->dialog, FALSE);
st=g_strdup(gtk_entry_get_text(GTK_ENTRY(s->search_entry)));

switch (s->stype) {
	case SEARCH_TYPE_POI:
		s->store=poi_list_store_new();
		gtk_tree_view_set_model(GTK_TREE_VIEW(s->list), GTK_TREE_MODEL(s->store));
		MACRO_BANNER_SHOW_INFO(s->dialog, _("Searching for POIs..."));
		sres=poi_search((cid==-1 && slen>0) ? POI_SEARCH_TEXT : (cid!=-1 && slen==0) ? POI_SEARCH_CAT : POI_SEARCH_TEXT_CAT, s->lat, s->lon, st, cid, &s->store);
	break;
	case SEARCH_TYPE_WAY:
		s->store=osm_search_store_new();
		gtk_tree_view_set_model(GTK_TREE_VIEW(s->list), GTK_TREE_MODEL(s->store));
		MACRO_BANNER_SHOW_INFO(s->dialog, _("Searching for streets..."));
		sres=osm_way_search(s->lat, s->lon, st, &s->store);
	break;
	case SEARCH_TYPE_PLACE:
		s->store=osm_search_store_new();
		gtk_tree_view_set_model(GTK_TREE_VIEW(s->list), GTK_TREE_MODEL(s->store));
		MACRO_BANNER_SHOW_INFO(s->dialog, _("Searching for places..."));
		sres=osm_place_search(s->lat, s->lon, st, &s->store);
	break;
}

if (sres==TRUE && st && !g_slist_find_custom(search_list, st, (GCompareFunc)strcmp)) {
	search_list=g_slist_prepend(search_list, g_strdup(st));
	gtk_list_store_insert_with_values(search_model, &iter, INT_MAX, 0, st, -1);
}
gtk_widget_set_sensitive(s->dialog, TRUE);

g_free(st);
return TRUE;
}

static gboolean
search_get_list_item_latlon(GtkWidget *list, gdouble *lat, gdouble *lon)
{
GtkTreeIter iter;
GtkTreeModel *model;

if (!gtk_tree_selection_get_selected(gtk_tree_view_get_selection(GTK_TREE_VIEW(list)), NULL, &iter))
	return FALSE;

model=gtk_tree_view_get_model(GTK_TREE_VIEW(list)); 
if (!model)
	return FALSE;

gtk_tree_model_get(model, &iter, ITEM_LAT, lat, -1);
gtk_tree_model_get(model, &iter, ITEM_LON, lon, -1);
return TRUE;
}

static guint
search_get_list_item_id(GtkWidget *list)
{
guint id;
GtkTreeIter iter;
GtkTreeModel *model;

if (!gtk_tree_selection_get_selected(gtk_tree_view_get_selection(GTK_TREE_VIEW(list)), NULL, &iter))
	return 0;

model=gtk_tree_view_get_model(GTK_TREE_VIEW(list)); 
if (!model)
	return 0;

gtk_tree_model_get(model, &iter, ITEM_ID, &id, -1);
return id;
}

static void
search_set_dialog_mode(search_dialog *s)
{
s->stype=gtk_combo_box_get_active(GTK_COMBO_BOX(s->cmb_type));
switch (s->stype) {
	case SEARCH_TYPE_POI:
		gtk_widget_show(s->cmb_category);
	break;
	case SEARCH_TYPE_WAY:
	case SEARCH_TYPE_PLACE:
		gtk_widget_hide(s->cmb_category);
	break;
	default:
		g_assert_not_reached();
	break;
}
}

static gboolean
search_type_changed_cb(GtkWidget *widget, gpointer data)
{
search_dialog *s=(search_dialog *)data;

search_set_dialog_mode(s);
if (s->store) {
	gtk_list_store_clear(s->store);
	g_object_unref(s->store);
}
s->store=NULL;
gtk_tree_view_set_model(GTK_TREE_VIEW(s->list), GTK_TREE_MODEL(s->store));

return TRUE;
}

static void
search_item_activated_cb(GtkTreeView *tree_view, GtkTreePath *path, GtkTreeViewColumn *column, gpointer data)
{
search_dialog *s=(search_dialog *)data;
gdouble lat, lon;
GtkTreeIter iter;
GtkTreeModel *model;

if (!gtk_tree_selection_get_selected(gtk_tree_view_get_selection(GTK_TREE_VIEW(tree_view)), NULL, &iter))
	return;

model=gtk_tree_view_get_model(GTK_TREE_VIEW(tree_view)); 
if (!model)
	return;

gtk_tree_model_get(model, &iter, ITEM_LAT, &lat, -1);
gtk_tree_model_get(model, &iter, ITEM_LON, &lon, -1);

map_center_latlon(lat, lon);
g_idle_add((GSourceFunc)map_update_location_from_center, NULL);
}

static void
poi_distance_data_func(GtkTreeViewColumn *col, GtkCellRenderer *renderer, GtkTreeModel *model, GtkTreeIter *iter, gpointer user_data)
{
gdouble dist;
gchar buf[20];

gtk_tree_model_get(model, iter, ITEM_DIST, &dist, -1);
g_snprintf(buf, sizeof(buf), "%.1f %s", dist, UNITS_TEXT[_units]);
g_object_set(renderer, "text", buf, NULL);
}

gboolean
mapper_search_dialog(mapper_search_type stype, gdouble lat, gdouble lon)
{
GtkWidget *sw, *hbox;
GtkTreeViewColumn *column;
GtkCellRenderer *renderer;
GtkEntryCompletion *search_comp;
gint res;

sd.store=NULL;
sd.lat=lat;
sd.lon=lon;
sd.stype=stype;

sd.dialog=gtk_dialog_new_with_buttons(_("Search"),
			GTK_WINDOW(mapp.mainwindow),
			GTK_DIALOG_MODAL,
			_("Search"), SBTN_SEARCH,
			_("Info"), SBTN_INFO,
			_("Goto"), SBTN_GOTO,
			_("Route to"), SBTN_ROUTETO,
			_("Edit"), SBTN_EDIT,
			GTK_STOCK_CLOSE, GTK_RESPONSE_REJECT, 
			NULL);

gtk_dialog_set_default_response(sd.dialog, SBTN_SEARCH);

if (search_model==NULL)
	search_model=gtk_list_store_new(1, G_TYPE_STRING);

gtk_window_set_default_size(GTK_WINDOW(sd.dialog), 710, 350);

hbox=gtk_hbox_new(FALSE, 4),
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(sd.dialog)->vbox), hbox, FALSE, FALSE, 0);

sd.cmb_type=gtk_combo_box_new_text();
gtk_combo_box_append_text(GTK_COMBO_BOX(sd.cmb_type), "POI");
gtk_combo_box_append_text(GTK_COMBO_BOX(sd.cmb_type), "Street");
gtk_combo_box_append_text(GTK_COMBO_BOX(sd.cmb_type), "Place");
gtk_box_pack_start(GTK_BOX(hbox), sd.cmb_type, FALSE, FALSE, 0);

sd.cmb_category=category_combo_new();
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(sd.dialog)->vbox), sd.cmb_category, FALSE, FALSE, 0);
poi_category_combo_populate(sd.cmb_category, 0, TRUE);

sd.search_entry=gtk_entry_new();
gtk_entry_set_activates_default(sd.search_entry, TRUE);
search_comp=gtk_entry_completion_new();
gtk_entry_completion_set_model(search_comp, GTK_TREE_MODEL(search_model));
gtk_entry_completion_set_text_column(search_comp, 0);
gtk_entry_set_completion(GTK_ENTRY(sd.search_entry), search_comp);
gtk_box_pack_start(GTK_BOX(hbox), sd.search_entry, TRUE, TRUE, 0);
gtk_widget_grab_focus(sd.search_entry);

sw=gtk_scrolled_window_new(NULL, NULL);
gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw), GTK_SHADOW_ETCHED_IN);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
#if defined(WITH_DEVICE_MAEMO) && defined(WITH_HILDON_1)
hildon_helper_set_thumb_scrollbar(sw, TRUE);
#endif
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(sd.dialog)->vbox), sw, TRUE, TRUE, 0);

sd.list=gtk_tree_view_new();
gtk_tree_selection_set_mode(gtk_tree_view_get_selection (GTK_TREE_VIEW(sd.list)), GTK_SELECTION_SINGLE);
gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(sd.list), TRUE);
gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(sd.list), TRUE);

renderer=gtk_cell_renderer_text_new();
column=gtk_tree_view_column_new_with_attributes(_("Location"), renderer, "text", ITEM_LATLON, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(sd.list), column);

renderer=gtk_cell_renderer_text_new();
column=gtk_tree_view_column_new_with_attributes(_("Distance"), renderer, "text", ITEM_DIST, NULL);
gtk_tree_view_column_set_sort_column_id (column, ITEM_DIST);
gtk_tree_view_append_column(GTK_TREE_VIEW(sd.list), column);
gtk_tree_view_column_set_cell_data_func(column, renderer, poi_distance_data_func, NULL, NULL);

renderer=gtk_cell_renderer_text_new();
column=gtk_tree_view_column_new_with_attributes(_("Label"), renderer, "text", ITEM_LABEL, NULL);
gtk_tree_view_column_set_sort_column_id (column, ITEM_LABEL);
gtk_tree_view_append_column(GTK_TREE_VIEW(sd.list), column);

renderer=gtk_cell_renderer_text_new();
column=gtk_tree_view_column_new_with_attributes(_("Category"), renderer, "text", ITEM_CATLAB, NULL);
gtk_tree_view_column_set_sort_column_id (column, ITEM_CATLAB);
gtk_tree_view_append_column(GTK_TREE_VIEW(sd.list), column);

gtk_tree_view_set_search_column (GTK_TREE_VIEW(sd.list), ITEM_LABEL);
gtk_container_add(GTK_CONTAINER(sw), sd.list);

sd.label=gtk_label_new("");
gtk_box_pack_start(GTK_BOX(hbox), sd.label, FALSE, FALSE, 0);

g_signal_connect(G_OBJECT(sd.cmb_type), "changed", G_CALLBACK(search_type_changed_cb), &sd);
g_signal_connect(G_OBJECT(sd.list), "row-activated", G_CALLBACK(search_item_activated_cb), &sd);

gtk_widget_show_all(sd.dialog);
gtk_combo_box_set_active(GTK_COMBO_BOX(sd.cmb_type), sd.stype);
search_set_dialog_mode(&sd);

while ((res=gtk_dialog_run(GTK_DIALOG(sd.dialog)))!=GTK_RESPONSE_REJECT) {
	gdouble lat, lon;
	gint id;

	switch (res) {
	case SBTN_SEARCH:
		search_do(&sd);
	break;
	case SBTN_INFO:
		switch (sd.stype) {
			case SEARCH_TYPE_POI:
				id=search_get_list_item_id(sd.list);
				if (id>0)
					poi_info_dialog(sd.dialog, id);
			break;
			default:
				popup_error(sd.dialog, _("No information available."));
			break;
		}
	break;
	case SBTN_GOTO:
		if (search_get_list_item_latlon(sd.list, &lat, &lon)==FALSE) {
#ifdef WITH_HILDON
			hildon_banner_show_information(sd.dialog, NULL, _("Select a location from the list."));
#else
			popup_error(sd.dialog, _("Select a location from the list."));
#endif
		} else {
			map_center_latlon(lat, lon);
			map_set_zoom(2);
			goto out;
		}
		continue;
	break;
	case SBTN_ROUTETO:
		if (search_get_list_item_latlon(sd.list, &_dest.lat, &_dest.lon)==FALSE) {
#ifdef WITH_HILDON
			hildon_banner_show_information(sd.dialog, NULL, _("Select a location from the list."));
#else
			popup_error(sd.dialog, _("Select a location from the list."));
#endif
		} else {
			_dest.valid=TRUE;
			goto out;
		}
		continue;
	break;
	case SBTN_EDIT:
		switch (sd.stype) {
			case SEARCH_TYPE_POI: {
				poi_info *p=NULL;
				guint id;

				id=search_get_list_item_id(sd.list);
				if (id>0) {
					p=poi_get_by_id(id);
					if (p) {
						if (poi_edit_dialog(ACTION_EDIT_POI, p));
							goto out;
					} else {
						popup_error(sd.dialog, _("Failed to fetch POI for editing."));
					}
				}
			}
			break;
			default:
				popup_error(sd.dialog, _("Only POIs can be modified."));
			break;
		}
		continue;
	break;
	case GTK_RESPONSE_DELETE_EVENT:
	case GTK_RESPONSE_REJECT:
		goto out;
	break;
	default:
		g_debug("RES: %d", res);
		break;
	break;
	}
}

out: ;

gtk_widget_destroy(sd.dialog);
return TRUE;
}
