/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
#include <config.h>

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stddef.h>
#include <locale.h>
#include <math.h>
#include <errno.h>
#include <sys/wait.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <fcntl.h>
#include <libintl.h>
#include <locale.h>
#include <gtkhtml/gtkhtml.h>
#include <gtkhtml/gtkhtml-stream.h>

#include "mapper.h"
#include "utils.h"
#include "poi.h"
#include "gps.h"
#include "map.h"
#include "mapper-types.h"
#include "latlon.h"
#include "ui-common.h"
#include "settings.h"
#include "poi-gui.h"
#include "osm.h"
#include "osm-db.h"
#include "dialogs.h"
#include "track.h"
#include "map-poi.h"

typedef struct _search_dialog search_dialog;
struct _search_dialog {
	GtkWidget *dialog;
	GtkWidget *list;
	GtkWidget *cmb_category;
	GtkWidget *search_entry;
	GtkListStore *store;
	gdouble lat;
	gdouble lon;
};

static search_dialog sd;

typedef struct _PoiCategoryEditInfo PoiCategoryEditInfo;
struct _PoiCategoryEditInfo {
	GtkWidget *cmb_category;
	guint cat_id;
};

/* XXX: Load this from somewhere instead of hc... */
struct _quick_poi_categories {
    node_type_t type;
    const gchar *name;
};

#define QPBS_X (4)
#define QPBS_Y (3)
#define POI_QUICK_BUTTONS (QPBS_X*QPBS_Y)

static struct _quick_poi_categories quick_poi_categories[] = {
	{ NODE_AMENITY_SPEEDCAM,	"Speedcam", },
	{ NODE_AMENITY_FUEL,		"Fuel", },
	{ NODE_AMENITY_PARKING,		"Parking", },
	{ NODE_AMENITY_TAXI,		"Taxi", },

	{ NODE_AMENITY_PUB,  		"Pub", },
	{ NODE_AMENITY_CAFE, 		"Cafe", },
	{ NODE_AMENITY_FOOD, 		"Fast Food", },
	{ NODE_AMENITY_RESTAURANT,	"Restaurant", },

	{ NODE_AMENITY_SHOP,  		"Shop", },
	{ NODE_AMENITY_BANK, 		"Bank", },
	{ NODE_AMENITY_ATM, 		"ATM", },
	{ NODE_AMENITY_POST_BOX, 	"Post box", },
};

static poi_quick_data qp;

GtkListStore *
poi_gui_get_category_store(gboolean reload, gboolean do_counts)
{
static GtkListStore *pcstore=NULL;

if (reload && pcstore) {
	g_object_unref(pcstore);
	pcstore=NULL;
}

if (!pcstore) {
	pcstore=poi_category_generate_store(do_counts);
	if (pcstore)
		g_object_ref(pcstore);
}
return pcstore;
}

static gboolean 
category_delete(GtkWidget *widget, delete_poi *dpoi)
{
GtkWidget *dialog;
guint i;
gchar *buffer;

buffer = g_strdup_printf("%s\n\t%s\n%s", _("Delete category?"),
			 dpoi->txt_label, _("WARNING: All POIs in that category will also be deleted!"));
dialog = hildon_note_new_confirmation(GTK_WINDOW(mapp.mainwindow), buffer);
g_free(buffer);
i = gtk_dialog_run(GTK_DIALOG(dialog));
gtk_widget_destroy(GTK_WIDGET(dialog));

if (i == GTK_RESPONSE_OK) {
	if (poi_category_delete(dpoi->id)==FALSE)
		popup_error(mapp.mainwindow, _("Problem deleting category or POI"));
	gtk_widget_hide_all(dpoi->dialog);
	map_poi_cache_clear();
	map_force_redraw();
}

return TRUE;
}

GtkWidget *
category_combo_new(void)
{
GtkWidget *cmb;
GtkCellRenderer *rtext, *rpixbuf;

/* Category ID, Label */
cmb=gtk_combo_box_new_with_model(GTK_TREE_MODEL(gtk_list_store_new(3, G_TYPE_INT, G_TYPE_STRING, GDK_TYPE_PIXBUF)));

/* Category icon */
rpixbuf=gtk_cell_renderer_pixbuf_new();
gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(cmb), rpixbuf, FALSE);

/* Set up the text view for the combo box. */
rtext=gtk_cell_renderer_text_new();
gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(cmb), rtext, TRUE);
gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(cmb), rtext, "text", 1, NULL);
gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(cmb), rpixbuf, "pixbuf", 2, NULL);

return cmb;
}

void 
poi_category_combo_populate(GtkWidget *cmb_category, guint cat_id, gboolean add_na)
{
GtkTreeIter active;
GtkListStore *store;
GtkTreeIter iter;
gboolean has_active = FALSE;

store = GTK_LIST_STORE(gtk_combo_box_get_model(GTK_COMBO_BOX(cmb_category)));
gtk_list_store_clear(store);

if (add_na) {
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter, 0, -1, 1, _("[No category]"), -1);	
}

while (SQLITE_ROW == sqlite3_step(poisql.selall_cat_fast)) {
	guint cid = sqlite3_column_int(poisql.selall_cat_fast, 0);
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter, 
		0, cid, 
		1, sqlite3_column_text(poisql.selall_cat_fast, 1), 
		2, poi_get_icon(sqlite3_column_text(poisql.selall_cat_fast, 4), TRUE),
		-1);
	if (cid==cat_id) {
		active=iter;
		has_active=TRUE;
	}
}
sqlite3_reset(poisql.selall_cat_fast);

if (!has_active)
	gtk_tree_model_get_iter_first(GTK_TREE_MODEL(store), &active);

gtk_combo_box_set_active_iter(GTK_COMBO_BOX(cmb_category), &active);
}

gboolean 
poi_category_dialog(guint cat_id)
{
GtkWidget *dialog;
GtkWidget *table;
GtkWidget *label;
GtkWidget *txt_label;
GtkWidget *txt_desc;
GtkWidget *btn_delete = NULL;
GtkWidget *txt_scroll;
GtkWidget *chk_enabled;
GtkTextBuffer *desc_txt;
GtkTextIter begin, end;
gboolean results = TRUE;
delete_poi dpoi = { NULL, NULL, 0 };
poi_category *c;

if (cat_id > 0) {
	if (poi_category_get(cat_id, &c)==FALSE) {
		popup_error(mapp.mainwindow, "Failed to get category.");
		return FALSE;
	}

	dialog = gtk_dialog_new_with_buttons(_("Edit Category"),
				     GTK_WINDOW(mapp.mainwindow),
				     GTK_DIALOG_MODAL,
				     GTK_STOCK_OK,
				     GTK_RESPONSE_ACCEPT, NULL);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->action_area),
			  btn_delete = gtk_button_new_with_label(_("Delete")));

	dpoi.dialog = dialog;
	dpoi.txt_label = g_strdup(c->label);
	dpoi.id = c->id;

	g_signal_connect(G_OBJECT(btn_delete), "clicked", G_CALLBACK(category_delete), &dpoi);
	gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT);
} else {
	c=poi_category_new();
	c->enabled = 1;
	c->label = g_strdup("");
	c->id = 0;
	c->desc = g_strdup("");

	dialog = gtk_dialog_new_with_buttons(_("Add Category"),
				     GTK_WINDOW(mapp.mainwindow),
				     GTK_DIALOG_MODAL,
				     GTK_STOCK_OK,
				     GTK_RESPONSE_ACCEPT,
				     GTK_STOCK_CANCEL,
				     GTK_RESPONSE_REJECT, NULL);
}

gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), table = gtk_table_new(6, 4, FALSE), TRUE, TRUE, 0);

gtk_table_attach(GTK_TABLE(table), label = gtk_label_new(_("Label")), 0, 1, 0, 1, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
gtk_table_attach(GTK_TABLE(table), txt_label = gtk_entry_new(), 1, 2, 0, 1, GTK_EXPAND | GTK_FILL, 0, 2, 4);
gtk_table_attach(GTK_TABLE(table), label = gtk_label_new(_("Description")), 0, 1, 1, 2, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
txt_scroll = gtk_scrolled_window_new(NULL, NULL);
gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(txt_scroll), GTK_SHADOW_IN);
gtk_table_attach(GTK_TABLE(table), txt_scroll, 1, 2, 1, 2, GTK_EXPAND | GTK_FILL, 0, 2, 4);

gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(txt_scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

txt_desc = gtk_text_view_new();
gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(txt_desc), GTK_WRAP_WORD);

gtk_container_add(GTK_CONTAINER(txt_scroll), txt_desc);
gtk_widget_set_size_request(GTK_WIDGET(txt_scroll), 400, 60);

gtk_table_attach(GTK_TABLE(table), chk_enabled = gtk_check_button_new_with_label(_("Enabled")), 0, 2, 2, 3, GTK_EXPAND | GTK_FILL, 0, 2, 4);

/* label */
gtk_entry_set_text(GTK_ENTRY(txt_label), c->label);

/* desc */
desc_txt = gtk_text_view_get_buffer(GTK_TEXT_VIEW(txt_desc));
gtk_text_buffer_set_text(desc_txt, c->desc, -1);

/* enabled */
gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(chk_enabled), (c->enabled == 1 ? TRUE : FALSE));

poi_category_free(c);

gtk_widget_show_all(dialog);

while (GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(dialog))) {
	if (strlen(gtk_entry_get_text(GTK_ENTRY(txt_label)))) {
		c=poi_category_new();
		c->label=g_strdup(gtk_entry_get_text(GTK_ENTRY(txt_label)));
	} else {
		popup_error(dialog,_("Please specify a name for the category."));
		continue;
	}

	gtk_text_buffer_get_bounds(desc_txt, &begin, &end);
	c->desc = gtk_text_buffer_get_text(desc_txt, &begin, &end, TRUE);
	c->enabled = (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(chk_enabled)) ? 1 : 0);

	results=poi_category_update(cat_id, c);

	if (results==FALSE)
		popup_error(mapp.mainwindow, _("Problem updating category"));
	poi_category_free(c);

	break;
}

if (dpoi.txt_label)
	g_free(dpoi.txt_label);
gtk_widget_destroy(dialog);

return results;
}

static void 
category_toggled_cb(GtkCellRendererToggle *cell, gchar *path, gpointer data)
{
GtkTreeIter iter;
gboolean cat_enabled;
guint cat_id;

GtkTreeModel *model = GTK_TREE_MODEL(data);
if (!gtk_tree_model_get_iter_from_string(model, &iter, path))
	return;

gtk_tree_model_get(model, &iter, CAT_ENABLED, &cat_enabled, -1);
gtk_tree_model_get(model, &iter, CAT_ID, &cat_id, -1);

cat_enabled ^= 1;

if (poi_category_toggle(cat_id, cat_enabled)==FALSE) {
	popup_error(mapp.mainwindow, _("Problem updating Category"));
} else {
	gtk_list_store_set(GTK_LIST_STORE(model), &iter, CAT_ENABLED, cat_enabled, -1);
}

}

static gboolean 
category_add_cb(GtkWidget *widget, GtkWidget *tree_view)
{
GtkListStore *store;

if (poi_category_dialog(0)) {
	store=poi_gui_get_category_store(TRUE, FALSE);
	gtk_tree_view_set_model(GTK_TREE_VIEW(tree_view), GTK_TREE_MODEL(store));
	g_object_unref(G_OBJECT(store));
}
return TRUE;
}

static gboolean 
category_edit_cb(GtkWidget *widget, GtkWidget *tree_view)
{
GtkTreeIter iter;
GtkTreeModel *store;
GtkTreeSelection *selection;

store = gtk_tree_view_get_model(GTK_TREE_VIEW(tree_view));
selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree_view));
if (gtk_tree_selection_get_selected(selection, &store, &iter)) {
	GValue val;
	memset(&val, 0, sizeof(val));
	gtk_tree_model_get_value(store, &iter, 0, &val);
	if (poi_category_dialog(g_value_get_uint(&val))) {
		GtkListStore *new_store=poi_gui_get_category_store(TRUE, FALSE);
		gtk_tree_view_set_model(GTK_TREE_VIEW(tree_view), GTK_TREE_MODEL(new_store));
		g_object_unref(G_OBJECT(new_store));
	}
}
return TRUE;
}

gboolean 
poi_category_list()
{
GtkWidget *dialog;
GtkWidget *tree_view;
GtkWidget *sw;
GtkWidget *btn_edit;
GtkWidget *btn_add;
GtkTreeViewColumn *column;
GtkCellRenderer *renderer;
GtkListStore *store;

store=poi_gui_get_category_store(FALSE, TRUE);
if (!store)
	return TRUE;

dialog = gtk_dialog_new_with_buttons(_("POI Categories"),
			     GTK_WINDOW(mapp.mainwindow),
			     GTK_DIALOG_MODAL, GTK_STOCK_CLOSE,
			     GTK_RESPONSE_ACCEPT, NULL);

gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->action_area), btn_edit = gtk_button_new_with_label(_("Edit")));
gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->action_area), btn_add = gtk_button_new_with_label(_("Add")));

sw=gtk_scrolled_window_new(NULL, NULL);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), sw, TRUE, TRUE, 0);

tree_view=gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
#ifdef WITH_HILDON
g_object_set(tree_view, "allow-checkbox-mode", FALSE, NULL);
#endif
gtk_container_add(GTK_CONTAINER(sw), tree_view);

gtk_tree_selection_set_mode(gtk_tree_view_get_selection (GTK_TREE_VIEW(tree_view)), GTK_SELECTION_SINGLE);
gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(tree_view), TRUE);

renderer = gtk_cell_renderer_text_new();
column = gtk_tree_view_column_new_with_attributes(_("ID"), renderer, "text", CAT_ID, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);

renderer = gtk_cell_renderer_toggle_new();
g_signal_connect(renderer, "toggled", G_CALLBACK(category_toggled_cb), store);
column = gtk_tree_view_column_new_with_attributes(_("Enabled"), renderer, "active", CAT_ENABLED, NULL);
gtk_tree_view_column_set_max_width(column, 32);
gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);
gtk_tree_view_column_set_sort_column_id (column, CAT_ENABLED);

renderer = gtk_cell_renderer_pixbuf_new();
column = gtk_tree_view_column_new_with_attributes(_("Icon"), renderer, "pixbuf", CAT_ICON, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);

renderer = gtk_cell_renderer_text_new();
column = gtk_tree_view_column_new_with_attributes(_("Label"), renderer, "text", CAT_LABEL, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);
gtk_tree_view_column_set_sort_column_id (column, CAT_LABEL);

renderer = gtk_cell_renderer_text_new();
column = gtk_tree_view_column_new_with_attributes(_("# POIs"), renderer, "text", CAT_POI_CNT, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);
gtk_tree_view_column_set_sort_column_id (column, CAT_POI_CNT);

renderer = gtk_cell_renderer_text_new();
g_object_set(G_OBJECT(renderer), "ellipsize", PANGO_ELLIPSIZE_END, NULL);
column = gtk_tree_view_column_new_with_attributes(_("Description"), renderer, "text", CAT_DESC, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);

g_signal_connect(G_OBJECT(btn_edit), "clicked", G_CALLBACK(category_edit_cb), tree_view);
g_signal_connect(G_OBJECT(btn_add), "clicked", G_CALLBACK(category_add_cb), tree_view);

gtk_window_set_default_size(GTK_WINDOW(dialog), 710, 350);

gtk_widget_show_all(dialog);
gtk_dialog_run(GTK_DIALOG(dialog));
gtk_widget_destroy(dialog);

return TRUE;
}

gboolean 
poi_delete_confirm(GtkWidget *widget, delete_poi *dpoi)
{
GtkWidget *dialog;
guint i;
gchar *buffer;

buffer = g_strdup_printf("%s\n%s", _("Delete POI?"), dpoi->txt_label);
dialog = hildon_note_new_confirmation(GTK_WINDOW(mapp.mainwindow), buffer);
g_free(buffer);
i = gtk_dialog_run(GTK_DIALOG(dialog));
gtk_widget_destroy(GTK_WIDGET(dialog));

if (i == GTK_RESPONSE_OK) {
	if (poi_delete(dpoi->id)==FALSE) {
		popup_error(mapp.mainwindow, _("Problem deleting POI"));
	} else {
		gtk_widget_hide_all(dpoi->dialog);
		map_poi_cache_clear();
		map_force_redraw();
	}
}
return TRUE;
}

static gboolean
poi_search_do_cb(GtkWidget *widget, gpointer data)
{
gchar *s;
guint cid;
GtkTreeIter iter;
search_dialog *sd=(search_dialog *)data;

if (gtk_combo_box_get_active_iter(GTK_COMBO_BOX(sd->cmb_category), &iter)==TRUE)
	gtk_tree_model_get(gtk_combo_box_get_model(GTK_COMBO_BOX(sd->cmb_category)), &iter, 0, &cid, -1);
else
	cid=-1;

if ((strlen(gtk_entry_get_text(GTK_ENTRY(sd->search_entry)))<2) && (cid==-1)) {
	popup_error(sd->dialog, _("Query string too short. Minimum 2 characters."));
	return TRUE;
}

gtk_tree_view_set_model(GTK_TREE_VIEW(sd->list), NULL);

s=g_strdup(gtk_entry_get_text(GTK_ENTRY(sd->search_entry)));
if (poi_search((cid==-1) ? POI_SEARCH_TEXT : POI_SEARCH_TEXT_CAT, sd->lat, sd->lon, s, cid, &sd->store)!=FALSE) {
	gtk_tree_view_set_model(GTK_TREE_VIEW(sd->list), GTK_TREE_MODEL(sd->store));
}
g_free(s);

return TRUE;
}

static gboolean
poi_goto_cb(GtkWidget *widget, gpointer data)
{
GtkTreeIter iter;
GtkTreeModel *model;
gdouble lat, lon;
search_dialog *sd=(search_dialog *)data;

if (!gtk_tree_selection_get_selected(gtk_tree_view_get_selection(GTK_TREE_VIEW(sd->list)), NULL, &iter)) {
	popup_error(sd->dialog, _("Select one POI from the list."));
	return TRUE;
}

model=gtk_tree_view_get_model(GTK_TREE_VIEW(sd->list));
if (!model)
	return TRUE;

gtk_tree_model_get(model, &iter, ITEM_LAT, &lat, -1);
gtk_tree_model_get(model, &iter, ITEM_LON, &lon, -1);

map_center_latlon(lat, lon);
map_update_location_from_center();

return TRUE;
}

static gboolean
poi_route_to_cb(GtkWidget *widget, gpointer data)
{
GtkTreeIter iter;
GtkTreeModel *model;
search_dialog *sd=(search_dialog *)data;

if (!gtk_tree_selection_get_selected(gtk_tree_view_get_selection(GTK_TREE_VIEW(sd->list)), NULL, &iter)) {
	popup_error(sd->dialog, _("Select one POI from the list."));
	return TRUE;
}

model=gtk_tree_view_get_model(GTK_TREE_VIEW(sd->list));
if (!model)
	return TRUE;

_dest.valid=TRUE;
gtk_tree_model_get(model, &iter, ITEM_LAT, &_dest.lat, -1);
gtk_tree_model_get(model, &iter, ITEM_LON, &_dest.lon, -1);

map_update_location_from_center();

return TRUE;
}

gboolean
poi_search_dialog(GtkListStore *store, poi_info *poi, gdouble lat, gdouble lon)
{
GtkWidget *sw, *hbox, *label, *btn_search, *btn_goto, *btn_route_to;
GtkTreeViewColumn *column;
GtkCellRenderer *renderer;
GtkTreeIter iter;
gboolean selected = FALSE;

sd.dialog=gtk_dialog_new_with_buttons(_("Select POI"),
		GTK_WINDOW(mapp.mainwindow),
		GTK_DIALOG_MODAL, 
		GTK_STOCK_OK,
		GTK_RESPONSE_ACCEPT,
		GTK_STOCK_CANCEL,
		GTK_RESPONSE_REJECT, NULL);
sd.lat=lat;
sd.lon=lon;

gtk_container_add(GTK_CONTAINER(GTK_DIALOG(sd.dialog)->action_area),
	btn_goto = gtk_button_new_with_label(_("Go")));

gtk_container_add(GTK_CONTAINER(GTK_DIALOG(sd.dialog)->action_area),
	btn_route_to = gtk_button_new_with_label(_("Route to")));

hbox = gtk_hbox_new(FALSE, 4),
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(sd.dialog)->vbox), hbox, FALSE, FALSE, 0);

label=gtk_label_new(_("Search:"));
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

sd.cmb_category = category_combo_new();
gtk_box_pack_start(GTK_BOX(hbox), sd.cmb_category, FALSE, FALSE, 0);
poi_category_combo_populate(sd.cmb_category, 0, TRUE);

sd.search_entry=gtk_entry_new();
gtk_box_pack_start(GTK_BOX(hbox), sd.search_entry, TRUE, TRUE, 0);

btn_search=gtk_button_new_with_label(_("Search"));
gtk_box_pack_start(GTK_BOX(hbox), btn_search, FALSE, FALSE, 0);

sw = gtk_scrolled_window_new(NULL, NULL);
gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw), GTK_SHADOW_ETCHED_IN);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(sd.dialog)->vbox), sw, TRUE, TRUE, 0);

if (store!=NULL) {
	sd.list=gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
	sd.store=store;
	g_object_unref(G_OBJECT(store));
} else {
	sd.list=gtk_tree_view_new();
	sd.store=NULL;
}
gtk_container_add(GTK_CONTAINER(sw), sd.list);

gtk_tree_selection_set_mode(gtk_tree_view_get_selection (GTK_TREE_VIEW(sd.list)), GTK_SELECTION_SINGLE);
gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(sd.list), TRUE);
gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(sd.list), TRUE);
gtk_tree_view_set_search_column(GTK_TREE_VIEW(sd.list), ITEM_LABEL);

renderer = gtk_cell_renderer_text_new();
column = gtk_tree_view_column_new_with_attributes(_("Location"), renderer, "text", ITEM_LATLON, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(sd.list), column);

renderer = gtk_cell_renderer_text_new();
column = gtk_tree_view_column_new_with_attributes(_("Label"), renderer, "text", ITEM_LABEL, NULL);
gtk_tree_view_column_set_sort_column_id (column, ITEM_LABEL);
gtk_tree_view_append_column(GTK_TREE_VIEW(sd.list), column);

renderer = gtk_cell_renderer_text_new();
column = gtk_tree_view_column_new_with_attributes(_("Category"), renderer, "text", ITEM_CATLAB, NULL);
gtk_tree_view_column_set_sort_column_id (column, ITEM_CATLAB);
gtk_tree_view_append_column(GTK_TREE_VIEW(sd.list), column);

g_signal_connect(G_OBJECT(btn_search), "clicked", G_CALLBACK(poi_search_do_cb), &sd);
g_signal_connect(G_OBJECT(btn_goto), "clicked", G_CALLBACK(poi_goto_cb), &sd);
g_signal_connect(G_OBJECT(btn_route_to), "clicked", G_CALLBACK(poi_route_to_cb), &sd);

gtk_window_set_default_size(GTK_WINDOW(sd.dialog), 710, 350);

gtk_widget_show_all(sd.dialog);

while (GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(sd.dialog))) {
	if (gtk_tree_selection_get_selected(gtk_tree_view_get_selection(GTK_TREE_VIEW(sd.list)), NULL, &iter)) {
		gtk_tree_model_get(GTK_TREE_MODEL(sd.store), &iter, 
			ITEM_ID, &(poi->poi_id), 
			ITEM_CATID, &(poi->cat_id), 
			ITEM_LAT, &(poi->lat),
			ITEM_LON, &(poi->lon),
			ITEM_LABEL, &(poi->label),
			ITEM_DESC, &(poi->desc),
			-1);
		selected = TRUE;
		break;
	} else {
		popup_error(sd.dialog, _("Select one POI from the list."));
	}
}

g_object_unref(G_OBJECT(store));
gtk_widget_destroy(sd.dialog);
return selected;
}

gboolean 
poi_select(guint unitx, guint unity, guint range, poi_info *poi)
{
GtkListStore *store=NULL;
guint num_pois;
GtkTreeIter iter;
gdouble lat, lon;

if (poi_get_list_near_unit(unitx, unity, range, &store, &num_pois)==FALSE)
	return FALSE;

switch (num_pois) {
case 0:
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("No POIs found."));
	g_object_unref(G_OBJECT(store));
	return FALSE;
break;
case 1:
	gtk_tree_model_get_iter_first(GTK_TREE_MODEL(store), &iter);
	gtk_tree_model_get(GTK_TREE_MODEL(store),
			&iter,
			ITEM_ID, &(poi->poi_id),
			ITEM_CATID, &(poi->cat_id),
			ITEM_LAT, &(poi->lat),
			ITEM_LON, &(poi->lon),
			ITEM_LABEL, &(poi->label),
			ITEM_DESC, &(poi->desc), -1);
	g_object_unref(G_OBJECT(store));
	return TRUE;
break;
}

unit2latlon(unitx, unity, lat, lon);
return poi_search_dialog(store, poi, lat, lon);
}

gboolean 
poi_edit_cat(GtkWidget * widget, PoiCategoryEditInfo * data)
{
if (poi_category_list())
	poi_category_combo_populate(data->cmb_category, data->cat_id, FALSE);
return TRUE;
}

/**
 * Url click callback
 */
static void
poi_info_url_clicked_cb(GtkHTML *html, const gchar *url, gpointer data)
{
#ifdef WITH_OSSO
osso_rpc_run_with_defaults(mapper_app.osso, "osso_browser",
	OSSO_BROWSER_OPEN_NEW_WINDOW_REQ, NULL,
	DBUS_TYPE_STRING, url, DBUS_TYPE_BOOLEAN, FALSE, DBUS_TYPE_INVALID);
#else
gtk_show_uri(NULL, url, GDK_CURRENT_TIME, NULL);
#endif
}


/**
 * Callback to load requested url
 */
static void
poi_info_url_requested_cb(GtkHTML *html, const char *url, GtkHTMLStream *stream)
{
g_debug("URLReq: %s\n", url);
gtk_html_stream_close(stream, GTK_HTML_STREAM_ERROR);
}

static void
poi_info_title_cb(GtkHTML *html, const gchar *title, gpointer data)
{
gtk_window_set_title(GTK_WINDOW(data), title);
}

#define WRITE_HTML(html, args...) \
	{ gchar *ph; \
	ph=g_markup_printf_escaped(html, ##args); \
	gtk_html_write(GTK_HTML(info), s, ph, strlen(ph)); \
	g_free(ph); }

/**
 * Display a nice POI information dialog
 *
 */
gboolean
poi_info_dialog(GtkWidget *window, guint poi_id)
{
GtkWidget *dialog;
GtkWidget *info;
GtkWidget *sw;
poi_info *p;
GtkHTMLStream *s;
gint ls;

p=poi_get_by_id(poi_id);
if (!p) {
	g_debug("Requested info for POI %d that does not exist!\n", poi_id);
	return FALSE;
}

dialog=gtk_dialog_new_with_buttons(_("POI"),
			GTK_WINDOW(window), GTK_DIALOG_MODAL,
			GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
			NULL);

/* XXX: Add edit button */

info=gtk_html_new();
gtk_html_set_editable(GTK_HTML(info), FALSE);
gtk_html_allow_selection(GTK_HTML(info), TRUE);
sw=gtk_scrolled_window_new(NULL, NULL);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
gtk_container_add(GTK_CONTAINER(sw), info);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), sw, TRUE, TRUE, 0);

g_signal_connect(G_OBJECT(info), "link_clicked", G_CALLBACK(poi_info_url_clicked_cb), NULL);
g_signal_connect(G_OBJECT(info), "url_requested", G_CALLBACK(poi_info_url_requested_cb), NULL);
g_signal_connect(G_OBJECT(info), "title_changed", G_CALLBACK(poi_info_title_cb), dialog);

s=gtk_html_begin(GTK_HTML(info));
ls=strlen(p->label);

/* XXX: Format lat/lon according to settings */

WRITE_HTML("<html><meta http-equiv=\"Content-Type\" content=\"text/html; charset=utf-8\">");
WRITE_HTML("<head><title>POI - %s</title></head><body><h3>%s</h3><i>%s</i></b><br/><table>" \
	"<tr><th align=\"left\">Location:</th><td>%.5f, %.5f</td></tr>",
	ls>0 ? p->label : _("(Unknown)"), 
	ls>0 ? p->label : _("(Unknown)"), p->cat_label, 
	p->lat, p->lon);

if (p->postal_code)
	WRITE_HTML("<tr><th align=\"left\">Postal Code:</th><td>%s</td></tr>", p->postal_code);
if (p->url)
	WRITE_HTML("<tr><th align=\"left\">Link:</th><td><a href=\"%s\">Web site</a></td></tr>", p->url);
if (p->desc)
	WRITE_HTML("<tr><th align=\"left\" colspan=\"2\">Description:</th><td colspan=\"2\">%s</td></tr>", p->desc);
if (p->osm_id>0)
	WRITE_HTML("<tr><th align=\"left\">OSM:</th><td><a href=\"http://www.openstreetmap.org/browse/node/%u\">View/Edit on OpenStreetMap.org</a></td></tr>", p->osm_id);
if (ls>0) {
	gchar *elabel;

	elabel=gnome_vfs_escape_string(p->label);
	WRITE_HTML("<tr><th>Search on:</th><td><ul>");
	WRITE_HTML("<li><a href=\"http://www.google.com/search?q=%s\">Google</a></li>", elabel);
	WRITE_HTML("<li><a href=\"http://en.wikipedia.org/wiki/Special:Search?search=%s\">Wikipedia</a></li>", elabel);
	WRITE_HTML("</ul></td></tr>");
	g_free(elabel);
}

WRITE_HTML("</table><br/>");

# if 0
if (p->image)
	WRITE_HTML("<img src=\"%s\">", p->image);
#endif

WRITE_HTML("</body></html>");
gtk_html_end(GTK_HTML(info), s, GTK_HTML_STREAM_OK);

poi_free(p);

gtk_window_set_default_size(GTK_WINDOW(dialog), 710, 350);
gtk_widget_show_all(dialog);
gtk_dialog_run(GTK_DIALOG(dialog));
gtk_widget_destroy(dialog);

return TRUE;
}

/**
 * poi_edit_dialog
 *
 * Edit or Add POI with given information in poi_info
 *
 */
gboolean 
poi_edit_dialog(POIAction action, poi_info *poi)
{
gchar slat1[10], slon1[10];
gchar *p_latlon;
gboolean res=FALSE;
GtkWidget *dialog;
GtkWidget *table;
GtkWidget *label;
GtkWidget *txt_label;
GtkWidget *cmb_category;
GtkWidget *txt_desc;
GtkWidget *txt_postal_code, *txt_url;
GtkWidget *btn_delete = NULL;
GtkWidget *btn_catedit;
GtkWidget *hbox;
GtkWidget *txt_scroll;
GtkTextBuffer *desc_txt;
GtkTextIter begin, end;
delete_poi dpoi = { NULL, NULL, 0 };
PoiCategoryEditInfo pcedit;
gchar tmp1[16], tmp2[16];

/* Fatal, poi must be set */

if (!poi)
	return FALSE;

dialog = gtk_dialog_new_with_buttons(action == ACTION_EDIT_POI ? _("Edit POI") : _("Add POI"),
		     GTK_WINDOW(mapp.mainwindow),
		     GTK_DIALOG_MODAL,
		     GTK_STOCK_OK,
		     GTK_RESPONSE_ACCEPT,
		     GTK_STOCK_CANCEL,
		     GTK_RESPONSE_REJECT, NULL);

if (action == ACTION_EDIT_POI) {
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->action_area), btn_delete = gtk_button_new_with_label(_("Delete")));

	dpoi.dialog=dialog;
	dpoi.txt_label=g_strdup(poi->label);
	dpoi.id=poi->poi_id;

	g_signal_connect(G_OBJECT(btn_delete), "clicked", G_CALLBACK(poi_delete_confirm), &dpoi);
}

/* Set the p_latlon string. */
lat_format(_degformat, poi->lat, tmp1);
lon_format(_degformat, poi->lon, tmp2);
p_latlon=g_strdup_printf("%s, %s", tmp1, tmp2);

gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), table = gtk_table_new(6, 4, FALSE), TRUE, TRUE, 0);

gtk_table_attach(GTK_TABLE(table), label = gtk_label_new(_("Lat, Lon")), 0, 1, 0, 1, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);

gtk_table_attach(GTK_TABLE(table), label = gtk_label_new(p_latlon), 1, 3, 0, 1, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 0.0f, 0.5f);

gtk_table_attach(GTK_TABLE(table), label = gtk_label_new(_("Label")), 0, 1, 1, 2, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
gtk_table_attach(GTK_TABLE(table), txt_label = gtk_entry_new(), 1, 2, 1, 2, GTK_EXPAND | GTK_FILL, 0, 2, 4);

gtk_table_attach(GTK_TABLE(table), label = gtk_label_new(_("Category")), 0, 1, 3, 4, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
gtk_table_attach(GTK_TABLE(table), hbox = gtk_hbox_new(FALSE, 4), 1, 2, 3, 4, GTK_EXPAND | GTK_FILL, 0, 2, 4);

gtk_box_pack_start(GTK_BOX(hbox), cmb_category = category_combo_new(), FALSE, FALSE, 4);

gtk_box_pack_start(GTK_BOX(hbox), btn_catedit = gtk_button_new_with_label(_("Edit Categories...")),	FALSE, FALSE, 4);

gtk_table_attach(GTK_TABLE(table), label = gtk_label_new(_("Description")), 0, 1, 5, 6, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);

txt_scroll = gtk_scrolled_window_new(NULL, NULL);
gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(txt_scroll), GTK_SHADOW_IN);
gtk_table_attach(GTK_TABLE(table), txt_scroll, 1, 2, 5, 6, GTK_EXPAND | GTK_FILL, 0, 2, 4);

gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(txt_scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

txt_desc=gtk_text_view_new();
gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(txt_desc), GTK_WRAP_WORD);

gtk_container_add(GTK_CONTAINER(txt_scroll), txt_desc);
gtk_widget_set_size_request(GTK_WIDGET(txt_scroll), 400, 60);

desc_txt=gtk_text_view_get_buffer(GTK_TEXT_VIEW(txt_desc));

/* label */
if (poi->label)
	gtk_entry_set_text(GTK_ENTRY(txt_label), poi->label);
if (poi->desc)
	gtk_text_buffer_set_text(desc_txt, poi->desc, -1);

poi_category_combo_populate(cmb_category, poi->cat_id, FALSE);

/* Connect Signals */
pcedit.cmb_category=cmb_category;
pcedit.cat_id=poi->cat_id;

g_signal_connect(G_OBJECT(btn_catedit), "clicked", G_CALLBACK(poi_edit_cat), &pcedit);
gtk_widget_show_all(dialog);

while (GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(dialog))) {
	GtkTreeIter iter;

	if (strlen(gtk_entry_get_text(GTK_ENTRY(txt_label))))
		poi->label=g_strdup(gtk_entry_get_text(GTK_ENTRY(txt_label)));
	else {
		popup_error(dialog, _("Please specify a name for the POI."));
		continue;
	}

	if (!gtk_combo_box_get_active_iter(GTK_COMBO_BOX(cmb_category), &iter)) {
		popup_error(dialog, _("Please specify a category for the POI."));
		continue;
	}

	gtk_text_buffer_get_bounds(desc_txt, &begin, &end);
	poi->desc=gtk_text_buffer_get_text(desc_txt, &begin, &end, TRUE);

	gtk_tree_model_get(gtk_combo_box_get_model(GTK_COMBO_BOX(cmb_category)), &iter, 0, &poi->cat_id, -1);

	if (poi_update_or_add(poi)==FALSE) {
		popup_error(mapp.mainwindow, _("Problem updating POI"));
	} else {
		map_poi_cache_clear();
		map_force_redraw();
		res=TRUE;
	}
	break;
}

poi_free(poi);
g_free(dpoi.txt_label);
g_free(p_latlon);

gtk_widget_destroy(dialog);

return res;
}

static gboolean
poi_quick_button_cb(GtkWidget *button, gpointer data)
{
poi_info *p;
gchar *txt;
poi_quick_data *qpdata=(poi_quick_data *)data;

g_return_val_if_fail(data, TRUE);

p=poi_new();
p->cat_id=GPOINTER_TO_INT(g_object_get_data(G_OBJECT(button), "pid"));

if (strlen(gtk_entry_get_text(GTK_ENTRY(qpdata->label)))>0) {
	p->label=g_strdup(gtk_entry_get_text(GTK_ENTRY(qpdata->label)));
} else {
	p->label=g_strdup("");
}

if (qpdata->fixed==TRUE) {
	p->lat=qpdata->lat;
	p->lon=qpdata->lon;
} else {
	p->lat=_gps->data.lat;	
	p->lon=_gps->data.lon;
}
p->desc=g_strdup("Quick POI, update information please.");

/* Add a text break to the current track */
txt=g_strdup_printf("QP(%d): %f %f %s", p->cat_id, p->lat, p->lon, p->label);
path_insert_mark_text(_track, txt);

/* poi_add frees the label and desc so strdup */
if (poi_add(p)==FALSE) {
	popup_error(mapp.mainwindow, _("Problem adding POI"));
} else {
	map_poi_cache_clear();
	map_force_redraw();
	if (qpdata->close==TRUE)
		gtk_widget_destroy(qpdata->dialog);
	if (qpdata->note)
		gtk_notebook_set_current_page(GTK_NOTEBOOK(qpdata->note), qpdata->tab);
	hildon_banner_show_information(mapp.mainwindow, NULL, _("POI added"));
}
poi_free(p);
return TRUE;
}

static void
poi_button_set_icon(GtkWidget *button, node_type_t t)
{
const gchar *iname;
GdkPixbuf *icon=NULL;
GtkSettings *settings;

iname=poi_get_icon_from_type(t);
if (iname)
	icon=poi_get_icon(iname, TRUE);
if (!icon)
	return;

gtk_button_set_image(GTK_BUTTON(button), gtk_image_new_from_pixbuf(icon));
#if GTK_CHECK_VERSION(2,10,0)
gtk_button_set_image_position(GTK_BUTTON(button), GTK_POS_TOP);
#endif
settings=gtk_widget_get_settings(button);
g_object_set(G_OBJECT(settings), "gtk-button-images", TRUE, NULL);
}

/**
 * Return a vbox filled with POI_QUICK_BUTTONS POI buttons and one for other POI, 
 * with an optinal POI label.
 */
GtkWidget *
poi_quick_button_box(poi_quick_data *qpdata)
{
GtkWidget *table, *vbox;
GtkWidget *buttons[POI_QUICK_BUTTONS];
GtkWidget *otherbtn;
guint x, y;

vbox=gtk_vbox_new(FALSE, 6);
table=gtk_table_new(QPBS_Y, QPBS_X, TRUE);
gtk_table_set_col_spacings(GTK_TABLE(table), 6);
gtk_table_set_row_spacings(GTK_TABLE(table), 6);

for (y=0;y<QPBS_Y;y++) {
	for (x=0;x<QPBS_X;x++) {
		guint p=y*(QPBS_X)+x;

		buttons[p]=gtk_button_new_with_label(quick_poi_categories[p].name);
		poi_button_set_icon(buttons[p], quick_poi_categories[p].type);
		gtk_table_attach(GTK_TABLE(table), buttons[p], x, x+1, y, y+1, GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 6, 6);
		g_object_set_data(G_OBJECT(buttons[p]), "pid", GINT_TO_POINTER(quick_poi_categories[p].type));
		g_signal_connect(G_OBJECT(buttons[p]), "clicked", G_CALLBACK(poi_quick_button_cb), qpdata);
	}
}

gtk_box_pack_start(GTK_BOX(vbox), table, TRUE, TRUE, 0);

/* Other POI button */
otherbtn=gtk_button_new_with_label("Other");
poi_button_set_icon(otherbtn, NODE_AMENITY_GENERIC);
gtk_box_pack_start(GTK_BOX(vbox), otherbtn, TRUE, FALSE, 6);
g_object_set_data(G_OBJECT(otherbtn), "pid", GINT_TO_POINTER(NODE_AMENITY_GENERIC));
g_signal_connect(G_OBJECT(otherbtn), "clicked", G_CALLBACK(poi_quick_button_cb), qpdata);

gtk_box_pack_start(GTK_BOX(vbox), qpdata->label = gtk_entry_new(), TRUE, FALSE, 0);

return vbox;
}

/**
 * Show simple dialog for adding a POI.
 * Uses the above helper to get the buttons. The location is given as paramters.
 */
gboolean
poi_quick_dialog(gdouble lat, gdouble lon)
{
GtkWidget *box;

qp.dialog=gtk_dialog_new_with_buttons(_("Quick POI"),
			GTK_WINDOW(mapp.mainwindow),
			GTK_DIALOG_MODAL,
			GTK_STOCK_CANCEL,
			GTK_RESPONSE_REJECT,
			NULL);

qp.fixed=TRUE;
qp.lat=lat;
qp.lon=lon;
qp.close=TRUE;

box=poi_quick_button_box(&qp);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(qp.dialog)->vbox), box, TRUE, TRUE, 0);

gtk_widget_show_all(qp.dialog);

if (gtk_dialog_run(GTK_DIALOG(qp.dialog))==GTK_RESPONSE_REJECT)
	gtk_widget_destroy(qp.dialog);

return TRUE;
}
