#include <config.h>

#ifndef _MAPPER_GPS_H
#define _MAPPER_GPS_H

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stddef.h>
#include <locale.h>
#include <math.h>
#include <errno.h>
#include <sys/wait.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/ip.h> 

#ifdef WITH_BLUEZ
#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>
#include <bluetooth/rfcomm.h>
#endif

#include "path.h"
#include "mapper-types.h"
#include "gpsdata.h"

#define GPS_READ_BUF_SIZE	(1024)

/**
 * Types of GPS data sources
 */
typedef enum {
	GPS_IO_TYPE_MIN=0,
	GPS_IO_RFCOMM=1,	/* Bluetooth rfcomm socket */
	GPS_IO_HILDON_DBUS,	/* Request rfcomm using hildon bluetooth D-Bus */
	GPS_IO_BLUEZ_DBUS,	/* Request rfcomm using bluez bluetooth D-Bus*/
	GPS_IO_FILE,		/* File or (serial) device node */
	GPS_IO_TCP,			/* TCP */
	GPS_IO_GPSD,		/* TCP connection to gpsd (as TCP, but will send a request for NMEA mode */
	GPS_IO_GYPSY,		/* Gypsy events */
	GPS_IO_LIBLOCATION,	/* Liblocation events */
	GPS_IO_SIMULATION,	/* Random movement */
	GPS_IO_TYPE_MAX
} GpsIOSourceType;

typedef struct _GpsTypes GpsTypes;
struct _GpsTypes {
	GpsIOSourceType type;
	const gchar *name;
	gboolean scan;		/* Can scan for device */
	gboolean address;	/* Needs address */
	gboolean port;		/* Needs port */
	guint port_def;		/* Default port */
	const gchar *address_def; /* Default mac/ip/device */
};

/** This enumerated type defines the possible connection states. */
typedef enum {
	RCVR_OFF,
	RCVR_DOWN,
	RCVR_UP,
	RCVR_FIXED
} GpsConnState;

/**
 * GPS IO struct
 *
 */
typedef struct _GpsIO GpsIO;
struct _GpsIO {
	gint fd;
	gchar *address; /* BT MAC, File path or IP */
	guint port;
	GpsIOSourceType type;
	GpsConnState conn;
	GIOChannel *channel;
	DBusConnection *dbus_conn;
	DBusGProxy *rfcomm_req_proxy;
#ifdef WITH_BLUEZ
	struct sockaddr_rc rcvr_addr_rc;
#endif
	struct sockaddr_in rcvr_addr_in;
	/* Channel callback IDs*/
	guint connect_sid;
	guint error_sid;
	guint input_sid;
	guint clater_sid;
	guint errors; /* G_IO_STATUS_ERRROR counter */
	guint again; /* G_IO_STATUS_AGAIN counter */
	/* Input buffer */
	gchar buffer[GPS_READ_BUF_SIZE];
	gchar *curr;
	gchar *last;
	/* Latest NMEA line, ready for parsing */
	guint nmea_cnt;
	gchar *nmea;
};

/**
 * Gps 
 */
typedef struct _Gps Gps;
struct _Gps {
	gchar *name;	/* Name of the connection */
	GpsIO io;
	GpsData data;
	/* Event callbacks */

	/* On errors */
	gboolean(* connection_error) (Gps *gps, const gchar *error_str);

	/* Connection retry */
	gboolean(* connection_retry) (Gps *gps, const gchar *error_str);

	/* Connection progress */
	gboolean(* connection_progress)(Gps *gps, gdouble fix);

	/* New location */
	gboolean(* update_location)(Gps *gps);
	gboolean(* update_satellite)(Gps *gps);
	gboolean(* update_info)(Gps *gps);
};

Gps *_gps;
gint _gmtoffset;

const GpsTypes *gps_get_supported_types(void);

Gps *gps_new(GpsIOSourceType type);
void gps_clear(Gps *gps);
gboolean gps_type_needs_config(GpsIOSourceType type);
void gps_set_address(Gps *gps, gchar *address, guint port);
void gps_set_type(Gps *gps, GpsIOSourceType type);
void gps_free(Gps *gps);
void gps_data_integerize(GpsData *gpsdata);
gboolean gps_connect(Gps *gps);
gboolean gps_connect_now(Gps *gps);
void gps_connect_later(Gps *gps);
void gps_disconnect(Gps *gps);

gboolean gps_location_update(Gps *gps);

#endif
