/*
 * poi.c - POI representation
 * Copyright (C) 2010 Collabora Ltd
 * @author Cosimo Alfarano <cosimo.alfarano@collabora.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include "config.h"
#include "poi.h"

#include "conf.h"

#include <glib.h>
#include <glib-object.h>
#include <clutter/clutter.h>

typedef struct
{
  guint id;
  gchar *name;
  gchar *feat;
  gchar *feat_type;

  gdouble lat, lon;
  gint srid;

  /* misc tags */
  GHashTable *tags;

  /* clutter icons */
  ClutterActor *icon_small, *icon_big;

} MapBuddyPoiPrivate;

struct _MapBuddyPoi
{
  GObject base;

  MapBuddyPoiPrivate *priv;
};

struct _MapBuddyPoiClass
{
  GObjectClass parent_class;
};

G_DEFINE_TYPE (MapBuddyPoi, map_buddy_poi, G_TYPE_OBJECT);

#define SPATIALITE_OSM_SRID 4326
#define POI_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE((obj), MAP_BUDDY_TYPE_POI, \
                               MapBuddyPoiPrivate))

static void
map_buddy_poi_dispose (GObject *object)
{
  MapBuddyPoi *self = MAP_BUDDY_POI (object);

  g_free (self->priv->name);
  self->priv->name = NULL;

  g_free (self->priv->feat);
  self->priv->feat = NULL;

  g_free (self->priv->feat_type);
  self->priv->feat_type = NULL;

  if (self->priv->tags != NULL)
    {
      g_hash_table_unref (self->priv->tags);
      self->priv->tags = NULL;
    }

  if (self->priv->icon_small != NULL)
    {
      g_object_unref (self->priv->icon_small);
      self->priv->icon_small = NULL;
    }

  if (self->priv->icon_big != NULL)
    {
      g_object_unref (self->priv->icon_big);
      self->priv->icon_big = NULL;
    }

  G_OBJECT_CLASS (map_buddy_poi_parent_class)->dispose (object);
}

static void
map_buddy_poi_class_init (MapBuddyPoiClass *cls)
{
  g_type_class_add_private (cls, sizeof (MapBuddyPoiPrivate));

  GObjectClass *object_class = G_OBJECT_CLASS (cls);
  object_class->dispose = map_buddy_poi_dispose;
}

static void
map_buddy_poi_init (MapBuddyPoi *poi)
{
  MapBuddyPoiPrivate *priv = POI_GET_PRIVATE (poi);
  poi->priv = priv;
  /* use it as default SRID, no need of adding API for it, for the moment.
   * If a different/more flexible way of import will be added, it will be
   * usedful */
  poi->priv->srid = SPATIALITE_OSM_SRID;
}

static void
map_buddy_poi_set_name (MapBuddyPoi *self,
    const gchar *name)
{
  g_return_if_fail (MAP_BUDDY_IS_POI (self));
  g_return_if_fail (self->priv->name == NULL);
  g_return_if_fail (name != NULL && name[0] != '\0');

  self->priv->name = g_strdup (name);
}

/** map_buddy_poi_set_feature:
 * @self: poi instance
 * @feat: a non-NULL/non-empty string for a OSM feature
 * @feat_type: a non-NULL/non-empty string for a OSM feature type
 *
 * set @feature and @feature_type to @self
 * It also set a big and a small icons (ClutterActors) for @self, if it's
 * possible.
 */
void
map_buddy_poi_set_feature (MapBuddyPoi *self,
    const gchar *feat,
    const gchar *feat_type)
{
  MapBuddyConf *conf;
  ClutterActor *icon_small = NULL;
  gchar *icon_path_small = NULL;
  GError *err = NULL;

  g_return_if_fail (MAP_BUDDY_IS_POI (self));
  g_return_if_fail (self->priv->feat == NULL);
  g_return_if_fail (feat != NULL && feat[0] != '\0');
  g_return_if_fail (self->priv->feat_type == NULL);
  g_return_if_fail (feat_type != NULL && feat_type[0] != '\0');

  self->priv->feat = g_strdup (feat);
  self->priv->feat_type = g_strdup (feat_type);

  conf = map_buddy_conf_dup ();

  icon_path_small = map_buddy_conf_get_osm_icon_for (conf, feat, feat_type,
      &err);
  if (err != NULL)
    {
      g_printerr ("%s: %s\n", G_STRFUNC, err->message);
      g_error_free (err);

      goto out;
    }
  icon_small = clutter_texture_new_from_file (icon_path_small, &err);
  if (err != NULL)
    {
      g_printerr ("%s: %s: %s\n", G_STRFUNC, icon_path_small, err->message);
      g_error_free (err);

      goto out;
    }

  self->priv->icon_small = icon_small;
  /* currently not used, just ref it as it were the right icon size */
  self->priv->icon_big = g_object_ref (icon_small);

out:
  g_free (icon_path_small);

  g_object_unref (conf);

  return;
}

const gchar *
map_buddy_poi_get_name (MapBuddyPoi *self)
{
  g_return_val_if_fail (MAP_BUDDY_IS_POI (self), NULL);

  return self->priv->name;
}

const gchar *
map_buddy_poi_get_feature (MapBuddyPoi *self)
{
  g_return_val_if_fail (MAP_BUDDY_IS_POI (self), NULL);

  return self->priv->feat;
}

const gchar *
map_buddy_poi_get_feature_type (MapBuddyPoi *self)
{
  g_return_val_if_fail (MAP_BUDDY_IS_POI (self), NULL);

  return self->priv->feat_type;
}

ClutterActor *
map_buddy_poi_get_icon_small (MapBuddyPoi *self)
{
  g_return_val_if_fail (MAP_BUDDY_IS_POI (self), NULL);

  return self->priv->icon_small;
}

ClutterActor *
map_buddy_poi_get_icon_big (MapBuddyPoi *self)
{
  g_return_val_if_fail (MAP_BUDDY_IS_POI (self), NULL);

  return self->priv->icon_big;
}

gint
map_buddy_poi_get_id (MapBuddyPoi *self)
{
  g_return_val_if_fail (MAP_BUDDY_IS_POI (self), -1);

  return self->priv->id;
}

gint
map_buddy_poi_get_srid (MapBuddyPoi *self)
{
  g_return_val_if_fail (MAP_BUDDY_IS_POI (self), -1);

  return self->priv->srid;
}


gdouble
map_buddy_poi_get_lat (MapBuddyPoi *self)
{
  g_return_val_if_fail (MAP_BUDDY_IS_POI (self), 0);

  return self->priv->lat;
}

gdouble
map_buddy_poi_get_lon (MapBuddyPoi *self)
{
  g_return_val_if_fail (MAP_BUDDY_IS_POI (self), 0);

  return self->priv->lon;
}

GHashTable *
map_buddy_poi_get_tags (MapBuddyPoi *self)
{
  g_return_val_if_fail (MAP_BUDDY_IS_POI (self), NULL);

  return self->priv->tags;
}

static void
map_buddy_poi_set_id (MapBuddyPoi *self,
    gint id)
{
  g_return_if_fail (MAP_BUDDY_IS_POI (self));

  self->priv->id = id;
}

static void
map_buddy_poi_set_lat (MapBuddyPoi *self,
    gdouble lat)
{
  g_return_if_fail (MAP_BUDDY_IS_POI (self));
  g_return_if_fail (lat <= 90 && lat >= -90);

  self->priv->lat = lat;
}

static void
map_buddy_poi_set_lon (MapBuddyPoi *self,
    gdouble lon)
{
  g_return_if_fail (MAP_BUDDY_IS_POI (self));
  g_return_if_fail (lon <= 180 && lon >= -180);

  self->priv->lon = lon;
}

MapBuddyPoi *
map_buddy_poi_new (guint id,
    const gchar *name,
    gdouble lat,
    gdouble lon)
{
  MapBuddyPoi *ret;

  ret = g_object_new (MAP_BUDDY_TYPE_POI, NULL);
  map_buddy_poi_set_id (ret, id);
  map_buddy_poi_set_name (ret, name);
  map_buddy_poi_set_lat (ret, lat);
  map_buddy_poi_set_lon (ret, lon);

  return ret;
}

void
map_buddy_poi_set_tags (MapBuddyPoi *self,
    GHashTable *tags)
{
  MapBuddyPoiPrivate *priv = POI_GET_PRIVATE (self);

  g_return_if_fail (MAP_BUDDY_IS_POI (self));

  if (priv->tags != NULL)
    g_hash_table_unref (priv->tags);

  if (tags != NULL)
    g_hash_table_ref (tags);

  priv->tags = tags;
}
