/*
 * Copyright (C) 2009 Pierre-Luc Beaudoin <pierre-luc@pierlux.com>
 * Copyright (C) 2010 Collabora
 * @author Pierre-Luc Beaudoin <pierre-luc@pierlux.com>
 * @author Alban Crequy <alban.crequy@collabora.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include "config.h"

#include "contact-layer.h"

#include "connection-watcher.h"
#include "contact-watcher.h"

#include <clutter/clutter.h>
#include <champlain/champlain.h>
#include <glib.h>
#include <glib-object.h>
#include <cairo.h>
#include <math.h>
#include <string.h>
#include <telepathy-glib/account.h>
#include <telepathy-glib/contact.h>


struct _MapBuddyContactLayerPrivate
{
  McAccountManager *mc_account_mgr;
  OssoABookAggregator *abook_aggregator;
  ChamplainLayer *contacts_layer;
  ContactWatcher *contact_watcher;
};

G_DEFINE_TYPE (MapBuddyContactLayer, map_buddy_contact_layer,
    G_TYPE_OBJECT);

#define MAP_BUDDY_CONTACT_LAYER_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE((obj), MAP_BUDDY_TYPE_CONTACT_LAYER, \
                               MapBuddyContactLayerPrivate))

static void
action_started_cb (OssoABookTouchContactStarter *starter,
                   GtkWidget                    *dialog)
{
  gtk_dialog_response (GTK_DIALOG (dialog), GTK_RESPONSE_CLOSE);
}

static gboolean
on_contact_marker_press (ClutterActor *actor,
                         ClutterEvent *event,
                         gpointer user_data)
{
  GQuark abook_contact_quark = g_quark_from_static_string ("MapBuddyWindowAbookContact");
  GtkWidget *starter, *dialog;
  OssoABookContact *abook_contact = g_object_get_qdata (G_OBJECT (actor),
      abook_contact_quark);
  
  g_debug ("Clicked on contact %s\n",
      champlain_marker_get_text (CHAMPLAIN_MARKER (actor)));

  if (!abook_contact)
    return TRUE;

  dialog = g_object_new (GTK_TYPE_DIALOG,
      "has-separator", FALSE,
      NULL);
  gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
  starter = osso_abook_touch_contact_starter_new_with_contact (
      NULL, abook_contact);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox),
      starter);
  g_signal_connect (starter, "action-started",
      G_CALLBACK (action_started_cb), dialog);

  gtk_widget_show (starter);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);

  return TRUE;
}

static void
on_contact_added (ContactWatcher *contact_watcher,
                  TpAccount *account,
                  TpConnection *connection,
                  TpContact *contact,
                  double lat, double lon,
                  gpointer user_data)
{
  MapBuddyContactLayer *self = MAP_BUDDY_CONTACT_LAYER (user_data);
  MapBuddyContactLayerPrivate *priv =
    MAP_BUDDY_CONTACT_LAYER_GET_PRIVATE (self);
  GQuark marker_quark = g_quark_from_static_string ("MapBuddyWindowContactMarker");
  GQuark abook_contact_quark = g_quark_from_static_string ("MapBuddyWindowAbookContact");
  ClutterActor *marker;
  ClutterColor white = {0xff, 0xff, 0xff, 0xff};
  ClutterColor black = {0x00, 0x00, 0x00, 0xff};
  GList *abook_contacts;
  const gchar *text = NULL;
  McAccount *mc_account;
  OssoABookContact *abook_contact = NULL;

  g_debug ("on_contact_added for '%s'\n",
      tp_contact_get_identifier (contact));

  mc_account = mc_account_manager_get_account (priv->mc_account_mgr,
      tp_proxy_get_object_path (TP_PROXY (account)));

  g_debug ("TpAccount objpath = %s\n",
      tp_proxy_get_object_path (TP_PROXY (account)));

  abook_contacts = osso_abook_aggregator_find_contacts_for_im_contact 
    (priv->abook_aggregator, tp_contact_get_identifier (contact), mc_account);
  g_debug ("abook_contacts: %u\n", g_list_length (abook_contacts));
  if (abook_contacts != NULL)
    abook_contact = OSSO_ABOOK_CONTACT (abook_contacts->data);
  g_list_free (abook_contacts);


  marker = g_object_get_qdata (G_OBJECT (contact), marker_quark);
  if (marker != NULL)
    {
      champlain_base_marker_set_position (CHAMPLAIN_BASE_MARKER (marker),
          lon, lat);
      return;
    }

  marker = champlain_marker_new ();

  if (abook_contact)
    {
      text = osso_abook_contact_get_display_name (abook_contact);
      g_debug ("Got name from address book: %s\n", text);

      g_object_set_qdata (G_OBJECT (marker), abook_contact_quark, abook_contact);
    }
  else
    {
      text = tp_contact_get_alias (contact);
      g_debug ("Got alias: %s\n", text);
    }

  champlain_marker_set_text (CHAMPLAIN_MARKER (marker), text);
  champlain_marker_set_color (CHAMPLAIN_MARKER (marker), &white);
  champlain_marker_set_text_color (CHAMPLAIN_MARKER (marker), &black);
  champlain_base_marker_set_position (CHAMPLAIN_BASE_MARKER (marker),
      lon, lat);
  g_signal_connect (CLUTTER_ACTOR (marker), "button-press-event",
      G_CALLBACK (on_contact_marker_press), self);

  champlain_layer_add_marker (priv->contacts_layer,
      CHAMPLAIN_BASE_MARKER (marker));

  g_object_set_qdata_full (G_OBJECT (contact), marker_quark, marker,
      (GDestroyNotify) clutter_actor_destroy);
}

static void
on_contact_removed (ContactWatcher *contact_watcher,
                    TpContact *contact,
                    gpointer user_data)
{
  GQuark marker_quark = g_quark_from_static_string ("MapBuddyWindowContactMarker");

  g_debug ("on_contact_removed\n");

  g_object_set_qdata (G_OBJECT (contact), marker_quark, NULL);
}

static void
aggregator_ready_cb (OssoABookWaitable *aggregator,
                     const GError      *error,
                     gpointer           user_data)
{
  MapBuddyContactLayer *self = MAP_BUDDY_CONTACT_LAYER (user_data);
  MapBuddyContactLayerPrivate *priv =
    MAP_BUDDY_CONTACT_LAYER_GET_PRIVATE (self);

  g_debug ("Contact aggregator ready\n");

  priv->contact_watcher = contact_watcher_new ();
  g_signal_connect (priv->contact_watcher, "contact-added",
      G_CALLBACK (on_contact_added), self);
  g_signal_connect (priv->contact_watcher, "contact-removed",
      G_CALLBACK (on_contact_removed), self);
  contact_watcher_start (priv->contact_watcher);
}

static void
map_buddy_contact_layer_dispose (GObject *object)
{

  MapBuddyContactLayer *marker = MAP_BUDDY_CONTACT_LAYER (object);
  MapBuddyContactLayerPrivate *priv = marker->priv;

  if (priv->mc_account_mgr)
    {
      g_object_unref (priv->mc_account_mgr);
      priv->mc_account_mgr = NULL;
    }

  if (priv->contact_watcher)
    {
      g_object_unref (priv->contact_watcher);
      priv->contact_watcher = NULL;
    }

  G_OBJECT_CLASS (map_buddy_contact_layer_parent_class)->dispose (object);
}

static void
map_buddy_contact_layer_class_init (MapBuddyContactLayerClass *map_buddyClass)
{
  g_type_class_add_private (map_buddyClass, sizeof (MapBuddyContactLayerPrivate));

  GObjectClass *object_class = G_OBJECT_CLASS (map_buddyClass);
  object_class->dispose = map_buddy_contact_layer_dispose;
}

static void
map_buddy_contact_layer_init (MapBuddyContactLayer *contact_layer)
{
  MapBuddyContactLayerPrivate *priv =
    MAP_BUDDY_CONTACT_LAYER_GET_PRIVATE (contact_layer);
  contact_layer->priv = priv;

  priv->mc_account_mgr = mc_account_manager_new (tp_dbus_daemon_dup (NULL));
  priv->abook_aggregator =
    OSSO_ABOOK_AGGREGATOR (osso_abook_aggregator_get_default (NULL));
}

void
map_buddy_contact_layer_start (MapBuddyContactLayer *self,
                               ChamplainLayer *champlain_layer)
{
  MapBuddyContactLayerPrivate *priv =
    MAP_BUDDY_CONTACT_LAYER_GET_PRIVATE (self);
  priv->contacts_layer = champlain_layer;

  osso_abook_waitable_call_when_ready (OSSO_ABOOK_WAITABLE (priv->abook_aggregator),
      aggregator_ready_cb, self, NULL);
}

MapBuddyContactLayer *
map_buddy_contact_layer_new (void)
{
  return MAP_BUDDY_CONTACT_LAYER (g_object_new (MAP_BUDDY_TYPE_CONTACT_LAYER, NULL));
}

