/*
 * Copyright (C) 2010 Pierre-Luc Beaudoin <pierre-luc@pierlux.com>
 *
 * This file is inspired by clutter-color.c which is
 * Copyright (C) 2006 OpenedHand, and has the same license.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "merchant.h"

GType
map_buddy_merchant_get_type (void)
{
  static GType type = 0;

  if (G_UNLIKELY (type == 0))
    {
      type = g_boxed_type_register_static (g_intern_static_string ("MapBuddyMerchant"),
          (GBoxedCopyFunc) map_buddy_merchant_copy,
          (GBoxedFreeFunc) map_buddy_merchant_free);
    }

  return type;
}

/**
 * map_buddy_merchant_copy:
 * @merchant: a #MapBuddyMerchant
 *
 * Makes a copy of the merchant structure.  The result must be
 * freed using #map_buddy_merchant_free.  All string fields will
 * be duplicated with #g_strdup.
 *
 * Return value: an allocated copy of @merchant.
 */
MapBuddyMerchant *
map_buddy_merchant_copy (const MapBuddyMerchant *merchant)
{
  MapBuddyMerchant *dest = NULL;
  if (G_UNLIKELY (merchant == NULL))
    return NULL;

  dest = g_slice_dup (MapBuddyMerchant, merchant);

  if (G_LIKELY (merchant->name != NULL))
    dest->name = g_strdup (merchant->name);

  if (G_LIKELY (merchant->description != NULL))
    dest->description = g_strdup (merchant->description);

  if (G_LIKELY (merchant->url != NULL))
    dest->url = g_strdup (merchant->url);

  if (G_LIKELY (merchant->address != NULL))
    dest->address = g_strdup (merchant->address);

  if (G_LIKELY (merchant->city != NULL))
    dest->city = g_strdup (merchant->city);

  if (G_LIKELY (merchant->address != NULL))
    dest->address = g_strdup (merchant->address);

  if (G_LIKELY (merchant->province != NULL))
    dest->province = g_strdup (merchant->province);

  if (G_LIKELY (merchant->postal_code != NULL))
    dest->postal_code = g_strdup (merchant->postal_code);

  if (G_LIKELY (merchant->country_code != NULL))
    dest->country_code = g_strdup (merchant->country_code);

  if (G_LIKELY (merchant->country != NULL))
    dest->country = g_strdup (merchant->country);

  if (G_LIKELY (merchant->phone != NULL))
    dest->phone = g_strdup (merchant->phone);

  return dest;
}

/**
 * map_buddy_merchant_free:
 * @merchant: a #MapBuddyMerchant
 *
 * Frees a merchant structure created with #map_buddy_merchant_new or
 * #map_buddy_merchant_copy. All strings will be freed with #g_free.
 * The data pointer will not be freed.
 */
void
map_buddy_merchant_free (MapBuddyMerchant *merchant)
{

  if (G_UNLIKELY (merchant == NULL))
    return;

  if (G_LIKELY (merchant->name != NULL))
    g_free (merchant->name);

  if (G_LIKELY (merchant->description != NULL))
    g_free (merchant->description);

  if (G_LIKELY (merchant->url != NULL))
    g_free (merchant->url);

  if (G_LIKELY (merchant->address != NULL))
    g_free (merchant->address);

  if (G_LIKELY (merchant->city != NULL))
    g_free (merchant->city);

  if (G_LIKELY (merchant->province != NULL))
    g_free (merchant->province);

  if (G_LIKELY (merchant->postal_code != NULL))
    g_free (merchant->postal_code);

  if (G_LIKELY (merchant->country_code != NULL))
    g_free (merchant->country_code);

  if (G_LIKELY (merchant->country != NULL))
    g_free (merchant->country);

  if (G_LIKELY (merchant->phone != NULL))
    g_free (merchant->phone);

  g_slice_free (MapBuddyMerchant, merchant);
}

/**
 * map_buddy_merchant_new:
 *
 * Returns: a newly allocated #MapBuddyMerchant to be freed with #map_buddy_merchant_free
 */
MapBuddyMerchant *
map_buddy_merchant_new ()
{
  return g_slice_new0 (MapBuddyMerchant);
}
