#ifndef LIBMCTACC_H
#define LIBMCTACC_H


#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#include <iostream>
#include <time.h>
#include <vector>
#include <cmath>

class AccThreadProxy;
class AccSensor;



class AccSensor //!Accelerometer sensor wrapper.
{
	private:
		pthread_t acc_thread; 						//!< Thread object.
		AccThreadProxy* proxy; 						//!< Proxy for creating the thread
		friend class AccThreadProxy; 					//!< We need this to have the starting mathod readData() private.
		int readAcc();							//!< Single readout for accelerometer data.
		void* readData();						//!< Looped reading of the data.


		int ax;								//!< Readings from the accelerometers, coordinate X.
		int ay;								//!< Readings from the accelerometers, coordinate Y.
		int az;								//!< Readings from the accelerometers, coordinate Z.
		
		FILE *fd;							//!< Descriptor of the file with readings.
		int rs;								//!< Number of read entries.


		AccSensor();							//!< Private constructor, access through New().
		static AccSensor* me;						//!< Used for singleton creation.
		
		int active;							//!< Actitvity flag.
		static const char *accelFilename; 				//!< Localization of the accelerometer data file.

		std:: vector <int> x;						//!< Vectors for last readings, coordinate X.
		std:: vector <int> y;						//!< Vectors for last readings, coordinate Y.
		std:: vector <int> z;						//!< Vectors for last readings, coordinate Z.

		std:: vector <int> timestamp;					//!< Vectors for last readings, timestamp.			
		struct timespec ts;						//!< Structure of the time interval.

		struct timespec burstTime;					//!< Structure of the burst interval.

		int capacity;							//!< Capacity of the vectors, how many last reading will be remembered.

		time_t aTimestamp;						//!< Timestamp of the last reading.

		int burstNumber;						//!< How many burst readings should be performed.

		

		enum orientation						//!< Orientation of the device.
			{	
				backDown,
				faceDown,
				bottomEdgeDown,
				bottomEdgeUp,
				rightEdgeDown,
				leftEdgeDown
			
			};

	public:			
		int start();							//!< Starting reading.
		int stop();							//!< Stopping reading.
		int getOrientation(); 						//!< Returns orientation.
		float getRotation();						//!< Returns rotation in rad.
		float getRotationDeg();						//!< Returns rotation in deg.
		float getPitch();						//!< Returns pitch in rad.
		float getPitchDeg();						//!< Returns pitch in deg.
		~AccSensor();							
		static AccSensor* New();					//!< Access to the constructor, create a singleton.
		
		int getX();							//!< Access methods to the data, coordinate X.
		int getY();							//!< Access methods to the data, coordinate Y.
		int getZ();							//!< Access methods to the data, coordinate Z.

		int getTimestamp();						//!< Access methods to the data, timestamp.
		int getX(int index);						//!< Accessing elements from memory, coordinate X.
		int getY(int index);						//!< Accessing elements from memory, coordinate Y.
		int getZ(int index);						//!< Accessing elements from memory, coordinate Z.
		int getTimestamp(int index);					//!< Accessing elements from memory, timestamp.

		int setInterval(int sec_,int nsec_);				//!< Setting reading interval.				
		int setCapacity(int capacity_);					//!< Setting capacity of the vector.
		int getCapacity();						//!< Getting current capacity.
		int setRealTime();						//!< Setting interval to 10ms, suitable for real-time applications.
		int setBurst(int sec_, int nsec_, int burstNumber_, 
				int burstSec_, int burstNsec_);			//!< Setting burst parameters.

};





class AccThreadProxy	//! Proxy for the accelerometer thread.
{
	private:
		AccThreadProxy();
		~AccThreadProxy();
                                                        
		static void* proxy(void* object);				//!< Static proxy function for accelerometer.
		friend class AccSensor;

};

#endif                                                          
