/*
 * Copyright (C) 2009 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of zeemote-conf.
 *
 * zeemote-conf is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * zeemote-conf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with zeemote-conf.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <gtk/gtk.h>

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#ifdef HILDON
#include <hildon/hildon.h>
#include <libosso.h>
#if (MAEMO_VERSION_MAJOR >= 5)
#include <hildon/hildon-gtk.h>
#include <hildon/hildon-pannable-area.h>
#endif
#endif

#include <zeemote.h>
#include "zeemote-conf-paths.h"
#include "zeemote-conf.h"

#define RESPONSE_SCAN   1
#define RESPONSE_TEST   2

/***************************************************************************/
/****************     device testing    ************************************/

typedef struct {
  GtkWidget *dialog, *sbar, *eventbox;
  zeemote_t *zeemote;
  zeemote_state_t *zeemote_state;
  gint sbid;

  GtkWidget *button[32];
  GtkWidget *area, *bat_bar;
  GdkPixmap *pixmap;

} context_t;

static const char *state_names[] = {
  "unknown",
  "connecting",
  "connection failed",
  "connected",
  "connection lost",
  "disconnected"
};

static void draw(GtkWidget *widget, context_t *context) {
  gint width = widget->allocation.width;
  gint height = widget->allocation.height;
  gint diameter = (height < width)?height:width;

  gint xcenter = width/2;
  gint ycenter = height/2;

  /* erase background */
  gdk_draw_rectangle(context->pixmap, 
		     widget->style->white_gc, TRUE,
		     0, 0, width, height);

  /* check if we got a valid state and if the zeemote is actually */
  /* connected */
  if(context->zeemote_state && 
     context->zeemote_state->state == ZEEMOTE_STATE_CONNECTED) {

    int i;
    for(i=0;i<zeemote_get_info(context->zeemote)->num_axes;i+=2) {
      const char *colors[2] = { "#800000", "#000080" };

      GdkGC *circle_gc = gdk_gc_new(context->pixmap);
      gdk_gc_copy(circle_gc, widget->style->black_gc);
      GdkColor color;
      gdk_color_parse(colors[i/2], &color);
      gdk_gc_set_rgb_fg_color(circle_gc, &color);
      
#define RADIUS 9
      gdk_draw_arc(context->pixmap, circle_gc, TRUE,
	xcenter + (context->zeemote_state->axis[i+0] * width)/65536 - RADIUS, 
	ycenter + (context->zeemote_state->axis[i+1] * height)/65536 - RADIUS, 
		   2*RADIUS+1, 2*RADIUS+1,
		   0, 360*64);
    }
  }
}

/* Create a new backing pixmap of the appropriate size */
static gint configure_event(GtkWidget *widget, GdkEventConfigure *event,
			    gpointer data) {
  context_t *context = (context_t*)data;

  if(context->pixmap)
    gdk_pixmap_unref(context->pixmap);
  
  context->pixmap = gdk_pixmap_new(widget->window,
				   widget->allocation.width,
				   widget->allocation.height,
				   -1);
  draw(widget, context);
  
  return TRUE;
}

/* Redraw the screen from the backing pixmap */
static gint expose_event(GtkWidget *widget, GdkEventExpose *event, 
			 gpointer data) {
  context_t *context = (context_t*)data;

  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                  context->pixmap,
                  event->area.x, event->area.y,
                  event->area.x, event->area.y,
                  event->area.width, event->area.height);

  return FALSE;
}

static gint test_timer(gpointer data) {
  context_t *context = (context_t*)data;

  context->zeemote_state = zeemote_get_state(context->zeemote);

  /* update statusbar */
  static int old_state = -1;
  if(context->zeemote_state->state != old_state) {
    GdkColor color;

    gtk_statusbar_push(GTK_STATUSBAR(context->sbar), context->sbid,
		       state_names[context->zeemote_state->state]);  
    old_state = context->zeemote_state->state;

    switch(context->zeemote_state->state) {
    case ZEEMOTE_STATE_CONNECTING:
      gdk_color_parse("orange", &color); 
      break;

    case ZEEMOTE_STATE_CONNECTED:
      gdk_color_parse("green", &color); 
      break;

    default:
      gdk_color_parse("red", &color); 
      break;
    }

    gtk_widget_modify_bg(context->eventbox,
			 GTK_STATE_NORMAL, &color);
  }

  /* update battery gauge */
  static struct { int b, s; } old_batt = { -1, -1};
  if((context->zeemote_state->battery != old_batt.b)||
     (context->zeemote_state->state != old_batt.s)) {
    gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(context->bat_bar),
				  context->zeemote_state->battery / 3000.0);
    char *text = NULL;
    if(!context->zeemote_state->battery)
      text = g_strdup_printf("please wait...");
    else
      text = g_strdup_printf("%d.%03d volts", 
			     context->zeemote_state->battery / 1000,
			     context->zeemote_state->battery % 1000
			     );

    gtk_progress_bar_set_text(GTK_PROGRESS_BAR(context->bat_bar), text);
    g_free(text);

    gtk_widget_set_sensitive(context->bat_bar, 
	     context->zeemote_state->state == ZEEMOTE_STATE_CONNECTED);

   old_batt.b = context->zeemote_state->battery;
   old_batt.s = context->zeemote_state->state;
  }

  /* update axis widget */
  static struct { int x[4], s; } old_axis = { -1, -1, -1, -1, -1};
  int i, d;
  /* check if one of the axes has changed */
  for(d=0,i=0;i<zeemote_get_info(context->zeemote)->num_axes;i++)
    if(context->zeemote_state->axis[i] != old_axis.x[i])
      d = 1;

  if(context->pixmap &&
     ((context->zeemote_state->state   != old_axis.s) || d)) {

    draw(context->area, context);
    gtk_widget_queue_draw_area(context->area, 0,0,
			       context->area->allocation.width, 
			       context->area->allocation.height);

    old_axis.s = context->zeemote_state->state;
    for(i=0;i<zeemote_get_info(context->zeemote)->num_axes;i++)
      old_axis.x[i] = context->zeemote_state->axis[i];
  }

  /* update button widget */
  static struct { int b, s; } old_button = { 0, -1};
  if((context->zeemote_state->buttons != old_button.b) ||
     (context->zeemote_state->state   != old_button.s)) {
    int i;
    
    for(i=0;i<zeemote_get_info(context->zeemote)->num_buttons;i++) {
      gtk_widget_set_sensitive(context->button[i], 
	       context->zeemote_state->state == ZEEMOTE_STATE_CONNECTED);
      
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(context->button[i]),
				   context->zeemote_state->buttons & (1<<i));
    }

    old_button.b = context->zeemote_state->buttons;
    old_button.s = context->zeemote_state->state;
  }

  return 1;
}

static gboolean button_press_event(GtkWidget *widget, GdkEventButton *event,
				   gpointer user_data) {
  return TRUE;
}


void test_device(zeemote_device_t *dev, GtkWidget *parent) {
  context_t context;
  memset(&context, 0, sizeof(context_t));

  context.dialog = 
    gtk_dialog_new_with_buttons("Zeemote test", GTK_WINDOW(parent),
		GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
		GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
		NULL);

  gtk_dialog_set_default_response(GTK_DIALOG(context.dialog), 
				  GTK_RESPONSE_CLOSE);
  gtk_window_set_default_size(GTK_WINDOW(context.dialog), 400, 200);

  context.zeemote = zeemote_connect(&dev->bdaddr);

  /* main control elements */
  GtkWidget *hbox = gtk_hbox_new(FALSE, 5);

  /* ------------ create stick box ------------ */

  GtkWidget *frame = gtk_frame_new("Stick");

  context.area = gtk_drawing_area_new();
  gtk_drawing_area_size(GTK_DRAWING_AREA(context.area), 100, 100);

  gtk_signal_connect(GTK_OBJECT(context.area), "expose_event",
		     G_CALLBACK(expose_event), &context);
  gtk_signal_connect(GTK_OBJECT(context.area),"configure_event",
		     G_CALLBACK(configure_event), &context);

  gtk_container_add(GTK_CONTAINER(frame), context.area);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), frame);

  GtkWidget *vbox = gtk_vbox_new(FALSE, 20);

  /* ------------ create battery box ------------ */

  GtkWidget *ivbox = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(ivbox), gtk_label_new("Battery:"), 
		     FALSE, FALSE, 0);
  context.bat_bar = gtk_progress_bar_new();
  gtk_box_pack_start(GTK_BOX(ivbox), context.bat_bar, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), ivbox, FALSE, FALSE, 0);

  /* ------------ create button box ------------ */

  ivbox = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(ivbox), gtk_label_new("Buttons:"), 
		     FALSE, FALSE, 0);

  int num_buttons = zeemote_get_info(context.zeemote)->num_buttons;

  /* 6 columns up to 24 buttons, 8 if more */
  int columns = (num_buttons <= 6)?num_buttons:((num_buttons<=24)?6:8);
  int rows = 1+(num_buttons-1)/columns;

  GtkWidget *table = gtk_table_new(rows, columns, TRUE);
  int i; 
  for(i=0;i<num_buttons;i++) {
    char *str = g_strdup_printf("%c", ((i<26)?'A':('a'-26))+i);
    context.button[i] = gtk_check_button_new_with_label(str);
    g_free(str);

    /* make button unresponsive to clicks */
    gtk_widget_set_sensitive(context.button[i], FALSE);
    gtk_signal_connect(GTK_OBJECT(context.button[i]), "button-press-event",
		       G_CALLBACK(button_press_event), NULL);

    gtk_table_attach_defaults(GTK_TABLE(table), context.button[i],
		      i%columns, i%columns+1, i/columns, i/columns+1);
  }

  gtk_box_pack_start(GTK_BOX(ivbox), table, FALSE, FALSE, 0);


  gtk_box_pack_start(GTK_BOX(vbox), ivbox, FALSE, FALSE, 0);

  gtk_box_pack_start_defaults(GTK_BOX(hbox), vbox);

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(context.dialog)->vbox), hbox);

  /* add statusbar */
  context.eventbox = gtk_event_box_new();
  context.sbar = gtk_statusbar_new();
  context.sbid = 
    gtk_statusbar_get_context_id(GTK_STATUSBAR(context.sbar), "id"); 
  gtk_container_add(GTK_CONTAINER(context.eventbox), context.sbar);
  g_object_set(context.sbar, "has-resize-grip", FALSE, NULL );

 gtk_box_pack_end(GTK_BOX(GTK_DIALOG(context.dialog)->vbox), 
		  context.eventbox, FALSE, FALSE, 0);

  /* to add a timer */
  gint timer_id = gtk_timeout_add(25, test_timer, &context);

  /* run the dialog */
  gtk_widget_show_all(context.dialog);
  gtk_dialog_run(GTK_DIALOG(context.dialog));

  gtk_timeout_remove(timer_id);

  zeemote_disconnect(context.zeemote);

  gtk_widget_destroy(context.dialog);
}

/***************************************************************************/
/****************      gconf handling   ************************************/

void store_scan_results_in_gconf(zeemote_scan_result_t *result) {
  GConfClient *gconf_client = gconf_client_get_default();

  gconf_client_set_int(gconf_client, GCONF_KEY_DEVICES, 
		       result->number_of_devices, NULL);

  int i;
  for(i=0;i<result->number_of_devices;i++) {
    char *key, addr[18]; 

    key = g_strdup_printf(GCONF_KEY_TYPE, i);
    gconf_client_set_int(gconf_client, key, result->device[i].type, NULL);
    g_free(key);

    key = g_strdup_printf(GCONF_KEY_BDADDR, i);
    ba2str(&result->device[i].bdaddr, addr);
    gconf_client_set_string(gconf_client, key, addr, NULL);
    g_free(key);
  }
}

/***************************************************************************/

/* scanning happens in background as it takes some time ... */
gpointer scan_thread(gpointer data) {
  zeemote_scan_result_t **result = data;
  *result = zeemote_scan();  
  return NULL;
}

static const char *get_type_name(int idx) {
  static const char *type_str[] = {
    "Zeemote JS1",
    ""
  };

  static const char unknown[] = "<unknown>";

  if(idx >= 0 && idx < 1) return type_str[idx];

  return unknown;
}

enum {
  COL_TYPE = 0,
  COL_BDADDR,
  COL_DATA,
  NUM_COLS
} ;

static void fill_model(GtkListStore *store, zeemote_scan_result_t *result) {
  int i;
  for(i=0;i<result->number_of_devices;i++) {
    GtkTreeIter iter;
    char addr[18];
    ba2str(&result->device[i].bdaddr, addr);

    /* Append a row and fill in some data */
    gtk_list_store_append (store, &iter);
    gtk_list_store_set(store, &iter,
		       COL_TYPE, get_type_name(result->device[i].type),
		       COL_BDADDR, addr,
		       COL_DATA, &result->device[i],
		       -1);
  }
}

static gboolean
selection_function(GtkTreeSelection *selection, GtkTreeModel *model,
		   GtkTreePath *path, gboolean path_currently_selected,
		   gpointer data) {
  GtkTreeIter iter;

  if(gtk_tree_model_get_iter(model, &iter, path)) {
    g_assert(gtk_tree_path_get_depth(path) == 1);

    //    if(gtk_tree_selection_get_selected(selection, &model, &iter)) 
    {
      GtkWidget *dialog = 
	gtk_widget_get_toplevel(GTK_WIDGET(
		   gtk_tree_selection_get_tree_view(selection)));

      gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog), 
					RESPONSE_TEST, TRUE); 
    }
  }
  
  return TRUE; /* allow selection state to change */
}

static GtkWidget 
*create_view_and_model(zeemote_scan_result_t *result) {
  GtkCellRenderer     *renderer;
  GtkTreeModel        *model;
  GtkWidget           *view;

#if (MAEMO_VERSION_MAJOR < 5)
  view = gtk_tree_view_new();
#else
  view = hildon_gtk_tree_view_new(HILDON_UI_MODE_EDIT);
#endif

  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), TRUE);
  gtk_tree_selection_set_select_function(
	 gtk_tree_view_get_selection(GTK_TREE_VIEW(view)), 
	 selection_function, NULL, NULL);

  /* --- type column --- */
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW (view),
	      -1, "Type", renderer, "text", COL_TYPE, NULL);

  /* --- bdaddr column --- */
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW (view),
	      -1, "BDADDR", renderer, "text", COL_BDADDR, NULL);

  GtkListStore *store = gtk_list_store_new(NUM_COLS, G_TYPE_STRING, 
			     G_TYPE_STRING, G_TYPE_POINTER);

  fill_model(store, result);

  gtk_tree_view_set_model(GTK_TREE_VIEW (view), GTK_TREE_MODEL(store));

  g_object_unref(store);

  return view;
}

#if (MAEMO_VERSION_MAJOR >= 5)
static GtkWidget*
create_device_list(zeemote_scan_result_t *result) { 

  /* put results inside a pannable area */
  GtkWidget *pannable_area = hildon_pannable_area_new();
  gtk_container_add(GTK_CONTAINER(pannable_area), 
		    create_view_and_model(result));

  /* build a zeemote_scan_result_t structure from the gconf contents */
  return pannable_area;
}
#else
static GtkWidget*
create_device_list(zeemote_scan_result_t *result) { 

  /* put results inside a scrolled view */
  GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
  				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  
  gtk_container_add(GTK_CONTAINER(scrolled_window), 
		    create_view_and_model(result));

  /* build a zeemote_scan_result_t structure from the gconf contents */
  return scrolled_window;
}
#endif

/**************************************************************************/

/* create the dialog box shown while worker is running */
static GtkWidget *busy_start(GtkWidget *parent) {
  GtkWidget *dialog = gtk_dialog_new();

  gtk_dialog_set_has_separator(GTK_DIALOG(dialog), FALSE);
  gtk_window_set_title(GTK_WINDOW(dialog), "Scanning...");
  gtk_window_set_default_size(GTK_WINDOW(dialog), 250, 10);

  gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
  gtk_window_set_transient_for(GTK_WINDOW(dialog), GTK_WINDOW(parent));

  GtkWidget *pbar = gtk_progress_bar_new();
  g_object_set_data(G_OBJECT(dialog), "pbar", pbar);
  gtk_progress_bar_set_pulse_step(GTK_PROGRESS_BAR(pbar), 0.1);

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), pbar);

  gtk_widget_show_all(dialog);

  return dialog;
}

static void busy_animate(GtkWidget *dialog) {
  GtkWidget *pbar = g_object_get_data(G_OBJECT(dialog), "pbar");
  gtk_progress_bar_pulse(GTK_PROGRESS_BAR(pbar));
}

static void busy_end(GtkWidget *dialog) {
  gtk_widget_destroy(dialog);
}

/**************************************************************************/

static void on_scan(GtkWidget *widget, GtkWidget *list) {

  zeemote_scan_result_t *result = NULL;

  GtkWidget *busy = busy_start(gtk_widget_get_toplevel(widget));

  GtkWidget *dialog = gtk_widget_get_toplevel(widget);
  gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog), 
				    RESPONSE_TEST, FALSE); 

  /* erase current list */
  GtkWidget *view = gtk_bin_get_child(GTK_BIN(list));
  GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(view));
  gtk_list_store_clear(GTK_LIST_STORE(model));

  g_thread_create(scan_thread, &result, FALSE, NULL);

  /* and just draw some idle thing while waiting */
  while(!result) {
    busy_animate(busy);
    usleep(100000);

    while(gtk_events_pending()) 
      gtk_main_iteration();
  }
  
  store_scan_results_in_gconf(result);

  fill_model(GTK_LIST_STORE(model), result);
  
  busy_end(busy);
}

/* user clicked the "test..." button */
static void on_test(GtkWidget *widget, GtkWidget *list) {
  GtkTreeModel     *model;
  GtkTreeIter       iter;

  GtkWidget *view = gtk_bin_get_child(GTK_BIN(list));
  GtkTreeSelection *selection = 
    gtk_tree_view_get_selection(GTK_TREE_VIEW(view));
  if(gtk_tree_selection_get_selected(selection, &model, &iter)) {
    zeemote_device_t *device = NULL;
    gtk_tree_model_get(model, &iter, COL_DATA, &device, -1);

    if(device)
      test_device(device, gtk_widget_get_toplevel(view));
  }
}

void zeemote_conf_dialog(void) {
#ifndef HILDON
  g_thread_init(NULL);
#endif

  GtkWidget *dialog = 
    gtk_dialog_new_with_buttons("Zeemote configuration", NULL,
		GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                NULL);

  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_CLOSE);
  gtk_window_set_default_size(GTK_WINDOW(dialog), 500, 300);

  zeemote_scan_result_t *result = zeemote_get_scan_results_from_gconf();

  GtkWidget *list = create_device_list(result);

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), 
		     gtk_label_new("Configured devices:"), FALSE, FALSE, 0);

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), list);

  /* ---- scan button ---- */
  GtkWidget *button = gtk_dialog_add_button(GTK_DIALOG(dialog),
				    "Scan...", RESPONSE_SCAN);
  

  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(on_scan), list);

  /* ---- test button ---- */
  button = gtk_dialog_add_button(GTK_DIALOG(dialog),
				 "Test...", RESPONSE_TEST);

  gtk_signal_connect(GTK_OBJECT(button), "clicked",
  		     GTK_SIGNAL_FUNC(on_test), list);

  gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog), 
				    RESPONSE_TEST, FALSE); 

  /* ---- close button ---- */
  gtk_dialog_add_button(GTK_DIALOG(dialog),
			GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE);

  /* run the dialog */
  gtk_widget_show_all(dialog);

  /* run gtk_dialog_run, but continue if e.g. the help button was pressed */
  int ret_code;
  do 
    ret_code = gtk_dialog_run(GTK_DIALOG(dialog));
  while((ret_code == RESPONSE_SCAN) || (ret_code == RESPONSE_TEST));

  gtk_widget_destroy(dialog);
}

#ifdef HILDON
osso_return_t execute(osso_context_t *osso, gpointer data, gboolean user_activated) {
  zeemote_conf_dialog();
  return OSSO_OK;
}

osso_return_t save_state(osso_context_t *osso, gpointer data) {
  /* ... save state ... */
  return OSSO_OK;
}
#else

// stand-alone test
int main(int argc, char *argv[]) {
  gtk_init (&argc, &argv);
  zeemote_conf_dialog();
  return 0;
}
#endif
