/*
 * tangle-texture.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2011 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-texture.h"
#include "tangle-misc.h"
#include "marshalers.h"
#include <string.h>

G_DEFINE_TYPE(TangleTexture, tangle_texture, TANGLE_TYPE_ACTOR);

enum {
	PROP_0,
	PROP_FILENAME,
	PROP_REPEAT_X,
	PROP_REPEAT_Y
};

enum {
	SIZE_CHANGED,
	LAST_SIGNAL
};

typedef struct {
	CoglMaterial* material;
	gint width;
	gint height;
	
	gchar* path;

	guint reference_count;
} CachedMaterial;

struct _TangleTexturePrivate {
	gchar* filename;
	
	CachedMaterial* cached_material;
	
	guint repeat_x : 1;
	guint repeat_y : 1;
};

static CoglMaterial* material_template = NULL;
static GHashTable* cached_materials;
static guint signals[LAST_SIGNAL] = { 0 };

static CachedMaterial* cached_material_new(const gchar* path, CoglHandle texture);
static void cached_material_unref(CachedMaterial* cached_material);
static CachedMaterial* cached_material_lookup(const gchar* path);

ClutterActor* tangle_texture_new(const gchar* filename) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_TEXTURE, "filename", filename, NULL));
}

void tangle_texture_set_from_file(TangleTexture* texture, const gchar* filename) {
	gchar* path;
	CachedMaterial* cached_material = NULL;
	CoglHandle new_texture;
	gboolean size_changed;
	
	g_return_if_fail(TANGLE_IS_TEXTURE(texture));

	path = tangle_lookup_filename(filename);
	
	if (!path) {
		g_free(texture->priv->filename);
		texture->priv->filename = NULL;
		
		if (texture->priv->cached_material) {
			cached_material_unref(texture->priv->cached_material);
			texture->priv->cached_material = NULL;
		}
	} else if (!(cached_material = cached_material_lookup(path)) &&
	           (new_texture = cogl_texture_new_from_file(path, COGL_TEXTURE_NONE, COGL_PIXEL_FORMAT_ANY, NULL)) != COGL_INVALID_HANDLE) {
		cached_material = cached_material_new(path, new_texture);
		cogl_handle_unref(new_texture);
	}
	
	if (cached_material) {
		size_changed = (!texture->priv->cached_material || cached_material->width != texture->priv->cached_material->width || cached_material->height != texture->priv->cached_material->height);

		if (texture->priv->cached_material) {
			cached_material_unref(texture->priv->cached_material);
		}
		texture->priv->cached_material = cached_material;
		
		g_free(texture->priv->filename);
		texture->priv->filename = g_strdup(filename);
		g_object_notify(G_OBJECT(texture), "filename");

		if (size_changed) {
			g_signal_emit(texture, signals[SIZE_CHANGED], 0, texture->priv->cached_material->width, texture->priv->cached_material->height);
			clutter_actor_queue_relayout(CLUTTER_ACTOR(texture));
		} else {
			clutter_actor_queue_redraw(CLUTTER_ACTOR(texture));
		}
	}
	
	g_free(path);	
}

void tangle_texture_get_repeat(TangleTexture* texture, gboolean* repeat_x_return, gboolean* repeat_y_return) {
	g_return_if_fail(TANGLE_IS_TEXTURE(texture));

	if (repeat_x_return) {
		*repeat_x_return = texture->priv->repeat_x;
	}
	if (repeat_y_return) {
		*repeat_y_return = texture->priv->repeat_y;
	}
}

void tangle_texture_set_repeat(TangleTexture* texture, gboolean repeat_x, gboolean repeat_y) {
	gboolean changed = FALSE;

	g_return_if_fail(TANGLE_IS_TEXTURE(texture));

	g_object_freeze_notify(G_OBJECT(texture));
	
	if (texture->priv->repeat_x != repeat_x) {
		texture->priv->repeat_x = repeat_x;
		g_object_notify(G_OBJECT(texture), "repeat-x");
		changed = TRUE;
	}
	if (texture->priv->repeat_y != repeat_y) {
		texture->priv->repeat_y = repeat_y;
		g_object_notify(G_OBJECT(texture), "repeat-y");
		changed = TRUE;
	}
	
	if (changed) {
		clutter_actor_queue_redraw(CLUTTER_ACTOR(texture));
	}
	
	g_object_thaw_notify(G_OBJECT(texture));
}

static void tangle_texture_get_preferred_width(TangleActor* actor, gfloat for_height, gboolean interacting, gfloat* min_width_return, gfloat* natural_width_return, gfloat* max_width_return) {
	TangleTexture* texture;
	
	texture = TANGLE_TEXTURE(actor);
	
	if (min_width_return) {
		*min_width_return = 0;
	}
	if (natural_width_return) {
		if (texture->priv->cached_material) {
			if (for_height >= 0) {
				*natural_width_return = (gfloat)texture->priv->cached_material->width / texture->priv->cached_material->height * for_height;
			} else {
				*natural_width_return = texture->priv->cached_material->width;
			}
		} else {
			*natural_width_return = 32;
		}
	}
	if (max_width_return) {
		*max_width_return = 0;
	}
}

static void tangle_texture_get_preferred_height(TangleActor* actor, gfloat for_width, gboolean interacting, gfloat* min_height_return, gfloat* natural_height_return, gfloat* max_height_return) {
	TangleTexture* texture;
	
	texture = TANGLE_TEXTURE(actor);
	
	if (min_height_return) {
		*min_height_return = 0;
	}
	if (natural_height_return) {
		if (texture->priv->cached_material) {
			if (for_width >= 0) {
				*natural_height_return = (gfloat)texture->priv->cached_material->height / texture->priv->cached_material->width * for_width;
			} else {
				*natural_height_return = texture->priv->cached_material->height;
			}
		} else {
			*natural_height_return = 32;
		}
	}
	if (max_height_return) {
		*max_height_return = 0;
	}
}

static void tangle_texture_paint_aligned(TangleActor* actor, gfloat width, gfloat height) {
	TangleTexture* texture;
	guint8 paint_opacity;
	gfloat w = 1.0, h = 1.0;
	
	texture = TANGLE_TEXTURE(actor);
	
	paint_opacity = clutter_actor_get_paint_opacity(CLUTTER_ACTOR(actor));

	if (texture->priv->cached_material) {
		if (texture->priv->repeat_x && texture->priv->cached_material->width > 0) {
			w = width / texture->priv->cached_material->width;
		}
		if (texture->priv->repeat_y && texture->priv->cached_material->height > 0) {
			h = height / texture->priv->cached_material->height;
		}

		/* TODO: workaround for a COGL bug
		 * Without this a (cached) material is rendered all blue. */
		cogl_set_source_color4ub(255, 255, 255, 0);
		cogl_rectangle(0, 0, width, height);

		cogl_material_set_color4ub(texture->priv->cached_material->material, paint_opacity, paint_opacity, paint_opacity, paint_opacity);
		cogl_set_source(texture->priv->cached_material->material);
		cogl_rectangle_with_texture_coords(0.0, 0.0, width, height, 0.0, 0.0, w, h);	
	} else {
		cogl_set_source_color4ub(255, 32, 64, 255);
		cogl_rectangle(0, 0, width, height);	
	}
}

static void tangle_texture_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleTexture* texture;
	
	texture = TANGLE_TEXTURE(object);

	switch (prop_id) {
		case PROP_FILENAME:
			tangle_texture_set_from_file(texture, g_value_get_string(value));
			break;
		case PROP_REPEAT_X:
			tangle_texture_set_repeat(texture, g_value_get_boolean(value), texture->priv->repeat_y);
			break;
		case PROP_REPEAT_Y:
			tangle_texture_set_repeat(texture, texture->priv->repeat_x, g_value_get_boolean(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_texture_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleTexture* texture;

	texture = TANGLE_TEXTURE(object);

        switch (prop_id) {
		case PROP_FILENAME:
			g_value_set_string(value, texture->priv->filename);
			break;
		case PROP_REPEAT_X:
			g_value_set_boolean(value, texture->priv->repeat_x);
			break;
		case PROP_REPEAT_Y:
			g_value_set_boolean(value, texture->priv->repeat_y);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_texture_finalize(GObject* object) {
	TangleTexture* texture;
	
	texture = TANGLE_TEXTURE(object);

	g_free(texture->priv->filename);	

	G_OBJECT_CLASS(tangle_texture_parent_class)->finalize(object);
}

static void tangle_texture_dispose(GObject* object) {
	TangleTexture* texture;
	
	texture = TANGLE_TEXTURE(object);

	if (texture->priv->cached_material) {
		cached_material_unref(texture->priv->cached_material);
		texture->priv->cached_material = NULL;
	}
	
	G_OBJECT_CLASS(tangle_texture_parent_class)->dispose(object);
}

static void tangle_texture_class_init(TangleTextureClass* texture_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(texture_class);
	TangleActorClass* actor_class = TANGLE_ACTOR_CLASS(texture_class);

	gobject_class->finalize = tangle_texture_finalize;
	gobject_class->dispose = tangle_texture_dispose;
	gobject_class->set_property = tangle_texture_set_property;
	gobject_class->get_property = tangle_texture_get_property;
	
	actor_class->get_preferred_width = tangle_texture_get_preferred_width;
	actor_class->get_preferred_height = tangle_texture_get_preferred_height;
	actor_class->paint_aligned = tangle_texture_paint_aligned;

	/**
	 * TangleTexture:filename:
	 *
	 * The filename of the image shown.
	 */
	g_object_class_install_property(gobject_class, PROP_FILENAME,
	                                g_param_spec_string("filename",
	                                "Filename",
	                                "The filename of the image shown",
	                                NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleTexture:repeat-x:
	 *
	 * Whether to repeat the contents rathen than scaling horizontally.
	 */
	g_object_class_install_property(gobject_class, PROP_REPEAT_X,
	                                g_param_spec_boolean("repeat-x",
	                                "Repeat X",
	                                "Whether to repeat the contents rathen than scaling horizontally",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleTexture:repeat-x:
	 *
	 * Whether to repeat the contents rathen than scaling vertically.
	 */
	g_object_class_install_property(gobject_class, PROP_REPEAT_Y,
	                                g_param_spec_boolean("repeat-y",
	                                "Repeat y",
	                                "Whether to repeat the contents rathen than scaling vertically",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	/**
	 * TangleTexture:size-changed:
	 * @texture: the object which received the signal
	 */
	signals[SIZE_CHANGED] = g_signal_new("size-changed", G_TYPE_FROM_CLASS(gobject_class),
	                                     G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(TangleTextureClass, size_changed),
					     NULL, NULL,
					     tangle_marshal_VOID__INT_INT,
					     G_TYPE_NONE, 2,
					     G_TYPE_INT,
					     G_TYPE_INT);

	g_type_class_add_private (gobject_class, sizeof (TangleTexturePrivate));

	cached_materials = g_hash_table_new(g_str_hash, g_str_equal);
}

static void tangle_texture_init(TangleTexture* texture) {
	CoglHandle texture_template;
	
	texture->priv = G_TYPE_INSTANCE_GET_PRIVATE(texture, TANGLE_TYPE_TEXTURE, TangleTexturePrivate);

	if (!material_template) {
		material_template = cogl_material_new();
		texture_template = cogl_texture_new_with_size(1, 1, COGL_TEXTURE_NO_SLICING, COGL_PIXEL_FORMAT_RGBA_8888_PRE);
		cogl_material_set_layer(material_template, 0, texture_template);
		cogl_handle_unref(texture_template);
	}
}

static CachedMaterial* cached_material_new(const gchar* path, CoglHandle texture) {
	CachedMaterial* cached_material;
	
	cached_material = g_slice_new(CachedMaterial);
	
	cached_material->material = cogl_material_copy(material_template);
	cogl_material_set_layer(cached_material->material, 0, texture);
	
	cached_material->width = cogl_texture_get_width(texture);
	cached_material->height = cogl_texture_get_height(texture);

	cached_material->path = g_slice_copy(strlen(path) + 1, path);

	cached_material->reference_count = 1;
		
	g_hash_table_insert(cached_materials, cached_material->path, cached_material);
	
	return cached_material;
}

static void cached_material_unref(CachedMaterial* cached_material) {
	cached_material->reference_count--;
	if (cached_material->reference_count == 0) {
		g_hash_table_remove(cached_materials, cached_material->path);

		cogl_handle_unref(cached_material->material);
		g_slice_free1(strlen(cached_material->path) + 1, cached_material->path);
		g_slice_free(CachedMaterial, cached_material);	
	}
}

static CachedMaterial* cached_material_lookup(const gchar* path) {
	CachedMaterial* cached_material;
	
	if ((cached_material = g_hash_table_lookup(cached_materials, path))) {
		cached_material->reference_count++;
	}
	
	return cached_material;
}
