/*
 * tangle-bufferer.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2011 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-bufferer.h"
#include "../config.h"

G_DEFINE_TYPE(TangleBufferer, tangle_bufferer, TANGLE_TYPE_WIDGET);

enum {
	PROP_0,
	PROP_N_BUFFERS
};

typedef struct {
	TangleBufferer* bufferer;
	ClutterActor* actor;
	guint cycle;
	glong prepaint_signal_handler;
	glong postpaint_signal_handler;
	CoglMatrix matrix;
	guint has_matrix : 1;
} Redrawable;

typedef struct {
	CoglHandle texture;
	Redrawable* redrawable;
} Buffer;

struct _TangleBuffererPrivate {
	guint n_buffers;

	guint cycle;
	GPtrArray* queued_actors;
	guint n_redrawable_actors;

	GArray* buffers;
	guint buffer_index;

	CoglHandle offscreen;
	
	gfloat absolute_x;
	gfloat absolute_y;
	gfloat absolute_width;
	gfloat absolute_height;
	
	gulong queue_redraw_hook_id;
	gulong allocation_changed_hook_id;
	
	guint buffering : 1;
	guint resources_created : 1;
};

static gboolean queue_redraw_hook(GSignalInvocationHint* hint, guint n_values, const GValue* values, gpointer user_data);
static gboolean allocation_changed_hook(GSignalInvocationHint* hint, guint n_values, const GValue* values, gpointer user_data);
static void begin_paint_actor(ClutterActor* actor, gpointer user_data);
static void end_paint_actor(ClutterActor* actor, gpointer user_data);
static void queue_actor(TangleBufferer* bufferer, ClutterActor* actor, gboolean allocation_changed);
static gboolean check_queued_actor(TangleBufferer* bufferer, ClutterActor* actor, guint* n_queued_redraws_pointer);
static void create_resources(TangleBufferer* bufferer);
static void destroy_resources(TangleBufferer* bufferer);
static void clear_buffers(TangleBufferer* bufferer);
static void switch_buffer(TangleBufferer* bufferer, ClutterActor* actor);
static void draw_buffers(TangleBufferer* bufferer);
static void remove_queued_actor(gpointer user_data, GObject* object);
static void get_absolute_allocation_box(ClutterActor* actor, ClutterActorBox* actor_box);

static TangleBufferer* active_bufferer = NULL;
static GQuark redrawable_quark;

ClutterActor* tangle_bufferer_new(void) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_BUFFERER, NULL));
}

ClutterActor* tangle_bufferer_new_with_layout(TangleLayout* layout) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_BUFFERER, "layout", layout, NULL));
}

guint tangle_bufferer_get_n_buffers(TangleBufferer* bufferer) {
	g_return_val_if_fail(TANGLE_IS_BUFFERER(bufferer), 0);
	
	return bufferer->priv->n_buffers;
}

void tangle_bufferer_set_n_buffers(TangleBufferer* bufferer, guint n_buffers) {
	g_return_if_fail(TANGLE_IS_BUFFERER(bufferer));
	
	if (bufferer->priv->n_buffers != n_buffers) {
		destroy_resources(bufferer);
		bufferer->priv->n_buffers = n_buffers;
		
		g_object_notify(G_OBJECT(bufferer), "n-buffers");
	}
}

void tangle_bufferer_begin_paint_actor(ClutterActor* actor) {
	Redrawable* redrawable;
	
	if (active_bufferer &&
	    active_bufferer->priv->buffering &&
	    (redrawable = (Redrawable*)g_object_get_qdata(G_OBJECT(actor), redrawable_quark)) &&
	    redrawable->prepaint_signal_handler) {
		switch_buffer(active_bufferer, actor);
	}
}

void tangle_bufferer_end_paint_actor(ClutterActor* actor) {
	Redrawable* redrawable;
	
	if (active_bufferer &&
	    active_bufferer->priv->buffering &&
	    (redrawable = (Redrawable*)g_object_get_qdata(G_OBJECT(actor), redrawable_quark)) &&
	    redrawable->prepaint_signal_handler) {
		switch_buffer(active_bufferer, NULL);
	}
}

static void tangle_bufferer_allocate(ClutterActor* actor, const ClutterActorBox* box, ClutterAllocationFlags flags) {
	g_return_if_fail(active_bufferer == NULL);
	
	active_bufferer = TANGLE_BUFFERER(actor);
	CLUTTER_ACTOR_CLASS(tangle_bufferer_parent_class)->allocate(actor, box, flags);
	active_bufferer = NULL;
}

static void tangle_bufferer_paint(ClutterActor* actor) {
	TangleBufferer* bufferer;
	gint i;
	ClutterActor* queued_actor;
	guint n_redrawable_actors = 0;
	ClutterActorBox box;
	gfloat width, height;
	GList* list;
	
	bufferer = TANGLE_BUFFERER(actor);

	active_bufferer = bufferer;
	
	for (i = 0; i < bufferer->priv->queued_actors->len; i++) {
		queued_actor = (ClutterActor*)g_ptr_array_index(bufferer->priv->queued_actors, i);
		if (check_queued_actor(bufferer, queued_actor, &n_redrawable_actors)) {
			g_object_set_qdata(G_OBJECT(queued_actor), redrawable_quark, NULL);
			i--;
		}
	}
	
	if (bufferer->priv->n_redrawable_actors != n_redrawable_actors) {
		bufferer->priv->n_redrawable_actors = n_redrawable_actors;
		bufferer->priv->buffering = TRUE;
	}

	get_absolute_allocation_box(CLUTTER_ACTOR(bufferer), &box);
	if (bufferer->priv->absolute_width != box.x2 - box.x1 ||
	    bufferer->priv->absolute_height != box.y2 - box.y1) {
		bufferer->priv->absolute_width = box.x2 - box.x1;
		bufferer->priv->absolute_height = box.y2 - box.y1;

		destroy_resources(bufferer);
	
		bufferer->priv->buffering = TRUE;
	}
	bufferer->priv->absolute_x = box.x1;
	bufferer->priv->absolute_y = box.y1;

	if (!bufferer->priv->resources_created) {
		create_resources(bufferer);
	}

	if (bufferer->priv->n_redrawable_actors * 2 + 1 > bufferer->priv->buffers->len) {
		bufferer->priv->n_redrawable_actors = 0;
		bufferer->priv->buffering = FALSE;

		CLUTTER_ACTOR_CLASS(tangle_bufferer_parent_class)->paint(actor);
	} else {
		if (bufferer->priv->buffering) {
			cogl_push_framebuffer(bufferer->priv->offscreen);
			cogl_push_matrix();

			clear_buffers(bufferer);

			bufferer->priv->buffer_index = 0;
			switch_buffer(bufferer, NULL);

			CLUTTER_ACTOR_CLASS(tangle_bufferer_parent_class)->paint(actor);

			cogl_pop_matrix();
			cogl_pop_framebuffer();
		}
		
		draw_buffers(bufferer);
	}
	
	bufferer->priv->cycle++;
	bufferer->priv->buffering = FALSE;
	
	active_bufferer = NULL;
}

static void tangle_bufferer_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleBufferer* bufferer;
	
	bufferer = TANGLE_BUFFERER(object);

	switch (prop_id) {
		case PROP_N_BUFFERS:
			tangle_bufferer_set_n_buffers(bufferer, g_value_get_uint(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_bufferer_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleBufferer* bufferer;

	bufferer = TANGLE_BUFFERER(object);

        switch (prop_id) {
		case PROP_N_BUFFERS:
			g_value_set_uint(value, bufferer->priv->n_buffers);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_bufferer_finalize(GObject* object) {
	TangleBufferer* bufferer;
	
	bufferer = TANGLE_BUFFERER(object);
	
	destroy_resources(bufferer);
	
	g_array_free(bufferer->priv->buffers, TRUE);
	g_ptr_array_free(bufferer->priv->queued_actors, TRUE);

	g_signal_remove_emission_hook(g_signal_lookup("queue-redraw", CLUTTER_TYPE_ACTOR),
	                              bufferer->priv->queue_redraw_hook_id);
	g_signal_remove_emission_hook(g_signal_lookup("allocation-changed", CLUTTER_TYPE_ACTOR),
	                              bufferer->priv->allocation_changed_hook_id);

	G_OBJECT_CLASS(tangle_bufferer_parent_class)->finalize(object);
}

static void tangle_bufferer_dispose(GObject* object) {
	G_OBJECT_CLASS(tangle_bufferer_parent_class)->dispose(object);
}

static void tangle_bufferer_class_init(TangleBuffererClass* bufferer_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(bufferer_class);
	ClutterActorClass* actor_class = CLUTTER_ACTOR_CLASS(bufferer_class);

	gobject_class->finalize = tangle_bufferer_finalize;
	gobject_class->dispose = tangle_bufferer_dispose;
	gobject_class->set_property = tangle_bufferer_set_property;
	gobject_class->get_property = tangle_bufferer_get_property;

#ifdef HAVE_COGL_OFFSCREEN_SET_TO_TEXTURE
	actor_class->allocate = tangle_bufferer_allocate;
	actor_class->paint = tangle_bufferer_paint;
#endif

	/**
	 * TangleBufferer:n-buffers:
	 *
	 * The number of offscreen buffers to be used.
	 */
	g_object_class_install_property(gobject_class, PROP_N_BUFFERS,
	                                g_param_spec_uint("n-buffers",
	                                                  "Number of buffers",
	                                                  "The number of offscreen buffers to be used",
	                                                  0, G_MAXUINT, 10,
	                                                  G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof(TangleBuffererPrivate));

	redrawable_quark = g_quark_from_static_string("tangle-bufferer-cycle");
}

static void tangle_bufferer_init(TangleBufferer* bufferer) {
	bufferer->priv = G_TYPE_INSTANCE_GET_PRIVATE(bufferer, TANGLE_TYPE_BUFFERER, TangleBuffererPrivate);

	bufferer->priv->n_buffers = 10;

	bufferer->priv->buffers = g_array_new(FALSE, FALSE, sizeof(Buffer));
	bufferer->priv->queued_actors = g_ptr_array_new();


#ifdef HAVE_COGL_OFFSCREEN_SET_TO_TEXTURE
	bufferer->priv->queue_redraw_hook_id =
		g_signal_add_emission_hook(g_signal_lookup("queue-redraw", CLUTTER_TYPE_ACTOR),
		                           0, queue_redraw_hook, bufferer, NULL);
	bufferer->priv->allocation_changed_hook_id =
		g_signal_add_emission_hook(g_signal_lookup("allocation-changed", CLUTTER_TYPE_ACTOR),
		                           0, allocation_changed_hook, bufferer, NULL);
#endif
}


static gboolean queue_redraw_hook(GSignalInvocationHint* hint, guint n_values, const GValue* values, gpointer user_data) {
	TangleBufferer* bufferer;
	ClutterActor* actor;
	ClutterActor* origin;
	
	bufferer = TANGLE_BUFFERER(user_data);	

	actor = CLUTTER_ACTOR(g_value_get_object(&values[0]));
	origin = CLUTTER_ACTOR(g_value_get_object(&values[1]));
	
	if (actor == origin && actor != CLUTTER_ACTOR(bufferer)) {
		queue_actor(bufferer, actor, FALSE);
	}

	return TRUE;
}


static gboolean allocation_changed_hook(GSignalInvocationHint* hint, guint n_values, const GValue* values, gpointer user_data) {
	ClutterActor* actor;
	
	actor = CLUTTER_ACTOR(g_value_get_object(&values[0]));
	
	if (active_bufferer && actor != CLUTTER_ACTOR(active_bufferer)) {
		queue_actor(active_bufferer, actor, TRUE);
	}

	return TRUE;
}

static void begin_paint_actor(ClutterActor* actor, gpointer user_data) {
	TangleBufferer* bufferer;
	
	bufferer = TANGLE_BUFFERER(user_data);
	
	if (bufferer->priv->buffering) {
		switch_buffer(bufferer, actor);
	}
}

static void end_paint_actor(ClutterActor* actor, gpointer user_data) {
	TangleBufferer* bufferer;
	
	bufferer = TANGLE_BUFFERER(user_data);
	
	if (bufferer->priv->buffering) {
		switch_buffer(bufferer, NULL);
	}
}

static void dequeue_actor(gpointer data) {
	Redrawable* redrawable;

	redrawable = (Redrawable*)data;	

	g_ptr_array_remove(redrawable->bufferer->priv->queued_actors, redrawable->actor);
	g_free(redrawable);
}

static void queue_actor(TangleBufferer* bufferer, ClutterActor* actor, gboolean allocation_changed) {
	Redrawable* redrawable;

	if (!(redrawable = (Redrawable*)g_object_get_qdata(G_OBJECT(actor), redrawable_quark))) {
		redrawable = g_new0(Redrawable, 1);
		redrawable->bufferer = bufferer;
		redrawable->actor = actor;

		g_object_set_qdata_full(G_OBJECT(actor), redrawable_quark, redrawable, dequeue_actor);
		g_ptr_array_add(bufferer->priv->queued_actors, actor);
	}
	
	g_return_if_fail(redrawable->bufferer = bufferer);

	redrawable->cycle = bufferer->priv->cycle;
	if (allocation_changed) {
		redrawable->has_matrix = FALSE;
	}
}

static gboolean check_queued_actor(TangleBufferer* bufferer, ClutterActor* actor, guint* n_redrawable_actors_pointer) {
	gboolean should_remove = FALSE;
	Redrawable* redrawable;
	gboolean should_redraw = TRUE;
	ClutterActor* ancestor;
	Redrawable* ancestor_redrawable;
	
	redrawable = (Redrawable*)g_object_get_qdata(G_OBJECT(actor), redrawable_quark);	
	if (redrawable->cycle != bufferer->priv->cycle) {
		should_redraw = FALSE;
		should_remove = TRUE;
	} else if (clutter_actor_get_opacity(actor) == 0 || !CLUTTER_ACTOR_IS_MAPPED(actor)) {
		should_redraw = FALSE;
	} else {
		ancestor = actor;
		while ((ancestor = clutter_actor_get_parent(ancestor)) && ancestor != CLUTTER_ACTOR(bufferer)) {
			if ((ancestor_redrawable = g_object_get_qdata(G_OBJECT(ancestor), redrawable_quark)) &&
			    ancestor_redrawable->cycle == bufferer->priv->cycle) {
				should_redraw = FALSE;
				break;
			}
		}
		if (ancestor != CLUTTER_ACTOR(bufferer)) {
			should_redraw = FALSE;
			should_remove = TRUE;
		}
	}
	
	if (should_redraw) {
		(*n_redrawable_actors_pointer)++;
		
		if (!redrawable->prepaint_signal_handler) {
			/* TODO: This could be universal, but the problem is that
			 * effects, such as ClutterOffscreenEffect, may have already
			 * pushed a new framebuffer, because paint is signaled after
			 * effects. Thus, handle switching in TangleWidget.
			 *
			redrawable->prepaint_signal_handler = g_signal_connect(actor, "paint", G_CALLBACK(begin_paint_actor), bufferer);
			redrawable->postpaint_signal_handler = g_signal_connect_after(actor, "paint", G_CALLBACK(end_paint_actor), bufferer);
			 */
			redrawable->prepaint_signal_handler = 1;

			bufferer->priv->buffering = TRUE;
		}
	} else {
		if (redrawable->prepaint_signal_handler) {
			/* TODO: see above.
			g_signal_handler_disconnect(actor, redrawable->prepaint_signal_handler);
			g_signal_handler_disconnect(actor, redrawable->postpaint_signal_handler);
			 */
			redrawable->prepaint_signal_handler = 0;

			bufferer->priv->buffering = TRUE;
		}
	}
	
	return should_remove;
}

static void create_resources(TangleBufferer* bufferer) {
	guint i;
	CoglHandle texture;
	Buffer* buffer;
	ClutterPerspective perspective;
	CoglMatrix projection_matrix;
	CoglMatrix modelview_matrix;
	gfloat width, height;
	
	for (i = 0; i < bufferer->priv->n_buffers; i++) {
		if ((texture = cogl_texture_new_with_size(bufferer->priv->absolute_width, bufferer->priv->absolute_height, COGL_TEXTURE_NO_SLICING, COGL_PIXEL_FORMAT_RGBA_8888_PRE)) == COGL_INVALID_HANDLE) {
			break;
		}
		
		g_array_append_val(bufferer->priv->buffers, texture);
		g_array_index(bufferer->priv->buffers, Buffer, i).texture = texture;
	}
	
	if (bufferer->priv->buffers->len > 0) {
		texture = g_array_index(bufferer->priv->buffers, Buffer, 0).texture;
		if ((bufferer->priv->offscreen = cogl_offscreen_new_to_texture(texture)) != COGL_INVALID_HANDLE) {
			clutter_stage_get_perspective(CLUTTER_STAGE(clutter_actor_get_stage(CLUTTER_ACTOR(bufferer))), &perspective);
			cogl_matrix_init_identity(&projection_matrix);
			cogl_matrix_perspective(&projection_matrix, perspective.fovy, perspective.aspect, perspective.z_near, perspective.z_far);
			cogl_get_modelview_matrix(&modelview_matrix);

			cogl_push_framebuffer(bufferer->priv->offscreen);

			clutter_actor_get_size(clutter_actor_get_stage(CLUTTER_ACTOR(bufferer)), &width, &height);
			cogl_set_viewport(-bufferer->priv->absolute_x, -bufferer->priv->absolute_y, width, height);
			cogl_set_projection_matrix(&projection_matrix);
			cogl_set_modelview_matrix(&modelview_matrix);

			cogl_pop_framebuffer();
		}
	}
	
	bufferer->priv->resources_created = TRUE;
	
	bufferer->priv->buffering = TRUE;
}

static void destroy_resources(TangleBufferer* bufferer) {
	guint i;

	for (i = 0; i < bufferer->priv->buffers->len; i++) {
		cogl_handle_unref(g_array_index(bufferer->priv->buffers, Buffer, i).texture);
	}
	g_array_set_size(bufferer->priv->buffers, 0);
	
	if (bufferer->priv->offscreen != COGL_INVALID_HANDLE) {
		cogl_handle_unref(bufferer->priv->offscreen);
		bufferer->priv->offscreen = COGL_INVALID_HANDLE;
	}
	
	bufferer->priv->resources_created = FALSE;
}

static void clear_buffers(TangleBufferer* bufferer) {
	guint i;
	Buffer* buffer;
	CoglColor color;
	
	for (i = 0; i < bufferer->priv->n_redrawable_actors * 2 + 1; i++) {
		buffer = &g_array_index(bufferer->priv->buffers, Buffer, i);

#ifdef HAVE_COGL_OFFSCREEN_SET_TO_TEXTURE
		g_return_if_fail(cogl_offscreen_set_to_texture(bufferer->priv->offscreen, buffer->texture));
#endif

		cogl_color_init_from_4ub(&color, 0, 0, 0, 0);
		cogl_clear(&color, COGL_BUFFER_BIT_COLOR | COGL_BUFFER_BIT_DEPTH);
	}
}

static void switch_buffer(TangleBufferer* bufferer, ClutterActor* actor) {
	Buffer* buffer;
	Redrawable* redrawable;
	CoglColor color;

	g_return_if_fail(bufferer->priv->buffer_index < bufferer->priv->buffers->len);

	buffer = &g_array_index(bufferer->priv->buffers, Buffer, bufferer->priv->buffer_index++);
	if (actor) {
		redrawable = (Redrawable*)g_object_get_qdata(G_OBJECT(actor), redrawable_quark);
		buffer->redrawable = redrawable;
		redrawable->has_matrix = FALSE;	
	} else {
		buffer->redrawable = NULL;
	}

#ifdef HAVE_COGL_OFFSCREEN_SET_TO_TEXTURE
	g_return_if_fail(cogl_offscreen_set_to_texture(bufferer->priv->offscreen, buffer->texture));
#endif

	cogl_color_init_from_4ub(&color, 0, 0, 0, 0);
	cogl_clear(&color, COGL_BUFFER_BIT_COLOR | COGL_BUFFER_BIT_DEPTH);
}

static void draw_buffers(TangleBufferer* bufferer) {
	guint i;
	Buffer* buffer;
	guint8 paint_opacity;
	ClutterActor* actor;
	CoglMatrix matrix;
	CoglMatrix result_matrix;
	gfloat x, y, w, h;

	for (i = 0; i < bufferer->priv->buffer_index; i++) {
		buffer = &g_array_index(bufferer->priv->buffers, Buffer, i);

		if (bufferer->priv->buffering || !buffer->redrawable) {
			cogl_set_source_texture(buffer->texture);
			cogl_rectangle_with_texture_coords(0, 0, bufferer->priv->absolute_width, bufferer->priv->absolute_height, 0.0, 0.0, 1.0, 1.0);
		} else {
			if (!buffer->redrawable->has_matrix) {
				cogl_matrix_init_identity(&buffer->redrawable->matrix);
				actor = buffer->redrawable->actor;
				while ((actor = clutter_actor_get_parent(actor)) && actor != CLUTTER_ACTOR(bufferer)) {
					clutter_actor_get_transformation_matrix(actor, &matrix);
					cogl_matrix_multiply(&result_matrix, &matrix, &buffer->redrawable->matrix);
					buffer->redrawable->matrix = result_matrix;
				}
				buffer->redrawable->has_matrix = TRUE;
			}

			cogl_push_matrix();
			cogl_transform(&buffer->redrawable->matrix);
			
			if (clutter_actor_has_clip(buffer->redrawable->actor)) {
				clutter_actor_get_clip(buffer->redrawable->actor, &x, &y, &w, &h);
				cogl_clip_push_rectangle(x, y, x + w, y + h);
				
				clutter_actor_paint(buffer->redrawable->actor);
				
				cogl_clip_pop();
			} else {
				clutter_actor_paint(buffer->redrawable->actor);
			}
			
			cogl_pop_matrix();
		}
	}
}

static void get_absolute_allocation_box(ClutterActor* actor, ClutterActorBox* actor_box) {
	ClutterVertex vertices[4];
	gfloat min_x = G_MAXFLOAT, min_y = G_MAXFLOAT;
	gfloat max_x = 0, max_y = 0;
	gint i;
	gfloat viewport[4];

	clutter_actor_get_abs_allocation_vertices(actor, vertices);
	for (i = 0; i < G_N_ELEMENTS(vertices); i++) {
		if (vertices[i].x < min_x) {
			min_x = vertices[i].x;
		}
		if (vertices[i].y < min_y) {
			min_y = vertices[i].y;
		}
		if (vertices[i].x > max_x) {
			max_x = vertices[i].x;
		}
		if (vertices[i].y > max_y) {
			max_y = vertices[i].y;
		}
	}

	cogl_get_viewport(viewport);

#define ROUND(x) ((x) >= 0 ? (long)((x) + 0.5) : (long)((x) - 0.5))

	actor_box->x1 = ROUND(min_x) - viewport[0];
	actor_box->x2 = ROUND(max_x) - viewport[1];
	actor_box->y1 = ROUND(min_y) - viewport[0];
	actor_box->y2 = ROUND(max_y) - viewport[1];

#undef ROUND

	if (actor_box->x2 - actor_box->x1 < 1) {
		actor_box->x2 = actor_box->x1 + 1;
	}
	if (actor_box->y2 - actor_box->y1 < 1) {
		actor_box->y2 = actor_box->y1 + 1;
	}
}
