/*
 * tangle-drop-action.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-drop-action.h"
#include "marshalers.h"

G_DEFINE_TYPE(TangleDropAction, tangle_drop_action, CLUTTER_TYPE_ACTION);

enum {
	PROP_0,
	PROP_DRAG_ACTIVE
};

enum {
	DRAG_BEGIN,
	DRAG_MOTION,
	DRAG_END,
	DROPPED,
	LAST_SIGNAL
};


struct _TangleDropActionPrivate {
	guint drag_active : 1;
};

static guint signals[LAST_SIGNAL] = { 0 };

ClutterAction* tangle_drop_action_new() {

	return CLUTTER_ACTION(g_object_new(TANGLE_TYPE_DROP_ACTION, NULL));
}

gboolean tangle_drop_action_drag_begin(TangleDropAction* drop_action, TangleDragAction* drag_action) {
	gboolean stop_drag = FALSE;

	g_return_val_if_fail(TANGLE_IS_DROP_ACTION(drop_action), FALSE);
	g_return_val_if_fail(!drop_action->priv->drag_active, FALSE);
	
	g_object_ref(drag_action);
	g_signal_emit(drop_action, signals[DRAG_BEGIN], 0, drag_action, &stop_drag);
	g_object_unref(drag_action);

	if (!stop_drag) {
		drop_action->priv->drag_active = TRUE;
		g_object_notify(G_OBJECT(drop_action), "drag-active");
	}
	
	return stop_drag;
}

gboolean tangle_drop_action_drag_motion(TangleDropAction* drop_action, TangleDragAction* drag_action) {
	gboolean stop_drag = FALSE;
	
	g_return_val_if_fail(TANGLE_IS_DROP_ACTION(drop_action), FALSE);
	g_return_val_if_fail(drop_action->priv->drag_active, FALSE);
	
	g_object_ref(drag_action);

	g_signal_emit(drop_action, signals[DRAG_MOTION], 0, drag_action, &stop_drag);

	if (stop_drag) {
		drop_action->priv->drag_active = FALSE;
		g_object_notify(G_OBJECT(drop_action), "drag-active");

		g_signal_emit(drop_action, signals[DRAG_END], 0, drag_action);
	}

	g_object_unref(drag_action);
	
	return stop_drag;
}

gboolean tangle_drop_action_drag_end(TangleDropAction* drop_action, TangleDragAction* drag_action) {
	gboolean stop_drag = FALSE;

	g_return_val_if_fail(TANGLE_IS_DROP_ACTION(drop_action), FALSE);
	g_return_val_if_fail(drop_action->priv->drag_active, FALSE);
	
	g_object_ref(drag_action);
	g_signal_emit(drop_action, signals[DRAG_END], 0, drag_action, &stop_drag);
	g_object_unref(drag_action);

	if (drop_action->priv->drag_active) {
		drop_action->priv->drag_active = FALSE;
		g_object_notify(G_OBJECT(drop_action), "drag-active");
	}
	
	return stop_drag;
}

void tangle_drop_action_dropped(TangleDropAction* drop_action, TangleDragAction* drag_action) {
	g_return_if_fail(TANGLE_IS_DROP_ACTION(drop_action));

	g_object_ref(drag_action);
	g_signal_emit(drop_action, signals[DROPPED], 0, drag_action);
	g_object_unref(drag_action);
}

static gboolean tangle_drop_action_real_drag_begin(TangleDropAction* drop_action, TangleDragAction* drag_action) {
	
	return FALSE;
}

static gboolean tangle_drop_action_real_drag_motion(TangleDropAction* drop_action, TangleDragAction* drag_action) {

	return FALSE;
}

static gboolean tangle_drop_action_real_drag_end(TangleDropAction* drop_action, TangleDragAction* drag_action) {

	return FALSE;
}

static void tangle_drop_action_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleDropAction* drop_action;
	
	drop_action = TANGLE_DROP_ACTION(object);

	switch (prop_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_drop_action_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleDropAction* drop_action;

	drop_action = TANGLE_DROP_ACTION(object);

        switch (prop_id) {
		case PROP_DRAG_ACTIVE:
			g_value_set_boolean(value, drop_action->priv->drag_active);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_drop_action_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_drop_action_parent_class)->finalize(object);
}

static void tangle_drop_action_dispose(GObject* object) {
	G_OBJECT_CLASS(tangle_drop_action_parent_class)->dispose(object);
}

static void tangle_drop_action_class_init(TangleDropActionClass* drop_action_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(drop_action_class);

	gobject_class->finalize = tangle_drop_action_finalize;
	gobject_class->dispose = tangle_drop_action_dispose;
	gobject_class->set_property = tangle_drop_action_set_property;
	gobject_class->get_property = tangle_drop_action_get_property;

	drop_action_class->drag_begin = tangle_drop_action_real_drag_begin;
	drop_action_class->drag_motion = tangle_drop_action_real_drag_motion;
	drop_action_class->drag_end = tangle_drop_action_real_drag_end;

	/**
	 * TangleDropAction:drag-active:
	 *
	 * Whether a drag is currently active i.e. a #ClutterActor with #TangleDragAction
	 * has been dragged over the #ClutterActor of this #TangleDropAction.
	 */
	g_object_class_install_property(gobject_class, PROP_DRAG_ACTIVE,
	                                g_param_spec_boolean("drag-active",
	                                "Drag active",
	                                "Whether a drag is currently active",
	                                TRUE,
	                                G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	/**
	 * TangleDropAction::drag-begin:
	 * @drop_action: the object which received the signal
	 * @drag_action: a #TangleDragAction that caused this
	 *
	 * When a #ClutterActor with #TangleDragAction is dragged over
	 * a #ClutterActor that has #TangleDropAction first time, this signal is emitted.
	 * Dragging can be accepted by returning FALSE, or rejected by returning TRUE.
	 *
	 * Return value: whether the dragging should be stopped
	 */
	signals[DRAG_BEGIN] = g_signal_new("drag-begin", G_TYPE_FROM_CLASS(gobject_class),
	                                   G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(TangleDropActionClass, drag_begin),
					   g_signal_accumulator_true_handled, NULL,
					   tangle_marshal_BOOLEAN__OBJECT,
					   G_TYPE_BOOLEAN, 1,
					   TANGLE_TYPE_DRAG_ACTION);
	/**
	 * TangleDropAction::drag-motion:
	 * @drop_action: the object which received the signal
	 * @drag_action: a #TangleDragAction that caused this
	 *
	 * When a #ClutterActor with #TangleDragAction is being dragged over
	 * a #ClutterActor that has #TangleDropAction, this signal is emitted.
	 * Dragging can be canceled by returning FALSE, otherwise return TRUE.
	 *
	 * Return value: whether the dragging should be stopped
	 */
	signals[DRAG_MOTION] = g_signal_new("drag-motion", G_TYPE_FROM_CLASS(gobject_class),
	                                    G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(TangleDropActionClass, drag_motion),
					    g_signal_accumulator_true_handled, NULL,
					    tangle_marshal_BOOLEAN__OBJECT,
					    G_TYPE_BOOLEAN, 1,
					    TANGLE_TYPE_DRAG_ACTION);
	/**
	 * TangleDropAction::drag-end:
	 * @drop_action: the object which received the signal
	 * @drag_action: a #TangleDragAction that caused this
	 *
	 * When a #ClutterActor with #TangleDragAction has been dragged away from
	 * a #ClutterActor that has #TangleDropAction, or the drag has been canceled
	 * during motion, this signal is emitted.
	 */
	signals[DRAG_END] = g_signal_new("drag-end", G_TYPE_FROM_CLASS(gobject_class),
	                                 G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(TangleDropActionClass, drag_end),
					 g_signal_accumulator_true_handled, NULL,
					 tangle_marshal_BOOLEAN__OBJECT,
					 G_TYPE_BOOLEAN, 1,
					 TANGLE_TYPE_DRAG_ACTION);
	/**
	 * TangleDropAction::dropped:
	 * @drop_action: the object which received the signal
	 * @drag_action: a #TangleDragAction that caused this
	 *
	 * When a #ClutterActor with #TangleDragAction is dropped over
	 * a #ClutterActor that has #TangleDropAction i.e. mouse button is released,
	 * over the actor, this signal is emitted.
	 */
	signals[DROPPED] = g_signal_new("dropped", G_TYPE_FROM_CLASS(gobject_class),
	                                G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(TangleDropActionClass, dropped),
					NULL, NULL,
					tangle_marshal_VOID__OBJECT,
					G_TYPE_NONE, 1,
					TANGLE_TYPE_DRAG_ACTION);

	g_type_class_add_private (gobject_class, sizeof (TangleDropActionPrivate));
}

static void tangle_drop_action_init(TangleDropAction* drop_action) {
	drop_action->priv = G_TYPE_INSTANCE_GET_PRIVATE(drop_action, TANGLE_TYPE_DROP_ACTION, TangleDropActionPrivate);
}

