/*
 * tangle-style.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-style.h"
#include "tangle-properties.h"
#include "tangle-stylable.h"
#include "tangle-template.h"
#include "tangle-action.h"
#include "tangle-vault.h"
#include "tangle-misc.h"
#include "marshalers.h"

/* We know that this is available in the Clutter library (from clutter-script-private.h). */
GType clutter_script_get_type_from_class (const gchar *name);

/**
 * SECTION:tangle-style
 * @Short_description: Style information related to a specied class of objects
 * @Title: TangleStyle
 */

static void clutter_scriptable_iface_init(ClutterScriptableIface *iface);

G_DEFINE_TYPE_WITH_CODE(TangleStyle, tangle_style, TANGLE_TYPE_OBJECT,
                        G_IMPLEMENT_INTERFACE(CLUTTER_TYPE_SCRIPTABLE, clutter_scriptable_iface_init););

enum {
	PROP_0,
	PROP_FOR_TYPE,
	PROP_FOR_CLASS,
	PROP_FOR_SUBCLASSES,
	PROP_FOR_NAME
};

enum {
	STYLE_CHANGED,
	LAST_SIGNAL
};

struct _TangleStylePrivate {
	GType for_type;
	gchar* for_name;
	TangleProperties* style_properties;
	ClutterScript* script;
	
	guint for_subclasses : 1;
};

static ClutterScriptableIface* parent_scriptable_iface = NULL;
static GQuark quark_object_style = 0;
static GQuark quark_object_style_properties = 0;
static guint signals[LAST_SIGNAL];

static void set_style_property(const gchar* name, const GValue* value, gpointer user_data);
static void unset_style_property(const gchar* name, const GValue* value, gpointer user_data);
static void on_notify(GObject* object, GParamSpec* param_spec, gpointer user_data);
static void on_style_changed(TangleStyle* style, const gchar* name, const GValue* old_value, const GValue* new_value, gpointer user_data);
static void disconnect_style_changed_handler(gpointer user_data, GObject* object);
static void set_object_property(GObject* object, const gchar* name, GType type, const GValue* value);
static void handle_object_reference(TangleStyle* style, const gchar* name, GValue* value);
static void foreach_style_property(const gchar* name, const GValue* value, gpointer user_data);

#define BLOCK_NOTIFY_HANDLER(o,h) g_signal_handlers_block_by_func(o, G_CALLBACK(on_notify), h);
#define UNBLOCK_NOTIFY_HANDLER(o,h) g_signal_handlers_unblock_by_func(o, G_CALLBACK(on_notify), h);

TangleStyle* tangle_style_new(GType for_type) {

	return TANGLE_STYLE(g_object_new(TANGLE_TYPE_STYLE, "for-type", for_type, NULL));
}

GType tangle_style_get_for_type(TangleStyle* style) {

	return style->priv->for_type;
}

const gchar* tangle_style_get_for_class(TangleStyle* style) {

	return g_type_name(style->priv->for_type);
}

const gchar* tangle_style_get_for_name(TangleStyle* style) {

	return style->priv->for_name;
}

void tangle_style_set_for_name(TangleStyle* style, const gchar* for_name) {
	g_free(style->priv->for_name);
	style->priv->for_name = g_strdup(for_name);
	g_object_notify(G_OBJECT(style), "for-name");
}

gboolean tangle_style_get_for_subclasses(TangleStyle* style) {

	return style->priv->for_subclasses;
}

void tangle_style_set_for_subclasses(TangleStyle* style, gboolean for_subclasses) {
	if (style->priv->for_subclasses != for_subclasses) {
		style->priv->for_subclasses = for_subclasses;
		g_object_notify(G_OBJECT(style), "for-subclasses");
	}
}

gboolean tangle_style_get_style_property(TangleStyle* style, const gchar* name, GValue* value) {

	return TANGLE_STYLE_GET_CLASS(style)->get_style_property(style, name, value);
}

void tangle_style_set_style_property(TangleStyle* style, const gchar* name, const GValue* value) {

	return TANGLE_STYLE_GET_CLASS(style)->set_style_property(style, name, value);
}

void tangle_style_foreach_style_property(TangleStyle* style, TanglePropertyCallback callback, gpointer user_data) {

	return TANGLE_STYLE_GET_CLASS(style)->foreach_style_property(style, callback, user_data);
}

gboolean tangle_style_is_for_object(TangleStyle* style, GObject* object) {
	
	return ((style->priv->for_subclasses && g_type_is_a(G_OBJECT_TYPE(object), style->priv->for_type)) ||
	        (!style->priv->for_subclasses && G_OBJECT_TYPE(object) == style->priv->for_type)) &&
	       (!CLUTTER_IS_ACTOR(object) ||
	        !style->priv->for_name ||
		(clutter_actor_get_name(CLUTTER_ACTOR(object)) &&
		 !strcmp(style->priv->for_name, clutter_actor_get_name(CLUTTER_ACTOR(object)))));
}

void tangle_style_apply(TangleStyle* style, GObject* object) {
	TangleStyle* old_style;
	GHashTable* hash_table;

	old_style = TANGLE_STYLE(g_object_get_qdata(object, quark_object_style));
	if (old_style != style) {
		g_object_ref(style);

		if (old_style) {
			/* TODO: only unset properties that are not part of a new style? */
			tangle_style_unapply(old_style, object);
		}

		hash_table = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
		g_object_set_qdata_full(object, quark_object_style_properties, hash_table, (GDestroyNotify)g_hash_table_unref);
		g_signal_connect(object, "notify", G_CALLBACK(on_notify), hash_table);

		g_object_set_qdata(object, quark_object_style, style);
		tangle_style_foreach_style_property(style, set_style_property, object);
				
		g_signal_connect(style, "style-changed", G_CALLBACK(on_style_changed), object);
		g_object_weak_ref(object, disconnect_style_changed_handler, style);
	}
}

void tangle_style_unapply(TangleStyle* style, GObject* object) {
	TangleStyle* old_style;
	GHashTable* hash_table;
	
	old_style = TANGLE_STYLE(g_object_get_qdata(object, quark_object_style));
	if (old_style == style) {
		tangle_style_foreach_style_property(style, unset_style_property, object);
		g_object_set_qdata(object, quark_object_style, NULL);
		
		hash_table = (GHashTable*)g_object_get_qdata(object, quark_object_style_properties);
		if (hash_table) {
			g_object_set_qdata(object, quark_object_style_properties, NULL);
			g_signal_handlers_disconnect_by_func(object, G_CALLBACK(on_notify), hash_table);
		}
		
		g_signal_handlers_disconnect_by_func(style, G_CALLBACK(on_style_changed), object);
		g_object_weak_unref(object, disconnect_style_changed_handler, style);

		g_object_unref(style);
	}
}

TangleStyle* tangle_object_get_style(GObject* object) {
	
	return TANGLE_STYLE(g_object_get_qdata(object, quark_object_style));
}

gboolean tangle_object_is_style_property(GObject* object, const gchar* name) {
	gboolean retvalue = FALSE;
	GHashTable* hash_table;
	
	hash_table = (GHashTable*)g_object_get_qdata(object, quark_object_style_properties);
	if (hash_table) {
		retvalue = g_hash_table_lookup_extended(hash_table, name, NULL, NULL);
	}
	
	return retvalue;
}

static gboolean tangle_style_get_style_property_impl(TangleStyle* style, const gchar* name, GValue* value) {
	gboolean retvalue;
	
	if (retvalue = tangle_properties_get_property(style->priv->style_properties, name, value)) {
		handle_object_reference(style, name, value);
	}

	return retvalue;
}

static void tangle_style_set_style_property_impl(TangleStyle* style, const gchar* name, const GValue* value) {
	GValue old_value = { 0 };

	tangle_properties_get_property(style->priv->style_properties, name, &old_value);
	tangle_properties_set_property(style->priv->style_properties, name, value);
	g_signal_emit(style, signals[STYLE_CHANGED], 0, name, old_value, value);
}

static void tangle_style_foreach_style_property_impl(TangleStyle* style, TanglePropertyCallback callback, gpointer user_data) {
	TangleVault* vault;
	
	vault = tangle_vault_new(3, TANGLE_TYPE_STYLE, style, G_TYPE_POINTER, callback, G_TYPE_POINTER, user_data);

	return tangle_properties_foreach(style->priv->style_properties, foreach_style_property, vault);
}

static void tangle_style_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleStyle* style;
	GType type;
	const gchar* string;
	
	style = TANGLE_STYLE(object);

	switch (prop_id) {
		case PROP_FOR_TYPE:
			if ((type = g_value_get_gtype(value)) != G_TYPE_NONE) {
				if (style->priv->for_type != G_TYPE_NONE) {
					g_warning("Trying to set a for-type of the style that already has it, ignored.");
				} else {
					style->priv->for_type = type;
				}
			}
			break;	
		case PROP_FOR_CLASS:
			if ((string = g_value_get_string(value))) {
				if (style->priv->for_type != G_TYPE_NONE) {
					g_warning("Trying to set a for-class of the style that already has for-type, ignored.");
				} else if ((type = g_type_from_name(string)) != G_TYPE_INVALID) {
					style->priv->for_type = type;
				} else if ((type = clutter_script_get_type_from_class(string)) != G_TYPE_INVALID) {
					style->priv->for_type = type;
				} else {
					g_warning("No such class: %s.", string);
				}
			}
			break;
		case PROP_FOR_SUBCLASSES:
			tangle_style_set_for_subclasses(style, g_value_get_boolean(value));
		case PROP_FOR_NAME:
			tangle_style_set_for_name(style, g_value_get_string(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_style_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleStyle* style;

	style = TANGLE_STYLE(object);

        switch (prop_id) {
		case PROP_FOR_TYPE:
			g_value_set_gtype(value, style->priv->for_type);
			break;
		case PROP_FOR_CLASS:
			g_value_set_string(value, tangle_style_get_for_class(style));
			break;
		case PROP_FOR_SUBCLASSES:
			g_value_set_boolean(value, style->priv->for_subclasses);
		case PROP_FOR_NAME:
			g_value_set_string(value, style->priv->for_name);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_style_constructed(GObject* object) {
	TangleStyle* style;
	
	style = TANGLE_STYLE(object);
	
	if (style->priv->for_type == G_TYPE_NONE) {
		g_critical("No valid for-type or for-class provided for style.");
	}
}

static void tangle_style_finalize(GObject* object) {
	TangleStyle* style;
	
	style = TANGLE_STYLE(object);

	g_free(style->priv->for_name);

	G_OBJECT_CLASS(tangle_style_parent_class)->finalize(object);
}

static void tangle_style_dispose(GObject* object) {
	TangleStyle* style;
	
	style = TANGLE_STYLE(object);
	
	TANGLE_UNREF_AND_NULLIFY_OBJECT(style->priv->style_properties);
	
	G_OBJECT_CLASS(tangle_style_parent_class)->dispose(object);
}

static void tangle_style_class_init(TangleStyleClass* style_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(style_class);

	style_class->get_style_property = tangle_style_get_style_property_impl;
	style_class->set_style_property = tangle_style_set_style_property_impl;
	style_class->foreach_style_property = tangle_style_foreach_style_property_impl;

	gobject_class->constructed = tangle_style_constructed;
	gobject_class->finalize = tangle_style_finalize;
	gobject_class->dispose = tangle_style_dispose;
	gobject_class->set_property = tangle_style_set_property;
	gobject_class->get_property = tangle_style_get_property;

	/**
	 * TangleStyle:for-type:
	 */
	g_object_class_install_property(gobject_class, PROP_FOR_TYPE,
	                                g_param_spec_gtype("for-type",
	                                "For type",
	                                "The type of the class that is target of this style",
	                                G_TYPE_NONE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleStyle:for-class:
	 */
	g_object_class_install_property(gobject_class, PROP_FOR_CLASS,
	                                g_param_spec_string("for-class",
	                                "For class",
	                                "The name of the class that is target of this style",
	                                NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleStyle:for-subclasses:
	 */
	g_object_class_install_property(gobject_class, PROP_FOR_SUBCLASSES,
	                                g_param_spec_boolean("for-subclasses",
	                                "For subclasses",
	                                "Whether the style is applied to subclasses of the for-type/for-class",
	                                TRUE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleStyle:for-name:
	 */
	g_object_class_install_property(gobject_class, PROP_FOR_NAME,
	                                g_param_spec_string("for-name",
	                                "For name",
	                                "The name of the clutter actor that is target of this style",
	                                NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	signals[STYLE_CHANGED] = g_signal_new("style-changed",
	                                      G_TYPE_FROM_CLASS(gobject_class),
					      G_SIGNAL_RUN_LAST,
					      0,
					      NULL, NULL,
					      tangle_marshal_VOID__STRING_POINTER_POINTER,
					      G_TYPE_NONE, 3,
					      G_TYPE_STRING,
					      G_TYPE_VALUE,
					      G_TYPE_VALUE);

	quark_object_style = g_quark_from_static_string("tangle-object-style");
	quark_object_style_properties = g_quark_from_static_string("tangle-object-style-properties");
	g_type_class_add_private(gobject_class, sizeof(TangleStylePrivate));
}

static void tangle_style_init(TangleStyle* style) {
	style->priv = G_TYPE_INSTANCE_GET_PRIVATE(style, TANGLE_TYPE_STYLE, TangleStylePrivate);
	style->priv->style_properties = tangle_properties_new();
	style->priv->for_type = G_TYPE_NONE;
	style->priv->for_subclasses = TRUE;	
}

static void tangle_style_set_custom_property(ClutterScriptable* scriptable, ClutterScript* script,  const gchar* name, const GValue* value) {
	TangleStyle* style;

	style = TANGLE_STYLE(scriptable);

	g_assert(!style->priv->script || style->priv->script == script);

	if (!style->priv->script) {
		/* TODO: This should be weak, should it not? */
		g_object_ref(script);
		style->priv->script = script;
	}

	if (g_object_class_find_property(G_OBJECT_CLASS(TANGLE_STYLE_GET_CLASS(style)), name)) {
		if (parent_scriptable_iface->set_custom_property) {
			parent_scriptable_iface->set_custom_property(scriptable, script, name, value);
		} else {
			g_object_set_property(G_OBJECT(scriptable), name, value);
		}
	} else {
		tangle_style_set_style_property(style, name, value);
	}
}

static void clutter_scriptable_iface_init(ClutterScriptableIface *iface) {
	if (!(parent_scriptable_iface = parent_scriptable_iface = g_type_interface_peek_parent (iface))) {
		parent_scriptable_iface = g_type_default_interface_peek(CLUTTER_TYPE_SCRIPTABLE);
	}

	iface->set_custom_property = tangle_style_set_custom_property;
}

static void set_style_property(const gchar* name, const GValue* value, gpointer user_data) {
	GObject* object;
	gboolean handled = FALSE;
	GHashTable* hash_table;
	GParamSpec* param_spec;
	GValue current_value = { 0 };

	object = G_OBJECT(user_data);
	if (TANGLE_IS_STYLABLE(object)) {
		handled = tangle_stylable_set_style_property(TANGLE_STYLABLE(object), name, value); 
	}
	if (!handled &&
	    (hash_table = (GHashTable*)g_object_get_qdata(object, quark_object_style_properties))) {
	    	if (G_VALUE_HOLDS(value, TANGLE_TYPE_ACTION)) {
			tangle_action_add(TANGLE_ACTION(g_value_get_object(value)), object, name);
		} else if (param_spec = g_object_class_find_property(G_OBJECT_GET_CLASS(object), name)) {
			g_value_init(&current_value, G_PARAM_SPEC_VALUE_TYPE(param_spec));
			g_object_get_property(object, name, &current_value);
			if (g_param_value_defaults(param_spec, &current_value)) {
				BLOCK_NOTIFY_HANDLER(object, hash_table);
				set_object_property(object, name, G_PARAM_SPEC_VALUE_TYPE(param_spec), value);
				UNBLOCK_NOTIFY_HANDLER(object, hash_table);
				g_hash_table_insert(hash_table, g_strdup(name), NULL);
			}
			g_value_unset(&current_value);
		} else {
			g_warning("%s: object does not have property called '%s'.", G_STRLOC, name);
		}
	}
}

static void unset_style_property(const gchar* name, const GValue* value, gpointer user_data) {
	GObject* object;
	gboolean handled = FALSE;
	GHashTable* hash_table;
	GParamSpec* param_spec;
	GValue current_value = { 0 };
	
	object = G_OBJECT(user_data);
	if (TANGLE_IS_STYLABLE(object)) {
		handled = tangle_stylable_unset_style_property(TANGLE_STYLABLE(object), name, value); 
	}
	if (!handled &&
	    (hash_table = (GHashTable*)g_object_get_qdata(object, quark_object_style_properties)) &&
	    g_hash_table_remove(hash_table, name)) {
		BLOCK_NOTIFY_HANDLER(object, hash_table);
		if (G_VALUE_HOLDS(value, TANGLE_TYPE_ACTION)) {
			tangle_action_remove(TANGLE_ACTION(g_value_get_object(value)), object, name);		
		} else if (param_spec = g_object_class_find_property(G_OBJECT_GET_CLASS(object), name)) {
			g_value_init(&current_value, G_PARAM_SPEC_VALUE_TYPE(param_spec));
			g_param_value_set_default(param_spec, &current_value);
			set_object_property(object, name, G_PARAM_SPEC_VALUE_TYPE(param_spec), &current_value);
			g_value_unset(&current_value);
		} else {
			g_warning("%s: object does not have property called '%s'.", G_STRLOC, name);
		}
		UNBLOCK_NOTIFY_HANDLER(object, hash_table);
	}
}

static void change_style_property(GObject* object, const gchar* name, const GValue* old_value, const GValue* new_value) {
	gboolean handled = FALSE;
	GHashTable* hash_table;
	GValue current_value = { 0 };
	GParamSpec* param_spec;

	if (TANGLE_IS_STYLABLE(object)) {
		handled = tangle_stylable_set_style_property(TANGLE_STYLABLE(object), name, new_value); 
	}
	if (!handled &&
	    (hash_table = (GHashTable*)g_object_get_qdata(object, quark_object_style_properties)) &&
	    g_hash_table_lookup_extended(hash_table, name, NULL, NULL)) {
		BLOCK_NOTIFY_HANDLER(object, hash_table);
		if (G_VALUE_HOLDS(old_value, TANGLE_TYPE_ACTION)) {
			tangle_action_remove(TANGLE_ACTION(g_value_get_object(old_value)), object, name);
			tangle_action_add(TANGLE_ACTION(g_value_get_object(new_value)), object, name);
		} else if (param_spec = g_object_class_find_property(G_OBJECT_GET_CLASS(object), name)) {
			set_object_property(object, name, G_PARAM_SPEC_VALUE_TYPE(param_spec), new_value);
		} else {
			g_warning("%s: object does not have property called '%s'.", G_STRLOC, name);
		}
		UNBLOCK_NOTIFY_HANDLER(object, hash_table);
	}
}

static void on_notify(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	GHashTable* hash_table;
	
	hash_table = (GHashTable*)user_data;
	g_hash_table_remove(hash_table, g_param_spec_get_name(param_spec));
}

static void on_style_changed(TangleStyle* style, const gchar* name, const GValue* old_value, const GValue* new_value, gpointer user_data) {
	GObject* object;
	GHashTable* hash_table;
	
	object = G_OBJECT(user_data);
	if (old_value && new_value) {
		change_style_property(object, name, old_value, new_value);
	} else if (new_value) {
		set_style_property(name, new_value, user_data);
	} else if (old_value) {
		unset_style_property(name, old_value, user_data);
	}
}

static void disconnect_style_changed_handler(gpointer user_data, GObject* object) {
	TangleStyle* style;
	
	style = TANGLE_STYLE(user_data);
	g_signal_handlers_disconnect_by_func(style, G_CALLBACK(on_style_changed), object);
}

static void set_object_property(GObject* object, const gchar* name, GType type, const GValue* value) {
	TangleTemplate* template;
	GObject* value_object;
	GValue object_value = { 0 };
	TangleAction* action;
	
	if (G_VALUE_HOLDS(value, TANGLE_TYPE_TEMPLATE) && !g_type_is_a(type, TANGLE_TYPE_TEMPLATE)) {
		template = TANGLE_TEMPLATE(g_value_get_object(value));
		value_object = tangle_template_create_object(template);
		g_object_ref_sink(value_object);
		g_value_init(&object_value, tangle_template_get_object_type(template));
		g_value_set_object(&object_value, value_object);
		g_object_set_property(object, name, &object_value);
		g_value_unset(&object_value);
		g_object_unref(value_object);
	} else {
		g_object_set_property(object, name, value);
	}
}

static void handle_object_reference(TangleStyle* style, const gchar* name, GValue* value) {
	gpointer type_class;
	GParamSpec* param_spec;
	GObject* object;

	if (G_VALUE_HOLDS_STRING(value) &&
	    style->priv->script &&
	    style->priv->for_type != G_TYPE_NONE) {
		type_class = g_type_class_ref(style->priv->for_type);

		if ((param_spec = g_object_class_find_property(G_OBJECT_CLASS(type_class), name)) &&
		    G_IS_PARAM_SPEC_OBJECT(param_spec) &&
		    (object = clutter_script_get_object(style->priv->script, g_value_get_string(value)))) {
			g_value_unset(value);
			g_value_init(value, G_PARAM_SPEC_VALUE_TYPE(param_spec));
			g_value_set_instance(value, object);

			tangle_properties_set_property(style->priv->style_properties, name, value);
		}

		g_type_class_unref(type_class);
	}
}

static void foreach_style_property(const gchar* name, const GValue* value, gpointer user_data) {
	TangleVault* vault;
	TangleStyle* style;
	gpointer callback;
	gpointer callback_data;
	
	vault = TANGLE_VAULT(user_data);
	tangle_vault_get(vault, 3, TANGLE_TYPE_STYLE, &style, G_TYPE_POINTER, &callback, G_TYPE_POINTER, &callback_data);
	
	/* We can omit the constness of value type because we actually own it. */
	handle_object_reference(style, name, (GValue*)value);
	
	TANGLE_PROPERTY_CALLBACK(callback)(name, value, callback_data);
}
