
/*
 * tangle-misc.h
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2009-2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-misc.h"
#include "tangle-stylesheet.h"
#include "tangle-vault.h"
#include <gobject/gvaluecollector.h>
#include <string.h>
#include <clutter/clutter.h>
#include <gmodule.h>
#include <glib.h>

/**
 * SECTION:tangle-misc
 * @Short_description: Miscellaneous utility function
 * @Title: Miscellaneous utilities
 */

static GHashTable* calculation_functions_hash_table = NULL;
static gchar** search_paths = { NULL };

static void on_weak_notify(gpointer user_data, GObject* other_object);

#define DEFINE_CALCULATION_FUNCTIONS(t) \
	static gint compare_##t(const GValue* value, const GValue* comparable) { g##t v; g##t c; \
	v = g_value_get_##t(value); c = g_value_get_##t(comparable); \
	return (v == c ? 0 : (v < c ? -1 : 1)); \
	} \
	static void add_##t(GValue* value, const GValue* addend) { g##t v; g##t a; \
	v = g_value_get_##t(value); a = g_value_get_##t(addend); \
	v += a; g_value_set_##t(value, v); \
	} \
	static void substract_##t(GValue* value, const GValue* substrahend) { g##t v; g##t s; \
	v = g_value_get_##t(value); s = g_value_get_##t(substrahend); \
	v -= s; g_value_set_##t(value, v); \
	} \
	static void multiply_##t(GValue* value, const GValue* multiplier) { g##t v; g##t m; \
	v = g_value_get_##t(value); m = g_value_get_##t(multiplier); \
	v *= m; g_value_set_##t(value, v); \
	} \
	static void divide_##t(GValue* value, const GValue* divider) { g##t v; g##t d; \
	v = g_value_get_##t(value); d = g_value_get_##t(divider); \
	v /= d; g_value_set_##t(value, v); \
	} \
	static TangleValueCalculationFunctions calculation_functions_##t = { \
	compare_##t, add_##t, substract_##t, multiply_##t, divide_##t };

DEFINE_CALCULATION_FUNCTIONS(char)
DEFINE_CALCULATION_FUNCTIONS(uchar)
DEFINE_CALCULATION_FUNCTIONS(int)
DEFINE_CALCULATION_FUNCTIONS(uint)
DEFINE_CALCULATION_FUNCTIONS(long)
DEFINE_CALCULATION_FUNCTIONS(ulong)
DEFINE_CALCULATION_FUNCTIONS(int64)
DEFINE_CALCULATION_FUNCTIONS(uint64)
DEFINE_CALCULATION_FUNCTIONS(float)
DEFINE_CALCULATION_FUNCTIONS(double)

gboolean tangle_init(int* argc, char*** argv) {
	gboolean success = FALSE;
	TangleStylesheet* stylesheet;
	
	if (clutter_init(argc, argv) == CLUTTER_INIT_SUCCESS) {
		tangle_value_register_calculation_functions(G_TYPE_CHAR, &calculation_functions_char);
		tangle_value_register_calculation_functions(G_TYPE_UCHAR, &calculation_functions_uchar);
		tangle_value_register_calculation_functions(G_TYPE_INT, &calculation_functions_int);
		tangle_value_register_calculation_functions(G_TYPE_UINT, &calculation_functions_uint);
		tangle_value_register_calculation_functions(G_TYPE_LONG, &calculation_functions_long);
		tangle_value_register_calculation_functions(G_TYPE_ULONG, &calculation_functions_ulong);
		tangle_value_register_calculation_functions(G_TYPE_INT64, &calculation_functions_int64);
		tangle_value_register_calculation_functions(G_TYPE_UINT64, &calculation_functions_uint64);
		tangle_value_register_calculation_functions(G_TYPE_FLOAT, &calculation_functions_float);
		tangle_value_register_calculation_functions(G_TYPE_DOUBLE, &calculation_functions_double);

		success = TRUE;
	}
	
	return success;
}

gpointer tangle_unref_object(gpointer object) {
	if (object) {
		g_object_unref(object);
	}
	
	return NULL;
}

TanglePointers* tangle_pointers_new(gint n, ...) {
	TanglePointers* pointers;
	va_list args;
	int i;
	gpointer p;

	pointers = (TanglePointers*)g_slice_alloc(n * sizeof(gpointer));

	va_start(args, n);
	for (i = 0; i < n; i++) {
		p = va_arg(args, gpointer);
		pointers[i] = p;		
	}
	va_end(args);

	return pointers;
}

void tangle_pointers_get(TanglePointers* pointers, gint n, ...) {
	va_list args;
	int i;
	gpointer* p_return;

	va_start(args, n);
	for (i = 0; i < n; i++) {
		p_return = va_arg(args, gpointer*);
		*p_return = pointers[i];		
	}
	va_end(args);
}

void tangle_pointers_free(TanglePointers* pointers, gint n) {
	g_slice_free1(n * sizeof(gpointer), pointers);
}

static GValue* find_property_value(guint n_properties, GObjectConstructParam* properties, GParamSpec* param_spec) {
	GValue* value = NULL;
	
	while (n_properties--) {
		if (properties->pspec == param_spec) {
			value = properties->value;
			break;
		}
		properties++;
	}
	
	return value;
}

gboolean tangle_lookup_construct_properties(GType type, guint n_construct_properties, GObjectConstructParam* construct_properties, const gchar* first_property_name, ...) {
	gboolean success = TRUE;
	GObjectClass* object_class;
	va_list args;
	const gchar* name;
	GParamSpec* param_spec;
	GValue* value;
	gchar* error;
	
	object_class = G_OBJECT_CLASS(g_type_class_ref(type));
	g_return_val_if_fail(object_class != NULL, FALSE);
	
	va_start(args, first_property_name);
	for (name = first_property_name; name; name = va_arg(args, const gchar*)) {
		param_spec = g_object_class_find_property(object_class, name);
		if (!param_spec) {
			g_warning("%s: No such property: %s\n", G_STRLOC, name);
			success = FALSE;
			break;
		}
		if (!(value = find_property_value(n_construct_properties, construct_properties, param_spec))) {
			success = FALSE;
		} else {
			error = NULL;
			G_VALUE_LCOPY(value, args, 0, &error);
			if (error) {
				g_warning("%s: %s", G_STRLOC, error);
				g_free(error);
				success = FALSE;
			}
		}
	}
	
	g_type_class_unref(object_class);

	return success;
}

GObject* tangle_construct_with_extra_properties(GObject* (*constructor)(GType, guint, GObjectConstructParam*), GType type, guint n_construct_properties, GObjectConstructParam* construct_properties, const gchar* first_extra_property_name, ...) {
	GObject* object = NULL;
	GList* extra_construct_properties = NULL;
	GObjectClass* object_class;
	va_list args;
	const gchar* name;
	GParamSpec* param_spec;
	GValue* value;
	gboolean existing_value;
	gchar* error;
	GObjectConstructParam* construct_param;
	guint n_all_construct_properties;
	GObjectConstructParam* all_construct_properties;
	GList* list;
	
	object_class = G_OBJECT_CLASS(g_type_class_ref(type));
	g_return_val_if_fail(object_class != NULL, NULL);
		
	va_start(args, first_extra_property_name);
	for (name = first_extra_property_name; name; name = va_arg(args, const gchar*)) {
		param_spec = g_object_class_find_property(object_class, name);
		if (!param_spec) {
			g_warning("%s: No such property: %s\n", G_STRLOC, name);
			break;
		}
		if (value = find_property_value(n_construct_properties, construct_properties, param_spec)) {
			existing_value = TRUE;
			g_value_unset(value);
		} else {
			existing_value = FALSE;
			value = g_slice_new0(GValue);
		}
		g_value_init(value, G_PARAM_SPEC_VALUE_TYPE(param_spec));			
		error = NULL;
		G_VALUE_COLLECT(value, args, 0, &error);
		if (error) {
			g_warning("%s: %s", G_STRLOC, error);
			g_free(error);
		} else if (!existing_value){
			construct_param = g_slice_new(GObjectConstructParam);
			construct_param->pspec = param_spec;
			construct_param->value = value;
			extra_construct_properties = g_list_prepend(extra_construct_properties, construct_param);
		}
	}
	va_end(args);
	
	n_all_construct_properties = n_construct_properties + g_list_length(extra_construct_properties);
	all_construct_properties = g_slice_alloc(n_all_construct_properties * sizeof(GObjectConstructParam));
	memcpy(all_construct_properties, construct_properties, n_construct_properties * sizeof(GObjectConstructParam));
	construct_param = all_construct_properties + n_construct_properties;
	for (list = extra_construct_properties; list; list = list->next) {
		*construct_param = *((GObjectConstructParam*)list->data);
		construct_param++;
	}
	object = constructor(type, n_all_construct_properties, all_construct_properties);

	g_slice_free1(n_all_construct_properties * sizeof(GObjectConstructParam), all_construct_properties);
	for (list = extra_construct_properties; list; list = list->next) {
		g_value_unset(((GObjectConstructParam*)list->data)->value);
		g_slice_free(GValue, ((GObjectConstructParam*)list->data)->value);
		g_slice_free(GObjectConstructParam, list->data);
	}	
	g_list_free(extra_construct_properties);
	g_type_class_unref(object_class);

	return object;
}

gboolean tangle_signal_accumulator_non_null_handled(GSignalInvocationHint* ihint, GValue* return_accu, const GValue* handler_return, gpointer user_data) {
	GObject* object;

	object = g_value_get_object(handler_return);
	g_value_set_object(return_accu, object);

	return !object;
}

static gboolean connect_signal_dynamically(GObject* object, const gchar* signal_name, const gchar* function_name, gpointer user_data, gboolean swapped, gboolean set_weak_notify) {
	gboolean retvalue = FALSE;
	GModule* module = NULL;
	gpointer function_pointer;
	TangleVault* vault;

	g_return_val_if_fail(G_IS_OBJECT(object), FALSE);	
	g_return_val_if_fail(signal_name != NULL, FALSE);
	g_return_val_if_fail(function_name != NULL, FALSE);
	
	if (!g_signal_parse_name(signal_name, G_TYPE_FROM_INSTANCE(object), NULL, NULL, FALSE)) {
		g_warning("No such signal: '%s'.", signal_name);
	} else if (!g_module_supported() || !(module = g_module_open(NULL, 0))) {
		g_warning("The current platform does not support modules.");
	} else if (!g_module_symbol(module, function_name, (gpointer*)&function_pointer)) {
		g_warning("No such function: '%s'", function_name);
	} else {
		if (swapped) {
			retvalue = g_signal_connect_swapped(object, signal_name, G_CALLBACK(function_pointer), user_data) != 0;
			if (set_weak_notify) {
				vault = tangle_vault_new(2, G_TYPE_OBJECT, object, G_TYPE_POINTER, function_pointer);
				g_object_weak_ref(G_OBJECT(user_data), on_weak_notify, vault);
			}
		} else {
			retvalue = g_signal_connect(object, signal_name, G_CALLBACK(function_pointer), user_data) != 0;
		}
		if (!retvalue) {
			g_warning("Failed to connect a handler function '%s' for signal '%s'.", function_name, signal_name);
		}
	}
	if (module) {
		g_module_close(module);
	}
	
	return retvalue;
}

gboolean tangle_signal_connect_dynamically(GObject* object, const gchar* signal_name, const gchar* function_name, gpointer user_data) {

	return connect_signal_dynamically(object, signal_name, function_name, user_data, FALSE, FALSE);
}

gboolean tangle_signal_connect_swapped_dynamically(GObject* object, const gchar* signal_name, const gchar* function_name, gpointer user_data) {

	return connect_signal_dynamically(object, signal_name, function_name, user_data, TRUE, FALSE);
}

gboolean tangle_signal_connect_object_swapped_dynamically(GObject* object, const gchar* signal_name, const gchar* function_name, GObject* other_object) {

	return connect_signal_dynamically(object, signal_name, function_name, other_object, TRUE, TRUE);
}

gboolean tangle_signal_connect_dynamically_from_script(GObject* object, const gchar* signal_name, const gchar* function_name, ClutterScript* script) {
	gboolean retvalue = FALSE;
	const gchar* colon;
	gchar* id;
	GObject* other_object;

	g_return_val_if_fail(CLUTTER_IS_SCRIPT(script), FALSE);
	
	if ((colon = strchr(signal_name, ':')) && colon != signal_name && colon[1] != 0 && colon[1] != ':') {
		id = g_strndup(signal_name, colon - signal_name);
		if ((other_object = clutter_script_get_object(script, id)) &&
		    tangle_signal_connect_object_swapped_dynamically(other_object, colon + 1, function_name, object)) {
			retvalue = TRUE;
		}			
		g_free(id);
	} else if (tangle_signal_connect_dynamically(object, signal_name, function_name, NULL)) {
		retvalue = TRUE;
	}
	
	return retvalue;
}

gint tangle_value_compare(const GValue* value, const GValue* comparable) {
	gint retvalue = 0;
	TangleValueCalculationFunctions* calculation_functions;
	
	g_return_if_fail(G_VALUE_TYPE(value) == G_VALUE_TYPE(comparable));
	
	if (!calculation_functions_hash_table || !(calculation_functions = (TangleValueCalculationFunctions*)g_hash_table_lookup(calculation_functions_hash_table, GUINT_TO_POINTER(G_VALUE_TYPE(value))))) {
		g_critical("No calculation functions for value type '%s'.", G_VALUE_TYPE_NAME(value));
	} else if (!calculation_functions->compare) {
		g_critical("No compare function for value type '%s'.", G_VALUE_TYPE_NAME(value));	
	} else {
		retvalue = calculation_functions->compare(value, comparable);
	}
	
	return retvalue;
}

void tangle_value_add(GValue* value, const GValue* addend) {
	TangleValueCalculationFunctions* calculation_functions;
	
	g_return_if_fail(G_VALUE_TYPE(value) == G_VALUE_TYPE(addend));
	
	if (!calculation_functions_hash_table || !(calculation_functions = (TangleValueCalculationFunctions*)g_hash_table_lookup(calculation_functions_hash_table, GUINT_TO_POINTER(G_VALUE_TYPE(value))))) {
		g_critical("No calculation functions for value type '%s'.", G_VALUE_TYPE_NAME(value));
	} else if (!calculation_functions->add) {
		g_critical("No add function for value type '%s'.", G_VALUE_TYPE_NAME(value));	
	} else {
		calculation_functions->add(value, addend);
	}
}

void tangle_value_substract(GValue* value, const GValue* subtrahend) {
	TangleValueCalculationFunctions* calculation_functions;
	
	g_return_if_fail(G_VALUE_TYPE(value) == G_VALUE_TYPE(subtrahend));
	
	if (!calculation_functions_hash_table || !(calculation_functions = (TangleValueCalculationFunctions*)g_hash_table_lookup(calculation_functions_hash_table, GUINT_TO_POINTER(G_VALUE_TYPE(value))))) {
		g_critical("No calculation functions for value type '%s'.", G_VALUE_TYPE_NAME(value));
	} else if (!calculation_functions->substract) {
		g_critical("No substract function for value type '%s'.", G_VALUE_TYPE_NAME(value));	
	} else {
		calculation_functions->substract(value, subtrahend);
	}
}

void tangle_value_multiply(GValue* value, const GValue* multiplier) {
	TangleValueCalculationFunctions* calculation_functions;
	
	g_return_if_fail(G_VALUE_TYPE(value) == G_VALUE_TYPE(multiplier));
	
	if (!calculation_functions_hash_table || !(calculation_functions = (TangleValueCalculationFunctions*)g_hash_table_lookup(calculation_functions_hash_table, GUINT_TO_POINTER(G_VALUE_TYPE(value))))) {
		g_critical("No calculation functions for value type '%s'.", G_VALUE_TYPE_NAME(value));
	} else if (!calculation_functions->multiply) {
		g_critical("No multiply function for value type '%s'.", G_VALUE_TYPE_NAME(value));	
	} else {
		calculation_functions->multiply(value, multiplier);
	}
}

void tangle_value_divide(GValue* value, const GValue* divider) {
	TangleValueCalculationFunctions* calculation_functions;
	
	g_return_if_fail(G_VALUE_TYPE(value) == G_VALUE_TYPE(divider));
	
	if (!calculation_functions_hash_table || !(calculation_functions = (TangleValueCalculationFunctions*)g_hash_table_lookup(calculation_functions_hash_table, GUINT_TO_POINTER(G_VALUE_TYPE(value))))) {
		g_critical("No calculation functions for value type '%s'.", G_VALUE_TYPE_NAME(value));
	} else if (!calculation_functions->divide) {
		g_critical("No divide function for value type '%s'.", G_VALUE_TYPE_NAME(value));	
	} else {
		calculation_functions->divide(value, divider);
	}
}

gboolean tangle_value_is_calculation_supported(GType value_type) {
	gboolean retvalue = FALSE;
	
	if (calculation_functions_hash_table && g_hash_table_lookup(calculation_functions_hash_table, GUINT_TO_POINTER(value_type))) {
		retvalue = TRUE;
	}
	
	return retvalue;
}

void tangle_value_register_calculation_functions(GType value_type, TangleValueCalculationFunctions* calculation_functions) {
	TangleValueCalculationFunctions* calculation_functions_copy;

	if (calculation_functions_hash_table && g_hash_table_lookup(calculation_functions_hash_table, GUINT_TO_POINTER(value_type))) {
		g_critical("Calculation functions already exists for value type '%s'.", g_type_name(value_type));
	} else {
		if (!calculation_functions_hash_table) {
			calculation_functions_hash_table = g_hash_table_new(NULL, NULL);
		}
		calculation_functions_copy = g_new(TangleValueCalculationFunctions, 1);
		*calculation_functions_copy = *calculation_functions;
		g_hash_table_insert(calculation_functions_hash_table, GUINT_TO_POINTER(value_type), calculation_functions_copy);
	}
}

gchar* tangle_lookup_filename(const gchar* filename) {
	gchar* path = NULL;
	gchar** csp;
	gchar* dir;
	
	if (g_path_is_absolute(filename)) {
		path = g_strdup(filename);
	} else {
		for (csp = search_paths; !path && *csp; csp++) {
			path = g_build_filename(*csp, filename, NULL);
			if (!g_file_test(path, G_FILE_TEST_EXISTS)) {
				g_free(path);
				path = NULL;
			}
		}
		
		if (!path) {
			dir = g_get_current_dir();
			path = g_build_filename(dir, filename, NULL);
			if (!g_file_test(path, G_FILE_TEST_EXISTS)) {
				g_free(path);
				path = NULL;
			}
			g_free(dir);
		}
	}
	
	return path;
}

void tangle_add_search_path(const gchar* search_path) {
	gsize search_paths_length;
	
	search_paths_length = (search_paths ? g_strv_length(search_paths) : 0);
	search_paths = g_renew(gchar*, search_paths, search_paths_length + 2);
	search_paths[search_paths_length] = g_strdup(search_path);
	search_paths[search_paths_length + 1] = NULL;
}

static void on_weak_notify(gpointer user_data, GObject* other_object) {
	TangleVault* vault;
	GObject* object;
	gpointer function_pointer;
	
	vault = TANGLE_VAULT(user_data);
	tangle_vault_get(vault, 2, G_TYPE_OBJECT, &object, G_TYPE_POINTER, &function_pointer);
	
	g_signal_handlers_disconnect_by_func(object, G_CALLBACK(function_pointer), other_object);
	
	tangle_vault_free(vault);
}
