/*
 * tangle-scrollbar.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2009-2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-scrollbar.h"
#include "tangle-misc.h"

/**
 * SECTION:tangle-scrollbar
 * @Short_description: A scrollbar to be used together with #TangleScrollingActor
 * @Title: TangleScrollbar
 */

G_DEFINE_TYPE(TangleScrollbar, tangle_scrollbar, TANGLE_TYPE_WIDGET);

enum {
	PROP_0,
	PROP_VERTICAL,
	PROP_BAR_ACTOR,
	PROP_BAR_OFFSET,
	PROP_BAR_SIZE,
	PROP_TOTAL_SIZE
};

struct _TangleScrollbarPrivate {
	ClutterActor* bar_actor;
	gfloat bar_offset;
	gfloat bar_size;
	gfloat total_size;

	guint vertical : 1;
};

ClutterActor* tangle_scrollbar_new(gboolean vertical) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_SCROLLBAR, "vertical", vertical, NULL));
}

ClutterActor* tangle_scrollbar_new_rectangle(gboolean vertical, const ClutterColor* color) {
	ClutterActor* rectangle;
	
	rectangle = clutter_rectangle_new_with_color(color);

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_SCROLLBAR, "vertical", vertical, "bar-actor", rectangle, NULL));
}

gboolean tangle_scrollbar_get_vertical(TangleScrollbar* scrollbar) {

	return scrollbar->priv->vertical;
}

ClutterActor* tangle_scrollbar_get_bar_actor(TangleScrollbar* scrollbar) {

	return scrollbar->priv->bar_actor;
}

void tangle_scrollbar_set_bar_actor(TangleScrollbar* scrollbar, ClutterActor* bar_actor) {
	if (scrollbar->priv->bar_actor != bar_actor) {
		if (scrollbar->priv->bar_actor) {
			clutter_container_remove_actor(CLUTTER_CONTAINER(scrollbar), scrollbar->priv->bar_actor);
		}
		scrollbar->priv->bar_actor = bar_actor;
		clutter_container_add_actor(CLUTTER_CONTAINER(scrollbar), scrollbar->priv->bar_actor);
		g_object_notify(G_OBJECT(scrollbar), "bar-actor");
	}
}

void tangle_scrollbar_bind_to_scrolling_actor(TangleScrollbar* scrollbar, TangleScrollingActor* scrolling_actor) {
	if (scrollbar->priv->vertical) {
		tangle_binding_new(G_OBJECT(scrollbar), "bar-offset", G_OBJECT(scrolling_actor), "scrolling-offset-y");
		tangle_binding_new(G_OBJECT(scrollbar), "bar-size", G_OBJECT(scrolling_actor), "height");
		tangle_binding_new(G_OBJECT(scrollbar), "total-size", G_OBJECT(scrolling_actor), "scrolling-height");			
	} else {
		tangle_binding_new(G_OBJECT(scrollbar), "bar-offset", G_OBJECT(scrolling_actor), "scrolling-offset-x");
		tangle_binding_new(G_OBJECT(scrollbar), "bar-size", G_OBJECT(scrolling_actor), "width");
		tangle_binding_new(G_OBJECT(scrollbar), "total-size", G_OBJECT(scrolling_actor), "scrolling-width");		
	}
}

void tangle_scrollbar_bind_to_scroller(TangleScrollbar* scrollbar, TangleScroller* scroller) {
	if (scrollbar->priv->vertical) {
		tangle_binding_new(G_OBJECT(scrollbar), "bar-offset", G_OBJECT(scroller), "scrolling-offset-y");
		tangle_binding_new(G_OBJECT(scrollbar), "bar-size", G_OBJECT(scroller), "height");
		tangle_binding_new(G_OBJECT(scrollbar), "total-size", G_OBJECT(scroller), "scrolling-height");			
	} else {
		tangle_binding_new(G_OBJECT(scrollbar), "bar-offset", G_OBJECT(scroller), "scrolling-offset-x");
		tangle_binding_new(G_OBJECT(scrollbar), "bar-size", G_OBJECT(scroller), "width");
		tangle_binding_new(G_OBJECT(scrollbar), "total-size", G_OBJECT(scroller), "scrolling-width");		
	}
}

static void tangle_scrollbar_allocate(ClutterActor* actor, const ClutterActorBox* box, ClutterAllocationFlags flags) {
	TangleScrollbar* scrollbar;
	gfloat size;
	ClutterActorBox actor_box;
	
	scrollbar = TANGLE_SCROLLBAR(actor);
	
	CLUTTER_ACTOR_CLASS(tangle_scrollbar_parent_class)->allocate(actor, box, flags);
	
	if (scrollbar->priv->bar_actor) {
		if (scrollbar->priv->vertical) {
			size = box->y2 - box->y1;
			actor_box.x1 = 0.0;
			actor_box.x2 = box->x2 - box->x1;
			actor_box.y1 = scrollbar->priv->bar_offset / scrollbar->priv->total_size * size;
			actor_box.y2 = actor_box.y1 + scrollbar->priv->bar_size / scrollbar->priv->total_size * size;
			if (actor_box.y1 < 0.0) {
				actor_box.y1 = 0.0;
			} else if (actor_box.y1 > size - 1.0) {
				actor_box.y1 = size - 1.0;
			}
			if (actor_box.y2 < 1.0) {
				actor_box.y2 = 1.0;
			} else if (actor_box.y2 > size) {
				actor_box.y2 = size;
			}
		} else {
			size = box->x2 - box->x1;
			actor_box.x1 = scrollbar->priv->bar_offset / scrollbar->priv->total_size * size;
			actor_box.x2 = actor_box.x1 + scrollbar->priv->bar_size / scrollbar->priv->total_size * size;
			actor_box.y1 = 0.0;
			actor_box.y2 = box->y2 - box->y1;
			if (actor_box.x1 < 0.0) {
				actor_box.x1 = 0.0;
			} else if (actor_box.x1 > size - 1.0) {
				actor_box.x1 = size - 1.0;
			}
			if (actor_box.x2 < 1.0) {
				actor_box.x2 = 1.0;
			} else if (actor_box.x2 > size) {
				actor_box.x2 = size;
			}
		}

		clutter_actor_allocate(scrollbar->priv->bar_actor, &actor_box, flags);
	}	
}

static void tangle_scrollbar_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleScrollbar* scrollbar;
	
	scrollbar = TANGLE_SCROLLBAR(object);

	switch (prop_id) {
		case PROP_VERTICAL:
			scrollbar->priv->vertical = g_value_get_boolean(value);
			break;
		case PROP_BAR_ACTOR:
			tangle_scrollbar_set_bar_actor(scrollbar, CLUTTER_ACTOR(g_value_get_object(value)));
			break;
		case PROP_BAR_OFFSET:
			scrollbar->priv->bar_offset = g_value_get_float(value);
			clutter_actor_queue_relayout(CLUTTER_ACTOR(scrollbar));
			break;
		case PROP_BAR_SIZE:
			scrollbar->priv->bar_size = g_value_get_float(value);
			clutter_actor_queue_relayout(CLUTTER_ACTOR(scrollbar));
			break;
		case PROP_TOTAL_SIZE:
			scrollbar->priv->total_size = g_value_get_float(value);
			clutter_actor_queue_relayout(CLUTTER_ACTOR(scrollbar));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_scrollbar_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleScrollbar* scrollbar;

	scrollbar = TANGLE_SCROLLBAR(object);

        switch (prop_id) {
		case PROP_VERTICAL:
			g_value_set_boolean(value, scrollbar->priv->vertical);
			break;
		case PROP_BAR_ACTOR:
			g_value_set_object(value, scrollbar->priv->bar_actor);
			break;
		case PROP_BAR_OFFSET:
			g_value_set_float(value, scrollbar->priv->bar_offset);
			break;
		case PROP_BAR_SIZE:
			g_value_set_float(value, scrollbar->priv->bar_size);
			break;
		case PROP_TOTAL_SIZE:
			g_value_set_float(value, scrollbar->priv->total_size);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_scrollbar_dispose(GObject* object) {
	TangleScrollbar* scrollbar;
	
	scrollbar = TANGLE_SCROLLBAR(object);
	
	TANGLE_UNREF_AND_NULLIFY_OBJECT_POINTER(scrollbar->priv->bar_actor);

	G_OBJECT_CLASS(tangle_scrollbar_parent_class)->dispose(object);
}

static void tangle_scrollbar_class_init(TangleScrollbarClass* scrollbar_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(scrollbar_class);
	ClutterActorClass* clutter_actor_class = CLUTTER_ACTOR_CLASS(scrollbar_class);

	gobject_class->dispose = tangle_scrollbar_dispose;
	gobject_class->set_property = tangle_scrollbar_set_property;
	gobject_class->get_property = tangle_scrollbar_get_property;

	clutter_actor_class->allocate = tangle_scrollbar_allocate;

	/**
	 * TangleScrollbar:vertical:
	 *
	 * Whether the scrollbar is vertical or horizontal.
	 */
	g_object_class_install_property(gobject_class, PROP_VERTICAL,
	                                g_param_spec_boolean("vertical",
	                                "Vertical",
	                                "Whether the scrollbar is vertical or horizontal",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollbar:bar-actor:
	 *
	 * The actor that is used to visualize a bar.
	 */
	g_object_class_install_property(gobject_class, PROP_BAR_ACTOR,
	                                g_param_spec_object("bar-actor",
	                                "Bar actor",
	                                "The actor that is used to visualize a bar",
	                                CLUTTER_TYPE_ACTOR,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollbar:bar-offset:
	 *
	 * The offset of the bar from the edge of the scrollbar. The value should be relative to the :total-size.
	 */
	g_object_class_install_property(gobject_class, PROP_BAR_OFFSET,
	                                g_param_spec_float("bar-offset",
	                                "Bar offset",
	                                "The offset of the bar from the edge of the scrollbar",
	                                -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollbar:bar-size:
	 *
	 * The size of the bar. The value should be relative to the :total-size.
	 */
	g_object_class_install_property(gobject_class, PROP_BAR_SIZE,
	                                g_param_spec_float("bar-size",
	                                "Bar size",
	                                "The size of the bar",
	                                0.0, G_MAXFLOAT, 0.0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollbar:total-size:
	 *
	 * The relative total size of the scrollbar that is used to compute actual size and offset of the bar
	 * based on relative :bar-offset and :bar-size respectively.
	 */
	g_object_class_install_property(gobject_class, PROP_TOTAL_SIZE,
	                                g_param_spec_float("total-size",
	                                "Total size",
	                                "The revaltie total size of the scrollbarbar",
	                                0.0, G_MAXFLOAT, 0.0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private (gobject_class, sizeof (TangleScrollbarPrivate));
}

static void tangle_scrollbar_init(TangleScrollbar* scrollbar) {
	scrollbar->priv = G_TYPE_INSTANCE_GET_PRIVATE(scrollbar, TANGLE_TYPE_SCROLLBAR, TangleScrollbarPrivate);
}
