/*
 * tangle-scrolling-actor.c
 *
 * This file is part of Tangle Toolkit - A graphical actor library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-scrolling-actor.h"
#include "tangle-vault.h"

/**
 * SECTION:tangle-scrolling-actor
 * @Short_description: A wrapper actor that scrolls its content
 * @Title: TangleScrollingActor
 */

G_DEFINE_TYPE(TangleScrollingActor, tangle_scrolling_actor, TANGLE_TYPE_WRAPPER_ACTOR);

enum {
	PROP_0,
	PROP_SCROLLING_OFFSET_X,
	PROP_SCROLLING_OFFSET_Y,
	PROP_SCROLLING_WIDTH,
	PROP_SCROLLING_HEIGHT,
	PROP_SCROLLING_THRESHOLD_X,
	PROP_SCROLLING_THRESHOLD_Y,
	PROP_MAX_OVERSHOOT_X,
	PROP_MAX_OVERSHOOT_Y,
	PROP_REPEAT_X,
	PROP_REPEAT_Y,
	PROP_CONTINUOUS_X,
	PROP_CONTINUOUS_Y,
	PROP_KINETIC_SCROLLING,
	PROP_SCROLLING_DECELERATION
};

enum {
	CLAMP_SCROLLING_OFFSET_X,
	CLAMP_SCROLLING_OFFSET_Y,
	LAST_SIGNAL
};

struct _TangleScrollingActorPrivate {
	gfloat scrolling_offset_x;
	gfloat scrolling_offset_y;
	gfloat scrolling_threshold_x;
	gfloat scrolling_threshold_y;
	gfloat max_overshoot_x;
	gfloat max_overshoot_y;
	gdouble scrolling_deceleration;

	gfloat scrolling_width;
	gfloat scrolling_height;
	
	gfloat start_motion_x;
	guint32 start_motion_x_time;
	gfloat start_motion_y;
	guint32 start_motion_y_time;
	gfloat farest_motion_x;
	guint32 farest_motion_x_time;
	gfloat farest_motion_y;
	guint32 farest_motion_y_time;
	
	guint32 previous_motion_time;

	gfloat start_scrolling_motion_x;
	gfloat start_scrolling_motion_y;
	gfloat start_scrolling_offset_x;
	gfloat start_scrolling_offset_y;

	gulong captured_event_handler_id;
	ClutterAnimation* animation_x;
	ClutterAnimation* animation_y;

	guint repeat_x : 1;
	guint repeat_y : 1;
	guint continuous_x : 1;
	guint continuous_y : 1;
	guint kinetic_scrolling : 1;

	guint descendant_interacting : 1;	
	guint descendant_dragging : 1;
	guint scrolling_effective : 1;
	guint going_right : 1;
	guint going_down : 1;
};

static guint signals[LAST_SIGNAL] = { 0 };

static gboolean on_captured_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data);
static void send_interaction_event(TangleActor* actor, gboolean began);
static gfloat get_width(TangleScrollingActor* 	scrolling_actor);
static gfloat get_height(TangleScrollingActor* scrolling_actor);
static gfloat normalise_scrolling_offset_x(TangleScrollingActor* scrolling_actor, gfloat scrolling_offset_x, gfloat* delta_return);
static gfloat normalise_scrolling_offset_y(TangleScrollingActor* scrolling_actor, gfloat scrolling_offset_y, gfloat* delta_return);

ClutterActor* tangle_scrolling_actor_new(ClutterActor* wrapped) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_SCROLLING_ACTOR, "wrapped", wrapped, NULL));
}

gfloat tangle_scrolling_actor_get_scrolling_offset_x(TangleScrollingActor* scrolling_actor) {

	return scrolling_actor->priv->scrolling_offset_x;
}

void tangle_scrolling_actor_set_scrolling_offset_x(TangleScrollingActor* scrolling_actor, gfloat scrolling_offset_x) {
	gfloat width;
	
	if (scrolling_actor->priv->scrolling_offset_x != scrolling_offset_x) {
		scrolling_actor->priv->scrolling_offset_x = normalise_scrolling_offset_x(scrolling_actor, scrolling_offset_x, NULL);
		clutter_actor_queue_relayout(CLUTTER_ACTOR(scrolling_actor));
		g_object_notify(G_OBJECT(scrolling_actor), "scrolling-offset-x");
	}	
}

gfloat tangle_scrolling_actor_get_scrolling_offset_y(TangleScrollingActor* scrolling_actor) {

	return scrolling_actor->priv->scrolling_offset_y;
}

void tangle_scrolling_actor_set_scrolling_offset_y(TangleScrollingActor* scrolling_actor, gfloat scrolling_offset_y) {
	gfloat height;
	
	if (scrolling_actor->priv->scrolling_offset_y != scrolling_offset_y) {
		scrolling_actor->priv->scrolling_offset_y = normalise_scrolling_offset_y(scrolling_actor, scrolling_offset_y, NULL);
		clutter_actor_queue_relayout(CLUTTER_ACTOR(scrolling_actor));
		g_object_notify(G_OBJECT(scrolling_actor), "scrolling-offset-y");
	}
}

gfloat tangle_scrolling_actor_get_scrolling_width(TangleScrollingActor* scrolling_actor) {

	return scrolling_actor->priv->scrolling_width;
}

gfloat tangle_scrolling_actor_get_scrolling_height(TangleScrollingActor* scrolling_actor) {

	return scrolling_actor->priv->scrolling_height;
}

void tangle_scrolling_actor_set_scrolling_threshold_x(TangleScrollingActor* scrolling_actor, gfloat scrolling_threshold_x) {
	if (scrolling_actor->priv->scrolling_threshold_x != scrolling_threshold_x) {
		scrolling_actor->priv->scrolling_threshold_x = scrolling_threshold_x;
		g_object_notify(G_OBJECT(scrolling_actor), "scrolling-threshold-x");
	}
}

gfloat tangle_scrolling_actor_get_scrolling_threshold_y(TangleScrollingActor* scrolling_actor) {

	return scrolling_actor->priv->scrolling_threshold_y;
}

void tangle_scrolling_actor_set_scrolling_threshold_y(TangleScrollingActor* scrolling_actor, gfloat scrolling_threshold_y) {
	if (scrolling_actor->priv->scrolling_threshold_y != scrolling_threshold_y) {
		scrolling_actor->priv->scrolling_threshold_y = scrolling_threshold_y;
		g_object_notify(G_OBJECT(scrolling_actor), "scrolling-threshold-y");
	}
}

void tangle_scrolling_actor_set_scrolling_threshold(TangleScrollingActor* scrolling_actor, gfloat scrolling_threshold) {
	tangle_scrolling_actor_set_scrolling_threshold_x(scrolling_actor, scrolling_threshold);
	tangle_scrolling_actor_set_scrolling_threshold_y(scrolling_actor, scrolling_threshold);
}

gboolean tangle_scrolling_actor_get_repeat_x(TangleScrollingActor* scrolling_actor) {

	return scrolling_actor->priv->repeat_x;
}

void tangle_scrolling_actor_set_repeat_x(TangleScrollingActor* scrolling_actor, gboolean repeat_x) {
	if (scrolling_actor->priv->repeat_x != repeat_x) {
		g_object_freeze_notify(G_OBJECT(scrolling_actor));

		scrolling_actor->priv->repeat_x = repeat_x;
		g_object_notify(G_OBJECT(scrolling_actor), "repeat-x");
		
		if (!repeat_x) {
			tangle_scrolling_actor_set_continuous_x(scrolling_actor, FALSE);
		}

		g_object_thaw_notify(G_OBJECT(scrolling_actor));
	}
}

gboolean tangle_scrolling_actor_get_repeat_y(TangleScrollingActor* scrolling_actor) {

	return scrolling_actor->priv->repeat_y;
}

void tangle_scrolling_actor_set_repeat_y(TangleScrollingActor* scrolling_actor, gboolean repeat_y) {
	if (scrolling_actor->priv->repeat_y != repeat_y) {
		g_object_freeze_notify(G_OBJECT(scrolling_actor));

		scrolling_actor->priv->repeat_y = repeat_y;
		g_object_notify(G_OBJECT(scrolling_actor), "repeat-y");
		
		if (!repeat_y) {
			tangle_scrolling_actor_set_continuous_y(scrolling_actor, FALSE);
		}

		g_object_thaw_notify(G_OBJECT(scrolling_actor));
	}
}

gboolean tangle_scrolling_actor_get_continuous_x(TangleScrollingActor* scrolling_actor) {

	return scrolling_actor->priv->continuous_x;
}

void tangle_scrolling_actor_set_continuous_x(TangleScrollingActor* scrolling_actor, gboolean continuous_x) {
	if (scrolling_actor->priv->continuous_x != continuous_x) {
		g_object_freeze_notify(G_OBJECT(scrolling_actor));

		scrolling_actor->priv->continuous_x = continuous_x;
		g_object_notify(G_OBJECT(scrolling_actor), "continuous-x");

		if (continuous_x) {
			tangle_scrolling_actor_set_repeat_x(scrolling_actor, TRUE);
		}

		g_object_thaw_notify(G_OBJECT(scrolling_actor));
	}
}

gboolean tangle_scrolling_actor_get_continuous_y(TangleScrollingActor* scrolling_actor) {

	return scrolling_actor->priv->continuous_y;
}

void tangle_scrolling_actor_set_continuous_y(TangleScrollingActor* scrolling_actor, gboolean continuous_y) {
	if (scrolling_actor->priv->continuous_y != continuous_y) {
		g_object_freeze_notify(G_OBJECT(scrolling_actor));

		scrolling_actor->priv->continuous_y = continuous_y;
		g_object_notify(G_OBJECT(scrolling_actor), "continuous-y");
		
		if (continuous_y) {
			tangle_scrolling_actor_set_repeat_y(scrolling_actor, TRUE);
		}

		g_object_thaw_notify(G_OBJECT(scrolling_actor));
	}
}

gboolean tangle_scrolling_actor_get_kinetic_scrolling(TangleScrollingActor* scrolling_actor) {

	return scrolling_actor->priv->kinetic_scrolling;
}

void tangle_scrolling_actor_set_kinetic_scrolling(TangleScrollingActor* scrolling_actor, gboolean kinetic_scrolling) {
	scrolling_actor->priv->kinetic_scrolling = kinetic_scrolling;
}

gdouble tangle_scrolling_actor_get_scrolling_deceleration(TangleScrollingActor* scrolling_actor) {

	return scrolling_actor->priv->scrolling_deceleration;
}

void tangle_scrolling_actor_set_scrolling_deceleration(TangleScrollingActor* scrolling_actor, gdouble scrolling_deceleration) {
	scrolling_actor->priv->scrolling_deceleration = scrolling_deceleration;
}

void tangle_scrolling_actor_clamp_scrolling_offset_x(TangleScrollingActor* scrolling_actor) {
	TangleClamp* clamp;
	
	clamp = tangle_clamp_new(TANGLE_CLAMP_HORIZONTAL, scrolling_actor->priv->scrolling_offset_x);
	g_signal_emit(scrolling_actor, signals[CLAMP_SCROLLING_OFFSET_X], 0, clamp);
	if (tangle_clamp_get_clamped_value_set(clamp)) {
		tangle_actor_animate(TANGLE_ACTOR(scrolling_actor), CLUTTER_EASE_IN_OUT_QUAD, 300, "scrolling-offset-x", tangle_clamp_get_clamped_value(clamp), NULL);		
	}
	g_object_unref(clamp);
}

void tangle_scrolling_actor_clamp_scrolling_offset_y(TangleScrollingActor* scrolling_actor) {
	TangleClamp* clamp;
	
	clamp = tangle_clamp_new(TANGLE_CLAMP_VERTICAL, scrolling_actor->priv->scrolling_offset_y);
	g_signal_emit(scrolling_actor, signals[CLAMP_SCROLLING_OFFSET_Y], 0, clamp);
	if (tangle_clamp_get_clamped_value_set(clamp)) {
		tangle_actor_animate(TANGLE_ACTOR(scrolling_actor), CLUTTER_EASE_IN_OUT_QUAD, 300, "scrolling-offset-y", tangle_clamp_get_clamped_value(clamp), NULL);		
	}
	g_object_unref(clamp);
}

void tangle_scrolling_actor_clamp_offsets(TangleScrollingActor* scrolling_actor) {
	tangle_scrolling_actor_clamp_scrolling_offset_x(scrolling_actor);
	tangle_scrolling_actor_clamp_scrolling_offset_y(scrolling_actor);
}

void tangle_scrolling_actor_clamp_page_boundaries(TangleScrollingActor* scrolling_actor, TangleClamp* clamp) {
	TangleClampDirection direction;
	gfloat original_value;
	gfloat page_size;
	
	g_return_if_fail(TANGLE_IS_SCROLLING_ACTOR(scrolling_actor));
	
	direction = tangle_clamp_get_direction(clamp);
	original_value = tangle_clamp_get_original_value(clamp);
	
	if (direction == TANGLE_CLAMP_HORIZONTAL) {
		if (original_value < 0.0) {
			tangle_clamp_set_clamped_value(clamp, 0.0);
		} else if (original_value > scrolling_actor->priv->scrolling_width) {
			tangle_clamp_set_clamped_value(clamp, scrolling_actor->priv->scrolling_width);
		} else {
			page_size = get_width(scrolling_actor);
			tangle_clamp_set_clamped_value(clamp, ((guint)((original_value + page_size / 2) / page_size)) * page_size);
		}
	} else if (direction == TANGLE_CLAMP_VERTICAL) {
		if (original_value < 0.0) {
			tangle_clamp_set_clamped_value(clamp, 0.0);
		} else if (original_value > scrolling_actor->priv->scrolling_height) {
			tangle_clamp_set_clamped_value(clamp, scrolling_actor->priv->scrolling_height);
		} else {
			page_size = get_height(scrolling_actor);
			tangle_clamp_set_clamped_value(clamp, ((guint)((original_value + page_size / 2) / page_size)) * page_size);
		}
	}
}

static gboolean tangle_scrolling_actor_button_press_event(ClutterActor* actor, ClutterButtonEvent* event) {
	TangleScrollingActor* scrolling_actor;
		
	scrolling_actor = TANGLE_SCROLLING_ACTOR(actor);

	if (!scrolling_actor->priv->captured_event_handler_id) {
		if (scrolling_actor->priv->animation_x) {
			tangle_actor_stop_animation(TANGLE_ACTOR(scrolling_actor), scrolling_actor->priv->animation_x);
		}
		if (scrolling_actor->priv->animation_y) {
			tangle_actor_stop_animation(TANGLE_ACTOR(scrolling_actor), scrolling_actor->priv->animation_y);
		}
		scrolling_actor->priv->start_scrolling_motion_x = event->x;
		scrolling_actor->priv->start_scrolling_motion_y = event->y;
		scrolling_actor->priv->scrolling_effective = FALSE;
		scrolling_actor->priv->captured_event_handler_id = g_signal_connect(clutter_actor_get_stage(actor), "captured-event", G_CALLBACK(on_captured_event), scrolling_actor);
	}

	return FALSE;
}


static void tangle_scrolling_actor_allocate_wrapped(TangleWrapperActor* wrapper_actor, ClutterActor* wrapped, const ClutterActorBox* box, ClutterAllocationFlags flags) {
	TangleScrollingActor* scrolling_actor;
	gfloat old_width, old_height;
	ClutterActorBox actor_box;

	scrolling_actor = TANGLE_SCROLLING_ACTOR(wrapper_actor);

	old_width = scrolling_actor->priv->scrolling_width;
	old_height = scrolling_actor->priv->scrolling_height;
	clutter_actor_get_preferred_size(wrapped, NULL, NULL, &scrolling_actor->priv->scrolling_width, &scrolling_actor->priv->scrolling_height);
	if (scrolling_actor->priv->scrolling_width < box->x2 - box->x1) {
		scrolling_actor->priv->scrolling_width = box->x2 - box->x1;
	}
	if (scrolling_actor->priv->scrolling_height < box->y2 - box->y1) {
		scrolling_actor->priv->scrolling_height = box->y2 - box->y1;
	}
	if (old_width != scrolling_actor->priv->scrolling_width) {
		g_object_notify(G_OBJECT(wrapper_actor), "scrolling-width");
	}
	if (old_height != scrolling_actor->priv->scrolling_height) {
		g_object_notify(G_OBJECT(wrapper_actor), "scrolling-height");
	}

	actor_box.x1 = box->x1 - scrolling_actor->priv->scrolling_offset_x;
	actor_box.y1 = box->y1 - scrolling_actor->priv->scrolling_offset_y;
	actor_box.x2 = actor_box.x1 + scrolling_actor->priv->scrolling_width;
	actor_box.y2 = actor_box.y1 + scrolling_actor->priv->scrolling_height;

	clutter_actor_allocate(wrapped, &actor_box, flags);
	
	scrolling_actor->priv->scrolling_width -= box->x2 - box->x1;
	scrolling_actor->priv->scrolling_height -= box->y2 - box->y1;
}

static void tangle_scrolling_actor_paint_wrapped(TangleWrapperActor* wrapper_actor, ClutterActor* wrapped) {
	TangleScrollingActor* scrolling_actor;
	ClutterActorBox actor_box;
	
	scrolling_actor = TANGLE_SCROLLING_ACTOR(wrapper_actor);

	TANGLE_WRAPPER_ACTOR_CLASS(tangle_scrolling_actor_parent_class)->paint_wrapped(wrapper_actor, wrapped);
	clutter_actor_get_allocation_box(CLUTTER_ACTOR(scrolling_actor), &actor_box);
	
	if (scrolling_actor->priv->repeat_x && scrolling_actor->priv->scrolling_offset_x < 0.0 &&
	    scrolling_actor->priv->repeat_y && scrolling_actor->priv->scrolling_offset_y < 0.0) {
		cogl_translate(-scrolling_actor->priv->scrolling_width - (actor_box.x2 - actor_box.x1), -scrolling_actor->priv->scrolling_height - (actor_box.y2 - actor_box.y1), 0.0);
		TANGLE_WRAPPER_ACTOR_CLASS(tangle_scrolling_actor_parent_class)->paint_wrapped(wrapper_actor, wrapped);
		cogl_translate(scrolling_actor->priv->scrolling_width + (actor_box.x2 - actor_box.x1), scrolling_actor->priv->scrolling_height + (actor_box.y2 - actor_box.y1), 0.0);
	}
	if (scrolling_actor->priv->repeat_x && scrolling_actor->priv->scrolling_offset_x < 0.0 &&
	    scrolling_actor->priv->repeat_y && scrolling_actor->priv->scrolling_offset_y > scrolling_actor->priv->scrolling_height) {
		cogl_translate(-scrolling_actor->priv->scrolling_width - (actor_box.x2 - actor_box.x1), scrolling_actor->priv->scrolling_height + (actor_box.y2 - actor_box.y1), 0.0);
		TANGLE_WRAPPER_ACTOR_CLASS(tangle_scrolling_actor_parent_class)->paint_wrapped(wrapper_actor, wrapped);
		cogl_translate(scrolling_actor->priv->scrolling_width + (actor_box.x2 - actor_box.x1), -scrolling_actor->priv->scrolling_height - (actor_box.y2 - actor_box.y1), 0.0);
	}
	if (scrolling_actor->priv->repeat_x && scrolling_actor->priv->scrolling_offset_x < 0.0) {
		cogl_translate(-scrolling_actor->priv->scrolling_width - (actor_box.x2 - actor_box.x1), 0.0, 0.0);
		TANGLE_WRAPPER_ACTOR_CLASS(tangle_scrolling_actor_parent_class)->paint_wrapped(wrapper_actor, wrapped);
		cogl_translate(scrolling_actor->priv->scrolling_width + (actor_box.x2 - actor_box.x1), 0.0, 0.0);
	}
	if (scrolling_actor->priv->repeat_y && scrolling_actor->priv->scrolling_offset_y < 0.0) {
		cogl_translate(0.0, -scrolling_actor->priv->scrolling_height - (actor_box.y2 - actor_box.y1), 0.0);
		TANGLE_WRAPPER_ACTOR_CLASS(tangle_scrolling_actor_parent_class)->paint_wrapped(wrapper_actor, wrapped);
		cogl_translate(0.0, scrolling_actor->priv->scrolling_height + (actor_box.y2 - actor_box.y1), 0.0);
	}
	if (scrolling_actor->priv->repeat_x && scrolling_actor->priv->scrolling_offset_x > scrolling_actor->priv->scrolling_width &&
	    scrolling_actor->priv->repeat_y && scrolling_actor->priv->scrolling_offset_y > scrolling_actor->priv->scrolling_height) {
		cogl_translate(scrolling_actor->priv->scrolling_width + (actor_box.x2 - actor_box.x1), scrolling_actor->priv->scrolling_height + (actor_box.y2 - actor_box.y1), 0.0);
		TANGLE_WRAPPER_ACTOR_CLASS(tangle_scrolling_actor_parent_class)->paint_wrapped(wrapper_actor, wrapped);
		cogl_translate(-scrolling_actor->priv->scrolling_width - (actor_box.x2 - actor_box.x1), -scrolling_actor->priv->scrolling_height - (actor_box.y2 - actor_box.y1), 0.0);
	}
	if (scrolling_actor->priv->repeat_x && scrolling_actor->priv->scrolling_offset_x > scrolling_actor->priv->scrolling_width &&
	    scrolling_actor->priv->repeat_y && scrolling_actor->priv->scrolling_offset_y < 0.0) {
		cogl_translate(scrolling_actor->priv->scrolling_width + (actor_box.x2 - actor_box.x1), -scrolling_actor->priv->scrolling_height - (actor_box.y2 - actor_box.y1), 0.0);
		TANGLE_WRAPPER_ACTOR_CLASS(tangle_scrolling_actor_parent_class)->paint_wrapped(wrapper_actor, wrapped);
		cogl_translate(-scrolling_actor->priv->scrolling_width - (actor_box.x2 - actor_box.x1), scrolling_actor->priv->scrolling_height + (actor_box.y2 - actor_box.y1), 0.0);
	}
	if (scrolling_actor->priv->repeat_x && scrolling_actor->priv->scrolling_offset_x > scrolling_actor->priv->scrolling_width) {
		cogl_translate(scrolling_actor->priv->scrolling_width + (actor_box.x2 - actor_box.x1), 0.0, 0.0);
		TANGLE_WRAPPER_ACTOR_CLASS(tangle_scrolling_actor_parent_class)->paint_wrapped(wrapper_actor, wrapped);
		cogl_translate(-scrolling_actor->priv->scrolling_width - (actor_box.x2 - actor_box.x1), 0.0, 0.0);
	}
	if (scrolling_actor->priv->repeat_y && scrolling_actor->priv->scrolling_offset_y > scrolling_actor->priv->scrolling_height) {
		cogl_translate(0.0, scrolling_actor->priv->scrolling_height + (actor_box.y2 - actor_box.y1), 0.0);
		TANGLE_WRAPPER_ACTOR_CLASS(tangle_scrolling_actor_parent_class)->paint_wrapped(wrapper_actor, wrapped);
		cogl_translate(0.0, -scrolling_actor->priv->scrolling_height - (actor_box.y2 - actor_box.y1), 0.0);
	}
}

static void tangle_scrolling_actor_handle_descendant_event(TangleActor* actor, TangleEvent* event) {
	TangleScrollingActor* scrolling_actor;
	
	scrolling_actor = TANGLE_SCROLLING_ACTOR(actor);
	
	if (event->any.sender != actor) {
		switch (event->type) {
			case TANGLE_EVENT_INTERACTION_BEGAN:
				scrolling_actor->priv->descendant_interacting = TRUE;
				if (event->interaction.dragging) {
					scrolling_actor->priv->descendant_dragging = TRUE;
				}
				break;
			case TANGLE_EVENT_INTERACTION_ENDED:
				scrolling_actor->priv->descendant_interacting = FALSE;
				scrolling_actor->priv->descendant_dragging = FALSE;
				break;
			default:
				break;
		}
	}

	TANGLE_ACTOR_CLASS(tangle_scrolling_actor_parent_class)->handle_descendant_event(actor, event);
}

static void tangle_scrolling_actor_clamp_scrolling_offset_x_impl(TangleScrollingActor* scrolling_actor, TangleClamp* clamp) {
	if (!tangle_clamp_get_clamped_value_set(clamp) && !scrolling_actor->priv->continuous_x) {
		if (scrolling_actor->priv->scrolling_offset_x < 0.0) {
			tangle_clamp_set_clamped_value(clamp, 0.0);
		} else if (scrolling_actor->priv->scrolling_offset_x > scrolling_actor->priv->scrolling_width) {
			tangle_clamp_set_clamped_value(clamp, scrolling_actor->priv->scrolling_width);
		}
	}
}

static void tangle_scrolling_actor_clamp_scrolling_offset_y_impl(TangleScrollingActor* scrolling_actor, TangleClamp* clamp) {
	if (!tangle_clamp_get_clamped_value_set(clamp) && !scrolling_actor->priv->continuous_y) {
		if (scrolling_actor->priv->scrolling_offset_y < 0.0) {
			tangle_clamp_set_clamped_value(clamp, 0.0);
		} else if (scrolling_actor->priv->scrolling_offset_y > scrolling_actor->priv->scrolling_height) {
			tangle_clamp_set_clamped_value(clamp, scrolling_actor->priv->scrolling_height);
		}
	}
}

static void tangle_scrolling_actor_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleScrollingActor* scrolling_actor;
	
	scrolling_actor = TANGLE_SCROLLING_ACTOR(object);

	switch (prop_id) {
		case PROP_SCROLLING_OFFSET_X:
			tangle_scrolling_actor_set_scrolling_offset_x(scrolling_actor, g_value_get_float(value));
			break;
		case PROP_SCROLLING_OFFSET_Y:
			tangle_scrolling_actor_set_scrolling_offset_y(scrolling_actor, g_value_get_float(value));
			break;
		case PROP_SCROLLING_THRESHOLD_X:
			tangle_scrolling_actor_set_scrolling_threshold_x(scrolling_actor, g_value_get_float(value));
			break;
		case PROP_SCROLLING_THRESHOLD_Y:
			tangle_scrolling_actor_set_scrolling_threshold_y(scrolling_actor, g_value_get_float(value));
			break;
		case PROP_MAX_OVERSHOOT_X:
			scrolling_actor->priv->max_overshoot_x = g_value_get_float(value);
			break;
		case PROP_MAX_OVERSHOOT_Y:
			scrolling_actor->priv->max_overshoot_y = g_value_get_float(value);
			break;
		case PROP_REPEAT_X:
			tangle_scrolling_actor_set_repeat_x(scrolling_actor, g_value_get_boolean(value));
			break;
		case PROP_REPEAT_Y:
			tangle_scrolling_actor_set_repeat_y(scrolling_actor, g_value_get_boolean(value));
			break;
		case PROP_CONTINUOUS_X:
			tangle_scrolling_actor_set_continuous_x(scrolling_actor, g_value_get_boolean(value));
			break;
		case PROP_CONTINUOUS_Y:
			tangle_scrolling_actor_set_continuous_y(scrolling_actor, g_value_get_boolean(value));
			break;
		case PROP_KINETIC_SCROLLING:
			tangle_scrolling_actor_set_kinetic_scrolling(scrolling_actor, g_value_get_boolean(value));
			break;
		case PROP_SCROLLING_DECELERATION:
			tangle_scrolling_actor_set_scrolling_deceleration(scrolling_actor, g_value_get_double(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_scrolling_actor_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleScrollingActor* scrolling_actor;

	scrolling_actor = TANGLE_SCROLLING_ACTOR(object);

        switch (prop_id) {
		case PROP_SCROLLING_OFFSET_X:
			g_value_set_float(value, scrolling_actor->priv->scrolling_offset_x);
			break;
		case PROP_SCROLLING_OFFSET_Y:
			g_value_set_float(value, scrolling_actor->priv->scrolling_offset_y);
			break;
		case PROP_SCROLLING_WIDTH:
			g_value_set_float(value, scrolling_actor->priv->scrolling_width);
			break;
		case PROP_SCROLLING_HEIGHT:
			g_value_set_float(value, scrolling_actor->priv->scrolling_height);
			break;
		case PROP_SCROLLING_THRESHOLD_X:
			g_value_set_float(value, scrolling_actor->priv->scrolling_threshold_x);
			break;
		case PROP_SCROLLING_THRESHOLD_Y:
			g_value_set_float(value, scrolling_actor->priv->scrolling_threshold_y);
			break;
		case PROP_MAX_OVERSHOOT_X:
			g_value_set_float(value, scrolling_actor->priv->max_overshoot_x);
			break;
		case PROP_MAX_OVERSHOOT_Y:
			g_value_set_float(value, scrolling_actor->priv->max_overshoot_y);
			break;
		case PROP_REPEAT_X:
			g_value_set_boolean(value, scrolling_actor->priv->repeat_x);
			break;
		case PROP_REPEAT_Y:
			g_value_set_boolean(value, scrolling_actor->priv->repeat_y);
			break;
		case PROP_CONTINUOUS_X:
			g_value_set_boolean(value, scrolling_actor->priv->continuous_x);
			break;
		case PROP_CONTINUOUS_Y:
			g_value_set_boolean(value, scrolling_actor->priv->continuous_y);
			break;
		case PROP_KINETIC_SCROLLING:
			g_value_set_boolean(value, scrolling_actor->priv->kinetic_scrolling);
			break;
		case PROP_SCROLLING_DECELERATION:
			g_value_set_double(value, scrolling_actor->priv->scrolling_deceleration);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_scrolling_actor_dispose(GObject* object) {
	TangleScrollingActor* scrolling_actor;
	
	scrolling_actor = TANGLE_SCROLLING_ACTOR(object);
	
	if (scrolling_actor->priv->captured_event_handler_id) {
		g_signal_handler_disconnect(scrolling_actor, scrolling_actor->priv->captured_event_handler_id);
		scrolling_actor->priv->captured_event_handler_id = 0;
	}

	G_OBJECT_CLASS(tangle_scrolling_actor_parent_class)->dispose(object);
}

static void tangle_scrolling_actor_class_init(TangleScrollingActorClass* scrolling_actor_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(scrolling_actor_class);
	ClutterActorClass* clutter_actor_class = CLUTTER_ACTOR_CLASS(scrolling_actor_class);
	TangleActorClass* actor_class = TANGLE_ACTOR_CLASS(scrolling_actor_class);
	TangleWrapperActorClass* wrapper_actor_class = TANGLE_WRAPPER_ACTOR_CLASS(scrolling_actor_class);

	gobject_class->dispose = tangle_scrolling_actor_dispose;
	gobject_class->set_property = tangle_scrolling_actor_set_property;
	gobject_class->get_property = tangle_scrolling_actor_get_property;
	
	clutter_actor_class->button_press_event = tangle_scrolling_actor_button_press_event;
	
	actor_class->handle_descendant_event = tangle_scrolling_actor_handle_descendant_event;

	wrapper_actor_class->allocate_wrapped = tangle_scrolling_actor_allocate_wrapped;
	wrapper_actor_class->paint_wrapped = tangle_scrolling_actor_paint_wrapped;
	
	scrolling_actor_class->clamp_scrolling_offset_x = tangle_scrolling_actor_clamp_scrolling_offset_x_impl;
	scrolling_actor_class->clamp_scrolling_offset_y = tangle_scrolling_actor_clamp_scrolling_offset_y_impl;
	
	/**
	 * TangleScrollingActor:scrolling-offset-x:
	 *
	 * The offset to the left side of the wrapped actor's bounding box.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_OFFSET_X,
	                                g_param_spec_float("scrolling-offset-x",
	                                                    "Scrolling offset X",
	                                                    "The offset to the left side of the wrapped actor's bounding box",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollingActor:scrolling-offset-y:
	 *
	 * The offset to the top side of the wrapped actor's bounding box.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_OFFSET_Y,
	                                g_param_spec_float("scrolling-offset-y",
	                                                    "Scrolling offset Y",
	                                                    "The offset to the top side of the wrapped actor's bounding box",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollingActor:scrolling-width:
	 *
	 * The maximum value for the scrollng-offset-x: property.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_WIDTH,
	                                g_param_spec_float("scrolling-width",
	                                                    "Scrolling width",
	                                                    "The maximum value for the scrollng-offset-y property",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollingActor:scrolling-height:
	 *
	 * The maximum value for the scrollng-offset-y: property.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_HEIGHT,
	                                g_param_spec_float("scrolling-height",
	                                                    "Scrolling height",
	                                                    "The maximum value for the scrollng-offset-y property",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollingActor:scrolling-threshold-x:
	 *
	 * The horizontal threshold before start scrolling.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_THRESHOLD_X,
	                                g_param_spec_float("scrolling-threshold-x",
	                                                    "Scrolling threshold X",
	                                                    "The horizontal threshold before start scrolling",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollingActor:scrolling-threshold-y:
	 *
	 * The vertical threshold before start scrolling.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_THRESHOLD_Y,
	                                g_param_spec_float("scrolling-threshold-y",
	                                                    "Scrolling threshold Y",
	                                                    "The vertical threshold before start scrolling",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollingActor:max-overshoot-x:
	 *
	 * The horizontal maximum of overshooting.
	 */
	g_object_class_install_property(gobject_class, PROP_MAX_OVERSHOOT_X,
	                                g_param_spec_float("max-overshoot-x",
	                                                    "Max overshoot X",
	                                                    "The horizontal maximum of overshooting",
	                                                   0.0, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollingActor:max-overshoot-y:
	 *
	 * The vertical maximum of overshooting.
	 */
	g_object_class_install_property(gobject_class, PROP_MAX_OVERSHOOT_Y,
	                                g_param_spec_float("max-overshoot-y",
	                                                    "Max overshoot Y",
	                                                    "The vertical maximum of overshooting",
	                                                   0.0, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollingActor:repeat-x:
	 *
	 * Whether to repeat the wrapped actor painting when overshooting horizontally.
	 * Setting this to FALSE sets also :continuous-x to FALSE as a side effect.
	 */
	g_object_class_install_property(gobject_class, PROP_REPEAT_X,
	                                g_param_spec_boolean("repeat-x",
	                                                    "Repeat X",
	                                                    "Whether to repeat the wrapped actor painting when overshooting horizontally",
	                                                    FALSE,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollingActor:repeat-y:
	 *
	 * Whether to repeat the wrapped actor painting when overshooting vertically.
	 * Setting this to FALSE sets also :continuous-y to FALSE as a side effect.
	 */
	g_object_class_install_property(gobject_class, PROP_REPEAT_Y,
	                                g_param_spec_boolean("repeat-y",
	                                                    "Repeat Y",
	                                                    "Whether to repeat the wrapped actor painting when overshooting vertically",
	                                                    FALSE,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollingActor:continuous-x:
	 *
	 * Whether to connect the left and right edges of the wrapped actor to enable infinite horizontal scrolling.
	 * Setting this to TRUE sets also :repeat-x to TRUE as a side effect.
	 */
	g_object_class_install_property(gobject_class, PROP_CONTINUOUS_X,
	                                g_param_spec_boolean("continuous-x",
	                                                    "Continuous X",
	                                                    "Whether to connect the left and right edges of the wrapped actor to enable infinite horizontal scrolling",
	                                                    FALSE,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollingActor:continuous-y:
	 *
	 * Whether to connect the top and bottom edges of the wrapped actor to enable infinite vertical scrolling.
	 * Setting this to TRUE sets also :repeat-y to TRUE as a side effect.
	 */
	g_object_class_install_property(gobject_class, PROP_CONTINUOUS_Y,
	                                g_param_spec_boolean("continuous-y",
	                                                    "Continuous Y",
	                                                    "Whether to connect the top and bottom edges of the wrapped actor to enable infinite vertical scrolling",
	                                                    FALSE,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollingActor:kinetic-scrolling:
	 *
	 * Whether kinetic scrolling is enabled or not.
	 */
	g_object_class_install_property(gobject_class, PROP_KINETIC_SCROLLING,
	                                g_param_spec_boolean("kinetic-scrolling",
	                                                    "Kinetic scrolling",
	                                                    "Whether kinetic scrolling is enabled or not",
	                                                    TRUE,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollingActor:scrolling-deceleration:
	 *
	 * Deceleration of kinetic scrolling.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_DECELERATION,
	                                g_param_spec_boolean("scrolling-deceleration",
	                                                    "Scrolling deceleration",
	                                                    "Deceleration of kinetic scrolling",
	                                                    0.7,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	signals[CLAMP_SCROLLING_OFFSET_X] = g_signal_new("clamp-scrolling-offset-x",
	                                       G_TYPE_FROM_CLASS(gobject_class),
					       G_SIGNAL_RUN_LAST,
					       G_STRUCT_OFFSET(TangleScrollingActorClass, clamp_scrolling_offset_x),
					       NULL, NULL,
					       g_cclosure_marshal_VOID__OBJECT,
					       G_TYPE_NONE, 1,
					       TANGLE_TYPE_CLAMP);
	signals[CLAMP_SCROLLING_OFFSET_Y] = g_signal_new("clamp-scrolling-offset-y",
	                                       G_TYPE_FROM_CLASS(gobject_class),
					       G_SIGNAL_RUN_LAST,
					       G_STRUCT_OFFSET(TangleScrollingActorClass, clamp_scrolling_offset_y),
					       NULL, NULL,
					       g_cclosure_marshal_VOID__OBJECT,
					       G_TYPE_NONE, 1,
					       TANGLE_TYPE_CLAMP);

	g_type_class_add_private(gobject_class, sizeof(TangleScrollingActorPrivate));
}

static void tangle_scrolling_actor_init(TangleScrollingActor* scrolling_actor) {
	scrolling_actor->priv = G_TYPE_INSTANCE_GET_PRIVATE(scrolling_actor, TANGLE_TYPE_SCROLLING_ACTOR, TangleScrollingActorPrivate);
	scrolling_actor->priv->kinetic_scrolling = TRUE;
	scrolling_actor->priv->scrolling_deceleration = 0.7;

	clutter_actor_set_reactive(CLUTTER_ACTOR(scrolling_actor), TRUE);
	g_object_set(CLUTTER_ACTOR(scrolling_actor), "clip-to-allocation", TRUE, NULL);
}

static void on_overshooting_animation_completed(ClutterAnimation* animation, gpointer user_data) {
	TangleVault* vault;
	TangleScrollingActor* scrolling_actor;
	const gchar* property;
	gfloat value;
	
	vault = TANGLE_VAULT(user_data);
	tangle_vault_get(vault, 3, TANGLE_TYPE_SCROLLING_ACTOR, &scrolling_actor, G_TYPE_STRING, &property, G_TYPE_FLOAT, &value);
	
	tangle_actor_animate(TANGLE_ACTOR(scrolling_actor), CLUTTER_EASE_IN_OUT_QUAD, 350, property, value, NULL);
}

static void handle_overshooting_animation(TangleScrollingActor* scrolling_actor, ClutterAnimation* animation, const gchar* property, gfloat value) {
	TangleVault* vault;

	vault = tangle_vault_new(3, TANGLE_TYPE_SCROLLING_ACTOR, scrolling_actor, G_TYPE_STRING, property, G_TYPE_FLOAT, value);
	tangle_signal_connect_vault_flags(animation, "completed", G_CALLBACK(on_overshooting_animation_completed), vault, G_CONNECT_AFTER);
}

#define ABS_F(x) ((x) < 0 ? -(x) : (x))
static gboolean on_captured_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data) {
	TangleScrollingActor* scrolling_actor;
	gboolean handled = TRUE;
	ClutterBackend* backend;
	guint32 event_time;
	gfloat x, y, dx, dy;
	gdouble vx, vy, ax, ay;
	guint tx, ty;
	TangleClamp* clamp;
	gfloat delta;

	if (event->type == CLUTTER_MOTION || event->type == CLUTTER_BUTTON_RELEASE) {
		scrolling_actor = TANGLE_SCROLLING_ACTOR(user_data);	
		
		g_object_ref(scrolling_actor);

		backend = clutter_get_default_backend();
		
		clutter_event_get_coords(event, &x, &y);
		event_time = clutter_event_get_time(event);
		dx = x - scrolling_actor->priv->start_scrolling_motion_x;
		dy = y - scrolling_actor->priv->start_scrolling_motion_y;

		if (scrolling_actor->priv->scrolling_effective ||
		    (!scrolling_actor->priv->descendant_dragging &&
		     (ABS_F(dx) > scrolling_actor->priv->scrolling_threshold_x ||
		      ABS_F(dy) > scrolling_actor->priv->scrolling_threshold_y))) {

			if (!scrolling_actor->priv->scrolling_effective ||
			    event_time - scrolling_actor->priv->previous_motion_time > clutter_backend_get_double_click_time(backend)) {
				if (!scrolling_actor->priv->scrolling_effective) {		
					scrolling_actor->priv->scrolling_effective = TRUE;
					scrolling_actor->priv->start_scrolling_motion_x = x;
					scrolling_actor->priv->start_scrolling_motion_y = y;
					scrolling_actor->priv->start_scrolling_offset_x = scrolling_actor->priv->scrolling_offset_x;
					scrolling_actor->priv->start_scrolling_offset_y = scrolling_actor->priv->scrolling_offset_y;
					dx = dy = 0;
				}
				scrolling_actor->priv->going_right = scrolling_actor->priv->start_motion_x < x;
				scrolling_actor->priv->going_down = scrolling_actor->priv->start_motion_y < y;
				scrolling_actor->priv->start_motion_x = scrolling_actor->priv->farest_motion_x = x;
				scrolling_actor->priv->start_motion_y = scrolling_actor->priv->farest_motion_y = y;
				scrolling_actor->priv->start_motion_x_time = scrolling_actor->priv->start_motion_y_time = scrolling_actor->priv->farest_motion_x_time = scrolling_actor->priv->farest_motion_y_time = event_time;
				
			} else {
				if ((scrolling_actor->priv->going_right && x > scrolling_actor->priv->farest_motion_x) ||
				    (!scrolling_actor->priv->going_right && x < scrolling_actor->priv->farest_motion_x)) {
					scrolling_actor->priv->farest_motion_x = x;
					scrolling_actor->priv->farest_motion_x_time = event_time;
				} else if (ABS_F(x - scrolling_actor->priv->farest_motion_x) > clutter_backend_get_double_click_distance(backend)) {				
					scrolling_actor->priv->start_motion_x = scrolling_actor->priv->farest_motion_x;
					scrolling_actor->priv->start_motion_x_time = scrolling_actor->priv->farest_motion_x_time;
					scrolling_actor->priv->farest_motion_x = x;
					scrolling_actor->priv->farest_motion_x_time = event_time;
					scrolling_actor->priv->going_right = scrolling_actor->priv->start_motion_x < x;
				}
				if ((scrolling_actor->priv->going_down && y > scrolling_actor->priv->farest_motion_y) ||
				    (!scrolling_actor->priv->going_down && y < scrolling_actor->priv->farest_motion_y)) {
					scrolling_actor->priv->farest_motion_y = y;
					scrolling_actor->priv->farest_motion_y_time = event_time;
				} else if (ABS_F(y - scrolling_actor->priv->farest_motion_y) > clutter_backend_get_double_click_distance(backend)) {				
					scrolling_actor->priv->start_motion_y = scrolling_actor->priv->farest_motion_y;
					scrolling_actor->priv->start_motion_y_time = scrolling_actor->priv->farest_motion_y_time;
					scrolling_actor->priv->farest_motion_y = y;
					scrolling_actor->priv->farest_motion_y_time = event_time;
					scrolling_actor->priv->going_down = scrolling_actor->priv->start_motion_y < y;
				}
			}

			tangle_scrolling_actor_set_scrolling_offset_x(scrolling_actor, scrolling_actor->priv->start_scrolling_offset_x - dx);
			tangle_scrolling_actor_set_scrolling_offset_y(scrolling_actor, scrolling_actor->priv->start_scrolling_offset_y - dy);

			if (event->type == CLUTTER_BUTTON_RELEASE && scrolling_actor->priv->kinetic_scrolling &&
			    event_time - scrolling_actor->priv->previous_motion_time < clutter_backend_get_double_click_time(backend)) {
				if (event_time - scrolling_actor->priv->start_motion_x_time > 0) {
					vx = (gdouble)(x - scrolling_actor->priv->start_motion_x) / (event_time - scrolling_actor->priv->start_motion_x_time);

					clamp = tangle_clamp_new(TANGLE_CLAMP_HORIZONTAL, normalise_scrolling_offset_x(scrolling_actor, scrolling_actor->priv->scrolling_offset_x + (scrolling_actor->priv->going_right ? -1 : 1) * vx * vx * 1000 / scrolling_actor->priv->scrolling_deceleration / 2, &delta));
					g_signal_emit(scrolling_actor, signals[CLAMP_SCROLLING_OFFSET_X], 0, clamp);
					if (tangle_clamp_get_clamped_value_set(clamp)) {
						ax = vx * vx * 1000 / ABS_F(tangle_clamp_get_clamped_value(clamp) - scrolling_actor->priv->scrolling_offset_x + delta) / 2;
					} else {
						ax = scrolling_actor->priv->scrolling_deceleration;

					}
					g_object_unref(clamp);

					tx = ABS_F(vx * 1000.0 / ax);

					if (tx > 0) {
						dx = (scrolling_actor->priv->going_right ? -1 : 1) * vx * vx * 1000 / ax / 2;
						if (scrolling_actor->priv->continuous_x) {
							scrolling_actor->priv->animation_x = tangle_actor_animate(TANGLE_ACTOR(scrolling_actor), CLUTTER_EASE_OUT_QUAD, tx,
														  "scrolling-offset-x", scrolling_actor->priv->scrolling_offset_x + dx,
														  NULL);
							g_object_add_weak_pointer(G_OBJECT(scrolling_actor->priv->animation_x), (gpointer*)&scrolling_actor->priv->animation_x);
							g_signal_connect_swapped(scrolling_actor->priv->animation_x, "completed", G_CALLBACK(tangle_scrolling_actor_clamp_scrolling_offset_x), scrolling_actor);
						} else {
							if (scrolling_actor->priv->scrolling_offset_x + dx < -scrolling_actor->priv->max_overshoot_x) {
								tx *= -(scrolling_actor->priv->scrolling_offset_x + scrolling_actor->priv->max_overshoot_x) / dx;
								dx = -(scrolling_actor->priv->scrolling_offset_x + scrolling_actor->priv->max_overshoot_x);
							} else if (scrolling_actor->priv->scrolling_offset_x + dx > scrolling_actor->priv->scrolling_width + scrolling_actor->priv->max_overshoot_x) {
								tx *= (scrolling_actor->priv->scrolling_width - scrolling_actor->priv->scrolling_offset_x + scrolling_actor->priv->max_overshoot_x) / dx;
								dx = (scrolling_actor->priv->scrolling_width - scrolling_actor->priv->scrolling_offset_x + scrolling_actor->priv->max_overshoot_x);					
							}

							if (tx > 0) {
								scrolling_actor->priv->animation_x = tangle_actor_animate(TANGLE_ACTOR(scrolling_actor), CLUTTER_EASE_OUT_QUAD, tx,
															  "scrolling-offset-x", scrolling_actor->priv->scrolling_offset_x + dx,
															  NULL);
								g_object_add_weak_pointer(G_OBJECT(scrolling_actor->priv->animation_x), (gpointer*)&scrolling_actor->priv->animation_x);
								g_signal_connect_swapped(scrolling_actor->priv->animation_x, "completed", G_CALLBACK(tangle_scrolling_actor_clamp_scrolling_offset_x), scrolling_actor);
								if (scrolling_actor->priv->scrolling_offset_x + dx < 0.0) {
									handle_overshooting_animation(scrolling_actor, scrolling_actor->priv->animation_x, "scrolling-offset-x", 0.0);
								} else if (scrolling_actor->priv->scrolling_offset_x + dx > scrolling_actor->priv->scrolling_width) {
									handle_overshooting_animation(scrolling_actor, scrolling_actor->priv->animation_x, "scrolling-offset-x", scrolling_actor->priv->scrolling_width);
								}					
							} else {
								tangle_scrolling_actor_clamp_scrolling_offset_x(scrolling_actor);
							}
						}
					} else {
						tangle_scrolling_actor_clamp_scrolling_offset_x(scrolling_actor);
					}
				}

				if (event_time - scrolling_actor->priv->start_motion_y_time > 0) {
					vy = (gdouble)(y - scrolling_actor->priv->start_motion_y) / (event_time - scrolling_actor->priv->start_motion_y_time);

					clamp = tangle_clamp_new(TANGLE_CLAMP_VERTICAL, normalise_scrolling_offset_y(scrolling_actor, scrolling_actor->priv->scrolling_offset_y + (scrolling_actor->priv->going_down ? -1 : 1) * vy * vy * 1000 / scrolling_actor->priv->scrolling_deceleration / 2, &delta));
					g_signal_emit(scrolling_actor, signals[CLAMP_SCROLLING_OFFSET_Y], 0, clamp);
					if (tangle_clamp_get_clamped_value_set(clamp)) {
						ay = vy * vy * 1000 / ABS_F(tangle_clamp_get_clamped_value(clamp) - scrolling_actor->priv->scrolling_offset_y + delta) / 2;
					} else {
						ay = scrolling_actor->priv->scrolling_deceleration;
					}
					g_object_unref(clamp);

					ty = ABS_F(vy * 1000.0 / ay);

					if (ty > 0) {
						dy = (scrolling_actor->priv->going_down ? -1 : 1) * vy * vy * 1000 / ay / 2;
						if (scrolling_actor->priv->continuous_y) {					
							scrolling_actor->priv->animation_y = tangle_actor_animate(TANGLE_ACTOR(scrolling_actor), CLUTTER_EASE_OUT_QUAD, ty,
														  "scrolling-offset-y", scrolling_actor->priv->scrolling_offset_y + dy,
														  NULL);
							g_object_add_weak_pointer(G_OBJECT(scrolling_actor->priv->animation_y), (gpointer*)&scrolling_actor->priv->animation_y);
							g_signal_connect_swapped(scrolling_actor->priv->animation_y, "completed", G_CALLBACK(tangle_scrolling_actor_clamp_scrolling_offset_y), scrolling_actor);
						} else {
							if (scrolling_actor->priv->scrolling_offset_y + dy < -scrolling_actor->priv->max_overshoot_y) {
								ty *= -(scrolling_actor->priv->scrolling_offset_y + scrolling_actor->priv->max_overshoot_y) / dy;
								dy = -(scrolling_actor->priv->scrolling_offset_y + scrolling_actor->priv->max_overshoot_y);
							} else if (scrolling_actor->priv->scrolling_offset_y + dy > scrolling_actor->priv->scrolling_height + scrolling_actor->priv->max_overshoot_y) {
								ty *= (scrolling_actor->priv->scrolling_height - scrolling_actor->priv->scrolling_offset_y + scrolling_actor->priv->max_overshoot_y) / dy;
								dy = (scrolling_actor->priv->scrolling_height - scrolling_actor->priv->scrolling_offset_y + scrolling_actor->priv->max_overshoot_y);					
							}

							if (ty > 0) {
								scrolling_actor->priv->animation_y = tangle_actor_animate(TANGLE_ACTOR(scrolling_actor), CLUTTER_EASE_OUT_QUAD, ty,
															  "scrolling-offset-y", scrolling_actor->priv->scrolling_offset_y + dy,
															  NULL);
								g_object_add_weak_pointer(G_OBJECT(scrolling_actor->priv->animation_y), (gpointer*)&scrolling_actor->priv->animation_y);
								g_signal_connect_swapped(scrolling_actor->priv->animation_y, "completed", G_CALLBACK(tangle_scrolling_actor_clamp_scrolling_offset_y), scrolling_actor);
								if (scrolling_actor->priv->scrolling_offset_y + dy < 0.0) {
									handle_overshooting_animation(scrolling_actor, scrolling_actor->priv->animation_y, "scrolling-offset-y", 0.0);
								} else if (scrolling_actor->priv->scrolling_offset_y + dy > scrolling_actor->priv->scrolling_height) {
									handle_overshooting_animation(scrolling_actor, scrolling_actor->priv->animation_y, "scrolling-offset-y", scrolling_actor->priv->scrolling_height);
								}					
							} else {
								tangle_scrolling_actor_clamp_scrolling_offset_y(scrolling_actor);
							}
						}					
					} else {
						tangle_scrolling_actor_clamp_scrolling_offset_y(scrolling_actor);
					}
				}
			} else if (event->type == CLUTTER_BUTTON_RELEASE) {
				tangle_scrolling_actor_clamp_offsets(scrolling_actor);
			}

			scrolling_actor->priv->previous_motion_time = event_time;
		}
		
		if (event->type == CLUTTER_BUTTON_RELEASE) {
			g_signal_handler_disconnect(actor, scrolling_actor->priv->captured_event_handler_id);
			scrolling_actor->priv->captured_event_handler_id = 0;
			send_interaction_event(TANGLE_ACTOR(scrolling_actor), FALSE);
		}

		g_object_unref(scrolling_actor);
	}

	return FALSE;
}

static void send_interaction_event(TangleActor* actor, gboolean began) {
	TangleEvent* event;
	
	event = tangle_event_new((began ? TANGLE_EVENT_INTERACTION_BEGAN : TANGLE_EVENT_INTERACTION_ENDED),
	                         TANGLE_ACTOR(actor));
	event->interaction.x_axis = TRUE;
	event->interaction.y_axis = TRUE;
	event->interaction.dragging = TRUE;
	tangle_actor_send_event(actor, event);
	tangle_event_free(event);
}

static gfloat get_width(TangleScrollingActor* 	scrolling_actor) {
	ClutterActorBox actor_box;
	gfloat width;
	
	clutter_actor_get_allocation_box(CLUTTER_ACTOR(scrolling_actor), &actor_box);
	if ((width = actor_box.x2 - actor_box.x1) == 0.0) {
		width = G_MINFLOAT;
	}
	
	return width;
}

static gfloat get_height(TangleScrollingActor* scrolling_actor) {
	ClutterActorBox actor_box;
	gfloat height;
	
	clutter_actor_get_allocation_box(CLUTTER_ACTOR(scrolling_actor), &actor_box);
	if ((height = actor_box.y2 - actor_box.y1) == 0.0) {
		height = G_MINFLOAT;
	}
	
	return height;
}

static gfloat normalise_scrolling_offset_x(TangleScrollingActor* scrolling_actor, gfloat scrolling_offset_x, gfloat* delta_return) {
	gfloat width;
	
	if (scrolling_actor->priv->continuous_x) {
		width = scrolling_actor->priv->scrolling_width + get_width(scrolling_actor);
		if (scrolling_offset_x < 0.0) {
			scrolling_offset_x -= (((gint)(scrolling_offset_x / width)) - 1) * width;
			if (delta_return) {
				*delta_return = (((gint)(scrolling_offset_x / width)) - 1) * width;
			}
		} else if (scrolling_offset_x > scrolling_actor->priv->scrolling_width + width) {
			scrolling_offset_x -= (((gint)(scrolling_offset_x / width))) * width;
			if (delta_return) {
				*delta_return = (((gint)(scrolling_offset_x / width))) * width;
			}
		}
	} else {
		if (scrolling_offset_x < -scrolling_actor->priv->max_overshoot_x) {
			if (delta_return) {
				*delta_return = -scrolling_actor->priv->max_overshoot_x - scrolling_offset_x;
			}
			scrolling_offset_x = -scrolling_actor->priv->max_overshoot_x;
		} else if (scrolling_offset_x > scrolling_actor->priv->scrolling_width + scrolling_actor->priv->max_overshoot_x) {
			if (delta_return) {
				*delta_return = scrolling_actor->priv->scrolling_width + scrolling_actor->priv->max_overshoot_x - scrolling_offset_x;
			}
			scrolling_offset_x = scrolling_actor->priv->scrolling_width + scrolling_actor->priv->max_overshoot_x;
		}
	}
	
	return scrolling_offset_x;
}

static gfloat normalise_scrolling_offset_y(TangleScrollingActor* scrolling_actor, gfloat scrolling_offset_y, gfloat* delta_return) {
	gfloat height;
	gfloat delta = 0.0;
	
	if (scrolling_actor->priv->continuous_y) {
			height = scrolling_actor->priv->scrolling_height + get_height(scrolling_actor);
		if (scrolling_offset_y < 0.0) {
			scrolling_offset_y -= (((gint)(scrolling_offset_y / height)) - 1) * height;
			if (delta_return) {
				*delta_return = (((gint)(scrolling_offset_y / height)) - 1) * height;
			}
		} else if (scrolling_offset_y > scrolling_actor->priv->scrolling_height + height) {
			scrolling_offset_y -= (((gint)(scrolling_offset_y / height))) * height;
			if (delta_return) {
				*delta_return = (((gint)(scrolling_offset_y / height))) * height;
			}
		}
	} else {
		if (scrolling_offset_y < -scrolling_actor->priv->max_overshoot_y) {
			if (delta_return) {
				*delta_return = -scrolling_actor->priv->max_overshoot_y - scrolling_offset_y;
			}
			scrolling_offset_y = -scrolling_actor->priv->max_overshoot_y;
		} else if (scrolling_offset_y > scrolling_actor->priv->scrolling_height + scrolling_actor->priv->max_overshoot_y) {
			if (delta_return) {
				*delta_return = scrolling_actor->priv->scrolling_height + scrolling_actor->priv->max_overshoot_y - scrolling_offset_y;
			}
			scrolling_offset_y = scrolling_actor->priv->scrolling_height + scrolling_actor->priv->max_overshoot_y;
		}
	}
	
	return scrolling_offset_y;
}
