/*
 * tangle-multiplexed-transition.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-multiplexed-transition.h"

/**
 * SECTION:tangle-multiplexed-transition
 * @Short_description: A transition that multiplexes different types of transitions
 * @Title: TangleMultiplexedTransition
 *
 * Normally a #TangleTransition handles all kinds of transitions in a similar way.
 * #TangleMultiplexedTransition can be used to separate the three different
 * cases: showing, moving and hiding of an actor. If an transition is not defined
 * (a property has null value), an actor is not animated at all in that specific case.
 */

G_DEFINE_TYPE(TangleMultiplexedTransition, tangle_multiplexed_transition, TANGLE_TYPE_TRANSITION);

enum {
	PROP_0,
	PROP_SHOW_TRANSITION,
	PROP_MOVE_TRANSITION,
	PROP_HIDE_TRANSITION
};

struct _TangleMultiplexedTransitionPrivate {
	TangleTransition* show_transition;
	TangleTransition* move_transition;
	TangleTransition* hide_transition;
};

TangleTransition* tangle_multiplexed_transition_new() {

	return TANGLE_TRANSITION(g_object_new(TANGLE_TYPE_MULTIPLEXED_TRANSITION, NULL));
}

TangleTransition* tangle_multiplexed_transition_new_full(TangleTransition* show_transition, TangleTransition* move_transition, TangleTransition* hide_transition) {

	return TANGLE_TRANSITION(g_object_new(TANGLE_TYPE_MULTIPLEXED_TRANSITION, "show-transition", show_transition, "move-transition", move_transition, "hide-transition", hide_transition, NULL));
}

TangleTransition* tangle_multiplexed_transition_get_show_transition(TangleMultiplexedTransition* multiplexed_transition) {

	return multiplexed_transition->priv->show_transition;
}

void tangle_multiplexed_transition_set_show_transition(TangleMultiplexedTransition* multiplexed_transition, TangleTransition* show_transition) {
	if (multiplexed_transition->priv->show_transition != show_transition) {
		if (show_transition) {
			g_object_ref(show_transition);
		}
		if (multiplexed_transition->priv->show_transition) {
			g_object_unref(multiplexed_transition->priv->show_transition);
		}
		multiplexed_transition->priv->show_transition = show_transition;
		g_object_notify(G_OBJECT(multiplexed_transition), "show-transition");
	}
}

TangleTransition* tangle_multiplexed_transition_get_move_transition(TangleMultiplexedTransition* multiplexed_transition) {

	return multiplexed_transition->priv->move_transition;
}

void tangle_multiplexed_transition_set_move_transition(TangleMultiplexedTransition* multiplexed_transition, TangleTransition* move_transition) {
	if (multiplexed_transition->priv->move_transition != move_transition) {
		if (move_transition) {
			g_object_ref(move_transition);
		}
		if (multiplexed_transition->priv->move_transition) {
			g_object_unref(multiplexed_transition->priv->move_transition);
		}
		multiplexed_transition->priv->move_transition = move_transition;
		g_object_notify(G_OBJECT(multiplexed_transition), "move-transition");
	}
}

TangleTransition* tangle_multiplexed_transition_get_hide_transition(TangleMultiplexedTransition* multiplexed_transition) {

	return multiplexed_transition->priv->hide_transition;
}

void tangle_multiplexed_transition_set_hide_transition(TangleMultiplexedTransition* multiplexed_transition, TangleTransition* hide_transition) {
	if (multiplexed_transition->priv->hide_transition != hide_transition) {
		if (hide_transition) {
			g_object_ref(hide_transition);
		}
		if (multiplexed_transition->priv->hide_transition) {
			g_object_unref(multiplexed_transition->priv->hide_transition);
		}
		multiplexed_transition->priv->hide_transition = hide_transition;
		g_object_notify(G_OBJECT(multiplexed_transition), "hide-transition");
	}
}

static ClutterTimeline* tangle_multiplexed_transition_animate_actor(TangleTransition* transition, TangleActor* actor, ClutterActorBox* current_box, ClutterActorBox* new_box) {
	ClutterAnimation* animation = NULL;
	TangleMultiplexedTransition* multiplexed_transition;
	
	multiplexed_transition = TANGLE_MULTIPLEXED_TRANSITION(transition);
	
	if (current_box && new_box) {
		if (multiplexed_transition->priv->move_transition) {
			tangle_transition_animate_actor(multiplexed_transition->priv->move_transition, actor, current_box, new_box);
		}
	} else if (current_box) {
		if (multiplexed_transition->priv->move_transition) {
			tangle_transition_animate_actor(multiplexed_transition->priv->hide_transition, actor, current_box, new_box);
		}	
	} else {
		if (multiplexed_transition->priv->move_transition) {
			tangle_transition_animate_actor(multiplexed_transition->priv->show_transition, actor, current_box, new_box);
		}	
	}
	
	return (animation ? clutter_animation_get_timeline(animation) : NULL);
}

static void tangle_multiplexed_transition_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleMultiplexedTransition* multiplexed_transition;
	
	multiplexed_transition = TANGLE_MULTIPLEXED_TRANSITION(object);

	switch (prop_id) {
		case PROP_SHOW_TRANSITION:
			tangle_multiplexed_transition_set_show_transition(multiplexed_transition, TANGLE_TRANSITION(g_value_get_object(value)));
			break;
		case PROP_MOVE_TRANSITION:
			tangle_multiplexed_transition_set_move_transition(multiplexed_transition, TANGLE_TRANSITION(g_value_get_object(value)));
			break;
		case PROP_HIDE_TRANSITION:
			tangle_multiplexed_transition_set_hide_transition(multiplexed_transition, TANGLE_TRANSITION(g_value_get_object(value)));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_multiplexed_transition_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleMultiplexedTransition* multiplexed_transition;

	multiplexed_transition = TANGLE_MULTIPLEXED_TRANSITION(object);

        switch (prop_id) {
		case PROP_SHOW_TRANSITION:
			g_value_set_object(value, multiplexed_transition->priv->show_transition);
			break;
		case PROP_MOVE_TRANSITION:
			g_value_set_object(value, multiplexed_transition->priv->move_transition);
			break;
		case PROP_HIDE_TRANSITION:
			g_value_set_object(value, multiplexed_transition->priv->hide_transition);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_multiplexed_transition_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_multiplexed_transition_parent_class)->finalize(object);
}

static void tangle_multiplexed_transition_dispose(GObject* object) {
	G_OBJECT_CLASS(tangle_multiplexed_transition_parent_class)->dispose(object);
}

static void tangle_multiplexed_transition_class_init(TangleMultiplexedTransitionClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);
	TangleTransitionClass* transition_class = TANGLE_TRANSITION_CLASS(klass);

	gobject_class->finalize = tangle_multiplexed_transition_finalize;
	gobject_class->dispose = tangle_multiplexed_transition_dispose;
	gobject_class->set_property = tangle_multiplexed_transition_set_property;
	gobject_class->get_property = tangle_multiplexed_transition_get_property;

	transition_class->animate_actor = tangle_multiplexed_transition_animate_actor;

	/**
	 * TangleMultiplexedTransition:show-transition:
	 *
	 * A transition that is used when showing an actor.
	 */
	g_object_class_install_property(gobject_class, PROP_SHOW_TRANSITION,
	                                g_param_spec_object("show-transition",
	                                "Show transition",
	                                "A transition that is used when showing an actor",
					TANGLE_TYPE_TRANSITION,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleMultiplexedTransition:move-transition:
	 *
	 * A transition that is used when moving an actor.
	 */
	g_object_class_install_property(gobject_class, PROP_MOVE_TRANSITION,
	                                g_param_spec_object("move-transition",
	                                "Move transition",
	                                "A transition that is used when moving an actor",
					TANGLE_TYPE_TRANSITION,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleMultiplexedTransition:hide-transition:
	 *
	 * A transition that is used when hiding an actor.
	 */
	g_object_class_install_property(gobject_class, PROP_HIDE_TRANSITION,
	                                g_param_spec_object("hide-transition",
	                                "Hide transition",
	                                "A transition that is used when hiding an actor",
					TANGLE_TYPE_TRANSITION,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private (gobject_class, sizeof (TangleMultiplexedTransitionPrivate));
}

static void tangle_multiplexed_transition_init(TangleMultiplexedTransition* multiplexed_transition) {
	multiplexed_transition->priv = G_TYPE_INSTANCE_GET_PRIVATE(multiplexed_transition, TANGLE_TYPE_MULTIPLEXED_TRANSITION, TangleMultiplexedTransitionPrivate);
}

