/*
 * tangle-droppable-actor.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-droppable-actor.h"
#include "marshalers.h"

/**
 * SECTION:tangle-droppable-actor
 * @Short_description: A wrapper actor that acts as drop area
 * @Title: TangleDroppableActor
 */

G_DEFINE_TYPE(TangleDroppableActor, tangle_droppable_actor, TANGLE_TYPE_WRAPPER_ACTOR);

enum {
	PROP_0,
	PROP_DROPPING_ENABLED
};

enum {
	HANDLE_DRAGGING,
	HANDLE_DROPPING,
	LAST_SIGNAL
};

struct _TangleDroppableActorPrivate {
	ClutterColor hilight_color;

	guint hilighted : 1;
	guint dropping_enabled : 1;
};

static guint signals[LAST_SIGNAL] = { 0 };

ClutterActor* tangle_droppable_actor_new(ClutterActor* wrapped) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_DROPPABLE_ACTOR, "wrapped", wrapped, NULL));
}

gboolean tangle_droppable_actor_is_dragging_droppable(TangleDroppableActor* droppable_actor, TangleDragging* dragging) {

	return droppable_actor->priv->dropping_enabled && TANGLE_DROPPABLE_ACTOR_GET_CLASS(droppable_actor)->is_dragging_droppable(droppable_actor, dragging);
}

void tangle_droppable_actor_handle_dragging(TangleDroppableActor* droppable_actor, TangleDragging* dragging) {
	gboolean handled;
	
	g_signal_emit(droppable_actor, signals[HANDLE_DRAGGING], 0, dragging, &handled);
}

void tangle_droppable_actor_handle_dropping(TangleDroppableActor* droppable_actor, TangleDragging* dragging, gboolean accepted) {
	gboolean handled;

	g_signal_emit(droppable_actor, signals[HANDLE_DROPPING], 0, dragging, accepted, &handled);
}

static gboolean tangle_droppable_actor_real_is_dragging_droppable(TangleDroppableActor* droppable_actor, TangleDragging* dragging) {

	return TRUE;
}

static gboolean tangle_droppable_actor_real_handle_dragging(TangleDroppableActor* droppable_actor, TangleDragging* dragging) {
	if (!droppable_actor->priv->hilighted) {
		droppable_actor->priv->hilighted = TRUE;
		clutter_actor_queue_redraw(CLUTTER_ACTOR(droppable_actor));
	}
	
	return TRUE;
}

static gboolean tangle_droppable_actor_real_handle_dropping(TangleDroppableActor* droppable_actor, TangleDragging* dragging, gboolean accepted) {
	if (droppable_actor->priv->hilighted) {
		droppable_actor->priv->hilighted = FALSE;
		clutter_actor_queue_redraw(CLUTTER_ACTOR(droppable_actor));
	}
	
	return TRUE;
}

static void tangle_droppable_actor_paint(ClutterActor* actor) {
	TangleDroppableActor* droppable_actor;
	gdouble scale_x;
	gdouble scale_y;
	guint8 alpha;
	ClutterActorBox box;

	droppable_actor = TANGLE_DROPPABLE_ACTOR(actor);

	if (CLUTTER_ACTOR_IS_MAPPED(actor)) {
		CLUTTER_ACTOR_CLASS(tangle_droppable_actor_parent_class)->paint(actor);

		if (droppable_actor->priv->hilighted) {
			cogl_push_matrix();

			g_object_get(actor, "transition-scale-x", &scale_x, "transition-scale-y", &scale_y, NULL);
			cogl_scale(scale_x, scale_y, 0);

			alpha = clutter_actor_get_paint_opacity(actor) * droppable_actor->priv->hilight_color.alpha / 255;
			cogl_set_source_color4ub(droppable_actor->priv->hilight_color.red,
			 			 droppable_actor->priv->hilight_color.green,
			 			 droppable_actor->priv->hilight_color.blue,
						 alpha);
			tangle_actor_get_aligned_allocation(TANGLE_ACTOR(actor), &box);
			cogl_rectangle(box.x1, box.y1, box.x2, box.y2);

			cogl_pop_matrix();
		}
	}
}

static void tangle_droppable_actor_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleDroppableActor* droppable_actor;
	
	droppable_actor = TANGLE_DROPPABLE_ACTOR(object);

	switch (prop_id) {
		case PROP_DROPPING_ENABLED:
			droppable_actor->priv->dropping_enabled = g_value_get_boolean(value);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_droppable_actor_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleDroppableActor* droppable_actor;

	droppable_actor = TANGLE_DROPPABLE_ACTOR(object);

        switch (prop_id) {
		case PROP_DROPPING_ENABLED:
			g_value_set_boolean(value, droppable_actor->priv->dropping_enabled);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_droppable_actor_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_droppable_actor_parent_class)->finalize(object);
}

static void tangle_droppable_actor_dispose(GObject* object) {
	G_OBJECT_CLASS(tangle_droppable_actor_parent_class)->dispose(object);
}

static void tangle_droppable_actor_class_init(TangleDroppableActorClass* droppable_actor_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(droppable_actor_class);
	ClutterActorClass* clutter_actor_class = CLUTTER_ACTOR_CLASS(droppable_actor_class);

	gobject_class->finalize = tangle_droppable_actor_finalize;
	gobject_class->dispose = tangle_droppable_actor_dispose;
	gobject_class->set_property = tangle_droppable_actor_set_property;
	gobject_class->get_property = tangle_droppable_actor_get_property;
	
	clutter_actor_class->paint = tangle_droppable_actor_paint;

	droppable_actor_class->is_dragging_droppable = tangle_droppable_actor_real_is_dragging_droppable;
	droppable_actor_class->handle_dragging = tangle_droppable_actor_real_handle_dragging;
	droppable_actor_class->handle_dropping = tangle_droppable_actor_real_handle_dropping;

	/**
	 * TangleDroppableActor:dropping-enabled:
	 *
	 * Whether the dropping actor accepts draggable actors or not.
	 */
	g_object_class_install_property(gobject_class, PROP_DROPPING_ENABLED,
	                                g_param_spec_boolean("dropping-enabled",
	                                "Dropping enabled",
	                                "Whether the dropping actor accepts draggable actors or not",
	                                TRUE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	/**
	 * TangleDroppableActor::handle-dragging:
	 * @actor: the object which received the signal
	 *
	 */
	signals[HANDLE_DRAGGING] = g_signal_new("handle-dragging", G_TYPE_FROM_CLASS(gobject_class),
	                                	G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(TangleDroppableActorClass, handle_dragging),
						g_signal_accumulator_true_handled, NULL,
						tangle_marshal_BOOLEAN__BOXED,
						G_TYPE_BOOLEAN, 1,
						TANGLE_TYPE_DRAGGING);
	/**
	 * TangleDroppableActor::handle-dropping:
	 * @actor: the object which received the signal
	 *
	 */
	signals[HANDLE_DROPPING] = g_signal_new("handle-dropping", G_TYPE_FROM_CLASS(gobject_class),
	                                	G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(TangleDroppableActorClass, handle_dropping),
						g_signal_accumulator_true_handled, NULL,
						tangle_marshal_BOOLEAN__BOXED_BOOLEAN,
						G_TYPE_BOOLEAN, 2,
						TANGLE_TYPE_DRAGGING,
						G_TYPE_BOOLEAN);

	g_type_class_add_private (gobject_class, sizeof (TangleDroppableActorPrivate));
}

static void tangle_droppable_actor_init(TangleDroppableActor* droppable_actor) {
	droppable_actor->priv = G_TYPE_INSTANCE_GET_PRIVATE(droppable_actor, TANGLE_TYPE_DROPPABLE_ACTOR, TangleDroppableActorPrivate);

	droppable_actor->priv->dropping_enabled = TRUE;

droppable_actor->priv->hilight_color.red = 255;
droppable_actor->priv->hilight_color.green = 255;
droppable_actor->priv->hilight_color.blue = 255;
droppable_actor->priv->hilight_color.alpha = 100;
}

