/*
 * tangle-draggable-actor.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-draggable-actor.h"
#include "marshalers.h"

/**
 * SECTION:tangle-draggable-actor
 * @Short_description: A wrapper actor that can be dragged around
 * @Title: TangleDraggableActor
 */

G_DEFINE_TYPE(TangleDraggableActor, tangle_draggable_actor, TANGLE_TYPE_FLOATING_ACTOR);

enum {
	PROP_0,
	PROP_FLOATING_MODE_DRAGGED,
	PROP_PLACEHOLDER_OPACITY_DRAGGED,
	PROP_DRAGGING_THRESHOLD_X,
	PROP_DRAGGING_THRESHOLD_Y,
	PROP_USE_ANCHOR_POINT
};

enum {
	BEGIN_DRAGGING,
	DO_DRAGGING,
	END_DRAGGING,
	LAST_SIGNAL
};

struct _TangleDraggableActorPrivate {
	TangleFloatingMode floating_mode_dragged;
	guchar placeholder_opacity_dragged;
	gfloat dragging_threshold_x;
	gfloat dragging_threshold_y;
	
	TangleDragging* dragging;
	gulong captured_event_handler_id;
	guint dragging_effective : 1;
	guint descendant_interacting : 1;
	guint descendant_dragging : 1;
	guint use_anchor_point : 1;
};

static guint signals[LAST_SIGNAL] = { 0 };

static gboolean on_captured_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data);
static void on_animation_completed(ClutterAnimation* animation, gpointer user_data);
static void handle_droppable_actor(TangleDragging* dragging);
static void send_interaction_event(TangleActor* actor, gboolean began);

ClutterActor* tangle_draggable_actor_new(ClutterActor* wrapped) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_DRAGGABLE_ACTOR, "wrapped", wrapped, NULL));
}

TangleDragging* tangle_dragging_new(TangleDraggableActor* draggable_actor) {
	TangleDragging* dragging;
	
	dragging = TANGLE_DRAGGING(g_new0(TangleDragging, 1));
	dragging->draggable_actor = draggable_actor;
	g_object_ref(draggable_actor);
	
	return dragging;
}

TangleDragging* tangle_dragging_copy(const TangleDragging* dragging) {
	TangleDragging* copy_of_dragging;

	copy_of_dragging = TANGLE_DRAGGING(g_new0(TangleDragging, 1));

	*copy_of_dragging = *dragging;
	g_object_ref(dragging->draggable_actor);
	if (dragging->droppable_actor) {
		//g_object_ref(dragging->droppable_actor);
	}
	
	return copy_of_dragging;
}

void tangle_dragging_free(TangleDragging* dragging) {
	g_object_unref(dragging->draggable_actor);
	if (dragging->droppable_actor) {
		//g_object_unref(dragging->droppable_actor);
	}
	g_free(dragging);
}

GType tangle_dragging_get_type(void) {
	static GType type = 0;

	if (G_UNLIKELY(type == 0)) {
		type = g_boxed_type_register_static("TangleDragging", (GBoxedCopyFunc)tangle_dragging_copy, (GBoxedFreeFunc)tangle_dragging_free);
	}
	
	return type;
}

void tangle_dragging_set_droppable_actor(TangleDragging* dragging, TangleDroppableActor* droppable_actor) {
	if (dragging->droppable_actor != droppable_actor) {
		if (dragging->droppable_actor) {
			g_object_unref(dragging->droppable_actor);
		}
		dragging->droppable_actor = droppable_actor;
		if (droppable_actor) {
			g_object_ref(droppable_actor);
		}
	}
}

gfloat tangle_draggable_actor_get_dragging_threshold_x(TangleDraggableActor* draggable_actor) {

	return draggable_actor->priv->dragging_threshold_x;
}

void tangle_draggable_actor_set_dragging_threshold_x(TangleDraggableActor* draggable_actor, gfloat dragging_threshold_x) {
	if (draggable_actor->priv->dragging_threshold_x != dragging_threshold_x) {
		draggable_actor->priv->dragging_threshold_x = dragging_threshold_x;
		g_object_notify(G_OBJECT(draggable_actor), "dragging-threshold-x");
	}
}

gfloat tangle_draggable_actor_get_dragging_threshold_y(TangleDraggableActor* draggable_actor) {

	return draggable_actor->priv->dragging_threshold_y;
}

void tangle_draggable_actor_set_dragging_threshold_y(TangleDraggableActor* draggable_actor, gfloat dragging_threshold_y) {
	if (draggable_actor->priv->dragging_threshold_y != dragging_threshold_y) {
		draggable_actor->priv->dragging_threshold_y = dragging_threshold_y;
		g_object_notify(G_OBJECT(draggable_actor), "dragging-threshold-y");
	}
}

void tangle_draggable_actor_set_dragging_threshold(TangleDraggableActor* draggable_actor, gfloat dragging_threshold) {
	tangle_draggable_actor_set_dragging_threshold_x(draggable_actor, dragging_threshold);
	tangle_draggable_actor_set_dragging_threshold_y(draggable_actor, dragging_threshold);
}

static gboolean tangle_draggable_actor_begin_dragging(TangleDraggableActor* draggable_actor, TangleDragging* dragging) {
	tangle_floating_actor_set_floating_mode(TANGLE_FLOATING_ACTOR(draggable_actor), draggable_actor->priv->floating_mode_dragged);
	tangle_floating_actor_set_placeholder_opacity(TANGLE_FLOATING_ACTOR(draggable_actor), draggable_actor->priv->placeholder_opacity_dragged);
	tangle_floating_actor_set_floating_position(TANGLE_FLOATING_ACTOR(draggable_actor), dragging->dx, dragging->dy);

	return TRUE;
}

static gboolean tangle_draggable_actor_do_dragging(TangleDraggableActor* draggable_actor, TangleDragging* dragging) {
	tangle_floating_actor_set_floating_position(TANGLE_FLOATING_ACTOR(draggable_actor), dragging->dx, dragging->dy);

	return TRUE;
}

static gboolean tangle_draggable_actor_end_dragging(TangleDraggableActor* draggable_actor, TangleDragging* dragging) {
	ClutterAnimation* animation;
	
	if (dragging->droppable_actor) {
		tangle_floating_actor_set_floating_scale_center(TANGLE_FLOATING_ACTOR(draggable_actor), dragging->anchor_x, dragging->anchor_y);
		if (draggable_actor->priv->floating_mode_dragged == TANGLE_FLOATING_MODE_PLACEHOLDER) {
			animation = tangle_actor_animate(TANGLE_ACTOR(draggable_actor), CLUTTER_EASE_OUT_QUAD, 250, "floating-scale-x", 0.0, "floating-scale-y", 0.0, "placeholder-opacity", 255, NULL);		
		} else {
			animation = tangle_actor_animate(TANGLE_ACTOR(draggable_actor), CLUTTER_EASE_OUT_QUAD, 250, "floating-scale-x", 0.0, "floating-scale-y", 0.0, NULL);		
		}		
	} else {
		if (draggable_actor->priv->floating_mode_dragged == TANGLE_FLOATING_MODE_COLLAPSE) {
			tangle_floating_actor_set_floating_mode(TANGLE_FLOATING_ACTOR(draggable_actor), TANGLE_FLOATING_MODE_PLACEHOLDER);
			tangle_floating_actor_set_placeholder_opacity(TANGLE_FLOATING_ACTOR(draggable_actor), 0);
		}
		animation = tangle_actor_animate(TANGLE_ACTOR(draggable_actor), CLUTTER_EASE_OUT_QUAD, 500, "floating-x", 0.0, "floating-y", 0.0, NULL);
	}
	g_signal_connect_after(animation, "completed", G_CALLBACK(on_animation_completed), draggable_actor);

	return FALSE;
}

static gboolean tangle_draggable_actor_button_press_event(ClutterActor* actor, ClutterButtonEvent* event) {
	TangleDraggableActor* draggable_actor;
	gboolean handled;
	gfloat x, y;
	gfloat width, height;
	
	draggable_actor = TANGLE_DRAGGABLE_ACTOR(actor);
	
	clutter_actor_get_transformed_position(actor, &x, &y);
	draggable_actor->priv->dragging = tangle_dragging_new(draggable_actor);
	draggable_actor->priv->dragging->x = event->x;
	draggable_actor->priv->dragging->y = event->y;
	draggable_actor->priv->dragging->anchor_x = event->x - x;
	draggable_actor->priv->dragging->anchor_y = event->y - y;
	draggable_actor->priv->dragging->dx = event->x - x - draggable_actor->priv->dragging->anchor_x;
	draggable_actor->priv->dragging->dy = event->y - y - draggable_actor->priv->dragging->anchor_y;
	clutter_actor_get_size(actor, &width, &height);
	draggable_actor->priv->dragging->center_x = x + width / 2;
	draggable_actor->priv->dragging->center_y = y + height / 2;

	draggable_actor->priv->dragging_effective = FALSE;
	draggable_actor->priv->captured_event_handler_id = g_signal_connect(clutter_actor_get_stage(actor), "captured-event", G_CALLBACK(on_captured_event), actor);

	return FALSE;
}

static void tangle_draggable_actor_handle_descendant_event(TangleActor* actor, TangleEvent* event) {
	TangleDraggableActor* draggable_actor;
	
	draggable_actor = TANGLE_DRAGGABLE_ACTOR(actor);
	
	if (event->any.sender != actor) {
		switch (event->type) {
			case TANGLE_EVENT_INTERACTION_BEGAN:
				draggable_actor->priv->descendant_interacting = TRUE;
				if (event->interaction.dragging) {
					draggable_actor->priv->descendant_dragging = TRUE;
				}
				break;
			case TANGLE_EVENT_INTERACTION_ENDED:
				draggable_actor->priv->descendant_interacting = FALSE;
				draggable_actor->priv->descendant_dragging = FALSE;
				break;
			default:
				break;
		}
	}

	TANGLE_ACTOR_CLASS(tangle_draggable_actor_parent_class)->handle_descendant_event(actor, event);
}

static void tangle_draggable_actor_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleDraggableActor* draggable_actor;
	
	draggable_actor = TANGLE_DRAGGABLE_ACTOR(object);

	switch (prop_id) {
		case PROP_DRAGGING_THRESHOLD_X:
			tangle_draggable_actor_set_dragging_threshold_x(draggable_actor, g_value_get_float(value));
			break;
		case PROP_DRAGGING_THRESHOLD_Y:
			tangle_draggable_actor_set_dragging_threshold_y(draggable_actor, g_value_get_float(value));
			break;
		case PROP_USE_ANCHOR_POINT:
			draggable_actor->priv->use_anchor_point = g_value_get_boolean(value);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_draggable_actor_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleDraggableActor* draggable_actor;

	draggable_actor = TANGLE_DRAGGABLE_ACTOR(object);

        switch (prop_id) {
		case PROP_DRAGGING_THRESHOLD_X:
			g_value_set_float(value, draggable_actor->priv->dragging_threshold_x);
			break;
		case PROP_DRAGGING_THRESHOLD_Y:
			g_value_set_float(value, draggable_actor->priv->dragging_threshold_y);
			break;
		case PROP_USE_ANCHOR_POINT:
			g_value_set_boolean(value, draggable_actor->priv->use_anchor_point);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_draggable_actor_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_draggable_actor_parent_class)->finalize(object);
}

static void tangle_draggable_actor_dispose(GObject* object) {
	G_OBJECT_CLASS(tangle_draggable_actor_parent_class)->dispose(object);
}

static void tangle_draggable_actor_class_init(TangleDraggableActorClass* draggable_actor_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(draggable_actor_class);
	ClutterActorClass* clutter_actor_class = CLUTTER_ACTOR_CLASS(draggable_actor_class);
	TangleActorClass* actor_class = TANGLE_ACTOR_CLASS(draggable_actor_class);

	gobject_class->finalize = tangle_draggable_actor_finalize;
	gobject_class->dispose = tangle_draggable_actor_dispose;
	gobject_class->set_property = tangle_draggable_actor_set_property;
	gobject_class->get_property = tangle_draggable_actor_get_property;
	
	clutter_actor_class->button_press_event = tangle_draggable_actor_button_press_event;

	actor_class->handle_descendant_event = tangle_draggable_actor_handle_descendant_event;

	draggable_actor_class->begin_dragging = tangle_draggable_actor_begin_dragging;
	draggable_actor_class->do_dragging = tangle_draggable_actor_do_dragging;
	draggable_actor_class->end_dragging = tangle_draggable_actor_end_dragging;

	/**
	 * TangleDraggableActor:dragging-threshold-x:
	 */
	g_object_class_install_property(gobject_class, PROP_DRAGGING_THRESHOLD_X,
	                                g_param_spec_float("dragging-threshold-x",
	                                                    "Dragging threshold X",
	                                                    "The horizontal threshold before dragging starts",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleDraggableActor:dragging-threshold-y:
	 */
	g_object_class_install_property(gobject_class, PROP_DRAGGING_THRESHOLD_Y,
	                                g_param_spec_float("dragging-threshold-y",
	                                                    "Dragging threshold Y",
	                                                    "The vertical threshold before dragging starts",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleDraggableActor:use-anchor-point:
	 *
	 * Whether to use anchor (touch) point or center point as a reference when checking droppable area.
	 */
	g_object_class_install_property(gobject_class, PROP_DRAGGING_THRESHOLD_Y,
	                                g_param_spec_boolean("use-anchor-point",
	                                                     "Use anchor point",
	                                                     "Whether to use anchor (touch) point or center point as a reference",
	                                                     FALSE,
	                                                     G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	/**
	 * TangleDraggableActor::begin-dragging:
	 * @actor: the object which received the signal
	 *
	 */
	signals[BEGIN_DRAGGING] = g_signal_new("begin-dragging", G_TYPE_FROM_CLASS(gobject_class),
	                                       G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(TangleDraggableActorClass, begin_dragging),
					       g_signal_accumulator_true_handled, NULL,
					       tangle_marshal_BOOLEAN__BOXED,
					       G_TYPE_BOOLEAN, 1,
					       TANGLE_TYPE_DRAGGING);
	/**
	 * TangleDraggableActor::do-dragging:
	 * @actor: the object which received the signal
	 *
	 */
	signals[DO_DRAGGING] = g_signal_new("do-dragging", G_TYPE_FROM_CLASS(gobject_class),
	                                    G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(TangleDraggableActorClass, do_dragging),
					    g_signal_accumulator_true_handled, NULL,
					    tangle_marshal_BOOLEAN__BOXED,
					    G_TYPE_BOOLEAN, 1,
					    TANGLE_TYPE_DRAGGING);
	/**
	 * TangleDraggableActor::end-dragging:
	 * @actor: the object which received the signal
	 *
	 */
	signals[END_DRAGGING] = g_signal_new("end-dragging", G_TYPE_FROM_CLASS(gobject_class),
	                                     G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(TangleDraggableActorClass, end_dragging),
					     g_signal_accumulator_true_handled, NULL,
					     tangle_marshal_BOOLEAN__BOXED,
					     G_TYPE_BOOLEAN, 1,
					     TANGLE_TYPE_DRAGGING);

	g_type_class_add_private (gobject_class, sizeof (TangleDraggableActorPrivate));
}

static void tangle_draggable_actor_init(TangleDraggableActor* draggable_actor) {
	draggable_actor->priv = G_TYPE_INSTANCE_GET_PRIVATE(draggable_actor, TANGLE_TYPE_DRAGGABLE_ACTOR, TangleDraggableActorPrivate);
	clutter_actor_set_reactive(CLUTTER_ACTOR(draggable_actor), TRUE);

	draggable_actor->priv->floating_mode_dragged = TANGLE_FLOATING_MODE_PLACEHOLDER;
	draggable_actor->priv->placeholder_opacity_dragged = 50;
}

#define ABS_F(x) ((x) < 0 ? -(x) : (x))
static gboolean on_captured_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data) {
	TangleDraggableActor* draggable_actor;
	gboolean handled = TRUE;
	gfloat x, y;
	gfloat width, height;
	
	if (event->type == CLUTTER_MOTION || event->type == CLUTTER_BUTTON_RELEASE) {
		draggable_actor = TANGLE_DRAGGABLE_ACTOR(user_data);
		
		g_object_ref(draggable_actor);

		clutter_event_get_coords(event, &draggable_actor->priv->dragging->x, &draggable_actor->priv->dragging->y);
		clutter_actor_get_transformed_position(CLUTTER_ACTOR(draggable_actor), &x, &y);
		draggable_actor->priv->dragging->dx = draggable_actor->priv->dragging->x - x - draggable_actor->priv->dragging->anchor_x;
		draggable_actor->priv->dragging->dy = draggable_actor->priv->dragging->y - y - draggable_actor->priv->dragging->anchor_y;
		clutter_actor_get_size(CLUTTER_ACTOR(draggable_actor), &width, &height);
		draggable_actor->priv->dragging->center_x = x + width / 2 + draggable_actor->priv->dragging->dx;
		draggable_actor->priv->dragging->center_y = y + height / 2 + draggable_actor->priv->dragging->dy;

		if (draggable_actor->priv->dragging_effective) {
			handle_droppable_actor(draggable_actor->priv->dragging);
			if (event->type == CLUTTER_BUTTON_RELEASE) {
				send_interaction_event(TANGLE_ACTOR(draggable_actor), FALSE);
			}
			g_signal_emit(draggable_actor, signals[(event->type == CLUTTER_MOTION ? DO_DRAGGING : END_DRAGGING)], 0, draggable_actor->priv->dragging, &handled);
			if (event->type == CLUTTER_BUTTON_RELEASE) {
				if (draggable_actor->priv->dragging->droppable_actor) {
					tangle_droppable_actor_handle_dropping(draggable_actor->priv->dragging->droppable_actor, draggable_actor->priv->dragging, TRUE);
				}
				tangle_dragging_free(draggable_actor->priv->dragging);
				draggable_actor->priv->dragging = NULL;
				g_signal_handler_disconnect(actor, draggable_actor->priv->captured_event_handler_id);
			}
		} else {
			if (event->type == CLUTTER_MOTION && !draggable_actor->priv->descendant_dragging &&
			    (!draggable_actor->priv->descendant_interacting ||
			     ABS_F(draggable_actor->priv->dragging->dy) > draggable_actor->priv->dragging_threshold_x ||
			     ABS_F(draggable_actor->priv->dragging->dx) > draggable_actor->priv->dragging_threshold_y)) {
				draggable_actor->priv->dragging_effective = TRUE;
				handle_droppable_actor(draggable_actor->priv->dragging);
				g_signal_emit(draggable_actor, signals[BEGIN_DRAGGING], 0, draggable_actor->priv->dragging, &handled);	
				if (handled) {
					send_interaction_event(TANGLE_ACTOR(draggable_actor), TRUE);
				}
			}
			if (event->type == CLUTTER_BUTTON_RELEASE || !handled) {
				tangle_dragging_free(draggable_actor->priv->dragging);
				draggable_actor->priv->dragging = NULL;
				g_signal_handler_disconnect(actor, draggable_actor->priv->captured_event_handler_id);
			}
		}

		g_object_unref(draggable_actor);
	}

	return FALSE;
}

static void on_animation_completed(ClutterAnimation* animation, gpointer user_data) {	
	tangle_floating_actor_set_floating_mode(TANGLE_FLOATING_ACTOR(user_data), TANGLE_FLOATING_MODE_NONE);
	tangle_floating_actor_set_floating_scale(TANGLE_FLOATING_ACTOR(user_data), 1.0, 1.0);
}

static TangleDroppableActor* get_droppable_actor_from_hierarchy(ClutterActor* actor) {
	TangleDroppableActor* droppable_actor = NULL;
	ClutterActor* parent;
	
	if (TANGLE_IS_DROPPABLE_ACTOR(actor)) {
		droppable_actor = TANGLE_DROPPABLE_ACTOR(actor);
	} else if ((parent = clutter_actor_get_parent(actor))) {
		droppable_actor = get_droppable_actor_from_hierarchy(parent);
	}
	
	return droppable_actor;
}
		
static void handle_droppable_actor(TangleDragging* dragging) {
	ClutterActor* actor;
	TangleDroppableActor* droppable_actor;
	
	if (dragging->draggable_actor->priv->use_anchor_point) {
		actor = clutter_stage_get_actor_at_pos(CLUTTER_STAGE(clutter_actor_get_stage(CLUTTER_ACTOR(dragging->draggable_actor))), CLUTTER_PICK_ALL, dragging->x, dragging->y);
	} else {
		actor = clutter_stage_get_actor_at_pos(CLUTTER_STAGE(clutter_actor_get_stage(CLUTTER_ACTOR(dragging->draggable_actor))), CLUTTER_PICK_ALL, dragging->center_x, dragging->center_y);
	}
	droppable_actor = get_droppable_actor_from_hierarchy(actor);
	if (dragging->droppable_actor != droppable_actor) {
		if (dragging->droppable_actor) {
			tangle_droppable_actor_handle_dropping(dragging->droppable_actor, dragging, FALSE);
		}
		if (droppable_actor && tangle_droppable_actor_is_dragging_droppable(droppable_actor, dragging)) {
			tangle_dragging_set_droppable_actor(dragging, droppable_actor);
			tangle_droppable_actor_handle_dragging(droppable_actor, dragging);
		} else {
			tangle_dragging_set_droppable_actor(dragging, NULL);
		}
	} else if (dragging->droppable_actor) {
		if (!tangle_droppable_actor_is_dragging_droppable(droppable_actor, dragging)) {
			tangle_droppable_actor_handle_dropping(dragging->droppable_actor, dragging, FALSE);
			tangle_dragging_set_droppable_actor(dragging, NULL);
		} else {
			tangle_droppable_actor_handle_dragging(droppable_actor, dragging);
		}
	}
}

static void send_interaction_event(TangleActor* actor, gboolean began) {
	TangleEvent* event;

	event = tangle_event_new((began ? TANGLE_EVENT_INTERACTION_BEGAN : TANGLE_EVENT_INTERACTION_ENDED),
	                         TANGLE_ACTOR(actor));
	event->interaction.x_axis = TRUE;
	event->interaction.y_axis = TRUE;
	event->interaction.dragging = TRUE;
	tangle_actor_send_event(actor, event);
	tangle_event_free(event);
}
