/*
 * tangle-button.h
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2009-2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-button.h"
#include "marshalers.h"

/**
 * SECTION:tangle-button
 * @Short_description: 	A widget with clicked functionality
 * @Title: TangleButton
 */

G_DEFINE_TYPE(TangleButton, tangle_button, TANGLE_TYPE_WIDGET);

enum {
	PROP_0,
	PROP_NORMAL_BACKGROUND_ACTOR,
	PROP_INTERACTIVE_BACKGROUND_ACTOR,
	PROP_CLICKED_ACTION
};

enum {
	CLICKED,
	LAST_SIGNAL
};

struct _TangleButtonPrivate {
	ClutterActor* normal_background_actor;
	ClutterActor* interactive_background_actor;
	TangleAction* clicked_action;
	guint button_release_event_handler_id;
	guint active : 1;
	guint inside : 1;
};

static void begin_interacting(TangleButton* button);
static void end_interacting(TangleButton* button);

static guint signals[LAST_SIGNAL];

ClutterActor* tangle_button_new(void) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_BUTTON, NULL));
}

ClutterActor* tangle_button_new_with_background_actor(ClutterActor* normal_background_actor) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_BUTTON, "normal-background-actor", normal_background_actor, "prefer-background-size", TRUE, NULL));
}

ClutterActor* tangle_button_new_with_background_actors(ClutterActor* normal_background_actor, ClutterActor* interactive_background_actor) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_BUTTON, "normal-background-actor", normal_background_actor, "interactive-background-actor", interactive_background_actor, "prefer-background-size", TRUE, NULL));
}

ClutterActor* tangle_button_get_normal_background_actor(TangleButton* button) {
	g_return_val_if_fail(TANGLE_IS_BUTTON(button), NULL);

	return button->priv->normal_background_actor;
}

ClutterActor* tangle_button_get_interactive_background_actor(TangleButton* button) {
	g_return_val_if_fail(TANGLE_IS_BUTTON(button), NULL);

	return button->priv->interactive_background_actor;
}

TangleAction* tangle_button_get_clicked_action(TangleButton* button) {

	return button->priv->clicked_action;
}

void tangle_button_set_normal_background_actor(TangleButton* button, ClutterActor* actor) {
	g_return_if_fail(TANGLE_IS_BUTTON(button));
        
	if (button->priv->normal_background_actor != actor) {
		if (button->priv->normal_background_actor) {
			g_object_unref(button->priv->normal_background_actor);
		}
		button->priv->normal_background_actor = actor;
		g_object_ref_sink(button->priv->normal_background_actor);

		if (!button->priv->active || !button->priv->interactive_background_actor) {
			tangle_widget_set_background_actor(TANGLE_WIDGET(button), button->priv->normal_background_actor);
		}
	}
}

void tangle_button_set_interactive_background_actor(TangleButton* button, ClutterActor* actor) {
	gboolean had_interactive_background_actor = FALSE;
	
	g_return_if_fail(TANGLE_IS_BUTTON(button));

	if (button->priv->interactive_background_actor != actor) {
		if (button->priv->interactive_background_actor) {
			g_object_unref(button->priv->interactive_background_actor);
			had_interactive_background_actor = TRUE;
		}
		button->priv->interactive_background_actor = actor;
		g_object_ref_sink(button->priv->interactive_background_actor);

		if (button->priv->active) {
			if (button->priv->interactive_background_actor) {
				tangle_widget_set_background_actor(TANGLE_WIDGET(button), button->priv->interactive_background_actor);
			} else if (had_interactive_background_actor) {
				tangle_widget_set_background_actor(TANGLE_WIDGET(button), button->priv->normal_background_actor);
			}
		}
	}
}

void tangle_button_set_clicked_action(TangleButton* button, TangleAction* action) {
	if (button->priv->clicked_action) {
		g_object_unref(button->priv->clicked_action);
	}
	g_object_ref(action);
	button->priv->clicked_action = action;
	g_object_notify(G_OBJECT(button), "clicked-action");
}

static void tangle_button_handle_ancestor_event(TangleActor* actor, TangleEvent* event) {
	if (event->any.sender != actor) {
		if (event->type == TANGLE_EVENT_INTERACTION_BEGAN) {
			end_interacting(TANGLE_BUTTON(actor));
			TANGLE_BUTTON(actor)->priv->active = FALSE;
		}
	}

	TANGLE_ACTOR_CLASS(tangle_button_parent_class)->handle_ancestor_event(actor, event);	
}

static gboolean tangle_button_clicked(TangleButton* button) {
	TangleProperties* properties;
	
	if (button->priv->clicked_action) {
		properties = tangle_properties_new();
		tangle_action_execute_full(button->priv->clicked_action, G_OBJECT(button), "clicked", properties);
		g_object_unref(G_OBJECT(properties));
	}
	
	return FALSE;
}

static void tangle_button_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleButton* button;
	
	button = TANGLE_BUTTON(object);

	switch (prop_id) {
		case PROP_NORMAL_BACKGROUND_ACTOR:
			tangle_button_set_normal_background_actor(button, CLUTTER_ACTOR(g_value_get_object(value)));
			break;
		case PROP_INTERACTIVE_BACKGROUND_ACTOR:
			tangle_button_set_interactive_background_actor(button, CLUTTER_ACTOR(g_value_get_object(value)));
			break;
		case PROP_CLICKED_ACTION:
			tangle_button_set_clicked_action(button, TANGLE_ACTION(g_value_get_object(value)));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_button_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleButton* button;

	button = TANGLE_BUTTON(object);

        switch (prop_id) {
		case PROP_NORMAL_BACKGROUND_ACTOR:
			g_value_set_object(value, button->priv->normal_background_actor);
			break;
		case PROP_INTERACTIVE_BACKGROUND_ACTOR:
			g_value_set_object(value, button->priv->interactive_background_actor);
			break;
		case PROP_CLICKED_ACTION:
			g_value_set_object(value, tangle_button_get_clicked_action(button));
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_button_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_button_parent_class)->finalize(object);
}

static void tangle_button_dispose(GObject* object) {
	G_OBJECT_CLASS(tangle_button_parent_class)->dispose(object);
}

static void tangle_button_class_init(TangleButtonClass* button_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS (button_class);
	ClutterActorClass* clutter_actor_class = CLUTTER_ACTOR_CLASS(button_class);
	TangleActorClass* actor_class = TANGLE_ACTOR_CLASS(button_class);

	gobject_class->finalize = tangle_button_finalize;
	gobject_class->dispose = tangle_button_dispose;
	gobject_class->set_property = tangle_button_set_property;
	gobject_class->get_property = tangle_button_get_property;

	actor_class->handle_ancestor_event = tangle_button_handle_ancestor_event;

	button_class->clicked = tangle_button_clicked;

	/**
	 * TangleClutterActor:normal-background-actor:
	 */
	g_object_class_install_property(gobject_class, PROP_NORMAL_BACKGROUND_ACTOR,
	                                g_param_spec_object("normal-background-actor",
	                                                    "Normal background actor",
	                                                    "The background actor used in normal (noninteractive) state",
	                                                    CLUTTER_TYPE_ACTOR,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_CONSTRUCT));

	/**
	 * TangleClutterActor:interactive-background-actor:
	 */
	g_object_class_install_property(gobject_class, PROP_INTERACTIVE_BACKGROUND_ACTOR,
	                                g_param_spec_object("interactive-background-actor",
	                                                    "Actor",
	                                                    "The background actor used in interactive (interacting) state",
	                                                    CLUTTER_TYPE_ACTOR,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_CONSTRUCT));
	/**
	 * TangleButton:on-clicked-action:
	 */
	g_object_class_install_property(gobject_class, PROP_CLICKED_ACTION,
	                                g_param_spec_object("clicked-action",
	                                "Clicked action",
	                                "Action that is executed on clicked signal",
	                                TANGLE_TYPE_ACTION,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	/**
	 * TangleButton:clicked:
	 * @actor: the object which received the signal
	 */
	signals[CLICKED] = g_signal_new("clicked", G_TYPE_FROM_CLASS(gobject_class),
	                                G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(TangleButtonClass, clicked),
					g_signal_accumulator_true_handled, NULL,
					tangle_marshal_BOOLEAN__VOID,
					G_TYPE_BOOLEAN, 0);

	g_type_class_add_private(gobject_class, sizeof(TangleButtonPrivate));
}

static gboolean on_button_release_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data);

static void start_interacting(TangleButton* button) {
	TangleEvent* event;
	
	tangle_actor_set_interacting(TANGLE_ACTOR(button), TRUE);
	if (button->priv->interactive_background_actor) {
		tangle_widget_set_background_actor(TANGLE_WIDGET(button), button->priv->interactive_background_actor);
	}
	button->priv->button_release_event_handler_id = g_signal_connect(clutter_actor_get_stage(CLUTTER_ACTOR(button)), "button-release-event", G_CALLBACK(on_button_release_event), button);

	event = tangle_event_new(TANGLE_EVENT_INTERACTION_BEGAN, TANGLE_ACTOR(button));
	event->interaction.x_axis = TRUE;
	event->interaction.y_axis = TRUE;
	tangle_actor_send_event(TANGLE_ACTOR(button), event);
	tangle_event_free(event);
}

static void end_interacting(TangleButton* button) {
	TangleEvent* event;
	
	tangle_actor_set_interacting(TANGLE_ACTOR(button), FALSE);
	if (button->priv->interactive_background_actor) {
		tangle_widget_set_background_actor(TANGLE_WIDGET(button), button->priv->normal_background_actor);
	}
	if (button->priv->button_release_event_handler_id) {
		g_signal_handler_disconnect(clutter_actor_get_stage(CLUTTER_ACTOR(button)), button->priv->button_release_event_handler_id);
		button->priv->button_release_event_handler_id = 0;
	}

	event = tangle_event_new(TANGLE_EVENT_INTERACTION_ENDED, TANGLE_ACTOR(button));
	event->interaction.x_axis = TRUE;
	event->interaction.y_axis = TRUE;
	tangle_actor_send_event(TANGLE_ACTOR(button), event);
	tangle_event_free(event);
}

static gboolean on_button_release_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data) {
	gboolean handled;

	end_interacting(TANGLE_BUTTON(user_data));
	TANGLE_BUTTON(user_data)->priv->active = FALSE;;

	if (TANGLE_BUTTON(user_data)->priv->inside) {
		g_signal_emit(user_data, signals[CLICKED], 0, &handled);
	}

	return FALSE;
}

static gboolean on_button_press_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data) {
	start_interacting(TANGLE_BUTTON(actor));
	TANGLE_BUTTON(actor)->priv->active = TRUE;
	TANGLE_BUTTON(actor)->priv->inside = TRUE;
	
	return FALSE;
}

static gboolean on_enter_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data) {
	if (TANGLE_BUTTON(actor)->priv->active) {
		start_interacting(TANGLE_BUTTON(actor));
	}
	TANGLE_BUTTON(actor)->priv->inside = TRUE;
	
	return FALSE;
}

static gboolean on_leave_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data) {
	if (TANGLE_BUTTON(actor)->priv->active) {
		end_interacting(TANGLE_BUTTON(actor));
	}
	TANGLE_BUTTON(actor)->priv->inside = FALSE;
	
	return FALSE;

}

static void tangle_button_init(TangleButton* button) {
	button->priv = G_TYPE_INSTANCE_GET_PRIVATE(button, TANGLE_TYPE_BUTTON, TangleButtonPrivate);
	clutter_actor_set_reactive(CLUTTER_ACTOR(button), TRUE);
	g_signal_connect(button, "button-press-event", G_CALLBACK(on_button_press_event), NULL);
	g_signal_connect(button, "enter-event", G_CALLBACK(on_enter_event), NULL);
	g_signal_connect(button, "leave-event", G_CALLBACK(on_leave_event), NULL);
}
