/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003-2010 LUT. .
 *
 * @name GPRSConnection.cc
 * @memo GPRS implementation of the MAbstractConnection interface.
 *
 * @version 0.2
 * date     02.10.2003
 * change   28.04.2010
 */

#include <cassert>
#include <sstream>
#include <stdio.h>
#include <iostream>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <assert.h>
#include <stdlib.h>
#include "GPRSConnection.h"

#warning "Temporary debug"
//temp
#include <syslog.h>

#define ERR(format, msg...) syslog(LOG_ERR, "ERROR: " format "\n" , ## msg)

#ifdef PH_DEBUG
#define DBG(format, msg...) syslog(LOG_DEBUG, format "\n" , ## msg)
#else
#define DBG( A... )
#endif
//temp

using namespace std;

/**
 * @memo Default constructor.
 * @doc Default constructor, sets the initial vatiables to their default values
 * and creates a new local socket.
 *
 * @return none
 */
CGPRSConnection::CGPRSConnection()
{
  iConnected = false;
  iIsListening = false;

  iSocket = socket(PF_UNIX, SOCK_STREAM, 0);
  assert(iSocket != -1);
}

/**
 * @memo Default Destructor.
 * @doc Default destructor, removes the local socket if listening. Similarly, the
 * possibly open connection is closed.
 *
 * @return none
 */

CGPRSConnection::~CGPRSConnection()
{
  if (iIsListening) unlink(iAddress.c_str());

  if (iConnected) Close();
}

/**
 * @memo Connects a gprs-plugin to a local service or
 * local service to a gprs-plugin.
 * @doc Creates connection between gprs-plugin and
 * provided service. Connection is established
 * using local sockets. When connection is established
 * from service to plugin, DATA message is sent. This
 * message contains receivers id and port information.
 *
 * @param aAddress The destination address. This can
 * be string "toService" for connecting gprs-plugin to service. Otherwise
 * connection is made from local service to gprs-plugin
 * @param aPort Target port. This is reserved for future use.
 *
 * @return true if the connection could be established
 */
bool CGPRSConnection::Connect(const std::string& aAddress, int aPort)
{
  struct sockaddr_un address;
  const char *parsedId;
  int id = 0;
  std::string serviceName;
  ostringstream addr;
    
  if (iConnected) {
    DBG("CGPRSConnection::Connect : Already connected");
    return false;
  }
 
  addr << "/tmp/phGPRS.";
  bzero(&address, sizeof(address));
  address.sun_family = PF_UNIX;
  
  if(aAddress == "toService")
    addr << aPort;
  else
    addr << 0;
  
  strcpy(address.sun_path, addr.str().c_str());
  
  if (connect(iSocket, (struct sockaddr *)&address, sizeof(address)) == -1) {
    ERR("CGPRSConnection::Connect : Connect failed");
    return false;
  }
  
  //if connection is made from local service to plugin, send connect info
  if(aAddress != "toService")
    {
      parsedId = strstr(aAddress.c_str(), "GPRS");
      id = atoi(parsedId+4);
      
      //send id
      if(send(iSocket, &id, sizeof(id), 0) <=0){
	return false;
      }
    }

  iRemoteAddress = std::string("GPRS:") + std::string(addr.str());

  iConnected = true;
  return true;
}

/**
 * @memo Kills the existing connection.
 * @doc Kills the existing connection. If there's no connection or it could not
 * be closed in a controlled way then the error code is returned. The existing
 * connection will be closed regardless of the return value.
 *
 * @return true if the connection was dropped in a controlled manner
 */
bool CGPRSConnection::Disconnect()
{
  if (!iConnected) {
    return false;
  }
  
  close(iSocket);
  iConnected = false;
  
  return true;
}


/**
 * @memo Explicitly closes the socket.
 * @doc Explicitly closes the socket without any additional state checking,
 * i.e. this function will execute even if there's no existing connection.
 * This function is used by the PeerHood engine to ensure that no file
 * descriptors are leaked.
 *
 * @return none
 */
void CGPRSConnection::Close()
{
  close(iSocket);
}

/**
 * @memo Sends data to the other end of the connection.
 * @doc Sends data to the other end of the connection. Note that this
 * function is not endianess-safe.
 *
 * @param aOutBuf Buffer containing the data that should be written.
 * @param aLength The number of bytes to be written.
 *
 * @return the number of bytes written or -1 in the case of an error
 */
int CGPRSConnection::Write(const void* aOutBuf, int aLength)
{
  int written = 0;

  if (!iConnected) {
    return -1;
  }

  write(iSocket, &aLength, sizeof(aLength));
  written = write(iSocket, (char *)aOutBuf, aLength);
  return written;
}

/**
 * @memo Reads data from an open connection.
 * @doc Reads data from an open connection. Just like 
 * <code>Read</code> this function is not endianess-safe so it should be
 * treated more like a standard Posix sequential socket.
 *
 * @param aInBuf Buffer where the data is read to.
 * @param aLength The number of bytes that should be read.
 *
 * @return the number of bytes read or -1 in the case of an error
 */
int CGPRSConnection::Read(void* aInBuf, int aLength)
{
   int retval = 0;

   if (!iConnected) {
     return -1;
   }
   retval = read(iSocket, (char *)aInBuf, aLength);
  return retval;
}

/**
 * @memo Returns connection's remote address.
 * @doc Returns connection's remote address. Note that the existence of the
 * address doesn't mean that there's a connection established.
 *
 * @return connection's remote address
 */
const std::string& CGPRSConnection::GetRemoteAddress()
{  
  return iRemoteAddress;
}

/**
 * @memo Returns connection's file descriptor.
 * @doc Returns connection's file descriptor. Other classes can have a direct
 * access to the connection's socket (or similar entity) via this function's
 * return value.
 *
 * @return connection's file descriptor or -1 if one isn't available
 */
int CGPRSConnection::GetFd()
{
  // if (!iConnected && !iIsListening) return -1;

  return iSocket;
}

/**
 * @memo Sets the connection to the listening state.
 * @doc Sets the connection to the listening state. This function must be
 * called before any incoming connections targeted to this object can be
 * accepted. This function will always listen for the service name specific
 * port for example "exampleService1.local"
 * If some other port is needed then the <code>Listen(int aPath)</code>
 * should be used instead.
 *
 * @return true if successfull
 */
bool CGPRSConnection::Listen()
{
  if(!Listen(0))
    return false;
  
  return true;
}

/**
 * @memo Sets the connection to the listening state.
 * @doc Sets the connection to the listening state. This function must be
 * called before any incoming connections targeted to this object can be
 * accepted.
 *
 * @param aPath The path can be 0 for listening
 * path for service, otherwise path for plugin
 * is listened.
 *
 * @return true if successfull
 */
bool CGPRSConnection::Listen(int aPort)
{
  ostringstream addr;
  struct sockaddr_un address;

  addr << "/tmp/phGPRS." << aPort;
  unlink(addr.str().c_str());
  
  memset(&address,0,sizeof(struct sockaddr_un));
  
  address.sun_family = PF_UNIX;
  strcpy(address.sun_path, addr.str().c_str());
  
  if (bind(iSocket, (struct sockaddr *)&address, sizeof(address)) == -1) {
    ERR("CGPRSConnection::Listen : bind failed");
    return false;
  }

  if (listen(iSocket, 16) == -1) {
    ERR("CGPRSConnection::Listen : listen failed");
    return false;
  }
   
  iIsListening = true;

  return true;
}

/**
 * @memo Accepts the incoming connection and returns a new connection object.
 * @doc Accepts the incoming connection and returns a new connection object. If
 * accepting fails then NULL is returned.
 *
 * @return new connection object or NULL in the case of an error
 */
MAbstractConnection* CGPRSConnection::AcceptL()
{
  int newSocket;
  int addressLength;
  struct sockaddr_un address;
  CGPRSConnection* retval;  
  char tempBuf[INET_ADDRSTRLEN];
  
  addressLength = sizeof(address);

  if ((newSocket = accept(iSocket, (struct sockaddr *)&address, (socklen_t *)&addressLength)) == -1) {
    ERR("CGPRSConnection::AcceptL : accept failed : ");
    return NULL;
  }

  retval = new CGPRSConnection();
  if (retval->iSocket < 0)
    {
      ERR("CGPRSConnection::AcceptL : socket failed");
      return NULL;
    }
  
  close(retval->iSocket);
  retval->iSocket = newSocket;
  retval->iConnected = true;
  retval->iRemoteAddress = std::string("GPRS:") + std::string(inet_ntop(AF_INET, &address, tempBuf, sizeof(tempBuf)));

  iConnected = true;
  return retval;
}


/**
 * @memo Tells if the connection is listening.
 * @doc Tells if the connection is in the listening state or not.
 *
 * @return true if the connection is in the listening state
 */
bool CGPRSConnection::IsListening()
{
  return iIsListening;
}


/**
 * @memo Tells if the connection is established.
 * @doc Tells if the connection is established or not.
 *
 * @return true if the connection is established
 */
bool CGPRSConnection::IsConnected()
{
  return iConnected;
}


/**
 * @memo Tells if the connection has data waiting
 * @doc Waits for the connection to have data to read, and returns true
 *
 * @return true if the data is ready
 */
bool CGPRSConnection::HasData()
{
  fd_set set;
  FD_ZERO(&set);
  FD_SET(GetFd(), &set);

  switch (select(GetFd() + 1, &set, NULL, NULL, NULL)) {
  case -1:
    
  case 0:
    break;

  default:
    if (FD_ISSET(GetFd(), &set)) {
      return true;
    }
    else {
      return false;
    }
  }

}

unsigned int CGPRSConnection::GetDeviceChecksum()
{
	return iChecksum;
}

void CGPRSConnection::SetDeviceChecksum(unsigned int aChecksum)
{
	iChecksum = aChecksum;	
}

