/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

// Emacs, hail the mighty -*-C++-*-

/**
 * Copyright 2003 LUT. .
 *
 * @name BTPlugin.h
 * @memo Bluetooth specific implementation of the PeerHood plugin interface.
 *
 * @version 0.1
 * date     08.04.2003
 * change   08.04.2003
 */

#ifndef __BTPLUGIN_H__
#define __BTPLUGIN_H__

#include <bluetooth/sdp.h>
#include <bluetooth/sdp_lib.h>
#include <string>
#include <list>
#include <pthread.h>
#include "AbstractPlugin.h"
#include "DaemonDevice.h"
#include "ServiceCheck.h"
#include "BTConnection.h"
#include "AbstractListener.h"

/**
 * @memo Bluetooth specific implementation of the PeerHood plugin interface.
 * @doc This class is Bluetooth specific implementation of the PeerHood plugin
 * interface. Other PeerHood capable devices are located using Bluetooth's 
 * device discovery method and Service Discovery Protocol (SDP).
 *
 * @see MAbstractPlugin
 */
class CBTPlugin : public MAbstractPlugin
{
 public:
  CBTPlugin();
  virtual ~CBTPlugin();
  bool Advert();
  bool Unadvert();
  bool Start();
  void Stop();
  const std::string& GetName();
  void UpdateState();
  void LoadListeners();
  
  // Abstract state changer
  void RegisterListener(MAbstractListener* aListener);
  void SetState(bool aActive);
  void TriggerShutdown();
  void SetAdapter(const std::string& aInterface, int aId);
  const std::string& GetAdapter();

 private:
  bool TestConnection(unsigned int aFd, unsigned int aBytes);
  bool LoadParameter(std::string& aParameter);
  static void* ThreadStarter(void *aArguments);
  static void* AdvertStarter(void *aArguments);
  bool AddDevice(const std::string& aDeviceName);

  bool FetchInformation(CDaemonDevice* aDevice, int aPort);
  bool FetchDeviceInfo(CDaemonDevice* aDevice, CBTConnection* aConnection);
  bool FetchServices(CDaemonDevice* aDevice);
  bool FetchServices(CDaemonDevice* aDevice, CBTConnection* aConnection);
  bool FetchPrototypes(CDaemonDevice* aDevice, CBTConnection* aConnection);
  bool FetchNeighbourhoodDevices(CDaemonDevice * aDevice, CBTConnection* aConnection);

  bool HasPeerHood(const std::string& aAddress, int* aPSM);
  void AdvertThread();
  void InquiryThread();
  bool MakeOlder();

  void RemoveListeners();
  
  int CreateLowlevelSocket();
  int ResetAdapter();
  int DoInquiry(int aLen, int aNrsp, const uint8_t *aLap, inquiry_info **aIi, long aFlags);
  
  volatile bool iStarted;
  volatile bool iActive;
  volatile bool iNeedsRestart;
  int iResetRounds;
  int iBTCtrlSock;
  int iDeviceId;
  
  volatile bool iInSdp;
 	volatile bool iListening; // For listening check in advert
  
  sdp_session_t* iSDP;
  uint32_t iSDPHandle;

  std::string iName;
  std::string iNDevice;
  pthread_t iInquiryThread;
  pthread_t iAdvertThread;

  std::list<CDaemonDevice *> iDeviceList;
  std::list<CServiceCheck> iServiceCheckList;

  // List of listeners
  std::list<MAbstractListener*> iListenerList;
  
  int iServiceCheckInterval, iInquiryInterval, iDaemonPort;
};

#endif // __BTPLUGIN_H__
