/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name BTPinger.cc
 * @memo Bluetooth implementation of the MAbstractPinger interface.
 *
 * @version 0.1
 * date     11.06.2003
 * change   11.06.2003
 */

#include <unistd.h>
#include <sys/socket.h>
#include <sys/poll.h>
#include <bluetooth/bluetooth.h>
#include <bluetooth/l2cap.h>
#include <asm/byteorder.h>
#include "BTPinger.h"

/**
 * @memo Constructor.
 * @doc Constructor, initializes the pinger object so that it can be used
 * immediately after the construction has finished.
 *
 * @param aAddress Remote device's address.
 *
 * @return none
 */
CBTPinger::CBTPinger(const std::string& aAddress) : MAbstractPinger(aAddress)
{
  iAddress = std::string(aAddress);
  iInRange = true;
}


/**
 * @memo Checks if a remote device is in range.
 * @doc Checks if a remote device is in range. The check is done by sending
 * the L2CAP_ECHO request via the Bluetooth signalling channel. If a response
 * isn't received during one second then the remote device is assumed to be out
 * of range.
 *
 * @return true if the remote device is in range, otherwise false
 */
bool CBTPinger::Ping()
{
  int fd;
  bdaddr_t bdaddr;
  struct sockaddr_l2 address;
  char buffer[2048];
  struct pollfd pf[1];

  iInRange = false;

  if ((fd = socket(PF_BLUETOOTH, SOCK_RAW, BTPROTO_L2CAP)) < 0 ) {
    return false;
  }
  
  baswap(&bdaddr, strtoba(const_cast<char*>(iAddress.c_str())));
  address.l2_family = AF_BLUETOOTH;
  address.l2_bdaddr = bdaddr;
  
  if (connect(fd, (struct sockaddr *)&address, sizeof(address)) < 0) {
    close(fd);
    return false;
  }

  for (unsigned int i = L2CAP_CMD_HDR_SIZE;i < sizeof(buffer);i++) buffer[i] = (i % 40) + 'A';
  l2cap_cmd_hdr *cmd = (l2cap_cmd_hdr *)buffer;
  cmd->code = L2CAP_ECHO_REQ;
  cmd->ident = 1;
  cmd->len = __cpu_to_le16(20);

  // ping
  if(send(fd, buffer, 20 + L2CAP_CMD_HDR_SIZE, 0) <= 0 ) {
    close(fd);
    return false;
  }
  
  pf[0].fd = fd; 
  pf[0].events = POLLIN;
  
  if (poll(pf, 1, 1000) <= 0) {
    close(fd);
    return false;
  }
  
  if (recv(fd, buffer, sizeof(buffer), 0) <= 0) {
    close(fd);
    return false;
  }

  close(fd);
  iInRange = true;

  return true;
}


/**
 * @memo Returns remote device's Bluetooth address.
 * @doc Return the Bluetooth address of the device under monitoring.
 *
 * @return remote device's address
 */
const std::string& CBTPinger::GetAddress()
{
  return iAddress;
}


/**
 * @memo Tells whether a device is in range or not.
 * @doc Tells whether a device is in range or not.
 *
 * @return true if the device is in range
 */
bool CBTPinger::InRange()
{
  return iInRange;
}

