/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

// Emacs, hail the mighty -*-C++-*-

/**
 * Copyright 2003 LUT. .
 *
 * @name PeerHood.h
 * @memo Definition of the PeerHood interface.
 *
 * @version 0.1
 * date     19.05.2003
 * change   19.05.2003
 */

#ifndef __PEERHOOD_H__
#define __PEERHOOD_H__

#include "BasicCallback.h"
#include "C_Callback.h"

#ifdef __cplusplus

#include <string>

typedef CPHList<MAbstractDevice*> TDeviceList;
typedef CPHIterator<MAbstractDevice*> TDeviceIterator;

/**
 * @memo Definition of the PeerHood interface.
 * @doc Definition of the PeerHood interface. This interface defines the public
 * API of the whole PeerHood library.
 *
 */
class MPeerHood
{
 public:
  /**
   * @memo Default destructor.
   * @doc Default destructor. Currently this does nothing but is here because
   * base classes without a destructor are <b><i>evil</i></b>.
   *
   * @return none
   */
  virtual ~MPeerHood() {};


  /**
   * @memo Method used to create a new instance of the PeerHood interface.
   * @doc Method used to create a new instance of the PeerHood interface. The
   * only way to create the instance is via this method. When this method is
   * called for the very first time a new instance of the interface is created.
   * Subsequent calls will return a pointer to the existing instance.
   *
   * @param aCallback Pointer to the callback that will receive notifications
   * from the PeerHood library.
   *
   * @return pointer to a PeerHood instance
   */
  static MPeerHood* GetInstance(CBasicCallback* aCallback = NULL);


  /**
   * @memo Initializes the PeerHood instance.
   * @doc Initializes the PeerHood instance. These routines include connecting
   * to the PeerHood daemon and setting up the debug output. In addition, all
   * internal variables are initialized. This method should be called only
   * once.
   *
   * @param aArgc The number of parameters.
   * @param aArgv Array containing the parameters.
   *
   * @return true if the object was initialized succesfully
   */
  virtual bool Init(int aArgc, char** aArgv) = 0;


  /**
   * @memo Gets a list of all nearby devices and their services.
   * @doc Gets a list containing all nearby devices and their services and 
   * resources. Note that this function reserves the memory required by the
   * list and it's caller's responsibility to free it. Also note that the 
   * returned list contains <i>all</i> devices in range - inluding those
   * without PeerHood capability. If no devices are found then the returned
   * list is empty.
   *
   * @return a list of found devices or NULL if an error occurred
   */
  virtual TDeviceList* GetDeviceListL() = 0;


  /**
   * @memo Returns a list of devices that offer the asked service.
   * @doc This function builds and returns a list that contains all devices
   * that offer the requested service. Note that the caller must free the 
   * memory allocated for the returned list. If no devices are found then the
   * returned list will be empty.
   *
   * @param aServiceName The service that should be looked for.
   *
   * @return a list of devices that offer the requested service or NULL if an
   * error occurred
   */
  virtual TDeviceList* GetDeviceListL(const std::string* aServiceName) = 0;


  /**
   * @memo Returns all locally registered services.
   * @doc Returns all locally registered services on a list. The memory
   * allocated for the returned list is not freed automatically so the caller
   * must take care of it. If no services are registered then the returned list
   * will be empty.
   *
   * @return a list of locally registered services or NULL in the case of an
   * error
   */
  virtual TServiceList* GetLocalServiceListL() = 0;

  /**
   * @memo Creates a connection to a local service.
   * @doc Creates a connetion to a local service. 
   * Destination address and technology prototype are taken from the parameters. If a
   * connection object is returned then it's caller's responsibility to delete
   * it in a controlled way.
   *
   * @param aService The service to connect to.
   *
   * @return a new connection object or NULL if an error happened
   */
  virtual MAbstractConnection* Connect(TServiceIterator& aService) = 0;
  
  /**
   * @memo Creates a connection to a local service.
   * @doc Creates a connetion to a local service. 
   * Destination address and technology prototype are taken from the parameters. If a
   * connection object is returned then it's caller's responsibility to delete
   * it in a controlled way. For supporting C-API of PeerHood.
   *
   * @param aService The service to connect to. CService class objects can be used
   * directly.
   *
   * @return a new connection object or NULL if an error happened
   */
  virtual MAbstractConnection* Connect(CService& aService) = 0;

  /**
   * @memo Creates a connection to a service on another PeerHood capable device.
   * @doc Creates a connetion to a service on another PeerHood capable device. 
   * Destination address and technology prototype are taken from the parameters. If a
   * connection object is returned then it's caller's responsibility to delete
   * it in a controlled way.
   *
   * @param aDevice The remote device.
   * @param aServiceName Remote service's name.
   *
   * @return a new connection object or NULL if an error happened
   */
  virtual MAbstractConnection* Connect(TDeviceIterator& aDevice, const std::string aServiceName) = 0;

	/**
   * @memo Creates a connection to a service on another PeerHood capable device.
   * @doc Creates a connetion to a service on another PeerHood capable device. 
   * Destination address and technology prototype are taken from the parameters. If a
   * connection object is returned then it's caller's responsibility to delete
   * it in a controlled way.
   *
   * @param aDevice The remote device.
   * @param aServiceName Remote service's name.
   *
   * @return a new connection object or NULL if an error happened
   */
	virtual MAbstractConnection* Connect(MAbstractDevice& aDevice, const std::string& aServiceName) = 0;
	
  /**
   * @memo Registers a service so that other PeerHood devices can find it. 
   * @doc Registers a service so that other PeerHood devices can find and use 
   * it. This method contacts the PeerHood daemon that in turns starts to 
   * advert the service through its currently running plugins.
   *
   * @param aName The name of the service.
   * @param aAttributes Service's attributes in one string.
   * @param aPort Service's port.
   *
   * @return port number if the service could be registered, otherwise 0
   */
  virtual unsigned short RegisterService(const std::string& aName, const std::string& aAttributes, const std::string& aPort) = 0;

  /**
   * @memo Registers a service so that other PeerHood devices can find it. 
   * @doc Registers a service so that other PeerHood devices can find and use 
   * it. This method contacts the PeerHood daemon that in turns starts to 
   * advert the service through its currently running plugins.
   *
   * @param aName The name of the service.
   * @param aAttributes Service's attributes in one string.
   *
   * @return port number if the service could be registered, otherwise 0
   */
  virtual unsigned short RegisterService(const std::string& aName, const std::string& aAttributes) = 0;


  /**
   * @memo  Unregisters a previously registered service.
   * @doc Unregisters a previously registered service. After unregistration
   * other devices are unable to find and call the unregistered service. Note
   * that the unregistration procedure doesn't delete the service object so
   * this should be done by the actual server application.
   *
   * @param aName The name of the service to be unregistered.
   *
   * @return true if the service could be unregistered
   */
  virtual bool UnregisterService(const std::string& aName) = 0;

  /**
   * @memo  Unregisters a previously registered service.
   * @doc Unregisters a previously registered service. After unregistration
   * other devices are unable to find and call the unregistered service. Note
   * that the unregistration procedure doesn't delete the service object so
   * this should be done by the actual server application.
   *
   * @param aName The name of the service to be unregistered.
   *
   * @return true if the service could be unregistered
   */
  virtual bool UnregisterService(const std::string& aName, const std::string& aPort) = 0;

  /**
   * @memo Sets a device under constant monitoring.
   * @doc Sets a device under constant monitoring. If a change (out of range, 
   * back in range) takes place then the registered callback interface is 
   * notified. An application must derive from the <code>CBasicCallback</code>
   * class and implement the defined methods in order to receive callback
   * events.
   *
   * @param aDevice The device that should be monitored.
   *
   * @return true if the monitoring could be started
   */
  virtual bool MonitorDevice(TDeviceIterator& aDevice) = 0;


  /**
   * @memo Stops the monitoring of a device.
   * @doc Stops the monitoring of a device. After this function is called the
   * given device is no longer monitored.
   *
   * @param aDevice The target device.
   *
   * @return true if the monitoring could be canceled succesfully
   */
  virtual bool UnmonitorDevice(TDeviceIterator& aDevice) = 0;

 /**
   * @memo Sets a device under constant monitoring using signal-level monitoring.
   * @doc Sets a device under constant monitoring using signal-level monitoring.
   * If a change (out of range, 
   * back in range) takes place then the registered callback interface is 
   * notified (Not currently used). An application must derive from the <code>CBasicCallback</code>
   * class and implement the defined methods in order to receive callback
   * events.
   *
   * @param aDevice The device that should be monitored.
   *
   * @return true if the monitoring could be started
   */
  virtual bool SignalMonitorDevice(TDeviceIterator& aDevice) = 0;


  /**
   * @memo Stops the signal-level monitoring of a device.
   * @doc Stops the signal-level monitoring of a device. After this function is called the
   * given device is no longer monitored.
   *
   * @param aDevice The target device.
   *
   * @return true if the monitoring could be canceled succesfully
   */
  virtual bool SignalUnmonitorDevice() = 0;


  /**
   * @memo Sets the plugin prefered by the current application (Not used currently).
   * @doc Sets the plugin prefered by the current application. This means that 
   * PeerHood will try to use the given plugin whenever possible. This method 
   * will override the value read from the configuration file. The prefered
   * plugin can be changed during runtime. However, it affects only the actions
   * performed after the call i.e. the running services are not affected.
   *
   * @param aPluginName The name of the prefered plugin.
   *
   * @return none
   */
  virtual void SetPreferedPlugin(const char* aPluginName) = 0;
};


// End of C++ (__cplusplus)
#else // C-api
#define TRUE 1
#define FALSE 0
typedef
	struct MPeerHood
		MPeerHood;
typedef
	struct TDeviceList
		TDeviceList;
typedef
	struct TDeviceIterator
		TDeviceIterator;
#endif // __cplusplus

#ifdef __cplusplus
extern "C" {
#endif

void ph_c_delete_peerhood(MPeerHood* _aPeerHood);
MPeerHood* ph_c_get_instance(C_Callback* _aCallback);
int ph_c_init(MPeerHood* _aPeerHood,int argc,char** argv);

TDeviceList* ph_c_get_devicelist(MPeerHood* _aPeerHood);
TDeviceList* ph_c_get_devicelist_with_services(MPeerHood* _aPeerHood, const char* _aServiceName);
void ph_c_delete_devicelist(TDeviceList* _aDeviceList);

TServiceList* ph_c_get_localservicelist(MPeerHood* _aPeerHood);
void ph_c_delete_servicelist(TServiceList* _aServiceList);

MAbstractConnection* ph_c_connect_localservice(MPeerHood* _aPeerHood, CService* _aService);
MAbstractConnection* ph_c_connect_remoteservice(MPeerHood* _aPeerHood, MAbstractDevice* _aDevice, const char* _aServiceName);

unsigned short ph_c_register_service(MPeerHood* _aPeerHood,const char* _aName, const char* _aAttributes);
unsigned short ph_c_register_service_with_port(MPeerHood* _aPeerHood, const char* _aName, const char* _aAttributes, const char* _aPort);

int ph_c_unregister_service(MPeerHood* _aPeerHood, const  char* _aName);
int ph_c_unregister_service_with_port(MPeerHood* _aPeerHood, const char* _aName, const char* _aPort);

int ph_c_monitor_device(MPeerHood* _aPeerHood, TDeviceIterator* _aDevice);
int ph_c_unmonitor_device(MPeerHood* _aPeerHood, TDeviceIterator* _aDevice);

int ph_c_signal_monitor_device(MPeerHood* _aPeerHood, TDeviceIterator* _aDevice);
int ph_c_signal_unmonitor_device(MPeerHood* _aPeerHood);

void ph_c_set_prefered_plugin(MPeerHood* _aPeerHood, const char* _aPluginName);

int ph_c_devicelist_is_empty(TDeviceList* _aDeviceList);
int ph_c_devicelist_size(TDeviceList* _aDeviceList);
int ph_c_servicelist_is_empty(TServiceList* _aServiceList);
int ph_c_servicelist_size(TServiceList* _aServiceList);

#ifdef __cplusplus
}
#endif

#endif // __PEERHOOD_H__
