/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2008 LUT. .
 * 
 * @name AbstractStateConverter.h
 * @memo Interface for daemon and plugins to implement. 
 * Used by different state listeners to update the state of the component.
 * 
 * @version 0.01
 * date 14.10.2008
 * change 14.10.2008
 */
#ifndef __ABSTRACTSTATECONVERTER_H__
#define __ABSTRACTSTATECONVERTER_H__

#include "AbstractListener.h"

/**
 * @memo Interface for daemon and plugins to implement for state changes
 * @doc Components that will support dynamic state changes will have to
 * implement this interface. This interface is used by Listeners for changing the
 * state of the owner object.
 */
class MAbstractStateConverter
{
public:
	
	/**
	 * @memo Register new listener for this component
	 * @doc Used for registering a new listener for the component that 
	 * implements this interface. 
	 * 
	 * @param MAbstractListener a reference to the listener to be registered
	 */
	virtual void RegisterListener(MAbstractListener* aListener) = 0;

	
	/**
		 * @memo Set state for implementing object
		 * @doc Set the implementing object to active or passive state.
		 * 
		 * @param bool New activity state: TRUE = active
		 */
	virtual void SetState(bool aActive) = 0;
	
	/**
		 * @memo Set the implementing object to shut down itself
		 * @doc Sets the object that implements this interface to shutdown state.
		 * 
		 */
	virtual void TriggerShutdown() = 0;
	
	/**
	 * @memo Sets the adapter for plugin to use.
	 * @doc Set the given adapter to be the adapter the plugin uses.
	 *
	 * @param aInterface The name of the interface, e.g. hci0, wlan0 etc.
	 * @param aId The id of the interface (the number at the end)
	 */
	virtual void SetAdapter(const std::string& aInterface, int aId) = 0;
	
	virtual const std::string& GetAdapter() = 0;
};

#endif /*__ABSTRACTSTATE_H__*/
