/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

// Emacs, hail the mighty -*-C++-*-

/**
 * Copyright 2003 LUT. .
 *
 * @name AbstractMonitor.h
 * @memo Definition of the PeerHood Monitor interface
 *
 * @version 0.1
 * date     31.07.2003
 * change   31.07.2003
 */

#ifndef __ABSTRACTMONITOR_H__
#define __ABSTRACTMONITOR_H__

#include <string>

/**
 * @memo Defines the PeerHood Monitor interface.
 * @doc Defines the PeerHood Monitor interface. The methods in this interface
 * are used when monitoring the presence of a remote device. Each networking
 * technology requires its own implementation of this class, so the actual
 * monitoring method is technology specific issue. Instances of this class are
 * created transparently via the corresponding Creator classes.
 *
 * @see MAbstractCreator
 */
class MAbstractMonitor
{
 public:
  /**
   * @memo Constructor.
   * @doc Constructor, initializes the class so that the <code>Monitor</code>
   * method can be called immediately after the construction has finished. The
   * target address is the same during the created object's lifetime.
   *
   * @param aAddress The address of the remote device.
   *
   * @return none
   */
  MAbstractMonitor(const std::string& aAddress) {};


  /**
   * @memo Default destructor.
   * @doc Default destructor. Currently this does nothing but is here because
   * virtual base classes without a destructor are <b><i>evil</i></b>.
   *
   * @return none
   */
  virtual ~MAbstractMonitor() {};


  /**
   * @memo Returns remote device's address.
   * @doc Return the address of the device under monitoring.
   *
   * @return remote device's address
   */
  virtual const std::string& GetAddress() = 0;


  /**
   * @memo Tells whether a device is in range or not.
   * @doc Tells whether a device is in range or not.
   *
   * @return true if the device is in range
   */
  virtual bool InRange() = 0;


  /**
   * @memo Monitors a remote device, i.e. checks the signal strength
   * @doc Monitors a remote device, i.e. checks the signal strengt
   *
   * @return true if the remote device is present
   */
  virtual int Monitor() = 0;
};

#endif // __ABSTRACTMONITOR_H__
