/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name LocalCreator.cc
 * @memo UNIX socket implementation of the MAbstractCreator interface.
 *
 * @version 0.1
 * date     23.04.2003
 * change   23.04.2003
 */

#include <cassert>
#include "LocalCreator.h"
#include "LocalConnection.h"
#include <Factory.h>

/**
 * @memo Default constructor.
 * @doc Default constructor, sets internal variables to their default values
 * and insert creator to the PeerHood factory.
 *
 * @return none
 */
CLocalCreator::CLocalCreator()
{
  iProto = std::string("local");
  Factory::Register(this);
}


/**
 * @memo Creates a new local socket specific connection object.
 * @doc Creates a new local socket specific connection object. The object is
 * created dynamically so the caller is responsible of deletion of the object.
 *
 * @param aProto Prototype of the desired object's type.
 *
 * @return new CLocalConnection object or NULL if the prototype was not recognized
 */
MAbstractConnection* CLocalCreator::CreateConnectionL(const std::string& aProto)
{
  if (iProto.compare(aProto) == 0) {
    return new CLocalConnection;
  }

  return NULL;
}


/**
 * @memo Does nothing but complies to the MAbstractCreator interface.
 * @doc Does nothing but is there in order to match the MAbstractCreator
 * interface. This method should <b>never</b> be called!
 *
 * @param aProto Prototype of the desired object's type.
 * @param aAddress Remote device's address.
 *
 * @return throws an assert if ever called
 */
MAbstractPinger* CLocalCreator::CreatePingerL(const std::string& aProto, const std::string& aAddress)
{
  assert("CLocalCreator::CreatePingerL - Should NEVER come here!");

  return NULL;
}

/**
 * @memo Does nothing but complies to the MAbstractCreator interface.
 * @doc Does nothing but is there in order to match the MAbstractCreator
 * interface. This method should <b>never</b> be called!
 *
 * @param aProto Prototype of the desired object's type.
 * @param aAddress Remote device's address.
 *
 * @return throws an assert if ever called
 */
MAbstractMonitor* CLocalCreator::CreateMonitorL(const std::string& aProto, const std::string& aAddress)
{
  assert("CLocalCreator::CreateMonitorL - Should NEVER come here!");

  return NULL;
}


/** 
 * @memo Returns creator's unique id string.
 * @doc Returns creator's unique id string. This creator's id is <i>local
 * </i>.
 *
 * @return creator's unique id string
 */
const std::string& CLocalCreator::GetPrototype()
{
  return iProto;
}


static CLocalCreator localcreator;
