/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2008 LUT. .
 *
 * @name MaemoBTListener.cc
 * @memo Maemo BTcond-specific implementation of MAbstractListener
 *
 * @version 0.19
 * date     10.10.2008
 * change   17.12.2008
 */

#include <bt-dbus.h>
#include "MaemoBTListener.h"

/**
 * @memo Default constructor - not in use
 * @doc Not used
 * 
 */
CMaemoBTListener::CMaemoBTListener()
{
	iConnection = NULL;
	iConverter = NULL;
	iName = std::string("MaemoBTListener");
}

/**
 * @memo Constructor
 * @doc Use this constructor to create instances of this class. A reference to the 
 * component that will own the instance will have to be given as a parameter. Created
 * component will be registered into that component. 
 * 
 * @param Reference to component that owns the instance. CANNOT BE NULL!
 */
CMaemoBTListener::CMaemoBTListener(MAbstractStateConverter* aConverter)
{
	iConverter = aConverter;
	iConnection = NULL;
	iName = std::string("MaemoBTListener");
	iConverter->RegisterListener(this);
}

/**
 * @memo Destructor, closes connection to D-Bus.
 * @doc Closes the connection if it wasn't closed.
 */
CMaemoBTListener::~CMaemoBTListener()
{
	if(iConnection != NULL)
		{
			// Returns void - cannot be checked
			dbus_connection_close(iConnection);
			iConnection = NULL;
		}
}

/**
 * @memo Connects listener to D-Bus and registers signals to listen
 * @doc Connects the listener to D-Bus using private connection. Register
 * signals to listen via D-Bus, registers itself to listen signals emitted by
 * btcond (signal interface). Use this only once.
 * 
 * @return bool, true if connection attempt was success
 */
bool CMaemoBTListener::Connect()
{
	if(!SetupConnection())	return false;
		
		if(!RegisterSignals()) return false;
		
		return true;
}

/** 
 * @memo Disconnects the listener from D-Bus
 * @doc Closes the connection to D-Bus if it hasn't been closed
 * earlier.
 */
void CMaemoBTListener::Disconnect()
{
	if(iConnection != NULL)
		{
			dbus_connection_close(iConnection);
			iConnection = NULL;
		}
}

/**
 * @memo No implementation of this function
 * @doc Not implemented - does nothing
 */
void CMaemoBTListener::CheckInitialState()
{
	/** Not implemented, btcond doesn't provide a method for checking the initial state.
	 *  The state can be checked via dbus from bluez - done by another listener!
	 */
	return;
}

/**
 * @memo Checks the newest message from the D-Bus queue
 * @doc Pops the first message from the queue and processes it,
 * message is passed to handler function (HandleMessage) that 
 * checks the message type. If the queue is empty nothing will be
 * done.
 */
void CMaemoBTListener::CheckState()
{
	DBusMessage* message = NULL;
		
	// Check connection
	if(iConnection == NULL) return;

	// Allows messages to be read and marshalled from the wire on non-blocking connection
	// Second parameter is blocking time in milliseconds
	dbus_connection_read_write(iConnection,0);

	// Pop the first message
	if((message = dbus_connection_pop_message(iConnection)) != NULL)
	{
		HandleMessage(message);
		dbus_message_unref(message);
		message = NULL;
	}
	else return;
}

/**
 * @memo Returns the name of the listener
 * @doc Returns the name set in the constructor.
 * 
 * @return Name of the listener
 */
const std::string& CMaemoBTListener::GetName()
{
	return iName;
}

/**
 * @memo Sets up the connection to D-Bus
 * @doc Sets up the private connection to D-Bus. Should be called
 * only once, used in Connect()-function. False is returned also when 
 * connection has already been created.
 * 
 * @return bool, true if success. 
 */
bool CMaemoBTListener::SetupConnection()
{
	DBusError error;

	// If connection already created.
	if(iConnection != NULL) return false;
	
	// Initialize error
	dbus_error_init(&error);

	// Get system bus, private connection
	iConnection = dbus_bus_get_private(DBUS_BUS_SYSTEM, &error);

	// Error
	if (iConnection == NULL)
	{
		if (dbus_error_is_set(&error))
		{
			ERR("CMaemoBTListener::SetupConnection: Connection error: %s", error.message);

			// Free error
			dbus_error_free(&error);
		}
		return false;
	}
	else return true;
}

/**
 * @memo Registers signals to listen 
 * @doc Registers this listener to listen the signal interface of
 * btcond in Maemo. Returns false if there is no connection. Listener
 * receives messages only from this interface.
 * 
 * @return bool, true if success
 */
bool CMaemoBTListener::RegisterSignals()
{
	DBusError error;
	std::string signal_and_if;

	// Check connection
	if(iConnection == NULL) return false;

	// Create signal that is listened
	// form: type='signal',interface='com.nokia.mce.signal'
	signal_and_if = "type=\'";
	signal_and_if += dbus_message_type_to_string(DBUS_MESSAGE_TYPE_SIGNAL);
	signal_and_if += "\',interface=\'";
	signal_and_if += BTCOND_SIG_INTERFACE;
	signal_and_if += "\'";

	// Initialize error variable
	dbus_error_init(&error);

	// Connect to com.nokia.btcond.signal interface listening for 'signal'
	dbus_bus_add_match(iConnection, signal_and_if.c_str(), &error);
	// Make sure it is sent
	dbus_connection_flush(iConnection);

	// If cannot listen 
	if (dbus_error_is_set(&error))
	{
		ERR("CMaemoBTListener::RegisterSignals: Cannot add listening to signal: %s, reason: %s", signal_and_if.c_str(), error.message);
		dbus_error_free(&error);
		return false;
	}

	signal_and_if.clear();
	
	return true;
}

/**
 * @memo Message handler, checks the type of the received message.
 * @doc Checks the received message type. If the signal is correct (incidating
 * the device up/down state) the state of the owner component will be changed
 * with SetState(). Currently uses hci0 device as hardcoded value.
 * 
 * @param message Message to be handled
 */
void CMaemoBTListener::HandleMessage(DBusMessage* message)
{
	DBusMessageIter msg_iter;
	int msgtype;
	bool state = true;
	// NOTE: no need to free, is a part of the DBusMessage
	const char* mdata = NULL;

	// Message iterator
	dbus_message_iter_init(message, &msg_iter);
	
	// Device brought up
	if(dbus_message_is_signal(message,BTCOND_SIG_INTERFACE,BTCOND_HCI_DEV_UP_SIG))
	{
		state = true;
	}
	
	// Device set off
	else if(dbus_message_is_signal(message,BTCOND_SIG_INTERFACE,BTCOND_HCI_DEV_DOWN_SIG))
	{
		state = false;
	}
	
	// Don't care about other messages
	else return;
	
	// Check the device from the message
	msgtype = dbus_message_iter_get_arg_type(&msg_iter);
	if(msgtype == DBUS_TYPE_STRING)
	{
		dbus_message_iter_get_basic(&msg_iter,&mdata);
		
		/** @TODO Remove hardcoding of the device - get it from e.g. owner component? */
		// If this is the right device
		if(strcmp(mdata,"hci0") == 0)
		{
			DBG("CMaemoBTListener::Handlemessage: Device hci0 state change.");
			iConverter->SetState(state);
		}
	}
}

