/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2008 LUT. .
 *
 * @name AbstractListener.h
 * @memo Definition of the PeerHood Listener interface.
 *
 * @version 0.02
 * date     25.08.2008
 * change   23.09.2008
 */

#ifndef __ABSTRACTLISTENER_H__
#define __ABSTRACTLISTENER_H__

#include <string>
/**
 * @memo Defines the PeerHood listener interface
 * @doc Defines a pure and virtual interface for different listeners, every
 * listener must implement this interface. Listener is used to monitor or
 * check state or environment changes in the device where PeerHood is
 * running. This interface hides underlying implementation from PeerHood,
 * new listeners can be added dynamically. Listeners are used by the object
 * that calls listener to be created, all listeners must register to owner by
 * using the RegisterListener()-function of the MAbstractStateConverter 
 * -interface. Functions should be called in order:
 * 1. Connect() after creation
 * 2. CheckInitialState() at start
 * 3. CheckState() periodically when running
 * 4. Disconnect() when shutting down removing listeners.
 */

class MAbstractListener
{
public:


	/**
	 * @memo Virtual destructor
	 * @doc
	 */
	virtual ~MAbstractListener() {};

	/**
	 * @memo Connect listener into event source
	 * @doc After successful creation of object this function is called to
	 * let the listener connect to the event/state source.
	 */
	virtual bool Connect() = 0;

	/**
	 * @memo Disconnect listener from event source
	 * @doc Called when removing listeners - all listeners must disconnect
	 * from the source (if needed) with this funcion!
	 */	 
	virtual void Disconnect() = 0;

	/**
	 * @memo Check the startup state of the listened event
	 * @doc Used to check the initial state of the system before plugins and
	 * other necessary modules are started.
	 */
	virtual void CheckInitialState() = 0;

	/**
	 * @memo Check the state of the listened event
	 * @doc Used to check the system state for changes, calls PeerHood daemon 
	 * functions (hardcoded at the moment - no interface yet) to change daemon
	 * state. 
	 */
	virtual void CheckState() = 0;
	
	/**
	 * @memo Returns the name of the listener
	 * @doc 
	 */
	virtual const std::string& GetName() = 0;
	
};

#endif /*__ABSTRACTLISTENER_H__*/
