/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name WLANPlugin.cc
 * @memo WLAN implementation of PeerHood's plugin interface.
 *
 * @version 0.1
 * date     30.06.2003
 * change   30.06.2003
 */

#include <cstdlib>
#include <ctime>
#include <unistd.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <errno.h>

#include <signal.h>
#include <setjmp.h>
#include <fstream>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "DeviceStorage.h"
#include "WLANPlugin.h"
#include "TCPConnection.h"
#include "UDPConnection.h"
#include "Daemon.h"
#include "ListenerFactory.h"
#include "AbstractStateConverter.h"
#include "PluginStorage.h"

const int MINIMUM_INQUIRY_DELAY = 1;
const float INQUIRY_VARIATION = 2.0;

//counter for keeping devices on device-list
const int KEEP_ALIVE_STAMP = 10;

const char* PLUGIN_NAME = "wlan-base";

//port for broadcasting HELLO messages (listening for unicast is WLAN_INQUIRY_PSM + 1)
const int WLAN_INQUIRY_PSM = 8000;

//port for listening incoming service requests
const int WLAN_PLUGIN_PSM = 8500;

/** @TODO Determine good values for error tolerance, this should be configurable somewhere! */
//Tolerance factor for WLAN errors
const int WLAN_ERROR_TOLERANCE = 10;

/**
 * @memo Default constructor, sets the internal variables to their default
 * values.
 * @doc Default constructor, sets the internal variables to their default
 * values. This method loads WLAN interface, monitoring and service checking interval variables fro
 * configuration file.
 *
 * @return none
 */
CWLANPlugin::CWLANPlugin()
{
	iStarted = false;
	iActive = true;
	iInSdp = false;
	iListen = false;
	iAdvertFailed = false;
	iName = std::string(PLUGIN_NAME);
#ifdef __PH_OLD_PLUGIN_STYLE__
	pluginMap[PLUGIN_NAME] = this;
#else
	PluginStorage::GetInstance()->AddPlugin(PLUGIN_NAME, this);
#endif
	srand(time(NULL));

	iNDevice = "WLANInterface";
	if(!LoadParameter(iNDevice))
	{
		ERR("CWLANPlugin::CWLANPlugin : Interface entry not found from .phconfig!");
		return;
	}

	std::string stringDelay = "WLANMonitoringInterval";
	if(!LoadParameter(stringDelay))
	{
		ERR("CWLANPlugin::InquiryThread : Can't load parameters, using defaults");
		iInquiryInterval = 10;
	}
	iInquiryInterval = atoi(stringDelay.c_str());

	stringDelay = "WLANServiceCheckInterval";
	if(!LoadParameter(stringDelay))
	{
		ERR("CWLANPlugin::ListenThread : Can't load parameters, using defaults");
		iServiceCheckInterval = 5;
	}
	iServiceCheckInterval = atoi(stringDelay.c_str());

}

CWLANPlugin::~CWLANPlugin()
{
	DBG("CWLANPlugin::Destructor");
	RemoveListeners();
}

/**
 * @memo Tests TCP connection for incoming data.
 * @doc  Tests TCP connection using ioctl. This method checks how manyt
 * bytes there are available for reading. This checking is necessary because
 * MSG_WAITALL flag is used for reading.
 *
 * @param aFd Socket filedescriptor which is checked
 * @param aBytes How many bytes are expected to be available
 *
 * @return false if there is not enough bytes pending
 */
bool CWLANPlugin::TestConnection(unsigned int aFd, unsigned int aBytes)
{
	int nPeek = 0;
	int testBuffer = 0;
	nPeek = recv(aFd, &testBuffer, sizeof(testBuffer),  MSG_PEEK);

	unsigned long pending = 0;
	int errvalue = 0;
	if(ioctl(aFd, FIONREAD, &pending) == -1)
	{
		ERR("CWLANPlugin::TestConnection : ioctl failed ");
		errvalue = errno;
		ERR("CWLANPlugin::TestConnection : The error generated was %d", errvalue );
		return false;
	}

	if(pending < aBytes)
	{
		ERR("CWLANPlugin::TestConnection : there is only %d bytes pending while expecting %d", (int)pending, aBytes);
		return false;
	}

	return true;
}

/**
 * @memo Loads a parameter from the configuration file.
 * @doc Loads a parameter from the configuration file. If the requested
 * parameter is found then the argument <code>aParameter</code> is replaced
 * with parameter's value.
 *
 * @param aParameter Parameter's name.
 *
 * @return true if the requested parameter was found
 */
bool CWLANPlugin::LoadParameter(std::string& aParameter)
{
	char* home;
	std::string line;

	if ((home = getenv("HOME")) == NULL) {
		return false;
	}

	std::string path(home);
	path.append("/.phconfig");

	std::ifstream in(path.c_str());

	if (!in.is_open()) {
		ERR("Failed to open the configuration file $HOME/.phconfig");
		in.close();
		return false;
	}

	while (!in.eof()) {
		getline(in, line);
		if (line[0] == '#') continue;
		std::string::size_type position = line.find_first_of(':');

		if (position != std::string::npos) {
			std::string parameter = line.substr(0, position);
			if (parameter.compare(aParameter) == 0) {
				aParameter = line.substr(++position);
				in.close();
				return true;
			}
		}
	}

	in.close();
	return false;
}


/**
 * @memo Starts the plugin's inquiry routine.
 * @doc Starts the plugin's inquiry routine. After this method is called the
 * plugin starts to monitor its environment and tries to locate nearby
 * PeerHood-capable devices. Whenever a change in the neighborhood status is
 * detected the plugin will update the device & service database . This plugin
 * uses protocol defined by virtual peerhood for WLAN devices to locate
 * PeerHood-capable devices in the neighborhood.
 *
 * @return boolean telling if the plugin could be started
 */
bool CWLANPlugin::Start()
{
	if(iStarted)
	{
		ERR("CWLANPlugin::Start : inquiry thread already started");
		return false;
	}

	iStarted = true;

	if (pthread_create(&iInquiryThread, NULL, *CWLANPlugin::ThreadStarter, this) != 0) {
		ERR("CWLANPlugin::Start : failed to create a thread");
		iStarted = false;
		return false;
	}

	return true;
}


/**
 * @memo Stops the plugin's inquiry routine.
 * @doc Stops the plugin's inquiry routine. After this call is handled the
 * plugin will stop monitoring the environment. This means that daemon's
 * device & service database will be deprecated as the time passes.
 *
 * @return none
 */
void CWLANPlugin::Stop()
{
	if (!iStarted)
	{
		ERR("CWLANPlugin::Stop : inquiry thread not running");
		return;
	}

	iStarted = false;

	if (pthread_join(iInquiryThread, NULL) != 0) {
		ERR("CWLANPlugin::Stop : failed to join the inquiring thread");
	}

	UnListen();
}

/**
 * @memo Starts the plugin's inquiry listen routine.
 * @doc Starts the plugin's inquiry listen routine.
 *
 * @return boolean telling if the plugin could be started
 */
bool CWLANPlugin::Listen()
{
	if(iListen)
	{
		ERR("CWLANPlugin::Listen : inquiry listen thread already started");
		return false;
	}

	iListen = true;

	if (pthread_create(&iListenThread, NULL, *CWLANPlugin::ListenStarter, this) != 0) {
		ERR("CWLANPlugin::Listen : failed to create a thread");
		iListen = false;
		return false;
	}


	return true;
}


/**
 * @memo Stops the plugin's inquiry listen routine.
 * @doc Stops the plugin's inquiry routine.
 *
 * @return none
 */
void CWLANPlugin::UnListen()
{
	if (!iListen)
	{
		ERR("CWLANPlugin::Unlisten : inquiry listen thread not running");
		return;
	}

	iListen = false;

	if (pthread_join(iListenThread, NULL) != 0) {
		ERR("CWLANPlugin::Unlisten : failed to join the inquiring listen thread");
	}

}


/**
 * @memo Returns plugin's unique id string.
 * @doc Returns plugin's unique id string. The id of this plugin is 'wlan-base'.
 *
 * @return plugin's unique id string ('wlan-base')
 */
const std::string& CWLANPlugin::GetName()
{
	return iName;
}

/**
 * @memo Updates the state
 * @doc Checks messages for every listener that is registered into this plugin.
 * 
 */
void CWLANPlugin::UpdateState()
{
	// Check states of the listeners
	std::list<MAbstractListener*>::iterator iter;
	for(iter = iListenerList.begin(); iter != iListenerList.end(); ++iter)
	{
		(*iter)->CheckState();
	}
}

/**
 * @memo Register a new listener for this plugin
 * @doc Adds a listener into the internal list of listener. Used by listeners only.
 * 
 * @param MAbstractlistener* aListener Reference to the listener that is added
 * 
 */
void CWLANPlugin::RegisterListener(MAbstractListener* aListener)
{
	if(aListener != NULL)
	{
		iListenerList.push_back(aListener);
		DBG("CWLANPlugin::RegisterListener: new listener %s.", aListener->GetName().c_str());
	}
}

/**
 * @memo Change the state of the plugin
 * @doc Sets new state for plugin (or no changes if there is no need)
 * 
 * @param aActive new state (true = active, false = passive)
 */
void CWLANPlugin::SetState(bool aActive)
{
	if(iActive != aActive)
	{
		iActive = aActive;
		DBG("CWLANPlugin::SetState: new state: %s", aActive ? "active" : "passive");
	}
	else DBG("CWLANPlugin::SetState: No change in activity state.");
}

/**
 * @memo Set to shut down - not used
 * @doc Not used - does nothing
 * 
 */
void CWLANPlugin::TriggerShutdown()
{
	DBG("CWLANPlugin::TriggerShutdown");
}

/**
 * @memo Loads all listeners for this plugin
 * @doc Uses the ListenerFactory for listener creation. When listeners have been created (listeners
 * automatically register themselves to the object that asked to create them) they are commanded
 * to connect to their sources and to check the initial state.
 */
void CWLANPlugin::LoadListeners()
{
	int listeners = 0;
	/** @TODO Get type name from somewhere else, currently hardcoded! */
	std::string listenerName = "wlan";
	listeners = ListenerFactory::GetInstance()->CreateListeners(listenerName, (MAbstractStateConverter*)this);
	if(listeners == 0) DBG("CWLANPlugin: No listeners for type \"%s\" found.", listenerName.c_str());

	// Connect listeners and check the initial state of the system, go through all listeners
	std::list<MAbstractListener*>::iterator iter;
	for(iter = iListenerList.begin(); iter != iListenerList.end(); ++iter)
	{
		if((*iter)->Connect())
		{
			(*iter)->CheckInitialState();
		}
	}
	DBG("CWLANPlugin: Listeners started");
}
/**
 * @memo Starts plugin's main (inquiry) thread.
 * @doc Starts plugin's main (inquiry) thread. This dummy function is required
 * because C++ doesn't allow non-static member functions to be started as a
 * thread. This function just calls plugin instance's real inquiry function.
 *
 * @param aArguments Pointer to the currently running instance.
 *
 * @return always NULL
 */
void *CWLANPlugin::ThreadStarter(void *aArguments)
{
	CWLANPlugin *fake = (CWLANPlugin *)aArguments;
	fake->InquiryThread();
	return NULL;
}

/**
 * @memo Starts plugin's inquiry listen thread.
 * @doc Starts plugin's inquiry listen thread. This dummy function is required
 * because C++ doesn't allow non-static member functions to be started as a
 * thread. This function just calls plugin instance's real inquiry function.
 *
 * @param aArguments Pointer to the currently running instance.
 *
 * @return always NULL
 */
void *CWLANPlugin::ListenStarter(void *aArguments)
{
	CWLANPlugin *fake = (CWLANPlugin *)aArguments;
	fake->ListenThread();
	return NULL;
}

/**
 * @memo Starts plugin's advertising thread.
 * @doc Starts plugin's advertising thread. This dummy function is required
 * because C++ doesn't allow non-static member functions to be started as a
 * thread. This function just calls plugin instance's real advertising
 * function.
 *
 * @param aArguments Pointer to the currently running instance.
 *
 * @return always NULL
 */
void *CWLANPlugin::AdvertStarter(void *aArguments)
{
	CWLANPlugin *fake = (CWLANPlugin *)aArguments;
	fake->AdvertThread();
	return NULL;
}

/**
 * @memo Plugin's main thread that performs WLAN inquiry at certain intervals.
 * @doc Plugin's main thread that performs WLAN inquiry at certain intervals. All
 * found devices are stored to the DeviceStorage shared with all plugins and
 * the PeerHood Daemon. Inquiry is performed in two-phases: First inquiring device
 * broadcastst HELLO -message using specific UDP port. Other devices are listening
 * to this port and send a reply back. Inquiring device receives these replies
 * using ListeningThread and adds devices to database.
 */
void CWLANPlugin::InquiryThread()
{
	int inquiryDelay = 0, on = 1, targetPort = 0;
	int writeFailures = 0, writeSuccess = 0;
	std::string transferMode;
	std::string targetAddress;

	// If the plugin is set to passive, wait
	while(!iActive)
	{
		if(!iStarted) return; // Plugin stopped
		else sleep(1); // otherwise sleep for second
	}

	CUDPConnection *udpConnection = new CUDPConnection(iNDevice);

	if (setsockopt(udpConnection->GetFd(), SOL_SOCKET, SO_BROADCAST, &on, sizeof(on)) < 0) {
		ERR("CWLANPlugin::InquiryThread : can't set broadcasting");
		delete udpConnection;
		return;
	}

	if(setsockopt(udpConnection->GetFd(), SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on)) == -1)
	{
		ERR("CWLANPlugin::InquiryThread :setsockopt failed  ");
		delete udpConnection;
		return;
	}

	if(!Listen())
	{
		ERR("CWLANPlugin::InquiryThread : can't start listening thread");
		delete udpConnection;
		return;
	}

	std::string message = "HELLO";
	targetPort = WLAN_INQUIRY_PSM;
	transferMode = "bcast";
	targetAddress = "none";

	DBG("CWLANPlugin::InquiryThread: start");
	
	while(iStarted && iActive)
	{
		
		if(iAdvertFailed)
		{
			ERR("CWLANPlugin::InquiryThread : AdvertThread has failed. Ending...");
			iStarted = false;
			break;
		}

		if (udpConnection->Write(message.c_str(), message.length(), targetPort, targetAddress, transferMode) == -1) {
			ERR("CWLANPlugin::InquiryThread : Write failed");
			
			// On error, increase counter, if enough errors -> break from loop
			writeFailures++;
			if(writeFailures == WLAN_ERROR_TOLERANCE)
			{
				ERR("CWLANPlugin::InquiryThread : Too many write failures.");
				break;
			}
		}
		// On successfull write, increase the counter of successfull writes
		else
		{
			writeSuccess++;
			
			// If enough successfull writes, we can consider the network conditions to be ok
			if(writeSuccess == WLAN_ERROR_TOLERANCE)
			{
				writeSuccess = 0;
				writeFailures = 0;
			}
		}

		inquiryDelay = iInquiryInterval + 1 + (int)(INQUIRY_VARIATION * rand() / (RAND_MAX + 1.0));
		for (int i = 0;i < inquiryDelay;i++) {
			sleep(1);
			if (!iStarted) break;
		}
	}
	delete udpConnection;
	
	// Reset if plugin was set to passive but the plugin should be still running
	if(!iActive && iStarted)
	{
		DBG("CWLANPlugin::InquiryThread: restarting...");
		UnListen(); // Stop the listening thread (join), causes small overhead
		InquiryThread(); // start again
	}
}


/**
 * @memo Plugin's inquiry thread that listens WLAN inquiries.
 * @doc Plugin's inquiry thread that listen WLAN inquiries. This thread
 * listens for UDP broadcast and unicast traffic using specific ports. If
 * someone sends UDP broadcast thread sends unicast back. If someone sends
 * unicast, thread adds the device into it's databse.
 *
 * @return none
 */
void CWLANPlugin::ListenThread()
{
	char buffer[10];
	int on = 1, targetPort = 0, maxFd = 0;
	sockaddr_in temp;
	char *addr;
	std::string saddr;
	fd_set inquirySet;
	struct timeval timeVal;

	std::string transferMode;
	std::string targetAddress;
	
	// If the plugin is set to passive, wait
//	while(!iActive)
//	{
//		if(!iListen) return;
//		else sleep(1);
//	}
	
	CUDPConnection udpConnection(iNDevice);
	CUDPConnection udpReplyConnection(iNDevice);

	//needed for later memcmp, could be fixed?
	CUDPConnection search(iNDevice);
	iIFaces = search.GetInterfaces();

	if(setsockopt(udpConnection.GetFd(), SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on)) == -1)
	{
		ERR("CWLANPlugin::ListenThread :setsockopt failed  ");
		delete iIFaces;
		return;
	}

	if(setsockopt(udpReplyConnection.GetFd(), SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on)) == -1)
	{
		ERR("CWLANPlugin::ListenThread :setsockopt failed  ");
		delete iIFaces;
		return;
	}
	targetPort = WLAN_INQUIRY_PSM;
	transferMode = "bcast";

	if (!udpConnection.Listen(targetPort, transferMode)) {
		ERR("CWLANPlugin::ListenThread : listening broadcast failed");
		delete iIFaces;
		return;
	}

	targetPort = WLAN_INQUIRY_PSM + 1;
	transferMode = "ucast";
	if (!udpReplyConnection.Listen(targetPort, transferMode)) {
		ERR("CWLANPlugin::ListenThread : listening unicast failed");
		delete iIFaces;
		return;
	}

	DBG("CWLANPlugin::ListenThread: start");
	while (iListen && iActive) {
		
		if(udpConnection.GetFd() > udpReplyConnection.GetFd())
			maxFd = udpConnection.GetFd();
		else
			maxFd = udpReplyConnection.GetFd();

		FD_ZERO(&inquirySet);
		FD_SET(udpReplyConnection.GetFd(), &inquirySet);
		FD_SET(udpConnection.GetFd(), &inquirySet);

		timeVal.tv_sec = 1;
		timeVal.tv_usec = 0;

		if(select(maxFd + 1, &inquirySet, NULL, NULL, &timeVal) ==-1){
			ERR("CWLANPlugin::ListenThread() : select failed");
			delete iIFaces;
			return;
		}

		for(int i=0;i<=maxFd;i++)
		{
			if(FD_ISSET(i, &inquirySet))
			{
				if(i==udpConnection.GetFd())
				{
					memset(buffer, 0, sizeof(buffer));
					if(udpConnection.Read(buffer, 10) == -1){
						ERR("CWLANPlugin::ListenThread : reading failed");
						delete iIFaces;
						return;
					}

					if(std::string(buffer).compare(std::string("HELLO")) == 0)
					{
						saddr = udpConnection.GetRemoteAddress();

						addr = (char *)saddr.c_str();

						if ((inet_aton(addr, &temp.sin_addr)) < 0)
						{
							ERR("CWLANPlugin::ListenThread : inet_aton failed");
							delete iIFaces;
							return;
						}

						if((memcmp(&temp.sin_addr, &iIFaces->iUcast.sin_addr, sizeof(in_addr)) != 0)
								&& (saddr.compare("127.0.0.1") != 0)) //this hack is needed only for iPAQ
						{
							std::string message = "HELLOREP";
							targetPort = WLAN_INQUIRY_PSM+1;
							transferMode = "ucast";
							targetAddress = saddr;

							if (udpConnection.Write(message.c_str(), message.length(), targetPort, targetAddress, transferMode) == -1) {
								ERR("CWLANPlugin::InquiryThread : Write HELLOREP failed");
								delete iIFaces;
								return;
							}
						}
					}
				}

				if(i==udpReplyConnection.GetFd())
				{
					memset(buffer, 0, sizeof(buffer));
					if(udpReplyConnection.Read(buffer, 10) == -1){
						ERR("CWLANPlugin::ListenThread : reading failed");
						delete iIFaces;
						return;
					}

					if(std::string(buffer).compare(std::string("HELLOREP")) == 0)
					{
						saddr = udpReplyConnection.GetRemoteAddress();
						if (AddDevice(saddr))
						{
							DBG("CWLANPlugin::ListenThread : Added new device %s\n", saddr.c_str());
							CDeviceStorage::GetInstance()->Update(PLUGIN_NAME, iDeviceList);
						}
					}
				}
			}//if(FD_ISSET)
		}//for

		if (MakeOlder()) {
			DBG("should update the device database");
			DBG("the size of the list is %zu", iDeviceList.size());
			CDeviceStorage::GetInstance()->Update(PLUGIN_NAME, iDeviceList);
		}

	}//while (iListen)
	delete iIFaces;
	
	udpConnection.Close();
	udpReplyConnection.Close();
}

/**
 * @memo Checks if the found device is new (i.e. not detected before).
 * @doc Checks if the found device is new (i.e. not detected before). If this
 * is the case then it is added to plugin's internal device list and the common
 * device & service database should be updated later on.
 * There two kinds of counters for device: timestamp is used for to mark "old" devices
 * and lastchecked is for performing service checking on specific interval.
 * timestamp is zerod every time when device is found. If timestamp reaches specific value
 * device is erased. If device is seen, lastchecked value is decreased.  If lastchecked
 * reaches zero, device services are checked.
 *
 * @param aDeviceName The name of the device to be added.
 *
 * @return true if the checked device is new
 */
bool CWLANPlugin::AddDevice(const std::string& aDeviceName)
{
	CDaemonDevice* temp;
	std::list<CDaemonDevice *>::iterator i;

	for (i = iDeviceList.begin();i != iDeviceList.end();++i) {
		if ((*i)->GetAddress().compare(aDeviceName) == 0) {
			(*i)->SetTimestamp(0);
			for (std::list<CServiceCheck>::iterator j = iServiceCheckList.begin();j != iServiceCheckList.end();++j) {
				if(aDeviceName.compare(j->iDeviceName) == 0)
				{
					if(j->iLastChecked > iServiceCheckInterval)
						j->iLastChecked = iServiceCheckInterval;
					j->iLastChecked--;
					if(j->iLastChecked <= 0)
					{
						(*i)->SetTimestamp(KEEP_ALIVE_STAMP);
						iServiceCheckList.erase(j);
					}
					return false;
				}
			}
		}
	}

	temp = new CDaemonDevice(aDeviceName);
	temp->SetPrototype(PLUGIN_NAME);
	temp->SetPeerHood(true);
	temp->SetTimestamp(0);

	if (!FetchInformation(temp))
		//  if(!FetchServices(temp))
	{
		//      ERR("CWLANPlugin::AddDevice : FetchServices failed");
		ERR("CWLANPlugin::AddDevice : FetchInformation failed");
		delete temp;
		return false;
	}
	iDeviceList.push_back(temp);

	CServiceCheck tempCheck;
	tempCheck.iDeviceName = aDeviceName;
	tempCheck.iLastChecked = iServiceCheckInterval;
	iServiceCheckList.push_back(tempCheck);

	return true;
}

/**
 * @memo Increases each device's timestamp by one and checks is some of them
 * exceeds the limit.
 * @doc Increases each device's timestamp value by one and checks if some of
 * them exceeds the limit defined byt KEEP_ALIVE_STAMP. In that case the "old"
 * device is removed from
 * the internal device list so the common device & service database should be
 * updated.
4 *
 * @return true if some device was removed from the internal device list
 */
bool CWLANPlugin::MakeOlder()
{
	bool retval = false;

	for (std::list<CDaemonDevice *>::iterator i = iDeviceList.begin();i != iDeviceList.end();) {
		if ((*i)->GetTimestamp() > KEEP_ALIVE_STAMP) {
			std::list<CDaemonDevice *>::iterator sentenced = i++;
			delete *sentenced;
			iDeviceList.erase(sentenced);
			retval = true;
		}
		else {
			(*i)->IncreaseTimestamp();
			i++;
		}
	}
	return retval;
}


/**
 * @memo Starts advertazing the PeerHood service by providing a lookup service.
 * @doc Starts advertazing the PeerHood service by providing a lookup service.Other WLAN-
 * enabled PeerHood devices will look PeerHood service from this well-known lookup service
 * when they try to locate the surrounding PeerHood devices. If this function fails other devices are
 * not capable of detecting PeerHood on this device.
 *
 * @return true if the advertising was started successfully
 */
bool CWLANPlugin::Advert()
{
	if (iInSdp) {
		ERR("CWLANPlugin::Advert : already in SDP");
		return false;
	}
	iInSdp = true;

	if (pthread_create(&iAdvertThread, NULL, *CWLANPlugin::AdvertStarter, this) != 0) {
		ERR("CWLANPlugin::Advert : failed to create the advertising thread");
		Unadvert();
		return false;
	}
	return true;
}

/**
 * @memo Removes the PeerHood tag from the lookup service.
 * @doc Removes the PeerHood tag from the lookup service. After this
 * function is invoked, other devices are not able to detect the existence of
 * the PeerHood service on this machine.
 *
 * @return true if the advertising was ended succesfully
 */
bool CWLANPlugin::Unadvert()
{
	if (!iInSdp) {
		ERR("CWLANPlugin::Unadvert : not in sdp at all?!?");
		return false;
	}

	iInSdp = false;

	if (pthread_join(iAdvertThread, NULL) != 0) {
		ERR("CWLANPlugin::Unadvert : failed to join the advertising thread");
	}

	return true;
}

/**
 * @memo Checks if the PeerHood tag can be found from remote device's
 * lookup service
 * @doc Checks if the PeerHood tag can be found from remote device's
 * lookup service. If the tag if found then the remote device is assumed to be
 * PeerHood-capable without any further checking.
 *
 * @param aAddress The address of the remote device.
 * @param aPSM On return contains the PSM of the remote address if PeerHood is
 * found, otherwise -1.
 *
 * @return true if the PeerHood tag was found on remote device's lookup service
 */
bool CWLANPlugin::HasPeerHood(const std::string& aAddress, int* aPSM)
{
	DBG("  PeerHood found at psm %d",*aPSM);
	return true;
}


/**
 * @memo Fetch the required information from other device
 * @doc
 *
 * @return true if all information is successfully fetched
 */

bool CWLANPlugin::FetchInformation(CDaemonDevice* aDevice)
{
	CTCPConnection* connection = NULL;

	connection = new CTCPConnection(iNDevice, "WLAN");

	if (!connection->Connect(aDevice->GetAddress(), WLAN_PLUGIN_PSM)) {
		ERR("CWLANPlugin::FetchInformation : can't connect");
		delete connection;
		return false;
	}

	// Here we need to set  flags for needed information
	int requestType = CDaemon::GetInstance()->GetFetchRequestType();

	DBG("--**-- ");

	// Write the requesttype
	DBG("Requesting %d from %s", requestType, aDevice->GetAddress().c_str());

	requestType = htonl(requestType);

	if (connection->Write(&requestType, sizeof(requestType)) == -1) {
		ERR("CWLANPlugin::FetchDeviceInfo : failed to write request type");
		connection->Disconnect();
		delete connection;
		return false;
	}

	requestType = ntohl(requestType);

	// Check here what information will be gathered (according to the requestType)
	// Take care of disconnects and deletes in here

	// Get the basic info
	if (requestType & D_GET_DEVICEINFO) {
		DBG("Requesting: device info");
		if(!FetchDeviceInfo(aDevice, connection)) {
			ERR("CBTPlugin::FetchDeviceInfo: Info not received correctly");
			return false;
		}
	}

	// Get services
	if (requestType & D_GET_SERVICEINFO) {
		DBG("Requesting: services");
		if(!FetchServices(aDevice, connection)) {
			ERR("CBTPlugin::FetchServices: Info not received correctly");
			return false;
		}
	}

	// Get supported prototypes
	if (requestType & D_GET_PROTOINFO) {
		DBG("Requesting: prototypes");
		if(!FetchPrototypes(aDevice, connection)){
			ERR("CBTPlugin::FetchPrototypes: Info not received correctly");
			return false;
		}
	}

	// Get neigborhoodinfo
	if (requestType & D_GET_NEIGHINFO) {
		DBG("Requesting: neighborhood");
		if (!FetchNeighbourhoodDevices(aDevice, connection)) {
			ERR("CBTPlugin::FetchNeighborhoodDevices: Info not received correctly");
			return false;
		}
	}

	DBG("--**-- ");

	connection->Disconnect();
	delete connection;
	return true;
}

/**
 * @memo Fetch services from the remote device
 * @doc Fetch services from remote device and add them to the list of this daemon device
 *
 * @return true if all information is successfully fetched
 */

bool CWLANPlugin::FetchDeviceInfo(CDaemonDevice* aDevice, CTCPConnection* connection)
{
	fd_set testSet;
	struct timeval timeVal;

	unsigned int nameLength = 0;
	unsigned int checksum = 0;
	char* buffer = NULL;


	FD_ZERO(&testSet);
	FD_SET(connection->GetFd(), &testSet);

	timeVal.tv_sec = 1;
	timeVal.tv_usec = 0;

	// Wait for the information
	if(select(connection->GetFd() + 1, &testSet, NULL, NULL, &timeVal) ==-1){
		ERR("CWLANPlugin::FetchDeviceInfo : select failed");
		connection->Disconnect();
		return false;
	}

	if (FD_ISSET(connection->GetFd(), &testSet)) {
		if (connection->Read(&nameLength, sizeof(nameLength)) == -1) {
			DBG("CWLANPlugin::FetchDeviceInfo : failed to read the device name length");
			connection->Disconnect();
			return false;
		}

		nameLength = ntohl(nameLength);

		if (FD_ISSET(connection->GetFd(), &testSet)) {
			buffer = new char[nameLength];

			if (connection->Read(buffer, nameLength) == -1) {
				ERR("CWLANPlugin::FetchDeviceInfo : failed to read the device name");
				connection->Disconnect();
				delete[] buffer;
				return false;
			}

			//      DBG("Got name %s", buffer);
			std::string deviceName(buffer);
			delete[] buffer;
			aDevice->SetDeviceName(deviceName);

			if (connection->Read(&checksum, sizeof(checksum)) == -1) {
				ERR("CWLANPlugin::FetchDeviceInfo : failed to read the device name checksum");
				connection->Disconnect();
				return false;
			}

			aDevice->SetChecksum(ntohl(checksum));

			DBG("CBTPlugin::FetchDeviceInfo: OK");
			DBG(" ");

			return true;
		}
		else {
			ERR("CBTPlugin::FetchDeviceInfo : FD_ISSET failed");
		}
	}
	else {
		ERR("CBTPlugin::FetchDeviceInfo : FD_ISSET failed");
	}

	return false;
}

/**
 * @memo Fecthes a list of services available on the given remote device.
 * @doc Fetches a list of services available on the given remote device. This
 * is done by establishing a new connection to the remote device using address
 * information given in the parameter <code>aDevice</code> and invoking the
 * service list request on the remote device. Founded services are added to
 * the device.
 *
 * @param aDevice The device whose service list should be fetched.
 *
 * @return none
 */
bool CWLANPlugin::FetchServices(CDaemonDevice* aDevice)
{
	DBG("FetchServices without connection .. we should not have this");
	return false;
}


/**
 * @memo Fetch services from the remote device
 * @doc Fetch services from remote device and add them to the list of this daemon device
 *
 * @return true if all information is successfully fetched
 */

bool CWLANPlugin::FetchServices(CDaemonDevice* aDevice, CTCPConnection* connection)
{
	fd_set testSet;
	struct timeval timeVal;

	char* buffer = NULL;

	unsigned short numberOfServices = 0;
	CService* temp = NULL;
	unsigned int size = 0;

	FD_ZERO(&testSet);
	FD_SET(connection->GetFd(), &testSet);

	timeVal.tv_sec = 1;
	timeVal.tv_usec = 0;

	bool foundService = false;

	// Wait for the information
	if(select(connection->GetFd() + 1, &testSet, NULL, NULL, &timeVal) ==-1){
		DBG("CBTPlugin::FetchServices : select failed");
		return false;
	}

	if (FD_ISSET(connection->GetFd(), &testSet)) {
		numberOfServices = 0;

		if (connection->Read(&numberOfServices, sizeof(numberOfServices)) == -1) {
			DBG("CBTPlugin::FetchServices : failed to read the number of services");
			return false;
		}

		numberOfServices = ntohs(numberOfServices);

		DBG("CWLANPlugin::FetchServices: Got %d services", numberOfServices);
		
		// TODO find out why in some cases on n810/n900 25701 or 25717 services is received
		// and create a better fix. This is just a quick fix attempting to prevent crash.
		if(numberOfServices > 1000)
		{
			DBG("CWLANPlugin::FetchServices : Amount of services is too big, must be a bug.");
			return false;
		}

		for (int i = 0;i < numberOfServices;i++) {
			foundService = false;

			if (FD_ISSET(connection->GetFd(), &testSet)) {
				if (connection->Read(&size, sizeof(size)) == -1) {
					ERR("CBTPlugin::FetchServices : failed to read the size of the service");
					return false;
				}

				size = ntohl(size);

				if (FD_ISSET(connection->GetFd(), &testSet)) {
					buffer = new char[size];

					if (connection->Read(buffer, size) == -1) {
						ERR("CBTPlugin::FetchServices : failed to read service info");
						delete[] buffer;
						return false;
					}

					temp = new CService(buffer);

					// Check is the service is already registered
					foundService = aDevice->CheckService(temp->GetPort());

					if (!foundService) {
						DBG("FetchServices: Found new service %s for %s", temp->GetName().c_str(), aDevice->GetAddress().c_str());
						aDevice->AddService(temp);
					}
					else {
						DBG("FetchServices: Found old service %s for %s", temp->GetName().c_str(), aDevice->GetAddress().c_str());
						delete temp;
					}

					delete[] buffer;

				}
				else {
					ERR("CBTPlugin::FetchServices : FD_ISSET failed");
					return false;
				}
			}
		}
		DBG("CBTPlugin::FetchServices: OK");
		DBG("Number of services in device is %d", aDevice->GetServiceListSize());
		DBG(" ");

		return true;
	}
	else {
		ERR("CBTPlugin::FetchServices : FD_ISSET failed");
	}

	return false;
}

/**
 * @memo Fetch prototypes from the remote device
 * @doc Fetch prototypes from remote device and add them to the list of this daemon device
 *
 * @return true if all information is successfully fetched
 */

bool CWLANPlugin::FetchPrototypes(CDaemonDevice* aDevice, CTCPConnection* connection)
{
	fd_set testSet;
	struct timeval timeVal;

	unsigned short numberOfPlugins = 0;
	unsigned short size = 0;
	char* newProto = NULL;

	FD_ZERO(&testSet);
	FD_SET(connection->GetFd(), &testSet);

	timeVal.tv_sec = 1;
	timeVal.tv_usec = 0;

	bool foundProto = false;

	// Wait for the information
	if(select(connection->GetFd() + 1, &testSet, NULL, NULL, &timeVal) ==-1){
		DBG("CBTPlugin::FetchPrototypes : select failed");
		return false;
	}

	if (FD_ISSET(connection->GetFd(), &testSet)) {
		numberOfPlugins = 0;

		if (connection->Read(&numberOfPlugins, sizeof(numberOfPlugins)) == -1) {
			DBG("CBTPlugin::FetchPrototypes : failed to read the number of plugins");
			return false;
		}

		numberOfPlugins = ntohs(numberOfPlugins);

		DBG("FetchPrototypes: Number of protos %d", numberOfPlugins);

		for (int i = 0;i < numberOfPlugins;i++) {
			foundProto = false;

			if (FD_ISSET(connection->GetFd(), &testSet)) {
				if (connection->Read(&size, sizeof(size)) == -1) {
					ERR("CBTPlugin::FetchPlugins : failed to read the size of the plugins");
					return false;
				}

				size = ntohs(size);

				if (FD_ISSET(connection->GetFd(), &testSet)) {

					newProto = new char[size];

					if (connection->Read(newProto, size) == -1) {
						ERR("CBTPlugin::FetchPrototypes : failed to read plugin info");
						delete[] newProto;
						return false;
					}

					// Check is the service is already registered
					foundProto = aDevice->CheckPrototype(newProto);

					if (!foundProto) {
						DBG("FetchPrototypes: Found new proto %s for %s", newProto, aDevice->GetAddress().c_str());
						aDevice->AddPrototype(newProto);
					}
					else {
						DBG("FetchPrototypes: Found old proto %s for %s", newProto, aDevice->GetAddress().c_str());
						delete [] newProto;
					}
				}
				else {
					ERR("CWLANPlugin::FetchPrototypes : FD_ISSET failed");
					return false;
				}
			}
			else {
				ERR("CWLANPlugin::FetchPrototypes : FD_ISSET failed");
				return false;
			}
		}

		DBG("CWLANPlugin::FetchPrototypes: OK");
		DBG("Number of fetched prototypes %d", aDevice->GetProtoListSize());
		DBG(" ");

		return true;
	}
	else {
		ERR("CWLANPlugin::FetchPrototypes : FD_ISSET failed");
	}

	return false;
}


bool CWLANPlugin::FetchNeighbourhoodDevices(CDaemonDevice* aDevice, CTCPConnection* connection)
{
	fd_set testSet;
	struct timeval timeVal;

	unsigned short numberOfNeighborhoodDevices = 0;
	CDaemonDevice* temp = NULL;

	char* newNeighbor = NULL;
	unsigned int size = 0;

	FD_ZERO(&testSet);
	FD_SET(connection->GetFd(), &testSet);

	timeVal.tv_sec = 1;
	timeVal.tv_usec = 0;

	bool foundNeighbor = false;

	if(select(connection->GetFd() + 1, &testSet, NULL, NULL, &timeVal) ==-1){
		ERR("CWLANPlugin::FetchNeighbourhoodDevices : select failed");
		return false;
	}

	// Start reading data (number of devices and then their info)
	if (FD_ISSET(connection->GetFd(), &testSet)) {
		if ((connection->Read(&numberOfNeighborhoodDevices, sizeof(numberOfNeighborhoodDevices))) == -1) {
			ERR("CWLANPlugin::FetchNeighbourhoodDevices : failed to read the number of devices");
			return false;
		}

		numberOfNeighborhoodDevices = ntohs(numberOfNeighborhoodDevices);

		DBG("FetchNeighbors: Number of neigh devices %d", numberOfNeighborhoodDevices);

		for (int i = 0;i < numberOfNeighborhoodDevices;i++) {
			foundNeighbor = false;

			if (FD_ISSET(connection->GetFd(), &testSet)) {
				if ((connection->Read(&size, sizeof(size))) == -1) {
					ERR("CWLANPlugin::FetchNeighbourhoodDevices : failed to read the size of the device");
					return false;
				}

				size = ntohl(size);

				if (FD_ISSET(connection->GetFd(), &testSet)) {
					newNeighbor = new char[size];

					if (connection->Read(newNeighbor, size) == -1) {
						ERR("CWLANPlugin::FetchNeighbourhoodDevices : failed to read device info");
						delete[] newNeighbor;
						return false;
					}

					temp = new CDaemonDevice(newNeighbor);
					temp->SetReferrerName(aDevice->GetName());

					foundNeighbor = aDevice->CheckNeighbors(temp->GetAddress());

					if(!foundNeighbor) {
						DBG("FetchNeighbors: Found new neighbor %s for %s", temp->GetAddress().c_str(), aDevice->GetAddress().c_str());
						aDevice->AddNeighboringDevice(temp);
					}
					else {
						DBG("FetchNeighbors: Found old neighbor %s for %s", temp->GetAddress().c_str(), aDevice->GetAddress().c_str());
						delete temp;
					}

					delete[] newNeighbor;
				}
				else {
					ERR("CWLANPlugin::FetchNeighbourhoodDevices : FD_ISSET failed");
					return false;
				}
			}
			else {
				ERR("CWLANPlugin::FetchNeighbourhoodDevices : FD_ISSET failed");
				return false;
			}
		}

		DBG("Number of fetched neighbors %d", aDevice->GetNeighborListSize());
		DBG("CWLANPlugin::FetchNeighborhoodDevices: OK");
		DBG(" ");

		return true;
	}
	else {
		ERR("CWLANPlugin::FetchNeighbourhoodDevices : FD_ISSET failed");
	}

	return false;
}


/**
 * @memo Fecthes a list of services available on the given remote device.
 * @doc Fetches a list of services available on the given remote device. This
 * is done by establishing a new connection to the remote device using address
 * information given in the parameter <code>aDevice</code> and invoking the
 * service list request on the remote device. Founded services are added to
 * the device.
 *
 * @param aDevice The device whose service list should be fetched.
 *
 * @return none
 */
// bool CWLANPlugin::FetchServices(CDaemonDevice* aDevice)
// {
//   unsigned short numberOfServices;
//   CTCPConnection* connection;
//   CService* temp;
//   char* buffer;
//   unsigned int size;
//   fd_set testSet;
//   struct timeval timeVal;
//   unsigned int checksum;

//   DBG("  fetching services from %s : %d", aDevice->GetAddress().c_str(), WLAN_PLUGIN_PSM);

//   connection = new CTCPConnection(iNDevice, "WLAN");

//   if (!connection->Connect(aDevice->GetAddress(), WLAN_PLUGIN_PSM)) {
//     ERR("CWLANPlugin::FetchServices : can't connect");
//     delete connection;
//     return false;
//   }

//   FD_ZERO(&testSet);
//   FD_SET(connection->GetFd(), &testSet);

//   timeVal.tv_sec = 10;
//   timeVal.tv_usec = 0;

//   if(select(connection->GetFd() + 1, &testSet, NULL, NULL, &timeVal) ==-1){
//     ERR("CWLANPlugin::FetchServices : select failed");
//     connection->Disconnect();
//     delete connection;
//     return false;
//   }

//   if (FD_ISSET(connection->GetFd(), &testSet))
//     {
//       unsigned int length = 0;
//       if(!TestConnection(connection->GetFd(), sizeof(length)))
// 	{
// 	  ERR("WLANPlugin::FetchServices : TestConnection failed");
// 	  connection->Disconnect();
// 	  delete connection;
// 	  return false;
// 	}

//       //device name length
//       if (connection->Read(&length, sizeof(length)) == -1) {
// 	ERR("WLANPlugin::FetchServices : failed to read the device name length");
// 	connection->Disconnect();
// 	delete connection;
// 	return false;
//       }

//       length = ntohl(length);

//       if (FD_ISSET(connection->GetFd(), &testSet))
// 	{
// 	  if(!TestConnection(connection->GetFd(), length))
// 	    {
// 	      ERR("WLANPlugin::FetchServices : TestConnection failed");
// 	      connection->Disconnect();
// 	      delete connection;
// 	      return false;
// 	    }

// 	  //receive device name
// 	  buffer = new char[length];
// 	  memset(buffer, 0, length);
// 	  if (connection->Read(buffer, length) == -1) {
// 	    ERR("WLANPlugin::FetchServices : failed to read the device name");
// 	    connection->Disconnect();
// 	    delete connection;
// 	    delete[] buffer;
// 	    return false;
// 	  }

// 	  std::string deviceName(buffer);
// 	  delete[] buffer;
// 	  aDevice->SetDeviceName(deviceName);

// 	  if (FD_ISSET(connection->GetFd(), &testSet))
// 	    {
// 	      if(!TestConnection(connection->GetFd(), sizeof(checksum)))
// 		{
// 		  ERR("WLANPlugin::FetchServices : TestConnection failed");
// 		  connection->Disconnect();
// 		  delete connection;
// 		  return false;
// 		}

// 	      //receive checksum
// 	      if (connection->Read(&checksum, sizeof(checksum)) == -1) {
// 		ERR("WLANPlugin::FetchServices : failed to read the checksum");
// 		connection->Disconnect();
// 		delete connection;
// 		return false;
// 	      }

// 	      checksum = ntohl(checksum);
// 	      aDevice->SetChecksum(checksum);

// 	      //receive number of servics
// 	      numberOfServices = 0;
// 	      if (FD_ISSET(connection->GetFd(), &testSet))
// 		{
// 		  if(!TestConnection(connection->GetFd(), sizeof(numberOfServices)))
// 		    {
// 		      ERR("WLANPlugin::FetchServices : TestConnection failed");
// 		      connection->Disconnect();
// 		      delete connection;
// 		      return false;
// 		    }

// 		  if (connection->Read(&numberOfServices, sizeof(numberOfServices)) == -1) {
// 		    ERR(" CWLANPlugin::FetchServices : failed to read the number of services");
// 		    connection->Disconnect();
// 		    delete connection;
// 		    return false;
// 		  }

// 		  numberOfServices = ntohs(numberOfServices);

// 		  for (int i = 0;i < numberOfServices;i++) {
// 		    if (FD_ISSET(connection->GetFd(), &testSet))
// 		      {
// 			if(!TestConnection(connection->GetFd(), sizeof(size)))
// 			  {
// 			    ERR("WLANPlugin::FetchServices : TestConnection failed");
// 			    connection->Disconnect();
// 			    delete connection;
// 			    return false;
// 			  }

// 			if (connection->Read(&size, sizeof(size)) == -1) {
// 			  ERR("CWLANPlugin::FetchServices : failed to read the size of the service");
// 			  connection->Disconnect();
// 			  delete connection;
// 			  return false;
// 			}

// 			size = ntohl(size);

// 			if (FD_ISSET(connection->GetFd(), &testSet))
// 			  {
// 			    if(!TestConnection(connection->GetFd(), size))
// 			      {
// 				ERR("WLANPlugin::FetchServices : TestConnection failed");
// 				connection->Disconnect();
// 				delete connection;
// 				return false;
// 			      }

// 			    buffer = new char[size];
// 			    memset(buffer, 0, length);
// 			    if (connection->Read(buffer, size) == -1) {
// 			      ERR("failed to read service info");
// 			      delete[] buffer;
// 			      return false;
// 			    }

// 			    temp = new CService(buffer);
// 			    aDevice->AddService(temp);
// 			    delete[] buffer;

// 			    DBG("  the service is : %s", temp->GetName().c_str());
// 			    DBG("  attributes are : %s", temp->GetAttributeString().c_str());
// 			    DBG("  port is : %d", temp->GetPort());
// 			  }
// 			else
// 			  {
// 			    ERR("CWLANPlugin::FetchServices : FD_ISSET failed");
// 			  }
// 		      }
// 		    else
// 		      {
// 			ERR("CWLANPlugin::FetchServices : FD_ISSET failed");
// 		      }
// 		  }//for (int i = 0;i < numberOfServices;i++)

// 		  if(!connection->Disconnect())
// 		    ERR("CWLANPlugin::FetchServices : disconection failed");

// 		  delete connection;
// 		  return true;
// 		}
// 	      else
// 		{
// 		  ERR("CWLANPlugin::FetchServices : FD_ISSET failed");
// 		}
// 	    }
// 	  else
// 	    {
// 	      ERR("CWLANPlugin::FetchServices : FD_ISSET failed");
// 	    }
// 	}
//       else
// 	{
// 	  ERR("CWLANPlugin::FetchServices : FD_ISSET failed");
// 	}
//     }
//   else
//     {
//       ERR("CWLANPlugin::FetchServices : FD_ISSET failed");
//     }

//   if(!connection->Disconnect())
//     ERR("CWLANPlugin::FetchServices : disconection failed");

//   delete connection;
//   return false;
//}



/**
 * @memo Listens a PSM for incoming SERVICE LIST
 * @doc Listens a PSM for incoming SERVICE LIST
 * If a new request is received then a new onnection object is created and passed to the
 * PeerHood Daemon that sends the list of available services.
 *
 * @return none
 */
void CWLANPlugin::AdvertThread()
{
	fd_set fdSet;
	struct timeval timeVal;
	int on = 1;
	int serviceType;
	
	while(!iActive)
	{
		if(!iInSdp) return;
		else sleep(1);
	}
	
	CTCPConnection* connection = new CTCPConnection(iNDevice, "WLAN");

	if (setsockopt(connection->GetFd(), SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on)) < 0) {
		ERR("CWLANPlugin::AdvertThread : can't set reuseaddr");
		delete connection;
		iAdvertFailed = true;
		return;
	}

	if (!connection->Listen(WLAN_PLUGIN_PSM)) {
		ERR("CWLANPlugin::AdvertThread : listening failed");
		delete connection;
		iAdvertFailed = true;
		return;
	}

	DBG("CWLANPlugin::AdvertThread: start");
	while (iInSdp && iActive) {
		
		timeVal.tv_sec = 1;
		timeVal.tv_usec = 0;

		FD_ZERO(&fdSet);
		FD_SET(connection->GetFd(), &fdSet);

		switch (select(connection->GetFd() + 1, &fdSet, NULL, NULL, &timeVal)) {
		// error
		case -1:
		ERR("CWLANPlugin::AdvertThread : select failed");
		iAdvertFailed = true;
		iInSdp = false;
		break;

		// timeout
		case 0:
			break;

			// data
		default:
			DBG("CWLANPlugin::AdvertThread : got SERVICE LIST request");
			CTCPConnection *newConnection = (CTCPConnection *)connection->AcceptL();
			if(newConnection == NULL)
			{
				ERR("CWLANPlugin::AdvertThread : accept failed");
				break;
			}

			if (newConnection->Read(&serviceType, sizeof(serviceType)) == -1) {
				ERR("CWLANPlugin::AdvertThread : failed to read request type");
				newConnection->Disconnect();
				delete newConnection;
				break;
			}

			serviceType = ntohl(serviceType);
			DBG("Got service request %d from %s", serviceType, newConnection->GetRemoteAddress().c_str());

			if (serviceType & D_GET_DEVICEINFO) {
				DBG("Sending device info");
				CDaemon::GetInstance()->SendDeviceInfo(newConnection);
			}


			if (serviceType & D_GET_SERVICEINFO) {
				DBG("Sending service list");
				CDaemon::GetInstance()->SendServiceList(newConnection);
			}

			if (serviceType & D_GET_PROTOINFO) {
				DBG("Sending prototypes");
				CDaemon::GetInstance()->SendPrototypes(newConnection);
			}

			if (serviceType & D_GET_NEIGHINFO) {
				DBG("Sending neighbourhood list");
				CDaemon::GetInstance()->SendNeighborList(newConnection);
			}

			DBG("All information sent");
			DBG("--**--");

			newConnection->Disconnect();
			delete newConnection;
			break;
		}

	}
	/** @TODO Check if both are really needed .. */
	if(!connection->Disconnect()) ERR("CWLANPlugin::AdvertThread: cannot disconnect listening socket.");
	connection->Close();
	delete connection;
	
	/* If set to passive */
	if(!iActive && iInSdp)
	{
		DBG("CWLANPlugin::AdvertThread: restarting...");
		AdvertThread();
	}
}

/**
 * @memo Clear list of listers
 * @doc Removes all listeners from the listener list. Called when this object is destroyed.
 * Listeners are loaded only once!
 */
void CWLANPlugin::RemoveListeners()
{
	std::list<MAbstractListener*>::iterator iter;
	// Go through all listeners and disconnect, delete object and erase from list
	for(iter = iListenerList.begin(); iter != iListenerList.end();)
	{
		(*iter)->Disconnect();
		delete *iter;
		iter = iListenerList.erase(iter);
	}
}

void CWLANPlugin::SetAdapter(const std::string& aInterface, int aId)
{
	// TODO implement adapter change
	DBG("CWLANPlugin::SetAdapter: called with parameters: interface = %s, id = %d",aInterface.c_str(), aId);
}

const std::string& CWLANPlugin::GetAdapter()
{
	return iNDevice;
}

static CWLANPlugin wlanplugin;
