/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of obdlib.
 *
 *  obdlib is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  obdlib is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with obdlib.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <sys/time.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "libobd.h"


int is_serial_device(char* devname)
{
	char* pstr;

	pstr = strstr(devname, "/dev/tty");
	if (pstr == devname)
		return 1;

	pstr = strstr(devname, "/dev/rfcomm");
	if (pstr == devname)
		return 1;

	return 0;
}

int calc_elapsed_time(struct timeval* stv, struct timeval* etv)
{
	if(etv->tv_usec > stv->tv_usec)
		return (etv->tv_usec - stv->tv_usec)/1000;
	else
		return (1000000 + etv->tv_usec - stv->tv_usec)/1000;
}

int send_pid(void* obd)
{
	struct obd_sensor_data sensor_data;
	struct timeval stv, etv;
	int j;

	gettimeofday(&stv, NULL);

	for(j = 0; j <0x20; j++) {
		sensor_data.pid = j;
		gettimeofday(&stv, NULL);
		if(obd_sendpid(obd, &sensor_data, 5)< 0) {
			printf("Error gettting pid %02X\n", j);
			return -1;
		} else {
			gettimeofday(&etv, NULL);

			printf("PID: %02X Valid: %d Value1: %.2f Value2: %.2f Time: %03dms\n", 
					sensor_data.pid, sensor_data.valid, 
					sensor_data.result[0], sensor_data.result[1], calc_elapsed_time(&stv, &etv));
		}
	}
	return 0;
}

int send_dtc(void* obd)
{
	struct obd_dtc_data dtc_data;
	struct timeval stv, etv;
	int i;

	gettimeofday(&stv, NULL);

	if(obd_senddtc(obd, &dtc_data, 5)< 0)
		printf("Error gettting dtc\n");
	else {
		if(dtc_data.valid < 0) {
			printf("Got a not valid DTC reply\n");
			return -1;
		} else {
			gettimeofday(&etv, NULL);

			printf("num DTCs: %d Time: %03dms ", dtc_data.valid, calc_elapsed_time(&stv, &etv));
			for(i=0; i < dtc_data.valid; i++)
				printf("\"%s\" ", dtc_data.code_list[i]);
			printf("\n");
		}
	}
	return 0;
}

int setup_device(void** obd, char* devname)
{
        if((*obd = obd_alloc()) == NULL) {
		printf("Sorry, cannot alocate obd data\n");
		goto error;
	}

	if(obd_open_dev(*obd, devname) < 0) {
		printf("problem with odb_open_dev\n");
		goto error;
	}

        if(obd_channel_setup(*obd) < 0) {
		printf("problem with obd_channel_setup\n");
		goto error;
	}

	return 0;

error:
	if(*obd) {
		obd_close(*obd);
		obd_dealloc(*obd);
		*obd = NULL;
	}
	return -1;
}

int setup_simulator(void** obd)
{
        if((*obd = obd_alloc()) == NULL) {
                printf("Sorry, cannot alocate obd data\n");
                goto error;
        }

        if ((obd_open_simulator(*obd)) < 0)
        {
                printf("problem with odb_open_simulator");
                goto error;
        }
	return 0;

error:
	if(*obd) {
		obd_close(*obd);
		obd_dealloc(*obd);
		*obd = NULL;
	}
	return -1;
}

int setup_file(void** obd, char* filename)
{
        if((*obd = obd_alloc()) == NULL) {
		printf("Sorry, cannot alocate obd data\n");
		goto error;
	}

	if(obd_open_file(*obd, filename) < 0) {
		printf("problem with odb_open_dev\n");
		goto error;
	}

	return 0;

error:
	if(*obd) {
		obd_close(*obd);
		obd_dealloc(*obd);
		*obd = NULL;
	}
	return -1;
}

void close_obd(void* obd)
{
	obd_close(obd);
	obd_dealloc(obd);
}

int main (int argc, char **argv)
{
	void *obd = NULL;

	obd_log_setup(stdout, OBD_DEBUG);

	if(argc == 1) {
		if(setup_simulator(&obd) < 0)
			return -1;
	} else {
		if(is_serial_device(argv[1])) {
			if(setup_device(&obd, argv[1]) < 0)
				return -1;
		} else {
			if(setup_file(&obd, argv[1]) < 0)
				return -1;
		}

	}

	if(send_pid(obd) < 0)
		return -1;

	if(send_dtc(obd) < 0)
		return -1;

	close_obd(obd);

	return 0;
}
