/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of libobd.
 *
 *  libobd is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  libobd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with libobd.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>

#include "obd-sensor.h"
#include "obd-log.h"

int obdsensor_process_obd(unsigned char* data, unsigned int size, struct obd_sensor_data* sensor)
{
	unsigned char mode, pid;

	if(size < 3) {
		LOG_ERROR("Data is too short - %d", size);
		sensor->valid |= OBD_INVALID_DATA;
		return 0;
	}

	mode = data[0];
	pid = data[1];

	if(mode != 0x41) {
		LOG_INFO("The reply for PID %02X is not valid - %X", sensor->pid, mode);
		sensor->valid |= OBD_INVALID_DATA;
		return 0;
		
	}

	if(pid != sensor->pid) {
		LOG_WARNING("PID from the reply is not equal to the requested one - %02X/%02X",
				pid, sensor->pid);
		sensor->valid |= OBD_INVALID_DATA;
		return 0;
	}


	data += 2;
	size -= 2;

	switch(pid) {
		case 0x00:
		case 0x20:
		case 0x40:
			if(size < 4) {
				sensor->valid |= OBD_INVALID_DATA;
				LOG_WARNING("Got the wrong number of bytes for PID %X", pid);
				return 0;
			}
			sensor->result[0] = data[0] << 24 | data[1] << 16 |
						data[2] << 8 | data[1];
			sensor->valid |= OBD_HAS_FIRST_VALUE;
			return 0;
		case 0x01:
			sensor->result[0] = data[0];
			sensor->valid |= OBD_HAS_FIRST_VALUE;
			if(size > 1) {
				sensor->result[1] = data[1] << 16 | data[2] << 8 | data[3];
				sensor->valid |= OBD_HAS_SECOND_VALUE;
			}
			return 0;
		case 0x03:
			if(size < 2) {
				sensor->valid |= OBD_INVALID_DATA;
				LOG_WARNING("Got the wrong number of bytes for PID %X", pid);
				return 0;
			}
			sensor->valid |= OBD_HAS_SECOND_VALUE | OBD_HAS_FIRST_VALUE;
			sensor->result[0] = data[0];
			sensor->result[1] = data[1];
			return 0;

		case 0x04:
		case 0x11:
		case 0x2C:
		case 0x2E:
		case 0x2F:
		case 0x45:
		case 0x47:
		case 0x48:
		case 0x49:
		case 0x4A:
		case 0x4B:
		case 0x4C:
			sensor->valid |= OBD_HAS_FIRST_VALUE;
			sensor->result[0] = (data[0] * 100) / 255.0;
			return 0;

		case 0x05:
		case 0x0F:
		case 0x46:
			sensor->valid |= OBD_HAS_FIRST_VALUE;
			sensor->result[0] = data[0] - 40;
			sensor->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			sensor->rimperial[0] = ((sensor->result[0] * 9) / 5.0) + 32;
			return 0;

		case 0x06:
		case 0x07:
		case 0x08:
		case 0x09:
			sensor->valid |= OBD_HAS_FIRST_VALUE;
			sensor->result[0] = 0.7812 * (data[0] - 128);
			return 0;

		case 0x0A:
			sensor->valid |= OBD_HAS_FIRST_VALUE;
			sensor->result[0] = data[0] * 3;
			sensor->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			sensor->rimperial[0] = (sensor->result[0] * 29) / 100000.0;
			return 0;

		case 0x0B:
		case 0x12:
		case 0x13:
		case 0x1C:
		case 0x1D:
		case 0x1E:
		case 0x30:
		case 0x33:
			sensor->valid |= OBD_HAS_FIRST_VALUE;
			sensor->result[0] = data[0];
			return 0;

		case 0x0C:
			if(size < 2) {
				sensor->valid |= OBD_INVALID_DATA;
				LOG_WARNING("Got the wrong number of bytes for PID %X", pid);
				return 0;
			}
			sensor->valid |= OBD_HAS_FIRST_VALUE;
			sensor->result[0] = ((256 * data[0]) + data[1])/4;
			return 0;

		case 0x0D:
			sensor->valid |= OBD_HAS_FIRST_VALUE;
			sensor->result[0] = data[0];
			sensor->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			sensor->rimperial[0] = (data[0] * 62) / 100.0;
			return 0;

		case 0x0E:
			sensor->valid |= OBD_HAS_FIRST_VALUE;
			sensor->result[0] = (data[0]/2.0) - 64;
			return 0;

		case 0x10:
			if(size < 2) {
				sensor->valid |= OBD_INVALID_DATA;
				LOG_WARNING("Got the wrong number of bytes for PID %X", pid);
				return 0;
			}
			sensor->valid |= OBD_HAS_FIRST_VALUE;
			sensor->result[0] = ((256 * data[0]) + data[1]) / 100.0;
			sensor->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			sensor->rimperial[0] = (sensor->result[0] * 132) / 1000.0;
			return 0;

		case 0x14:
		case 0x15:
		case 0x16:
		case 0x17:
		case 0x18:
		case 0x19:
		case 0x1A:
		case 0x1B:
			if(size < 2) {
				sensor->valid |= OBD_INVALID_DATA;
				LOG_WARNING("Got the wrong number of bytes for PID %X", pid);
				return 0;
			}
			sensor->valid |= OBD_HAS_SECOND_VALUE | OBD_HAS_FIRST_VALUE;
			sensor->result[0] = data[0] * 0.005;
			sensor->result[1] = (data[1]-128) * 0.7812;
			return 0;

		case 0x1F:
		case 0x4D:
		case 0x4E:
			if(size < 2) {
				sensor->valid |= OBD_INVALID_DATA;
				LOG_WARNING("Got the wrong number of bytes for PID %X", pid);
				return 0;
			}
			sensor->valid |= OBD_HAS_FIRST_VALUE;
			sensor->result[0] = 256 * data[0] + data[1];
			return 0;

		case 0x21:
		case 0x31:
			if(size < 2) {
				sensor->valid |= OBD_INVALID_DATA;
				LOG_WARNING("Got the wrong number of bytes for PID %X", pid);
				return 0;
			}
			sensor->valid |= OBD_HAS_FIRST_VALUE;
			sensor->result[0] = 256 * data[0] + data[1];
			sensor->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			sensor->rimperial[0] = sensor->result[0] * 0.6213;
			return 0;

		case 0x22:
			if(size < 2) {
				sensor->valid |= OBD_INVALID_DATA;
				LOG_WARNING("Got the wrong number of bytes for PID %X", pid);
				return 0;
			}
			sensor->valid |= OBD_HAS_FIRST_VALUE;
			sensor->result[0] = (256 * data[0] + data[1]) * 0.079;
			sensor->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			sensor->rimperial[0] = (sensor->result[0] * 29) / 100000.0;
			return 0;

		case 0x23:
			if(size < 2) {
				sensor->valid |= OBD_INVALID_DATA;
				LOG_WARNING("Got the wrong number of bytes for PID %X", pid);
				return 0;
			}
			sensor->valid |= OBD_HAS_FIRST_VALUE;
			sensor->result[0] = (256 * data[0] + data[1]) * 10.0;
			sensor->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			sensor->rimperial[0] = (sensor->result[0] * 29) / 100000.0;
			return 0;

		case 0x24:
		case 0x25:
		case 0x26:
		case 0x27:
		case 0x28:
		case 0x29:
		case 0x2A:
		case 0x2B:
			if(size < 4) {
				sensor->valid |= OBD_INVALID_DATA;
				LOG_WARNING("Got the wrong number of bytes for PID %X", pid);
				return 0;
			}
			sensor->result[0] = (256 * data[0] + data[1]) * 0.0000305;
			sensor->result[1] = (256 * data[2] + data[3]) * 0.000122;
			sensor->valid |= OBD_HAS_SECOND_VALUE | OBD_HAS_FIRST_VALUE;
			return 0;

		case 0x2D:
			sensor->valid |= OBD_HAS_FIRST_VALUE;
			sensor->result[0] = data[0] * 0.78125 - 100;
			return 0;

		case 0x32:
			if(size < 2) {
				sensor->valid |= OBD_INVALID_DATA;
				LOG_WARNING("Got the wrong number of bytes for PID %X", pid);
				return 0;
			}
			sensor->valid |= OBD_HAS_FIRST_VALUE;
			sensor->result[0] = ((256 * data[0] + data[1])/4) - 8192;
			sensor->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			sensor->rimperial[0] = (sensor->result[0] * 29) / 100000.0;
			return 0;

		case 0x34:
		case 0x35:
		case 0x36:
		case 0x37:
		case 0x38:
		case 0x39:
		case 0x3A:
		case 0x3B:
			if(size < 4) {
				sensor->valid |= OBD_INVALID_DATA;
				LOG_WARNING("Got the wrong number of bytes for PID %X", pid);
				return 0;
			}
			sensor->result[0] = (256 * data[0] + data[1]) * 0.0000305;
			sensor->result[1] = ((256 * data[2] + data[3]) * 0.00391) - 128;
			sensor->valid |= OBD_HAS_SECOND_VALUE | OBD_HAS_FIRST_VALUE;
			return 0;

		case 0x3C:
		case 0x3D:
		case 0x3E:
		case 0x3F:
			if(size < 2) {
				sensor->valid |= OBD_INVALID_DATA;
				LOG_WARNING("Got the wrong number of bytes for PID %X", pid);
				return 0;
			}
			sensor->valid |= OBD_HAS_FIRST_VALUE;
			sensor->result[0] = ((256 * data[0] + data[1])/10) - 40;
			sensor->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			sensor->rimperial[0] = ((sensor->result[0] * 9) / 5.0) + 32;
			return 0;

		default:
			sensor->valid |= OBD_INVALID_DATA;
			return 0;
	}

}

int obdsensor_process_dtc(unsigned char* data, unsigned int size, struct obd_dtc_data* dtc)
{
	unsigned char mode;
	unsigned char chr1, chr2, chr3, chr4, chr5;
	int i;

	if(size < 3) {
		LOG_WARNING("DTC Data is too short - %d", size);
		dtc->valid = -1;
		return 0;
	}

	mode = data[0];

	if(mode != 0x43 && mode != 0x47) {
		LOG_WARNING("The dtc reply is not valid - %X", mode);
		dtc->valid = -1;
		return 0;
	}

	data++;
	size--;

	if(size%2 == 1) {
		LOG_WARNING("The dtc reply is not valid as we do not have an even number of bytes - %d", size);
		size--;
	}

	dtc->valid = 0;

	for(i = 0; i < size && dtc->valid < MAX_DTCS; i+=2) {
		if(data[i] == 0 && data[i+1] == 0) /* Ignore P0000 */
			continue;

		chr1 = data[i] >> 6;
		chr2 = (data[i] & 0x30) >> 4;
		chr3 = data[i] & 0x0F;
		chr4 = (data[i+1] & 0xF0) >> 4;
		chr5 = data[i+1] & 0x0F;
		if(chr3 > 9 || chr4 > 9 || chr5 >9) {
			LOG_WARNING("The DTC data is invalid - %02X %02X %02X", chr3, chr4, chr5);
			continue;
		}

		switch(chr1) {
			case 0:
				dtc->code_list[dtc->valid][0] = 'P';
				break;
			case 1:
				dtc->code_list[dtc->valid][0] = 'C';
				break;
			case 2:
				dtc->code_list[dtc->valid][0] = 'B';
				break;
			case 3:
				dtc->code_list[dtc->valid][0] = 'U';
				break;
		}

		snprintf(dtc->code_list[dtc->valid] + 1, 5, "%d%d%d%d", chr2, chr3, chr4, chr5);
		dtc->valid++;
	}
	return 0;
}
